package com.elitesland.yst.production.inv.application.service.impl;


import com.elitesland.yst.production.inv.application.facade.vo.ioc.InvIocParamVO;
import com.elitesland.yst.production.inv.application.facade.vo.ioc.InvIocRespVO;
import com.elitesland.yst.production.inv.application.facade.vo.ioc.InvIocSaveVO;
import com.elitesland.yst.production.inv.application.out.SystemService;
import com.elitesland.yst.production.inv.application.service.InvIocService;
import com.elitesland.yst.production.inv.domain.convert.InvIocConvert;
import com.elitesland.yst.production.inv.domain.entity.ioc.InvIoc;
import com.elitesland.yst.production.inv.domain.service.InvIocDomainService;
import com.elitesland.yst.production.inv.infr.dto.InvIocDTO;
import com.elitesland.yst.production.inv.utils.InvUdcEnum;
import com.elitescloud.cloudt.common.annotation.SysCodeProc;
import com.elitescloud.cloudt.common.base.ApiCode;
import com.elitescloud.cloudt.common.base.ApiResult;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitescloud.boot.exception.BusinessException;
import lombok.AllArgsConstructor;
import lombok.AllArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import lombok.val;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;
import org.springframework.util.StringUtils;

import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.stream.Collectors;

/**
 * <p>
 * 功能说明:库存操作码
 * </p>
 *
 * @Author Darren
 * @Date 2022/04/19
 * @Version 1.0
 * @Content:
 */
@Slf4j
@Service("InvIocService")
@AllArgsConstructor
public class InvIocServiceImpl implements InvIocService {

    private final InvIocDomainService InvIocDomainService;

    private final SystemService systemService;


    /**
     * 新增库存操作码数据
     *
     * @param saveVo 新增库存操作码数据
     * @return 新增库存操作码ID
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public ApiResult<Long> createOne(InvIocSaveVO saveVo) {

        InvIoc InvIoc = InvIocConvert.INSTANCE.saveVoToEn(saveVo);
        //新增时默认启用
        if(org.apache.commons.lang3.StringUtils.isEmpty(InvIoc.getStatus())){
            InvIoc.setStatus(InvUdcEnum.INV_ACTIVE_STATUS_ACTIVE.getValueCode());
        }
        //检查必填项
        checkIocMandatory(InvIoc);
        //校验编码和名称是否已存在
        checkIocCodeAndName(InvIoc);
        //保存
        Long id = InvIocDomainService.saveInvIoc(InvIoc, Optional.empty());

        return ApiResult.ok(id);
    }

    /**
     * 修改库存操作码数据
     *
     * @param saveVo 修改库存操作码对象
     * @return 库存操作码ID
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public ApiResult<Long> update(InvIocSaveVO saveVo) {

        val iocDTOOptional = InvIocDomainService.findById(saveVo.getId());
        if (iocDTOOptional.isEmpty()) {
            return ApiResult.fail("未查询到库存操作码信息!");
        }

        InvIoc InvIoc = InvIocConvert.INSTANCE.saveVoToEn(saveVo);
        //检查必填项
        checkIocMandatory(InvIoc);
        //校验编码和名称是否已存在
        checkIocCodeAndName(InvIoc);
        //保存
        Long id = InvIocDomainService.saveInvIoc(InvIoc, iocDTOOptional);

        return ApiResult.ok(id);
    }


    /**
     * 检查必填项
     *
     * @param InvIoc 库存操作码对象
     */
    private void checkIocMandatory(InvIoc InvIoc) {
        if (StringUtils.isEmpty(InvIoc.getIoCode())) {
            throw new BusinessException(ApiCode.FAIL, "操作码不能为空!");
        }
        if (StringUtils.isEmpty(InvIoc.getIoName())) {
            throw new BusinessException(ApiCode.FAIL, "操作码名称不能为空!");
        }
        if (StringUtils.isEmpty(InvIoc.getIoType())) {
            throw new BusinessException(ApiCode.FAIL, "操作类型编码不能为空!");
        }
        if (StringUtils.isEmpty(InvIoc.getStatus())) {
            throw new BusinessException(ApiCode.FAIL, "是否启用不能为空!");
        }


    }

    /**
     * 校验编码和名称是否已存在
     *
     * @param invIoc 库存操作码对象
     */
    private void checkIocCodeAndName(InvIoc invIoc) {
        InvIocParamVO param = new InvIocParamVO();
        param.setIoCode(invIoc.getIoCode());
        List<InvIocDTO> invIocDTOList = InvIocDomainService.selectByParam(param);

        val invIocDTOOptional = invIocDTOList.stream().filter(dto ->
                Objects.equals(invIoc.getIoCode(), dto.getIoCode()) && !Objects.equals(invIoc.getId(), dto.getId())
        ).findFirst();
        invIocDTOOptional.ifPresent(iocDTO -> {
            throw new BusinessException(ApiCode.FAIL, "库存操作码已存在-(" + iocDTO.getIoName() + ")!");
        });

        param.setIoCode(null);
        param.setIoName(invIoc.getIoName());
        List<InvIocDTO> invIocNameList = InvIocDomainService.selectByParam(param);
        val invIocNameOptional = invIocNameList.stream().filter(dto ->
                Objects.equals(invIoc.getIoName(), dto.getIoName()) && !Objects.equals(invIoc.getId(), dto.getId())
        ).findFirst();
        invIocNameOptional.ifPresent(iocDTO -> {
            throw new BusinessException(ApiCode.FAIL, "操作码名称已存在-(" + iocDTO.getIoCode() + ")!");
        });
    }


    /**
     * 通过库存操作码ID查询单条数据
     *
     * @param id 库存操作码ID
     * @return 单条库存操作码数据
     */
    @Override
    @SysCodeProc
    public ApiResult<InvIocRespVO> findIdOne(Long id) {
        val invIocDTOOptional = InvIocDomainService.findById(id);
        if (invIocDTOOptional.isEmpty()) {
            return ApiResult.fail("未查询到库存操作码信息!");
        }
        InvIocDTO InvIocDTO = invIocDTOOptional.get();

        //转换RespVO
        InvIocRespVO InvIocRespVO = InvIocConvert.INSTANCE.dtoToRespVo(InvIocDTO);

        return ApiResult.ok(InvIocRespVO);
    }

    /**
     * 通过库存操作码查询单条数据
     *
     * @param code 库存操作码编码
     * @return 单条库存操作码数据
     */
    @Override
    @SysCodeProc
    public Optional<InvIocRespVO> findByCode(String code) {
        val invIocDTOOptional = InvIocDomainService.findByCode(code);
        if (invIocDTOOptional.isEmpty()) {
            return Optional.empty();
        }
        InvIocDTO InvIocDTO = invIocDTOOptional.get();

        //转换RespVO
        InvIocRespVO InvIocRespVO = InvIocConvert.INSTANCE.dtoToRespVo(InvIocDTO);

        return Optional.of(InvIocRespVO);
    }

    /**
     * 通过库存操作码集合查询数据
     *
     * @param codeList 库存操作码编码集合
     * @return 库存操作码数据集合
     */
    @Override
    @SysCodeProc
    public List<InvIocRespVO> findByCodeBatch(List<String> codeList) {
        List<InvIocDTO> InvIocDTOList = InvIocDomainService.findByCodeBatch(codeList);
        if (CollectionUtils.isEmpty(InvIocDTOList)) {
            return Collections.EMPTY_LIST;
        }

        //转换RespVO
        List<InvIocRespVO> invIocRespVOList = InvIocDTOList.stream().map(
                InvIocConvert.INSTANCE::dtoToRespVo).collect(Collectors.toList());

        return invIocRespVOList;
    }

    /**
     * 分页查询库存操作码
     *
     * @param param 分类查询、分页和排序对象
     * @return 多条库存操作码数据
     */
    @Override
    @SysCodeProc
    public PagingVO<InvIocRespVO> search(InvIocParamVO param) {

        val pagingVo = InvIocDomainService.search(param);
        if (CollectionUtils.isEmpty(pagingVo.getRecords())) {
            return PagingVO.<InvIocRespVO>builder().total(0L).records(Collections.EMPTY_LIST).build();
        }
        val vos = pagingVo.getRecords();


        //转换RespVO
        List<InvIocRespVO> whAreaSettingRespVOList = vos.stream().map(
                InvIocConvert.INSTANCE::dtoToRespVo).collect(Collectors.toList());


        return PagingVO.<InvIocRespVO>builder()
                .total(pagingVo.getTotal())
                .records(whAreaSettingRespVOList)
                .build();
    }

    /**
     * 根据入参查询库存操作码
     *
     * @param param 分类查询、分页和排序对象
     * @return 多条库存操作码数据
     */
    @Override
    @SysCodeProc
    public List<InvIocRespVO> select(InvIocParamVO param) {
        List<InvIocDTO> invIocDTOList = InvIocDomainService.selectByParam(param);

        if (CollectionUtils.isEmpty(invIocDTOList)) {
            return Collections.EMPTY_LIST;
        }

        //转换RespVO
        List<InvIocRespVO> invIocRespVOList = invIocDTOList.stream().map(
                InvIocConvert.INSTANCE::dtoToRespVo).collect(Collectors.toList());

        return invIocRespVOList;
    }


    /**
     * 根据库存操作码ID批量逻辑删除
     *
     * @param ids 库存操作码ID集合
     * @return 库存操作码ID集合
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public ApiResult<String> deleteBatch(List<Long> ids) {

        InvIocDomainService.updateDeleteFlagByIds(1, ids);
        return ApiResult.ok();
    }

}
