package com.elitesland.yst.production.inv.application.service.impl;


import com.elitesland.yst.production.inv.utils.BeanCopyUtil;
import com.elitesland.yst.production.inv.application.facade.vo.lot.InvLotCommon21FilterParamVO;
import com.elitesland.yst.production.inv.application.facade.vo.lot.InvLotCommon21InVO;
import com.elitesland.yst.production.inv.application.facade.vo.lot.InvLotCommonSaveVO;
import com.elitesland.yst.production.inv.application.facade.vo.lot.InvLotParam;
import com.elitesland.yst.production.inv.application.facade.vo.lot.InvLotRespVO;
import com.elitesland.yst.production.inv.application.facade.vo.lot.InvLotSaveVO;
import com.elitesland.yst.production.inv.application.out.ItmOutService;
import com.elitesland.yst.production.inv.application.out.SystemService;
import com.elitesland.yst.production.inv.application.service.InvLotCommonService;
import com.elitesland.yst.production.inv.application.service.InvLotService;
import com.elitesland.yst.production.inv.domain.convert.InvLotConvert;
import com.elitesland.yst.production.inv.domain.entity.lot.InvLot;
import com.elitesland.yst.production.inv.domain.service.InvLotDomainService;
import com.elitesland.yst.production.inv.utils.UdcEnum;
import com.elitescloud.cloudt.common.base.ApiResult;
import com.elitescloud.boot.exception.BusinessException;
import com.elitesland.yst.production.support.provider.item.dto.ItmItemRpcDTO;
import com.elitesland.yst.production.support.provider.item.param.ItmItemRpcDtoParam;
import lombok.AllArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import lombok.val;
import org.springframework.beans.BeanUtils;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;
import org.springframework.util.StringUtils;

import java.io.IOException;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.temporal.ChronoUnit;
import java.util.*;
import java.util.stream.Collectors;

/**
 * <p>
 * 批次（共通方法）
 * </p >
 *
 * @author ssy
 * @date 2020/7/29
 */
@Slf4j
@Service
@AllArgsConstructor
public class InvLotCommonServiceImpl implements InvLotCommonService {


    private final InvLotService invLotService;
    private final InvLotDomainService invLotDomainService;
    private final SystemService systemService;
    private final ItmOutService itmOutService;

    @Override
    public Map<Integer, List<InvLotRespVO>> getInvLotVOList(List<InvLotCommon21InVO> inVOList) {
        log.info("共通方法21，时间：{}，入参：{}", LocalDateTime.now(), String.join(",", inVOList.stream().map(InvLotCommon21InVO::toString).collect(Collectors.toList())));

        Map<Integer, List<InvLotRespVO>> result = Collections.synchronizedMap(new HashMap<>());
        if (CollectionUtils.isEmpty(inVOList)) {
            List<InvLotRespVO> invLotList = invLotService.findByParams(new InvLotParam());

            result.put(0, invLotList);
        } else {
            inVOList.stream().forEach(i -> {
                InvLotParam queryParam = new InvLotParam();
                queryParam.setItemId(i.getItemId());
                queryParam.setVariId(i.getVariId());
                queryParam.setLotNo(i.getLotNo());
                queryParam.setOuId(i.getOuId());
                List<InvLotRespVO> invLotList = invLotService.findByParams(queryParam);
                if (!CollectionUtils.isEmpty(invLotList)) {
                    result.put(inVOList.indexOf(i), invLotList);
                }
            });
        }
        return result;
    }

    @Override
    public Map<String, List<InvLotRespVO>> getInvLotVOListByKey(List<String> concatKey) {
        log.info("getInvLotVOListByKey，时间：{}，入参：{}", LocalDateTime.now(), concatKey);
        if (CollectionUtils.isEmpty(concatKey)) {
            return new HashMap<>();
        }
        List<InvLotRespVO> list = invLotService.findByQueryParam(concatKey);
        return list.stream().collect(Collectors.groupingBy(lot -> {
            return lot.getItemId() == null ? "" : lot.getItemId() + "" + (lot.getVariId() == null ? "" : lot.getVariId()) + "" + lot.getLotNo()+"" + lot.getOuId();
        }));
    }

    @Override
    public List<InvLotRespVO> getInvLotVOListByParam(InvLotParam queryParam) {
        List<InvLotRespVO> invLotRespVOS = invLotService.findByParams(queryParam);
        return invLotRespVOS;
    }


    @Override
    public List<InvLotRespVO> getInvLotCheckedList(List<InvLotRespVO> source, InvLotCommon21FilterParamVO filterVO) {

        return source.stream()
                .filter(i -> {
                    if (!StringUtils.isEmpty(filterVO.getLotStatus())) {
                        return filterVO.getLotStatus().equals(i.getLotStatus());
                    }
                    return true;
                })
                .filter(i -> {
                    if (!StringUtils.isEmpty(filterVO.getQcStatus())) {
                        return filterVO.getQcStatus().equals(i.getQcStatus());
                    }
                    return true;
                })
                .filter(i -> {
                    // 批次号 [最早批次号]决定必须要晚于[最早批次号]的批次号
                    if (!StringUtils.isEmpty(filterVO.getFirstLotNo())) {
                        if (filterVO.getFirstLotNo().compareTo(i.getLotNo()) <= 0) {
                            return true;
                        }
                        return false;
                    }
                    return true;

                }).filter(i -> {
                    if (StringUtils.isEmpty(filterVO.getFressType())) {
                        return true;
                    } else {
                        return filterVO.getFressType().equals(i.getFressType());
                    }
                }).filter(i -> {
                    // 新鲜度要求（最低允收期天数）
                    if (filterVO.getFressTypeDays() != null) {
                        // 调整日期
                        if (filterVO.getAdjustDays() == null) {
                            return (i.getUntilExpireDays() == null ? 0D : i.getUntilExpireDays()) >= filterVO.getFressTypeDays();
                        } else {
                            return (i.getUntilExpireDays() == null ? 0D : i.getUntilExpireDays()) >= (filterVO.getFressTypeDays() + filterVO.getAdjustDays());
                        }
                    }
                    return true;
                }).collect(Collectors.toList());
    }

    @Override
    @Transactional
    public ApiResult createInvLotInfo(List<InvLotSaveVO> source) {
        log.info("共通方法22，时间：{}，入参：{}", LocalDateTime.now(), String.join(",", source.stream().map(InvLotSaveVO::toString).collect(Collectors.toList())));
        // 查询批次主信息
        List<InvLotCommonSaveVO> lotVOS = source.stream().map(i -> {
            InvLotCommonSaveVO vo = new InvLotCommonSaveVO();
            BeanUtils.copyProperties(i, vo);
            return vo;
        }).collect(Collectors.toList());
        Map<Integer, List<InvLotRespVO>> map = checkIn(lotVOS);
        try {
            List<InvLotCommonSaveVO> sourceClone = BeanCopyUtil.deepCopy(lotVOS);
            map.keySet().stream().forEach(key -> {
                // 去除已经存在批次数据
                List<InvLotRespVO> list = map.get(key);

                list.stream().forEach(v -> {
                    String a = v.getItemId() + "_" + v.getVariId() + "_" + v.getLotNo()+ "_" + v.getOuId();
                    Iterator ite = sourceClone.iterator();
                    while (ite.hasNext()) {
                        InvLotCommonSaveVO s = (InvLotCommonSaveVO) ite.next();
                        String b = s.getItemId() + "_" + s.getVariId() + "_" + s.getLotNo()+ "_" + v.getOuId();
                        if (a.equals(b)) {
                            ite.remove();
                        }
                    }
                });

            });
            sourceClone.forEach(i -> i.setFirstInDate(LocalDateTime.now()));
            List<InvLot> invLotList = new ArrayList<>();
            for (InvLotCommonSaveVO invLotCommonSaveVO : sourceClone) {
                invLotCommonSaveVO.setCreateTime(LocalDateTime.now());
                InvLot invLot = InvLotConvert.INSTANCE.comSaveVoToEn(invLotCommonSaveVO);
                invLotList.add(invLot);
            }
            List<InvLot> lotList = invLotList.stream().filter(i -> i.getLotNo() != null && ! "".equals(i.getLotNo())).collect(Collectors.toList());
            if(!CollectionUtils.isEmpty(lotList)) {
                ArrayList<InvLot> lotSaveVOS = invLotList.stream().collect(Collectors.collectingAndThen(Collectors.toCollection(
                        () -> new TreeSet<>(Comparator.comparing(o -> o.getItemId() + "_" + o.getLotNo()+ "_" + o.getOuId() + "_" + o.getVariId()))),
                        ArrayList::new));
                invLotDomainService.createBatch(lotSaveVOS);
            }
            return ApiResult.builder().success(true).code(0).data(source).msg("操作成功").build();
        } catch (IOException e) {
            e.printStackTrace();
            return ApiResult.builder().success(false).data(source)
                    .msg("系统异常：" + e.getMessage())
                    .build();
        } catch (ClassNotFoundException e) {
            e.printStackTrace();
            return ApiResult.builder().success(false).data(source)
                    .msg("系统异常：" + e.getMessage())
                    .build();
        }
    }

    @Override
    @Transactional
    public ApiResult updateInvLotInfo(List<InvLotCommonSaveVO> source) {
        log.info("共通方法23，时间：{}，入参：{}", LocalDateTime.now(), String.join(",", source.stream().map(InvLotCommonSaveVO::toString).collect(Collectors.toList())));
        // 查询批次主信息
        Map<Integer, List<InvLotRespVO>> map = checkIn(source);
        // 校验数据
        List<String> errMsgList = Collections.synchronizedList(new ArrayList<>());
        map.keySet().stream().forEach(key -> {
            List<InvLotRespVO> list = map.get(key);
            if (CollectionUtils.isEmpty(list)) {
                InvLotCommonSaveVO invLotCommonSaveVO = source.get(key);
                errMsgList.add(String.format("[商品：%s、VARI_ID:%s、批次号:%s、公司:%s]", invLotCommonSaveVO.getItemId(), invLotCommonSaveVO.getVariId(), invLotCommonSaveVO.getLotNo(),invLotCommonSaveVO.getOuId()));
            }
        });
        if (!CollectionUtils.isEmpty(errMsgList)) {
            throw new BusinessException("维护批次" + errMsgList.stream().collect(Collectors.joining(",")) + "不存在，请确认。");
        }


        // 批量修改
        List<InvLot> invLotList = new ArrayList<>();
        for (InvLotCommonSaveVO invLotCommonSaveVO : source) {
            // InvLot invLot = new InvLot();
            //BeanUtils.copyProperties(invLotCommonSaveVO,invLot);
            InvLot invLot = InvLotConvert.INSTANCE.comSaveVoToEn(invLotCommonSaveVO);
            invLotList.add(invLot);
        }
        //invLotRepo.saveAll(invLotDOS);
        invLotDomainService.createBatch(invLotList);
        return ApiResult.builder().success(true).code(0).data(source).msg("操作成功").build();

    }

    /**
     * 判断是否超过保质期天数阈值，设置 剩余效期天数 和新鲜度
     * 且返回相应警告信息
     *
     * @return
     */
    private List<String> checkStoreExpireDaysAndSetLockReason(List<InvLot> source) {

        // 获取全部的商品主数据
        List<Long> itemLists = source.stream().map(InvLot::getItemId).distinct().collect(Collectors.toList());
        ItmItemRpcDtoParam partParam = new ItmItemRpcDtoParam();
        partParam.setItemIds(itemLists);
        val itemVOList = itmOutService.findItemRpcDtoByParam(partParam);
        // TODO 外部数据 - 获取商品主数据的保质期天数阈值配置
//       InvSettingRespVO invSettingRespVO = systemService.findBySettingName("保质期天数设定阈值");
        long storeExpireDaysSetting = 0L;
//        if (invSettingRespVO != null) {
//            storeExpireDaysSetting = Math.abs(StringUtils.isEmpty(invSettingRespVO.getSettingVal()) ? 0L : Long.parseLong(invSettingRespVO.getSettingVal()));
//        }

        String warnMsg = "[商品:%s、VARI_ID:%s、批次号:%s、公司:%s]";
        List<String> warnMsgList = new ArrayList<>();
        long finalStoreExpireDaysSetting = storeExpireDaysSetting;
        source.forEach(i -> {
            if (!CollectionUtils.isEmpty(itemVOList)) {
                ItmItemRpcDTO itmItemVO = itemVOList.stream().filter(item -> item.getId().equals(i.getItemId())).findAny().get();
                if (itmItemVO != null) {
                    i.setItemCode(itmItemVO.getItemCode());
                }
            }
            if ("LOCK".equals(i.getQcStatus())) {
                // 当入参qcStatus为“LOCK”时，设置lotStatus为“1”
                i.setLotStatus("1"); // 锁定
            } else {
                if (i.getManuDate() != null && i.getExpireDate() != null) {
                    long between = ChronoUnit.DAYS.between(i.getManuDate().toLocalDate(), i.getExpireDate().toLocalDate());
                    i.setLotStatus("0");// 批次状态，默认为 0
                    // 当保质期天数之差的绝对值 与阈值的绝对值 比较，大于时，锁定
                    if (Math.abs(between - (i.getExpireDays() == null ? 0 : i.getExpireDays().longValue())) > finalStoreExpireDaysSetting) {
                        i.setLotStatus("1"); // 锁定
                        i.setLockReason(UdcEnum.INV_LOT_LOCK_REASON_EXP.getValueCode());
                        warnMsgList.add(String.format(warnMsg, i.getItemId(), i.getVariId(), i.getLotNo(), i.getOuId()));
                        log.info("共通方法22(出现锁定状态数据)，时间：{}，入参：{}", LocalDateTime.now(), i.toString());
                    }
                }
            }
            if (i.getExpireDate() != null) {
                // 计算 剩余有效期天数 和 新鲜度
                Long expireDays = i.getExpireDate().toLocalDate().toEpochDay() - LocalDate.now().toEpochDay();// 有效天数
                i.setUntilExpireDays(expireDays < 0L ? 0 : expireDays.intValue());
                Long days = LocalDate.now().toEpochDay() - i.getManuDate().toLocalDate().toEpochDay();// 从生产到现在的天数
                // 如果商品保质期天数查询不到默认设置9999
                double fress = days.doubleValue() / (null == i.getExpireDays() ? 9999D : i.getExpireDays().doubleValue());
                if (fress < 1D / 3D) {
                    i.setFressType(UdcEnum.COM_FRESS_TYPE_1.getValueCode());
                } else if (fress >= 1D / 3D && fress < 1D / 2D) {
                    i.setFressType(UdcEnum.COM_FRESS_TYPE_2.getValueCode());
                } else if (fress >= 1D / 2D && fress < 2D / 3D) {
                    i.setFressType(UdcEnum.COM_FRESS_TYPE_3.getValueCode());
                } else if (fress >= 2D / 3D && fress < 1D) {
                    i.setFressType(UdcEnum.COM_FRESS_TYPE_4.getValueCode());
                } else {
                    i.setFressType(UdcEnum.COM_FRESS_TYPE_5.getValueCode());
                }
                i.setDeleteFlag(0);
            }
        });
        return warnMsgList;
    }

    /**
     * 查询批次主信息
     *
     * @param lotVOList
     * @return
     */
    private Map<Integer, List<InvLotRespVO>> checkIn(List<InvLotCommonSaveVO> lotVOList) {
        List<InvLotCommon21InVO> invLotCommon21InVOS = Collections.synchronizedList(new ArrayList<>());
        lotVOList.stream().forEach(i -> {
            //校验库存批次生产日期和失效日期
            if (i.getManuDate() != null && i.getExpireDate() != null) {
                if (i.getManuDate().isAfter(i.getExpireDate())) {
                    throw new BusinessException("失效日期设置不能小于生产日期，请检查！");
                }
            }
            InvLotCommon21InVO inVO = new InvLotCommon21InVO();
            inVO.setItemId(i.getItemId());
            inVO.setItemCode(i.getItemCode());
            inVO.setVariId(i.getVariId());
            inVO.setLotNo(i.getLotNo());
            inVO.setOuId(i.getOuId());
            invLotCommon21InVOS.add(inVO);
        });
        return getInvLotVOList(invLotCommon21InVOS);
    }


}
