package com.elitesland.yst.production.inv.utils;


import org.apache.commons.collections4.CollectionUtils;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

/**
 * @author Chris
 * @date 2021-03-11
 */
public class TreeUtils {

    /**
     * 工具类，构造方法设置为 private，防止初始化
     */
    private TreeUtils() {
    }

    /**
     * 将传入的树列表转换为树形结构，并且排序
     * @param id 最开始构建的顶端ID
     * @param sort 是否需要排序
     * @param treeNodes 树类列表
     * @param <T> 实现 TreeNode 的类
     * @return 最终树形结构
     */
    public static <T extends TreeNode<T>> List<T> buildTree(Long id, boolean sort, List<T> treeNodes) {
        Map<Long, T> nodeIdMap = treeNodes.stream().collect(Collectors.toMap(TreeNode::getId, t -> t));
        Map<Long, List<T>> nodePidMap = treeNodes.stream()
                .collect(Collectors.groupingBy(t -> t.getPid() == null ? 0L : t.getPid()));
        return buildTree(id, sort, nodeIdMap, nodePidMap);
    }

    /**
     * 将传入的 Map 转换为树形结构，并且不排序
     * map 一般由外部使用 redis 存储，如果没有存储，可调用 buildTree(List<T> treeNodes, boolean sort) 方法，但是转换会影响效率
     * @param id 最开始构建的顶端ID
     * @param sort 是否需要排序
     * @param nodeIdMap Map, key: ID, value: 节点
     * @param nodePidMap Map, key: 父ID, value: 相同父ID的树列表
     * @param <T> 实现 TreeNode 的类
     * @return 最终树形结构
     */
    public static <T extends TreeNode<T>> List<T> buildTree(Long id, boolean sort, Map<Long, T> nodeIdMap, Map<Long, List<T>> nodePidMap) {

        id = id == null ? 0L : id;
        T topTreeNode = null;
        List<T> pTree = null;
        // 先获取指定的顶级节点
        if (nodeIdMap.containsKey(id)) {
            topTreeNode = nodeIdMap.get(id);
            if (topTreeNode != null) {
                pTree = Collections.singletonList(topTreeNode);
            }
        }
        // 如果没有指定的顶级节点，看是否有当前的下一级节点，作为顶级节点列表
        if (pTree == null && nodePidMap.containsKey(id)) {
            pTree = nodePidMap.get(id);
        }
        // 如果仍然没有顶级节点则返回
        if (pTree == null || pTree.size() == 0) {
            return null;
        }

        List<T> treeList = build(pTree, nodePidMap);
        if (sort) {
            sort(treeList);
        }

        return treeList;
    }

    /**
     * 递归构建树
     * @param treeNodes
     * @param pTreeNodeMap
     * @param <T>
     * @return
     */
    private static <T extends TreeNode<T>> List<T> build(List<T> treeNodes, Map<Long, List<T>> pTreeNodeMap) {
        for (TreeNode<T> treeNode : treeNodes) {
            if (pTreeNodeMap.containsKey(treeNode.getId())) {
                List<T> subTrees = build(pTreeNodeMap.get(treeNode.getId()), pTreeNodeMap);
                treeNode.setIsLeaf(false);
                treeNode.setTreeNodes(subTrees);
            } else {
                treeNode.setIsLeaf(true);
            }
        }
        return treeNodes;
    }



    /**
     * 排序
     * @param treeNodes 树形结构数据
     * @param <T> 树
     */
    private static <T extends TreeNode<T>> void sort(List<T> treeNodes) {
        if (treeNodes != null && !treeNodes.isEmpty()) {
            Collections.sort(treeNodes);
            for (TreeNode<T> treeNode : treeNodes) {
                if (treeNode.getTreeNodes() != null && !treeNode.getTreeNodes().isEmpty()) {
                    sort(treeNode.getTreeNodes());
                }
            }
        }
    }

    /**
     * 特殊构建方式，从下往上构建树形结构
     *
     * @param treeNodes 树节点
     * @param <T>       树
     * @return 树形结构
     */
    public <T extends TreeNode<T>> List<T> buildTree(List<T> treeNodes) {
        // 排序
        Collections.sort(treeNodes);
        // 构建
        var idMap = treeNodes.stream().collect(Collectors.toMap(TreeNode::getCode, t -> t, (t1, t2) -> t1));
        List<T> tree = new ArrayList<>();

        treeNodes.forEach(node -> {
            var path = node.getPath();
            if (path.startsWith("/")) {
                path = path.substring(1);
            }
            var pathCodes = path.split("/");
            if (pathCodes.length > 1) {
                boolean isRoot = true;
                for (int i = pathCodes.length - 2; i >= 0; i--) {
                    var code = pathCodes[i];
                    if (idMap.containsKey(code)) {
                        isRoot = false;
                        T pNode = idMap.get(code);
                        if (pNode.getTreeNodes() == null) {
                            List<T> list = new ArrayList<>();
                            list.add(node);
                            pNode.setTreeNodes(list);
                        } else {
                            pNode.getTreeNodes().add(node);
                        }

                        break;
                    }
                }
                if (isRoot) {
                    tree.add(node);
                }

            } else {
                tree.add(node);
            }
        });

        return tree;
    }


    /**
     * 树转
     * @param treeNodes
     * @param pid
     * @return
     */
    public static <T extends TreeNode<T>> List<T> treeToList(List<T> treeNodes,List<T> result,Long pid){
        // 设置排序序号
        int sortNo = 1;
        for (T treeNode : treeNodes) {
            // 获取子节点
            List<T> childNodes = treeNode.getTreeNodes();
            // 设置pid
            treeNode.setPid(pid);
            treeNode.setSortNo(sortNo);
            sortNo++;

            if (CollectionUtils.isNotEmpty(childNodes)){
                Long id = treeNode.getId();
                treeToList(childNodes,result,id);
            }
            treeNode.setTreeNodes(null);
            result.add(treeNode);
        }
        return result;
    }

}
