package com.elitesland.yst.production.inv.application.service.stk;

import com.elitescloud.boot.exception.BusinessException;
import com.elitescloud.cloudt.common.base.ApiCode;
import com.elitescloud.cloudt.common.util.RedLockUtils;
import com.elitesland.yst.production.inv.application.facade.vo.base.InvBaseModel;
import com.elitesland.yst.production.inv.application.facade.vo.invwh.InvWhRespVO;
import com.elitesland.yst.production.inv.application.facade.vo.lot.InvLotRespVO;
import com.elitesland.yst.production.inv.application.out.ItmOutService;
import com.elitesland.yst.production.inv.application.out.OrgOutService;
import com.elitesland.yst.production.inv.application.out.SystemService;
import com.elitesland.yst.production.inv.application.service.InvIoService;
import com.elitesland.yst.production.inv.application.service.InvLotCommonService;
import com.elitesland.yst.production.inv.application.service.InvStkService;
import com.elitesland.yst.production.inv.application.service.InvWhService;
import com.elitesland.yst.production.inv.constants.InvRedisConstant;
import com.elitesland.yst.production.inv.domain.convert.invstk.InvCommonConvert;
import com.elitesland.yst.production.inv.domain.entity.invstk.InvIoDO;
import com.elitesland.yst.production.inv.domain.entity.invstk.InvStkDO;
import com.elitesland.yst.production.inv.enums.InvOptEnum;
import com.elitesland.yst.production.inv.enums.WhetherEnum;
import com.elitesland.yst.production.inv.infr.dto.*;
import com.elitesland.yst.production.inv.utils.NumSendObjectEnum;
import com.elitesland.yst.production.inv.utils.UdcEnum;
import com.elitesland.yst.production.support.provider.item.dto.ItmItemRpcDTO;
import com.elitesland.yst.production.support.provider.item.dto.ItmItemUomConvRpcDTO;
import com.elitesland.yst.production.support.provider.item.param.ItmItemRpcDtoParam;
import com.elitesland.yst.production.support.provider.item.param.ItmItemUomConvRpcDtoParam;
import com.elitesland.yst.production.support.provider.org.dto.OrgOuRpcDTO;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.redisson.RedissonRedLock;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import org.springframework.transaction.annotation.Propagation;
import org.springframework.transaction.annotation.Transactional;

import javax.persistence.EntityManager;
import javax.persistence.PersistenceContext;
import javax.validation.constraints.NotNull;
import java.math.BigDecimal;
import java.time.LocalDateTime;
import java.time.ZoneOffset;
import java.util.*;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

/**
 * @author Tom.su
 * @program yst-inv
 * @description
 * @date 2022/04/25 10:14
 */
@Component
@Slf4j
@Transactional(rollbackFor = Exception.class)
public abstract class AbstractInvStkOptService implements InvStkOptService {
    @Autowired
    protected ItmOutService itmOutService;
    @Autowired
    protected InvWhService invWhService;
    @Autowired
    protected InvLotCommonService invLotCommonService;
    @Autowired
    protected RedLockUtils redLockUtils;
    @Autowired
    protected InvStkService invStkService;
    @Autowired
    private OrgOutService outouService;
    @Autowired
    private SystemService systemService;
    @Autowired
    private InvIoService invIoService;

    @PersistenceContext
    private EntityManager entityManager;
    /**
     * 库存出入表正负处理
     *
     * @param qty 操作库存的数量
     * @return 处理后的库存数量
     */
    protected abstract BigDecimal setInvIoQty(BigDecimal qty);

    /**
     * 库存操作数量 赋值
     *
     * @param invStkDO 数据库库存持久化对象，已经确定存在数据库中
     * @param qty      操作库存的数量
     */
    protected abstract void updateQty(InvStkDO invStkDO, BigDecimal qty);

    /**
     * 库存操作入口
     */
    @Override
    public List<InvStkQtyResultDTO> stkOperatePortal(InvSceneConfigDtlDTO config,String sceneCode, List<InvStkCommonOperateBodyDTO> param) {

        long begin = System.currentTimeMillis();
        log.info("stkOperatePortal begin,config:{},param:{}", config, param);
        //业务参数校验
        validateStkOperateList(config, param);
        //组装其他参数
        resolveStkOperateParam(config,param);
        //获取批次信息
        //TODO 此方法上游未加公司维度
        List<InvLotRespVO> invLotList = this.getInvLotList(param);
        //获取商品信息
        List<ItmItemDTO> itmItemVOList = this.getItmItemVOList(param);
        //获取仓库信息
        List<InvWhRespVO> stkWhList = this.getStkWhList(param);
        //获取ou信息
        Map<Long, OrgOuRpcDTO> orgMap = outouService.findOuByIds(param.stream().map(InvStkCommonOperateBodyDTO::getOuId).collect(Collectors.toList())).stream().collect(Collectors.toMap(OrgOuRpcDTO::getId, v -> v, (e1, e2) -> e2));
        //根据仓库、功能区、商品、批次号、合作伙伴类型、合作伙伴编码创建key
        Map<String, List<InvStkCommonOperateBodyDTO>> listMap = param.stream()
                .collect(Collectors.groupingBy(dtl -> dtl.getWhId() + "-" + dtl.getOptDeter2()
                        + "-" + dtl.getItemId() + "-" + dtl.getLotNo()
                        + "-" + dtl.getDeter1() + "-" + dtl.getPType() +
                        "-" + dtl.getPCode() + "-"+dtl.getUom()
                        +"-" + dtl.getSnNo() + "-" + dtl.getLimit1() + "-" + dtl.getVariId() ) );
        //具体操作库存
        this.doStkOpt(config, sceneCode, listMap, invLotList, itmItemVOList, stkWhList, orgMap);
        log.info("stkOperatePortal end,{}ms", begin - System.currentTimeMillis());
        //封装结果集
        return buildStkResult(param);
    }

    /**
     * 库存操作，考虑并发场景，库存操作维度：仓库、功能区、商品、批次号、合作伙伴类型、合作伙伴编码
     *
     * @param config        库存场景配置
     * @param listMap       库存操作明细
     * @param invLotList    库存批次列表
     * @param itmItemVOList 商品列表
     * @param stkWhList     仓库列表
     */
    public void doStkOpt(InvSceneConfigDtlDTO config,
                         String sceneCode,
                         Map<String, List<InvStkCommonOperateBodyDTO>> listMap,
                         List<InvLotRespVO> invLotList,
                         List<ItmItemDTO> itmItemVOList,
                         List<InvWhRespVO> stkWhList,
                         Map<Long, OrgOuRpcDTO> orgOuMap
    ) {
        //生成批次号
        Long ioBatchId = LocalDateTime.now().toInstant(ZoneOffset.ofHours(8)).toEpochMilli();
        List<InvIoDO> ioList = new ArrayList<>();
        for (String key : listMap.keySet()) {
            log.info("开始处理库存，仓库唯一值：{}", key);
            List<InvStkCommonOperateBodyDTO> common28InSaveVO = listMap.get(key);
            RedissonRedLock redLock = redLockUtils.getRedLock(InvRedisConstant.INV_STK_OPERATE_FOR_IO_TYPE + key);
            Map<String, InvStkDO> dbStkMap = new HashMap<>(16);
            for (InvStkCommonOperateBodyDTO operateBodyDTO : common28InSaveVO) {
                ItmItemDTO itmItemVO = itmItemVOList.stream().filter(i -> i.getId().equals(operateBodyDTO.getItemId())).findAny().get();
                InvWhRespVO invWhRespVO = stkWhList.stream().filter(i -> i.getId().equals(operateBodyDTO.getWhId())).findAny().get();
                InvLotRespVO lotRespVO = new InvLotRespVO();
                if (!org.springframework.util.CollectionUtils.isEmpty(invLotList)) {
                    List<InvLotRespVO> lotRespVOList = invLotList.stream().filter(lot -> (lot.getItemId() + "_" + lot.getLotNo() + "_" + lot.getVariId())
                            .equals(operateBodyDTO.getItemId() + "_" + operateBodyDTO.getLotNo() + "_" + operateBodyDTO.getVariId())).collect(Collectors.toList());
                    if(!CollectionUtils.isEmpty(lotRespVOList)){
                        lotRespVO = lotRespVOList.get(0);
                    }
                }
                String uniqueKey = operateBodyDTO.getItemId() + "_" + operateBodyDTO.getWhId()
                        + "_" + operateBodyDTO.getOptDeter2() + "_"
                        + operateBodyDTO.getDeter1() + "_"
                        + operateBodyDTO.getUom() +"_"
                        + operateBodyDTO.getSnNo() + "_"
                        + operateBodyDTO.getPType() +"_"
                        + operateBodyDTO.getPCode() + "_"
                        + operateBodyDTO.getLotNo() + "_"
                        + operateBodyDTO.getLimit1() + "_"
                        + operateBodyDTO.getVariId();
                InvStkDO invStkDO = dbStkMap.get(uniqueKey);
                if (invStkDO == null) {
                    invStkDO = this.queryInvStkAsync(operateBodyDTO);
                    if (invStkDO == null) {
                        try {
                            boolean lockFlag = redLock.tryLock(3, 60, TimeUnit.SECONDS);
                            if (!lockFlag) {
                                throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "库存并发过大,请稍后再试！");
                            }
                            invStkDO = this.queryInvStkAsync(operateBodyDTO);
                            if (invStkDO == null) {
                                //创建库存
                                invStkDO = createStk(operateBodyDTO, itmItemVO, invWhRespVO, lotRespVO);
                                log.info("doStkOpt create invStkDO successful...,key:{}", uniqueKey);
                            }
                        } catch (InterruptedException e) {
                            log.error("invStkCommonOperate InterruptedException,cause->", e);
                            Thread.currentThread().interrupt();
                            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "系统线程中断！");
                        } finally {
                            redLock.unlock();
                        }
                        //校验唯一条件：商品id+异构ID+批次号+仓库ID+功能区+单位
                        dbStkMap.put(uniqueKey, invStkDO);
                    }
                    BigDecimal qty = operateBodyDTO.getQty() == null ? BigDecimal.ZERO : operateBodyDTO.getQty().abs();
                    BigDecimal avalQty = invStkDO.getAvalQty() == null ? BigDecimal.ZERO : invStkDO.getAvalQty();
                    // 扣减库存为负
                    if(InvOptEnum.INV_IO_TYPE_T.getType().equals(config.getIoType())
//                            && InvOptEnum.INV_IO_TYPE_O.getType().equals(config.getIoType())
                    ){
                        if ((avalQty.subtract(qty)).compareTo(BigDecimal.ZERO) < 0) {
                            log.error("库存可用量不足,单据信息：{},库存信息:{},", operateBodyDTO, invStkDO);
                            throw new BusinessException(ApiCode.PARAMETER_EXCEPTION, "库存可用量不足");
                        }
                    }
                    clearStk(invStkDO);
                    log.info("clearStk jps object sts success...");
                    var updateBean = new InvStkDO();
                    BeanUtils.copyProperties(invStkDO,updateBean);
                    updateBean.setDeter2(operateBodyDTO.getOptDeter2());
                    log.info("ready to update stk...");
                    this.updateQty(updateBean, qty);
                    int result = invStkService.updateStkQty(updateBean);
                    log.info("update stk success...");
                    // 允许库存为负-所以注释以下代码
//                    if (result != 1) {
//                        log.error("库存操作失败(库存数量不足)。单据信息：" + operateBodyDTO.toString());
//                        throw new BusinessException("库存操作失败(库存数量不足)");
//                    }
                    //插入出入库履历
                    ioList.add(getInvIoLog(operateBodyDTO,
                            invStkDO.getId(),
                            config,
                            sceneCode,
                            itmItemVO,
                            orgOuMap.get(operateBodyDTO.getOuId()),
                            ioBatchId,
                            1D,
                            invWhRespVO));
                }
            }

        }
        invIoService.createBatch(ioList);
    }

    private void clearStk(InvStkDO invStkDO) {
        entityManager.detach(invStkDO);
        invStkDO.setOhQty(BigDecimal.ZERO);
        invStkDO.setOhQty2(BigDecimal.ZERO);
        invStkDO.setRsvQty(BigDecimal.ZERO);
        invStkDO.setRsvQty2(BigDecimal.ZERO);
        invStkDO.setRsvQty3(BigDecimal.ZERO);
        invStkDO.setRsvQty4(BigDecimal.ZERO);
        invStkDO.setLockQty(BigDecimal.ZERO);
        invStkDO.setLockQty2(BigDecimal.ZERO);
        invStkDO.setLockQty3(BigDecimal.ZERO);
        invStkDO.setLockQty4(BigDecimal.ZERO);

        invStkDO.setOwQty(BigDecimal.ZERO);
        invStkDO.setOwQty2(BigDecimal.ZERO);
        invStkDO.setOwQty3(BigDecimal.ZERO);
        invStkDO.setOwQty4(BigDecimal.ZERO);
        invStkDO.setAvalQty(BigDecimal.ZERO);
        invStkDO.setAvalQty2(BigDecimal.ZERO);
    }

    protected InvIoDO getInvIoLog(InvStkCommonOperateBodyDTO common28InVO, Long stkId, InvSceneConfigDtlDTO invStk28Enum, String sceneCode,
                                  ItmItemDTO itmItemVO, OrgOuRpcDTO ou, Long ioBatchId, Double lineNo, InvWhRespVO invWhRespVO) {

        InvIoDO invIo = InvCommonConvert.INSTANCE.invStkCommonOperateBodyDTOToInvIoDO(common28InVO);
        invIo.setIoBatchId(ioBatchId);
        invIo.setLineNo(lineNo);
        invIo.setDeleteFlag(0);
        invIo.setStkId(stkId);
        invIo.setWhCode(invWhRespVO.getWhCode());
        invIo.setIoCode(invStk28Enum.getIoCode());
        //库存出入表 IO qty 正负处理
        invIo.setQty(this.setInvIoQty(common28InVO.getQty().abs()));
        invIo.setIoDate(common28InVO.getOpDate());
        invIo.setCreateUserId(common28InVO.getCreateUserId());
        //流水发号
        ArrayList<String> runtimeValues = new ArrayList<>();
        String generateCode = systemService.sysNumberRuleGenerateCode(NumSendObjectEnum.INV_MR.getCode(), runtimeValues);
        invIo.setDocNo(generateCode);
        invIo.setUom2(itmItemVO.getUom2());
        invIo.setSceneCode(sceneCode);
        if (itmItemVO.getVolume() != null) {
            invIo.setVolume(Double.valueOf(itmItemVO.getVolume()));
        }
        if (itmItemVO.getNetWeight() != null) {
            invIo.setNetWeight(Double.valueOf(itmItemVO.getNetWeight()));
        }
        if (itmItemVO.getGrossWeight() != null) {
            invIo.setGrossWeight(Double.valueOf(itmItemVO.getGrossWeight()));
        }
        invIo.setWeightUomCode(itmItemVO.getWeightUom());
        if (ou != null) {
            invIo.setOuId(ou.getId());
        }
//        invIo.setBuId(ou.getBuId());
        invIo.setItemCode(itmItemVO.getItemCode());
        invIo.setItemCateCode(itmItemVO.getItemCateCode());
        invIo.setBrand(itmItemVO.getBrand());
//        invIo.setHomeCurr(ou.getOuCurr());
        invIo.setIoStatus(UdcEnum.COM_STATUS_ACTIVEORNO_ACTIVE.getValueCode());
        invIo.setSrcDocNo(common28InVO.getDocNo());
        //库存合作伙伴保存
        invIo.setPType(common28InVO.getPType());
        invIo.setPCode(common28InVO.getPCode());
        invIo.setSecBuId(invWhRespVO.getSecBuId());
        invIo.setSecOuId(invWhRespVO.getSecOuId());
        //库存移动记录时间
        invIo.setCreateTime(LocalDateTime.now());
        return invIo;
    }

    private void resolveStkOperateParam(InvSceneConfigDtlDTO config, List<InvStkCommonOperateBodyDTO> param) {
        //单位转换
        this.convertUom(param);
        //指定功能区
        this.convertDeter2(config,param);
        //指定来源单号
        this.convertDocNo(config,param);
    }

    protected abstract void convertDeter2(InvSceneConfigDtlDTO config, List<InvStkCommonOperateBodyDTO> param);

    protected abstract void convertDocNo(InvSceneConfigDtlDTO config, List<InvStkCommonOperateBodyDTO> param);

    @Transactional(propagation = Propagation.REQUIRES_NEW, rollbackFor = Exception.class)
    public InvStkDO queryInvStkAsync(InvStkCommonOperateBodyDTO bodyDTO) {
        InvBaseModel invBaseModel = new InvBaseModel();
        BeanUtils.copyProperties(bodyDTO, invBaseModel);
        invBaseModel.setDeter2(bodyDTO.getOptDeter2());
        return invStkService.getInvStkDo(invBaseModel);
    }

    public void validateStkOperateList(InvSceneConfigDtlDTO config, List<InvStkCommonOperateBodyDTO> bodyList) {
        bodyList = bodyList.stream().filter(inVO -> inVO.getQty() != null && inVO.getQty().compareTo(BigDecimal.ZERO) != 0).collect(Collectors.toList());
        bodyList.forEach(body -> validateStkOperate(config, body));
    }

    private List<InvWhRespVO> getStkWhList(List<InvStkCommonOperateBodyDTO> bodyList) {
        //传入的仓库whID
        List<Long> whIdList = bodyList.stream().map(InvStkCommonOperateBodyDTO::getWhId).distinct().collect(Collectors.toList());
        //查询到的数据库仓库
        List<InvWhRespVO> orgWhVOList = invWhService.findIdBatch(whIdList);
        if (CollectionUtils.isEmpty(orgWhVOList)) {
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "仓库不存在，请检查!");
        }
        //查询到的数据库仓库WHid
        List<Long> dbWhIds = orgWhVOList.stream().map(InvWhRespVO::getId).distinct().collect(Collectors.toList());
        // 传入的仓库whID - 查询到的数据库仓库WHid =
        whIdList.removeAll(dbWhIds);
        if (CollectionUtils.isNotEmpty(whIdList)) {
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "仓库不存在，请检查!,仓库号：" + StringUtils.join(whIdList, ","));
        }
        return orgWhVOList;

    }

    private void validateStkOperate(InvSceneConfigDtlDTO config, InvStkCommonOperateBodyDTO body) {
        //关联单据必传
        if (WhetherEnum.Y.name().equals(config.getIsRelDocReq())) {
            validateSrcDoc(body);
        }
    }

    private List<InvLotRespVO> getInvLotList(List<InvStkCommonOperateBodyDTO> bodyList) {
        List<InvLotRespVO> rs = new ArrayList<>();

        List<String> invLotStringList = bodyList.stream()
                .filter(i -> !org.springframework.util.StringUtils.isEmpty(i.getLotNo()))
                .map(i -> i.getItemId() == null ? "" : i.getItemId() + "" + (i.getVariId() == null ? "" : i.getVariId()) + "" + i.getLotNo()+ "" +i.getOuId())
                .distinct().collect(Collectors.toList());
        if(CollectionUtils.isEmpty(invLotStringList)){
            return rs;
        }
        Map<String, List<InvLotRespVO>> invLotListMap = invLotCommonService.getInvLotVOListByKey(invLotStringList);
        if (invLotListMap == null || invLotListMap.size() == 0) {
            throw new BusinessException("库存批次信息不存在！,key:" + invLotStringList);
        }
        invLotStringList.removeAll(invLotListMap.keySet());
        if (CollectionUtils.isNotEmpty(invLotStringList)) {
            throw new BusinessException("库存批次信息不存在！,key:" + invLotStringList);
        }
        Collection<List<InvLotRespVO>> values = invLotListMap.values();
        for (List<InvLotRespVO> value : values) {
            rs.addAll(value);
        }
        return rs;
    }

    /**
     * 校验关联单据相关
     *
     * @param body
     */
    private void validateSrcDoc(InvStkCommonOperateBodyDTO body) {
        if (StringUtils.isBlank(body.getDocNo())) {
            throw new BusinessException("单据号不能为空！");
        }
        if (StringUtils.isBlank(body.getSrcDocCls())) {
            throw new BusinessException("来源单据类别不能为空！");
        }
        if (body.getSrcDocId() == null) {
            throw new BusinessException("来源单据ID不能为空！");
        }
        if (body.getSrcDocDid() == null) {
            throw new BusinessException("来源单据明细ID不能为空！");
        }
    }

    /**
     * 单位转换
     *
     * @param bodyList
     */
    private void convertUom(List<InvStkCommonOperateBodyDTO> bodyList) {
        List<@NotNull Long> itemIds = bodyList.stream().map(InvStkCommonOperateBodyDTO::getItemId).collect(Collectors.toList());
        List<ItmItemUomConvRpcDTO> uomRpcList = this.getUomRpcList(itemIds);
        ItmItemRpcDtoParam param = new ItmItemRpcDtoParam();
        param.setItemIds(itemIds);
        List<ItmItemRpcDTO> itemRpcList = itmOutService.findItemRpcDtoByParam(param);
        bodyList.stream().map(m -> {
            itemRpcList.stream().filter(v -> v.getId().equals(m.getItemId())).findAny().ifPresent(v -> {
                InvUomQueryDTO uomQueryParamVO = new InvUomQueryDTO();
                uomQueryParamVO.setFromUom(m.getUom());
                uomQueryParamVO.setToUom(v.getUom());
                uomQueryParamVO.setItemCode(v.getItemCode());
                uomQueryParamVO.setItemId(m.getItemId());
                BigDecimal uomRatio = this.getUomRatio(uomRpcList, uomQueryParamVO);
                //商品主单位数量转换
                if (uomRatio.compareTo(BigDecimal.ZERO) > 0) {
                    m.setQty(m.getQty().multiply(uomRatio));
                } else {
                    throw new BusinessException(ApiCode.FAIL, String.format("商品【%s]单位从【%s】到 【%s】单位转换系数未配置，请检查", v.getItemCode(), m.getUom(), v.getUom()));
                }
                //商品主单位
                m.setUom(v.getUom());
            });
            return m;
        }).collect(Collectors.toList());
    }

    /**
     * 查询商品信息
     *
     * @param bodyList
     * @return
     */
    private List<ItmItemDTO> getItmItemVOList(List<InvStkCommonOperateBodyDTO> bodyList) {
        List<Long> itemIdList = bodyList.stream().map(InvStkCommonOperateBodyDTO::getItemId).distinct().collect(Collectors.toList());
        log.info("商品ID集合是：{}", itemIdList);
        if (CollectionUtils.isEmpty(itemIdList)) {
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "商品id不能为空!");
        }
        ItmItemRpcDtoParam itmItemPartParam = new ItmItemRpcDtoParam();
        itmItemPartParam.setItemIds(itemIdList);
        // 查询全部商品
        itmItemPartParam.setItemStatus3("ALL");
        List<ItmItemRpcDTO> itmItemRpcList = itmOutService.findItemRpcDtoByParam(itmItemPartParam);
        if (CollectionUtils.isEmpty(itmItemRpcList)) {
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "商品信息查询失败!");
        }
        List<ItmItemDTO> itmItemList = itmItemRpcList.stream().map(e -> {
            ItmItemDTO itmItemVO = new ItmItemDTO();
            BeanUtils.copyProperties(e, itmItemVO);
            return itmItemVO;
        }).collect(Collectors.toList());
        return itmItemList;
    }

    private List<ItmItemUomConvRpcDTO> getUomRpcList(List<Long> itemIds) {
        ItmItemUomConvRpcDtoParam param = new ItmItemUomConvRpcDtoParam();
        param.setItemIds(itemIds);
        List<ItmItemUomConvRpcDTO> uomRpcList = itmOutService.findItemUomConvDtoByParam(param);
        return uomRpcList;
    }

    private BigDecimal getUomRatio(List<ItmItemUomConvRpcDTO> uomRpcList, InvUomQueryDTO paramVO) {
        log.info("单位转换参数为1：{},参数2：{}", uomRpcList, paramVO);
        BigDecimal uomRatio = BigDecimal.ONE;
        if (paramVO.getFromUom() == null || paramVO.getToUom() == null || paramVO.getFromUom().equals(paramVO.getToUom())) {
            return uomRatio;
        }
      /*  ItmItemuomConvRatioQueryParm queryParm = new ItmItemuomConvRatioQueryParm();
        queryParm.setItemCode(paramVO.getItemCode());
        queryParm.setFromUom(paramVO.getFromUom());
        queryParm.setToUom(paramVO.getToUom());*/
        //商品单位转换
        Optional<ItmItemUomConvRpcDTO> uomRpc = uomRpcList.stream().filter(u -> u.getItemId().equals(paramVO.getItemId())
                && u.getFromUom().equals(paramVO.getFromUom())
                && u.getToUom().equals(paramVO.getToUom())).findFirst();
        if (uomRpc.isPresent() && uomRpc.get().getRatio() != null) {
            if (uomRpc.get().getRatio().compareTo(BigDecimal.ZERO) <= 0) {
                throw new BusinessException(ApiCode.FAIL, "商品【" + paramVO.getItemCode() + "】单位转换" + paramVO.getFromUom() + "->" + paramVO.getToUom() + "异常,单位转换系数为0，请检查商品主数据");
            }
            return uomRpc.get().getRatio();
        } else {
            //商品单位转换
            Optional<ItmItemUomConvRpcDTO> uomRpcDto = uomRpcList.stream().filter(u -> u.getItemId().equals(paramVO.getItemId())
                    && u.getFromUom().equals(paramVO.getToUom())
                    && u.getToUom().equals(paramVO.getFromUom())).findFirst();
            if (uomRpcDto.isPresent() && uomRpcDto.get().getRatio() != null) {
                if (uomRpcDto.get().getRatio().compareTo(BigDecimal.ZERO) <= 0) {
                    throw new BusinessException(ApiCode.FAIL, "商品【" + paramVO.getItemCode() + "】单位转换" + paramVO.getFromUom() + "->" + paramVO.getToUom() + "异常,单位转换系数为0，请检查商品主数据");
                }
                return uomRpcDto.get().getRatio().negate();

            } else {
                throw new BusinessException(ApiCode.FAIL, "商品【" + paramVO.getItemCode() + "】单位转换" + paramVO.getFromUom() + "->" + paramVO.getToUom() + "异常，请检查商品主数据");
            }
        }
    }

    public InvStkDO createStk(InvStkCommonOperateBodyDTO common28InVO, ItmItemDTO itmItemVO, InvWhRespVO invWhRespVO, InvLotRespVO invLotRespVO) {
        InvStkDO invStkDO = new InvStkDO();
        invStkDO.setItemId(itmItemVO.getId());
        invStkDO.setItemCode(itmItemVO.getItemCode());
        invStkDO.setUom(itmItemVO.getUom());
        invStkDO.setUom2(itmItemVO.getUom2());
        invStkDO.setDeleteFlag(0);
        invStkDO.setItemName(itmItemVO.getItemName());
        invStkDO.setItemCateCode(itmItemVO.getItemCateCode());
        invStkDO.setBrand(itmItemVO.getBrand());
        invStkDO.setWhId(invWhRespVO.getId());
        invStkDO.setWhCode(invWhRespVO.getWhCode());
        invStkDO.setOuId(invWhRespVO.getOuId());
        invStkDO.setBuId(itmItemVO.getBuId());
        invStkDO.setDesId(itmItemVO.getBuId());
        invStkDO.setSecBuId(invWhRespVO.getSecBuId());
        invStkDO.setSecOuId(invWhRespVO.getSecOuId());
        invStkDO.setVariId(common28InVO.getVariId());
        invStkDO.setDeter1(common28InVO.getDeter1());
        invStkDO.setDeter2(common28InVO.getOptDeter2());
        invStkDO.setDeter3(common28InVO.getDeter3());
        invStkDO.setDeter4(itmItemVO.getItemType5());
        invStkDO.setDeter5(itmItemVO.getItemType2());
        invStkDO.setLotNo(common28InVO.getLotNo());
        invStkDO.setSnNo(common28InVO.getSnNo());
        invStkDO.setLimit1(common28InVO.getLimit1());
        invStkDO.setLimit1(common28InVO.getLimit2());
        invStkDO.setLimit1(common28InVO.getLimit3());
        invStkDO.setInDate(common28InVO.getOpDate());
        invStkDO.setPCode(itmItemVO.getSpuCode());
        invStkDO.setPType(common28InVO.getPType());
        invStkDO.setOhQty(BigDecimal.ZERO);
        invStkDO.setOhQty2(BigDecimal.ZERO);
        invStkDO.setRsvQty(BigDecimal.ZERO);
        invStkDO.setRsvQty2(BigDecimal.ZERO);
        invStkDO.setRsvQty3(BigDecimal.ZERO);
        invStkDO.setRsvQty4(BigDecimal.ZERO);
        invStkDO.setLockQty(BigDecimal.ZERO);
        invStkDO.setLockQty2(BigDecimal.ZERO);
        invStkDO.setLockQty3(BigDecimal.ZERO);
        invStkDO.setLockQty4(BigDecimal.ZERO);
        invStkDO.setOwQty(BigDecimal.ZERO);
        invStkDO.setOwQty2(BigDecimal.ZERO);
        invStkDO.setOwQty3(BigDecimal.ZERO);
        invStkDO.setOwQty4(BigDecimal.ZERO);
        invStkDO.setAvalQty(BigDecimal.ZERO);
        invStkDO.setAvalQty2(BigDecimal.ZERO);
        invStkDO.setDeleteFlag(0);
        invStkDO.setSrcDocCls(common28InVO.getSrcDocCls());
        invStkDO.setSrcDocId(common28InVO.getSrcDocId());
        invStkDO.setSrcDocDid(common28InVO.getSrcDocDid());
        invStkDO.setCreateUserId(common28InVO.getCreateUserId());
        invStkDO.setUntilExpireDays(invLotRespVO.getUntilExpireDays());
        invStkDO.setFressType(invLotRespVO.getFressType());
        if (itmItemVO.getVolume() != null) {
            invStkDO.setVolume(BigDecimal.valueOf(itmItemVO.getVolume()));
        }
        if (itmItemVO.getNetWeight() != null) {
            invStkDO.setNetWeight(BigDecimal.valueOf((itmItemVO.getNetWeight())));
        }
        if (itmItemVO.getGrossWeight() != null) {
            invStkDO.setGrossWeight(BigDecimal.valueOf((itmItemVO.getGrossWeight())));
        }
        invStkService.createInvStk(invStkDO);
        return invStkDO;
    }

    private List<InvStkQtyResultDTO> buildStkResult(List<InvStkCommonOperateBodyDTO> param) {
        ArrayList<InvStkQtyResultDTO> list = new ArrayList<>();
        param.forEach(i -> {
            InvStkQtyResultDTO resultDTO = new InvStkQtyResultDTO();
            resultDTO.setLotNo(i.getLotNo());
            resultDTO.setQty(i.getQty());
            resultDTO.setSrcDocNo(i.getDocNo());
            resultDTO.setSrcDocCls(i.getSrcDocCls());
            resultDTO.setSrcDocId(i.getSrcDocId());
            resultDTO.setSrcDocDid(i.getSrcDocDid());
            list.add(resultDTO);
        });
        return list;
    }

    /**
     * 对 null的qty附 初值 0D
     *
     * @param qty
     * @return
     */
    protected BigDecimal setQty(BigDecimal qty) {
        return qty == null ? BigDecimal.ZERO : qty;
    }


    /**
     * 指定功能区校验
     *
     * @param transferDeter2 配置中的指定功能区
     *
     */
    protected void convertDeter2(String transferDeter2,InvStkCommonOperateBodyDTO dto) {
        dto.setOptDeter2(dto.getDeter2());
        if(StringUtils.isNotBlank(transferDeter2)){
            dto.setOptDeter2(transferDeter2);
        }
    }


    /**
     *指定库存操作来源单据
     * @param isUseRelDoc
     * @param dto
     */
    protected void convertDocNo(String isUseRelDoc,InvStkCommonOperateBodyDTO dto) {
        if(isUseRelDoc != null && WhetherEnum.Y.equals(isUseRelDoc)) {
            dto.setDocNo(dto.getSrcDocNo2());
            dto.setSrcDocId(dto.getSrcDocId2());
            dto.setSrcDocDid(dto.getSrcDocDid2());
            dto.setSrcDocCls(dto.getSrcDocCls2());
            dto.setOptDeter2(dto.getDeter2());
        }
    }
}
