package com.elitesland.yst.production.inv.utils;

import com.alibaba.excel.EasyExcel;
import com.alibaba.excel.ExcelWriter;
import com.alibaba.excel.write.builder.ExcelWriterBuilder;
import com.alibaba.excel.write.handler.CellWriteHandler;
import com.alibaba.excel.write.handler.RowWriteHandler;
import com.alibaba.excel.write.merge.AbstractMergeStrategy;
import com.alibaba.excel.write.metadata.WriteSheet;
import com.alibaba.excel.write.metadata.WriteTable;
import com.alibaba.excel.write.metadata.fill.FillConfig;
import com.alibaba.excel.write.metadata.style.WriteCellStyle;
import com.alibaba.excel.write.metadata.style.WriteFont;
import com.alibaba.excel.write.style.HorizontalCellStyleStrategy;
import com.alibaba.excel.write.style.column.LongestMatchColumnWidthStyleStrategy;
import com.elitesland.yst.production.inv.utils.excel.ExcelTitleHandler;
import com.elitesland.yst.production.inv.utils.excel.LocalDateConverter;
import com.elitesland.yst.production.inv.utils.excel.LocalDateTimeConverter;
import com.google.common.collect.Lists;
import org.apache.poi.ss.usermodel.IndexedColors;
import org.springframework.core.io.ClassPathResource;
import org.springframework.util.ObjectUtils;

import javax.servlet.http.HttpServletResponse;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.URLEncoder;
import java.nio.charset.StandardCharsets;
import java.util.List;
import java.util.Map;

/**
 * @Description:
 * @Auther: menghua
 * @Date: 2021/2/18
 */
public class ExcelWriteUtil {

    /**
     * 获取输出流
     * @param response
     * @param filename  导出文件名
     * @return
     * @throws IOException
     */
    public OutputStream getOutputStream(HttpServletResponse response, String filename) throws IOException {
        OutputStream outputStream = response.getOutputStream();
        response.reset();
        filename = URLEncoder.encode(filename, StandardCharsets.UTF_8);
        response.setHeader("Content-disposition", "attachment; filename=" + filename + ".xlsx");
        response.setCharacterEncoding("UTF-8");
        response.setContentType("application/x-download");
        /*response.setContentType("application/msexcel;charset=UTF-8");//设置类型
        response.setHeader("Pragma", "No-cache");//设置头
        response.setHeader("Cache-Control", "no-cache");//设置头
        response.setDateHeader("Expires", 0);//设置日期头*/
        return outputStream;
    }

    /**
     * 根据模板创建ExcelWriter
     * @param outputStream 输出流
     * @param classpath 模板路径
     * @return
     * @throws IOException
     */
    public ExcelWriter createExcelWriter (OutputStream outputStream, String classpath, CellWriteHandler customWriteCellStype,
                                          AbstractMergeStrategy customWriteCellMerge, RowWriteHandler rowWriteHandler) throws IOException {
        InputStream inputStream = new ClassPathResource(classpath).getInputStream();
        ExcelWriter excelWriter = EasyExcel.write(outputStream).registerWriteHandler(customWriteCellStype)
                .registerWriteHandler(customWriteCellMerge).registerWriteHandler(rowWriteHandler).withTemplate(inputStream).build();
        return excelWriter;
    }

    /**
     * 创建ExcelSheet
     * @return
     */
    public WriteSheet createWriteSheet (){
        return EasyExcel.writerSheet().build();
    }

    /**
     * 填充list数据
     * @param excelWriter   写入的workbook
     * @param writeSheet    写入的sheet
     * @param datas          写入的数据
     */
    public void fillList(ExcelWriter excelWriter, WriteSheet writeSheet,List<List<?>> datas){
        FillConfig fillConfig = FillConfig.builder().forceNewRow(Boolean.TRUE).build();
        //填充list
        for (List<?> data : datas) {
            excelWriter.fill(data, fillConfig, writeSheet);
        }
    }

    /**
     * 填充普通变量
     * @param excelWriter   写入的workbook
     * @param writeSheet    写入的sheet
     * @param data          写入的数据
     */
    public void fillMap(ExcelWriter excelWriter, WriteSheet writeSheet, Map data){
        excelWriter.fill(data, writeSheet);
    }

    public void closeExcelWriter(HttpServletResponse response,ExcelWriter excelWriter, OutputStream outputStream) throws IOException {
        if(!ObjectUtils.isEmpty(excelWriter)){
            excelWriter.finish();
        }
        if(!ObjectUtils.isEmpty(outputStream)){
            outputStream.flush();
        }
        if(!ObjectUtils.isEmpty(response)){
            response.getOutputStream().close();
        }
    }

    /**
     * @param response  响应
     * @param list      出参数据集
     * @param clazz     出参对象类型
     * @param fileName  excel 文件名称
     * @param sheetName excel Sheet页名称
     */
    public static void excelWrite(HttpServletResponse response,
                                  List<?> list,
                                  Class<?> clazz,
                                  String fileName,
                                  String sheetName) throws IOException {
        response.setContentType("application/octet-stream");
        fileName = URLEncoder.encode(fileName, StandardCharsets.UTF_8);
        response.setHeader("Content-disposition", "attachment;filename*=utf-8''" + fileName + ".xlsx");
        ExcelWriterBuilder write = EasyExcel.write(response.getOutputStream(), clazz);
        write.registerWriteHandler(new LongestMatchColumnWidthStyleStrategy())
                .sheet(0,sheetName).doWrite(list);
    }

    /**
     * @Author zhangxinsheng
     * @Date  2020/11/20  14:38
     * @Description  销售模块导出专用工具类-张鑫昇
     **/
    public static void excelWrite1(HttpServletResponse response,
                                   List<?> list,
                                   Class<?> clazz,
                                   String fileName,
                                   String sheetName) throws IOException {
        // 这里注意 有同学反应使用swagger 会导致各种问题，请直接用浏览器或者用postman
        response.setContentType("application/vnd.ms-excel");
        response.setCharacterEncoding("utf-8");
        // 这里URLEncoder.encode可以防止中文乱码 当然和easyexcel没有关系
        fileName = URLEncoder.encode(fileName, StandardCharsets.UTF_8);
        response.setHeader("Content-disposition", "attachment;filename=" + fileName + ".xlsx");
        EasyExcel.write(response.getOutputStream(), clazz)
                .registerConverter(new LocalDateTimeConverter())
                .registerWriteHandler(new LongestMatchColumnWidthStyleStrategy())
                .sheet(sheetName + "1")
                .doWrite(list);
//        EasyExcel.write(fileName, clazz).registerConverter(new LocalDateTimeConverter()).sheet("模板").doWrite(list);
    }

    /**
     * 动态生成表头 姚蒙羊  2020/11/19
     * @param response
     * @param list
     * @param list1
     * @param clazz
     * @param fileName
     * @param sheetName
     * @throws IOException
     */
    public static void excelWriteDynamic(HttpServletResponse response,
                                         List<?> list, List<?> list1,
                                         Class<?> clazz,
                                         String fileName,
                                         String sheetName) throws IOException {
        // 这里注意 有同学反应使用swagger 会导致各种问题，请直接用浏览器或者用postman
        response.setContentType("application/vnd.ms-excel");
        response.setCharacterEncoding("utf-8");
        // 这里URLEncoder.encode可以防止中文乱码 当然和easyexcel没有关系
        fileName = URLEncoder.encode(fileName, StandardCharsets.UTF_8);
        response.setHeader("Content-disposition", "attachment;filename=" + fileName + ".xlsx");
        EasyExcel.write(response.getOutputStream(), clazz).head((List<List<String>>) list1).registerConverter(new LocalDateTimeConverter()).registerWriteHandler(new LongestMatchColumnWidthStyleStrategy()).sheet(sheetName + "1").doWrite(list);
//        EasyExcel.write(fileName, clazz).registerConverter(new LocalDateTimeConverter()).sheet("模板").doWrite(list);
    }
    /**
     * 明细+汇总双表头
     */
    public static void excelWriteDuoHeader(HttpServletResponse response,
                                           List<?> dataA,
                                           List<?> dataB,
                                           Class<?> typeA,
                                           Class<?> typeB,
                                           String fileName,
                                           String sheetName) throws IOException{
        // 这里注意 有同学反应使用swagger 会导致各种问题，请直接用浏览器或者用postman
        response.setContentType("application/vnd.ms-excel");
        response.setCharacterEncoding("utf-8");
        // 这里URLEncoder.encode可以防止中文乱码 当然和easyexcel没有关系
        fileName = URLEncoder.encode(fileName, StandardCharsets.UTF_8);
        response.setHeader("Content-disposition", "attachment;filename=" + fileName + ".xlsx");
        ExcelWriter writer= EasyExcel.write(response.getOutputStream()).registerWriteHandler(new LongestMatchColumnWidthStyleStrategy()).build();
        WriteSheet writeSheet= EasyExcel.writerSheet(sheetName).build();
        WriteTable writeTableA= EasyExcel.writerTable(0).head(typeA).build();
        WriteTable writeTableB= EasyExcel.writerTable(1).head(typeB).build();
        writer.write(dataA,writeSheet,writeTableA);
        writer.write(dataB,writeSheet,writeTableB);
        writer.finish();

    }

    /**
     * @Author zhangxinsheng
     * @Date  2020/11/20  15:04
     * @Description  diver27
     **/
    public static void excelWriteRpt(HttpServletResponse response,
                                     List<?> list,
                                     Class<?> clazz,
                                     String fileName,
                                     String sheetName) throws IOException {
        // 这里注意 有同学反应使用swagger 会导致各种问题，请直接用浏览器或者用postman
        response.setContentType("application/vnd.ms-excel");
        response.setCharacterEncoding("utf-8");
        // 这里URLEncoder.encode可以防止中文乱码 当然和easyexcel没有关系
        fileName = URLEncoder.encode(fileName, StandardCharsets.UTF_8);
        response.setHeader("Content-disposition", "attachment;filename=" + fileName + ".xlsx");
        // 头策略
        WriteCellStyle headWriteCellStyle = new WriteCellStyle();
        headWriteCellStyle.setFillForegroundColor(IndexedColors.PALE_BLUE.getIndex());
        WriteFont headWriteFont = new WriteFont();
        headWriteFont.setColor(IndexedColors.BLACK.getIndex());
        headWriteFont.setFontHeightInPoints((short)10);
        headWriteFont.setBold(true);
        headWriteFont.setFontName("Arial");
        headWriteCellStyle.setWriteFont(headWriteFont);
        // 内容策略
        WriteCellStyle contentWriteCellStyle = new WriteCellStyle();
        WriteFont contentWriteFont = new WriteFont();
        contentWriteFont.setFontHeightInPoints((short)10);
        contentWriteFont.setFontName("Arial");
        contentWriteFont.setBold(false);
        contentWriteCellStyle.setWriteFont(contentWriteFont);
        HorizontalCellStyleStrategy horizontalCellStyleStrategy = new HorizontalCellStyleStrategy(headWriteCellStyle, contentWriteCellStyle);
        ExcelWriterBuilder write = EasyExcel.write(response.getOutputStream(), clazz);
        write.registerConverter(new LocalDateTimeConverter())
                //根据内容自适应宽度
                .registerWriteHandler(new LongestMatchColumnWidthStyleStrategy())
                .registerWriteHandler(horizontalCellStyleStrategy)
                .registerWriteHandler(new RptCustomWriterStyle())
                .registerWriteHandler(new CustomWriteCellValue())
                .sheet(sheetName + "1").doWrite(list);
    }


    private final static Integer GROUP_SIZE = 10000;//按10000分组

    /**
     * 分sheet页导出
     *
     * @param response  响应
     * @param list      出参数据集
     * @param clazz     出参对象类型
     * @param fileName  excel 文件名称
     * @param sheetName excel Sheet页名称
     */
    public static void excelWriteGroup(HttpServletResponse response,
                                       List<?> list,
                                       Class<?> clazz,
                                       String fileName,
                                       String sheetName) throws IOException {
        response.setContentType("application/vnd.ms-excel");
        response.setCharacterEncoding("utf-8");
        fileName = URLEncoder.encode(fileName, StandardCharsets.UTF_8);
        response.setHeader("Content-disposition", "attachment;filename=" + fileName + ".xlsx");
        ExcelWriter excelWriter = EasyExcel.write(response.getOutputStream()).build();
        List<? extends List<?>> lists = Lists.partition(list, GROUP_SIZE);
        for (int i = 0; i < lists.size(); i++) {
            WriteSheet writeSheet1 = EasyExcel.writerSheet(i, sheetName + (i + 1)).head(clazz).build();
            excelWriter.write(lists.get(i), writeSheet1);
        }
        excelWriter.finish();
    }
}


