package com.elitesland.yst.production.sale.repo.shop;

import com.elitesland.yst.production.sale.api.vo.param.shop.MktDiscountGiftQueryVO;
import com.elitesland.yst.production.sale.api.vo.resp.shop.*;
import com.elitesland.yst.production.sale.common.constant.ConstantsSale;
import com.elitesland.yst.production.sale.common.constant.UdcEnum;
import com.elitesland.yst.production.sale.entity.QBipItemSkuDO;
import com.elitesland.yst.production.sale.entity.QMktDiscountGiftDDO;
import com.elitesland.yst.production.sale.entity.QMktDiscountGiftDO;
import com.elitesland.yst.production.sale.entity.QMktGiftDO;
import com.querydsl.core.Tuple;
import com.querydsl.core.types.ExpressionUtils;
import com.querydsl.core.types.Predicate;
import com.querydsl.core.types.Projections;
import com.querydsl.jpa.impl.JPAQuery;
import com.querydsl.jpa.impl.JPAQueryFactory;
import lombok.val;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Repository;
import org.springframework.util.StringUtils;

import java.math.BigDecimal;
import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;

/**
 * <p>
 * 折扣满减赠 proc
 * </p>
 *
 * @author ssy
 * @since 2021/8/23
 */
@Repository
public class MktDiscountGiftRepoProc {

    @Autowired
    private JPAQueryFactory jpaQueryFactory;

    private final QMktDiscountGiftDO qMktDiscountGiftDO = QMktDiscountGiftDO.mktDiscountGiftDO;

    private final QMktDiscountGiftDDO qMktDiscountGiftDDO = QMktDiscountGiftDDO.mktDiscountGiftDDO;

    private final QMktGiftDO qMktGiftDO = QMktGiftDO.mktGiftDO;

    private final QBipItemSkuDO bipItemSku = QBipItemSkuDO.bipItemSkuDO;

    /**
     * 根据 入参，查询符合条件的商品促销信息
     *
     * @param ouId        公司ID
     * @param customLevel 可为空，为空则查询全部客户级别
     * @param date        促销日期
     * @param itemIds     商品ID 集合
     * @return 查询jpaQuery
     */
    public JPAQuery<MktDiscountGiftVO> findMktDisCountGift(Long ouId, String customLevel, LocalDateTime date, List<Long> itemIds) {

        JPAQuery<MktDiscountGiftVO> jpaQuery = jpaQueryFactory.select(Projections.bean(MktDiscountGiftVO.class,
                qMktDiscountGiftDO.id.as("mktDiscountGiftId"),
                qMktDiscountGiftDO.code,
                qMktDiscountGiftDO.name,
                qMktDiscountGiftDO.validStime,
                qMktDiscountGiftDO.validEtime,
                qMktDiscountGiftDO.customLevel,
                qMktDiscountGiftDO.ouId,
                qMktDiscountGiftDO.ouCode,
                qMktDiscountGiftDO.ouName,
                qMktDiscountGiftDO.buId,
                qMktDiscountGiftDO.buCode,
                qMktDiscountGiftDO.buName,
                qMktDiscountGiftDO.type,
                qMktDiscountGiftDO.status,

                qMktDiscountGiftDDO.id.as("mktDiscountGiftDId"),
                qMktDiscountGiftDDO.itemId,
                qMktDiscountGiftDDO.itemCode,
                qMktDiscountGiftDDO.discountGiftCode,
                qMktDiscountGiftDDO.itemName,
                qMktDiscountGiftDDO.itemAttr,
                qMktDiscountGiftDDO.uom,
                qMktDiscountGiftDDO.limitAmt,
                qMktDiscountGiftDDO.limitNum

        ))
                .from(qMktDiscountGiftDO)
                .join(qMktDiscountGiftDDO).on(qMktDiscountGiftDO.id.eq(qMktDiscountGiftDDO.mktDiscountGiftId))
                .where(qMktDiscountGiftDO.deleteFlag.eq(ConstantsSale.COMMON_DELETE_NO))
                .where(qMktDiscountGiftDDO.deleteFlag.eq(ConstantsSale.COMMON_DELETE_NO))
                .where(qMktDiscountGiftDO.status.eq(UdcEnum.MKT_DISCOUNT_OFFSET_STATUS_ENABLE.getValueCode()))
                .where(qMktDiscountGiftDO.ouId.eq(ouId))
                .where(qMktDiscountGiftDDO.itemId.in(itemIds))
                .where(qMktDiscountGiftDO.validStime.before(date).or(qMktDiscountGiftDO.validStime.eq(date)))
                .where(qMktDiscountGiftDO.validEtime.after(date).or(qMktDiscountGiftDO.validEtime.eq(date)));
        if (!StringUtils.isEmpty(customLevel)) {
            jpaQuery.where(qMktDiscountGiftDO.customLevel.eq(customLevel).or(qMktDiscountGiftDO.customLevel.isNull()));
        }
        return jpaQuery;
    }

    public JPAQuery<MktDiscountGiftVO> findMktDisCountGiftByLQ(Long ouId, String customLevel, LocalDateTime date, List<Long> itemIds) {

        JPAQuery<MktDiscountGiftVO> jpaQuery = jpaQueryFactory.select(Projections.bean(MktDiscountGiftVO.class,
                qMktDiscountGiftDO.id.as("mktDiscountGiftId"),
                qMktDiscountGiftDO.code,
                qMktDiscountGiftDO.name,
                qMktDiscountGiftDO.validStime,
                qMktDiscountGiftDO.validEtime,
                qMktDiscountGiftDO.customLevel,
                qMktDiscountGiftDO.ouId,
                qMktDiscountGiftDO.ouCode,
                qMktDiscountGiftDO.ouName,
                qMktDiscountGiftDO.buId,
                qMktDiscountGiftDO.buCode,
                qMktDiscountGiftDO.buName,
                qMktDiscountGiftDO.type,
                qMktDiscountGiftDO.status,

                qMktDiscountGiftDDO.id.as("mktDiscountGiftDId"),
                qMktDiscountGiftDDO.itemId,
                qMktDiscountGiftDDO.itemCode,
                qMktDiscountGiftDDO.discountGiftCode,
                qMktDiscountGiftDDO.itemName,
                qMktDiscountGiftDDO.itemAttr,
                qMktDiscountGiftDDO.uom,
                qMktDiscountGiftDDO.limitAmt,
                qMktDiscountGiftDDO.limitNum

        ))
                .from(qMktDiscountGiftDO)
                .join(qMktDiscountGiftDDO).on(qMktDiscountGiftDO.id.eq(qMktDiscountGiftDDO.mktDiscountGiftId))
                .where(qMktDiscountGiftDO.deleteFlag.eq(ConstantsSale.COMMON_DELETE_NO))
                .where(qMktDiscountGiftDDO.deleteFlag.eq(ConstantsSale.COMMON_DELETE_NO))
                .where(qMktDiscountGiftDO.status.eq(UdcEnum.MKT_DISCOUNT_OFFSET_STATUS_ENABLE.getValueCode()))
                .where(qMktDiscountGiftDO.ouId.eq(ouId))
                .where(qMktDiscountGiftDDO.itemId.in(itemIds))
                .where(qMktDiscountGiftDO.validStime.before(date).or(qMktDiscountGiftDO.validStime.eq(date)))
                .where(qMktDiscountGiftDO.validEtime.after(date).or(qMktDiscountGiftDO.validEtime.eq(date)));
        if (!StringUtils.isEmpty(customLevel)) {
            jpaQuery.where(qMktDiscountGiftDO.customLevel.eq(customLevel));
        }else {
            jpaQuery.where(qMktDiscountGiftDO.customLevel.eq("").or(qMktDiscountGiftDO.customLevel.isNull()));
        }
        return jpaQuery;
    }

    /**
     * 查询主表
     *
     * @param queryParam 查询条件,可为空
     * @return jpaQuery
     */
    public JPAQuery<MktDiscountGiftRespVO> select(MktDiscountGiftQueryVO queryParam) {
        JPAQuery<MktDiscountGiftRespVO> jpaQuery = jpaQueryFactory.select(Projections.bean(MktDiscountGiftRespVO.class,
                qMktDiscountGiftDO.id, qMktDiscountGiftDO.code, qMktDiscountGiftDO.name, qMktDiscountGiftDO.validStime,
                qMktDiscountGiftDO.validEtime, qMktDiscountGiftDO.ouId, qMktDiscountGiftDO.ouCode, qMktDiscountGiftDO.ouName,
                qMktDiscountGiftDO.buId, qMktDiscountGiftDO.buCode, qMktDiscountGiftDO.buName, qMktDiscountGiftDO.customLevel,
                qMktDiscountGiftDO.type, qMktDiscountGiftDO.status
        ))
                .from(qMktDiscountGiftDO)
                .where(qMktDiscountGiftDO.deleteFlag.eq(ConstantsSale.COMMON_DELETE_NO))
                // .where(DataAuthJpaUtil.dataAuthJpaPredicate(qMktDiscountGiftDO.getMetadata())) //数据权限
                ;
        if (queryParam != null) {
            jpaQuery.where(buildCondition(queryParam));
        }
        return jpaQuery;
    }

    /**
     * 查询条件
     *
     * @param queryParam 查询条件
     * @return predicate
     */
    private Predicate buildCondition(MktDiscountGiftQueryVO queryParam) {
        Predicate predicate = qMktDiscountGiftDO.isNotNull();
        if (!StringUtils.isEmpty(queryParam.getCode())) {
            predicate = ExpressionUtils.and(predicate, qMktDiscountGiftDO.code.like("%" + queryParam.getCode() + "%"));
        }
        if (!StringUtils.isEmpty(queryParam.getItemCode())){
            predicate = ExpressionUtils.and(predicate, qMktDiscountGiftDO.code.in(queryParam.getCodeByItem()));
        }
        if (!StringUtils.isEmpty(queryParam.getGiftCode())){
            predicate = ExpressionUtils.and(predicate, qMktDiscountGiftDO.code.in(queryParam.getCodeByGift()));
        }
        if (!StringUtils.isEmpty(queryParam.getName())) {
            predicate = ExpressionUtils.and(predicate, qMktDiscountGiftDO.name.like("%" + queryParam.getName() + "%"));
        }
        if (!StringUtils.isEmpty(queryParam.getValidStime())) {
            predicate = ExpressionUtils.and(predicate, qMktDiscountGiftDO.validStime.eq(queryParam.getValidStime())
                    .or(qMktDiscountGiftDO.validStime.after(queryParam.getValidStime())));
        }
        if (!StringUtils.isEmpty(queryParam.getValidEtime())) {
            predicate = ExpressionUtils.and(predicate, qMktDiscountGiftDO.validEtime.eq(queryParam.getValidEtime())
                    .or(qMktDiscountGiftDO.validEtime.before(queryParam.getValidEtime())));
        }
        if (!StringUtils.isEmpty(queryParam.getCustomLevel())) {
            predicate = ExpressionUtils.and(predicate, qMktDiscountGiftDO.customLevel.eq(queryParam.getCustomLevel()).or(qMktDiscountGiftDO.customLevel.isNull()));
        }
        if (!StringUtils.isEmpty(queryParam.getOuId())) {
            predicate = ExpressionUtils.and(predicate, qMktDiscountGiftDO.ouId.eq(queryParam.getOuId()));
        }
        if (!StringUtils.isEmpty(queryParam.getBuId())) {
            predicate = ExpressionUtils.and(predicate, qMktDiscountGiftDO.buId.eq(queryParam.getBuId()));
        }
        if (!StringUtils.isEmpty(queryParam.getType())) {
            predicate = ExpressionUtils.and(predicate, qMktDiscountGiftDO.type.eq(queryParam.getType()));
        }
        if (!StringUtils.isEmpty(queryParam.getStatus())) {
            predicate = ExpressionUtils.and(predicate, qMktDiscountGiftDO.status.eq(queryParam.getStatus()));
        }
        return predicate;
    }


    /**
     * 查询明细
     *
     * @param id   主表ID，可为空
     * @param code 主表CODE，可为空
     * @return jpaQuery
     */
    public JPAQuery<MktDiscountGiftRespDVO> selectDetailsByMasIdOrCode(Long id, String code) {
        JPAQuery<MktDiscountGiftRespDVO> jpaQuery = jpaQueryFactory.select(Projections.bean(MktDiscountGiftRespDVO.class,
                qMktDiscountGiftDDO.id, qMktDiscountGiftDDO.mktDiscountGiftId, qMktDiscountGiftDDO.itemId, qMktDiscountGiftDDO.itemCode,
                qMktDiscountGiftDDO.itemName, qMktDiscountGiftDDO.itemAttr, qMktDiscountGiftDDO.uom, qMktDiscountGiftDDO.limitAmt,
                qMktDiscountGiftDDO.limitNum))
                .from(qMktDiscountGiftDDO)
                .where(qMktDiscountGiftDDO.deleteFlag.eq(ConstantsSale.COMMON_DELETE_NO))
                // .where(DataAuthJpaUtil.dataAuthJpaPredicate(qMktDiscountGiftDDO.getMetadata())) //数据权限
                ;
        if (id != null) {
            jpaQuery.where(qMktDiscountGiftDDO.mktDiscountGiftId.eq(id));
        }
        if (!StringUtils.isEmpty(code)) {
            jpaQuery.where(qMktDiscountGiftDDO.discountGiftCode.eq(code));
        }
        return jpaQuery;
    }

    /**
     * 查询赠品明细
     * 用法: 1.只传code: 查询该活动全部的赠品
     * 2.只传discountGiftItemId：查询该discountGiftItemId 绑定的全部赠品信息（若多活动配置过该discountGiftItemId，则查全部活动的赠品）
     * 3.传code和discountGiftItemId：查询该code活动下discountGiftItemId 绑定的赠品
     * 4.不传参数：查询全部的赠品信息
     *
     * @param code               主表code
     * @param discountGiftItemId 明细表商品ID， 查询该商品ID的全部赠品
     * @return jpaQuery
     */
    public JPAQuery<MktGiftRespVO> selectGiftDetailsByMasCodeOrItemId(String code, Long discountGiftItemId) {
        JPAQuery<MktGiftRespVO> jpaQuery = jpaQueryFactory.select(Projections.bean(MktGiftRespVO.class,
                qMktGiftDO.id, qMktGiftDO.itemId, qMktGiftDO.itemCode, qMktGiftDO.itemName,
                qMktGiftDO.validStime, qMktGiftDO.validEtime, qMktGiftDO.stockNum, qMktGiftDO.stockNowNum, qMktGiftDO.discountGiftItemId,
                qMktGiftDO.discountGiftCode, qMktGiftDO.skuCode, qMktGiftDO.skuName, qMktGiftDO.materiel, qMktGiftDO.type
        ))
                .from(qMktGiftDO)
                .where(qMktGiftDO.deleteFlag.eq(ConstantsSale.COMMON_DELETE_NO))
                // .where(DataAuthJpaUtil.dataAuthJpaPredicate(qMktGiftDO.getMetadata())) //数据权限
                ;
        if (!StringUtils.isEmpty(code)) {
            jpaQuery.where(qMktGiftDO.discountGiftCode.eq(code));
        }
        if (discountGiftItemId != null) {
            jpaQuery.where(qMktGiftDO.discountGiftItemId.eq(discountGiftItemId));
        }
        return jpaQuery;
    }

    /**
     * 更新促销状态 标记
     *
     * @param ids    id集合
     * @param status 促销状态
     */
    public void updateStatusBatch(List<Long> ids, String status) {
        jpaQueryFactory.update(qMktDiscountGiftDO)
                .set(qMktDiscountGiftDO.status, status)
                .where(qMktDiscountGiftDO.id.in(ids))
                // .where(DataAuthJpaUtil.dataAuthJpaPredicate(qMktDiscountGiftDO.getMetadata()))
                .execute();
    }

    /**
     * 更新逻辑删除 标记
     *
     * @param codeList   code集合
     * @param deleteFlag 删除状态
     */
    public void updateDeleteFlagBatch(List<String> codeList, int deleteFlag) {
        jpaQueryFactory.update(qMktDiscountGiftDO)
                .set(qMktDiscountGiftDO.deleteFlag, deleteFlag)
                .where(qMktDiscountGiftDO.code.in(codeList))
                // .where(DataAuthJpaUtil.dataAuthJpaPredicate(qMktDiscountGiftDO.getMetadata()))
                .execute();

        jpaQueryFactory.update(qMktDiscountGiftDDO)
                .set(qMktDiscountGiftDDO.deleteFlag, deleteFlag)
                .where(qMktDiscountGiftDDO.discountGiftCode.in(codeList))
                // .where(DataAuthJpaUtil.dataAuthJpaPredicate(qMktDiscountGiftDDO.getMetadata()))
                .execute();
        jpaQueryFactory.update(qMktGiftDO)
                .set(qMktGiftDO.deleteFlag, deleteFlag)
                .where(qMktGiftDO.discountGiftCode.in(codeList))
                // .where(DataAuthJpaUtil.dataAuthJpaPredicate(qMktGiftDO.getMetadata()))
                .execute();
    }

    /**
     * 根据id获取 code集合
     *
     * @param idList id集合
     * @return code集合
     */
    public List<String> getCodeById(List<Long> idList) {
        return jpaQueryFactory.select(qMktDiscountGiftDO.code).distinct().from(qMktDiscountGiftDO)
                .where(qMktDiscountGiftDO.id.in(idList))
                .where(qMktDiscountGiftDO.deleteFlag.eq(ConstantsSale.COMMON_DELETE_NO)).fetch()
//                .where(DataAuthJpaUtil.dataAuthJpaPredicate(qMktDiscountGiftDDO.getMetadata()))
                ;
    }

    public List<Long> getOuIdById(Long id){
        return jpaQueryFactory.select(qMktDiscountGiftDO.ouId).from(qMktDiscountGiftDO)
                .where(qMktDiscountGiftDO.id.eq(id)).fetch();
    }

    public List<Long> getGiftId(Long ouId){
        return jpaQueryFactory.select(qMktDiscountGiftDO.id).from(qMktDiscountGiftDO)
                .where(qMktDiscountGiftDO.ouId.eq(ouId))
                .where(qMktDiscountGiftDO.status.eq("ENABLE"))
                .where(qMktDiscountGiftDO.deleteFlag.eq(0))
                .fetch();
    }

    public List<Long> getGiftDetailSkuId(List<Long> ids){
        return jpaQueryFactory.select(qMktDiscountGiftDDO.itemId).from(qMktDiscountGiftDDO)
                .where(qMktDiscountGiftDDO.mktDiscountGiftId.in(ids))
                .where(qMktDiscountGiftDDO.deleteFlag.eq(0))
                .fetch();
    }

    public List<String> getGiftDetailCode(List<Long> ids){
        return jpaQueryFactory.select(qMktDiscountGiftDDO.itemCode).from(qMktDiscountGiftDDO)
                .where(qMktDiscountGiftDDO.mktDiscountGiftId.in(ids))
                .where(qMktDiscountGiftDDO.deleteFlag.eq(0))
                .fetch();
    }


    public List<Long> getGiftSkuId(Long id){
        return jpaQueryFactory.select(qMktDiscountGiftDDO.itemId).from(qMktDiscountGiftDDO)
                .where(qMktDiscountGiftDDO.mktDiscountGiftId.eq(id)).fetch();
    }

    public List<String> getGiftSkuCode(Long id){
        return jpaQueryFactory.select(qMktDiscountGiftDDO.itemCode).from(qMktDiscountGiftDDO)
                .where(qMktDiscountGiftDDO.mktDiscountGiftId.eq(id)).fetch();
    }


    /**
     * 删除(主/附)表数据
     *
     * @param codeList 删除的主表编码集合
     */
    public void deleteIdBatch(List<String> codeList) {

        jpaQueryFactory.delete(qMktDiscountGiftDO)
                .where(qMktDiscountGiftDO.code.in(codeList))
                // .where(DataAuthJpaUtil.dataAuthJpaPredicate(qMktDiscountGiftDO.getMetadata()))
                .execute();
        jpaQueryFactory.delete(qMktDiscountGiftDDO)
                .where(qMktDiscountGiftDDO.discountGiftCode.in(codeList))
                // .where(DataAuthJpaUtil.dataAuthJpaPredicate(qMktDiscountGiftDDO.getMetadata()))
                .execute();
        jpaQueryFactory.delete(qMktGiftDO)
                .where(qMktGiftDO.discountGiftCode.in(codeList))
                // .where(DataAuthJpaUtil.dataAuthJpaPredicate(qMktDiscountGiftDDO.getMetadata()))
                .execute();

    }

    /**
     * 根据 id 删除明细信息
     *
     * @param id
     */
    public void deleteDetailInfo(Long id) {
        jpaQueryFactory.delete(qMktDiscountGiftDDO)
                .where(qMktDiscountGiftDDO.id.eq(id))
                // .where(DataAuthJpaUtil.dataAuthJpaPredicate(qMktDiscountGiftDDO.getMetadata()))
                .execute();
    }

    /**
     * 根据 discountGiftCode、discountGiftItemId 删除赠品信息
     *
     * @param discountGiftCode
     * @param discountGiftItemId
     */
    public void deleteGiftInfo(String discountGiftCode, Long discountGiftItemId) {
        jpaQueryFactory.delete(qMktGiftDO)
                .where(qMktGiftDO.discountGiftCode.eq(discountGiftCode))
                .where(qMktGiftDO.discountGiftItemId.eq(discountGiftItemId))
                // .where(DataAuthJpaUtil.dataAuthJpaPredicate(qMktGiftDO.getMetadata()))
                .execute();
    }

    /**
     * 根据 id 删除赠品信息
     *
     * @param idList
     */
    public void deleteGiftInfo(List<Long> idList) {
        jpaQueryFactory.delete(qMktGiftDO)
                .where(qMktGiftDO.id.in(idList))
                // .where(DataAuthJpaUtil.dataAuthJpaPredicate(qMktGiftDO.getMetadata()))
                .execute();
    }

    /**
     * 校验  有效的时间区间内，是否存在相同的商品规则
     *
     * @param masId       可为null
     * @param ouId        公司ID
     * @param customLevel 客户级别
     * @param validStime  生效日期
     * @param validEtime  失效日期
     * @param itemIds     商品集合
     * @return 重复的促销编号、商品编码
     */
    public List<Tuple> checkItemExitsInValidDate(Long masId, Long ouId, String customLevel, LocalDateTime validStime, LocalDateTime validEtime, List<Long> itemIds) {
        JPAQuery<Tuple> jpaQuery = jpaQueryFactory.select(qMktDiscountGiftDO.code, qMktDiscountGiftDDO.itemCode).distinct()
                .from(qMktDiscountGiftDDO)
                .join(qMktDiscountGiftDO)
                .on(qMktDiscountGiftDDO.mktDiscountGiftId.eq(qMktDiscountGiftDO.id))
                .where(qMktDiscountGiftDO.deleteFlag.eq(ConstantsSale.COMMON_DELETE_NO))
                .where(qMktDiscountGiftDDO.deleteFlag.eq(ConstantsSale.COMMON_DELETE_NO))
                .where(qMktDiscountGiftDO.ouId.eq(ouId))
                .where(qMktDiscountGiftDDO.itemId.in(itemIds))
                .where(qMktDiscountGiftDO.validStime.between(validStime, validEtime)
                        .or(qMktDiscountGiftDO.validEtime.between(validStime, validEtime))
                        .or(qMktDiscountGiftDO.validStime.before(validStime).and(qMktDiscountGiftDO.validEtime.after(validEtime)))
                );
        if (!StringUtils.isEmpty(customLevel)) {
            jpaQuery.where(qMktDiscountGiftDO.customLevel.eq(customLevel).or(qMktDiscountGiftDO.customLevel.isNull()));
        }
        if (masId != null) {
            jpaQuery.where(qMktDiscountGiftDO.id.ne(masId));
        }
        return jpaQuery.fetch();
    }


    /**
     * 查询到符合天健的折扣
     * @param ouId
     * @param customLevel
     * @param date
     * @param itemId
     * @param num
     * @param amt
     * @return
     */
    public JPAQuery<MktDiscountGiftVO> findMatchMktDisCountGift(Long ouId, String customLevel, LocalDateTime date, Long itemId, Long num, BigDecimal amt) {

        JPAQuery<MktDiscountGiftVO> jpaQuery = jpaQueryFactory.select(Projections.bean(MktDiscountGiftVO.class,
                qMktDiscountGiftDO.id.as("mktDiscountGiftId"),
                qMktDiscountGiftDO.code,
                qMktDiscountGiftDO.name,
                qMktDiscountGiftDO.validStime,
                qMktDiscountGiftDO.validEtime,
                qMktDiscountGiftDO.customLevel,
                qMktDiscountGiftDO.ouId,
                qMktDiscountGiftDO.ouCode,
                qMktDiscountGiftDO.ouName,
                qMktDiscountGiftDO.buId,
                qMktDiscountGiftDO.buCode,
                qMktDiscountGiftDO.buName,
                qMktDiscountGiftDO.type,
                qMktDiscountGiftDO.status,

                qMktDiscountGiftDDO.id.as("mktDiscountGiftDId"),
                qMktDiscountGiftDDO.itemId,
                qMktDiscountGiftDDO.itemCode,
                qMktDiscountGiftDDO.itemName,
                qMktDiscountGiftDDO.itemAttr,
                qMktDiscountGiftDDO.uom,
                qMktDiscountGiftDDO.limitAmt,
                qMktDiscountGiftDDO.limitNum

        ))
                .from(qMktDiscountGiftDO)
                .join(qMktDiscountGiftDDO).on(qMktDiscountGiftDO.id.eq(qMktDiscountGiftDDO.mktDiscountGiftId))
                .where(qMktDiscountGiftDO.deleteFlag.eq(ConstantsSale.COMMON_DELETE_NO))
                .where(qMktDiscountGiftDDO.deleteFlag.eq(ConstantsSale.COMMON_DELETE_NO))
                .where(qMktDiscountGiftDO.status.eq(UdcEnum.MKT_DISCOUNT_OFFSET_STATUS_ENABLE.getValueCode()))
                .where(qMktDiscountGiftDO.ouId.eq(ouId))
                .where(qMktDiscountGiftDDO.itemId.in(itemId))
                .where(qMktDiscountGiftDO.validStime.before(date).or(qMktDiscountGiftDO.validStime.eq(date)))
                .where(qMktDiscountGiftDO.validEtime.after(date).or(qMktDiscountGiftDO.validEtime.eq(date)));
        if (!StringUtils.isEmpty(customLevel)) {
            jpaQuery.where(qMktDiscountGiftDO.customLevel.eq(customLevel).or(qMktDiscountGiftDO.customLevel.isNull()));
        }

        if(Objects.nonNull(num)){
            jpaQuery.where(qMktDiscountGiftDO.type.eq("0"));
            jpaQuery.where(qMktDiscountGiftDDO.limitNum.eq(num));

        }
        if(Objects.nonNull(amt)){
            jpaQuery.where(qMktDiscountGiftDO.type.goe("1"));
            jpaQuery.where(qMktDiscountGiftDDO.limitAmt.goe(amt));
        }
        return jpaQuery;
    }

    /**
     * 获取买赠赠品信息
     */
    public JPAQuery<BipGiftVO> getGiftList(Long ouId, String customLevel, LocalDateTime date, List<Long> skuids){

        val jpaQuery = jpaQueryFactory.select(Projections.bean(BipGiftVO.class,
                bipItemSku.bipItemId.as("itemId"),
                bipItemSku.id.as("skuId"),

                qMktDiscountGiftDDO.limitAmt,
                qMktDiscountGiftDDO.limitNum,

                qMktGiftDO.id,
                qMktGiftDO.itemId.as("giftId"),
                qMktGiftDO.itemCode.as("giftCode"),
                qMktGiftDO.itemName.as("giftName"),
                qMktGiftDO.stockNum,
                qMktGiftDO.stockNowNum,

                qMktDiscountGiftDO.type,
                qMktDiscountGiftDO.status,
                qMktDiscountGiftDO.code.as("mktCode"),
                qMktDiscountGiftDO.name.as("mktName")
        )).from(bipItemSku);
        jpaQuery.leftJoin(qMktDiscountGiftDDO).on(bipItemSku.id.eq(qMktDiscountGiftDDO.itemId));
        jpaQuery.leftJoin(qMktDiscountGiftDO).on(qMktDiscountGiftDO.id.eq(qMktDiscountGiftDDO.mktDiscountGiftId));
        jpaQuery.leftJoin(qMktGiftDO).on(qMktGiftDO.discountGiftCode.eq(qMktDiscountGiftDO.code).and(qMktGiftDO.discountGiftItemId.eq(bipItemSku.id)));
        jpaQuery.where(qMktGiftDO.isNotNull())
                .where(qMktDiscountGiftDO.ouId.eq(ouId))
                .where(qMktDiscountGiftDO.status.eq("ENABLE"))
                .where(bipItemSku.id.in(skuids))
                .where(qMktDiscountGiftDO.validStime.before(date).or(qMktDiscountGiftDO.validStime.eq(date)))
                .where(qMktDiscountGiftDO.validEtime.after(date).or(qMktDiscountGiftDO.validEtime.eq(date)));
        if (!StringUtils.isEmpty(customLevel)) {
            jpaQuery.where(qMktDiscountGiftDO.customLevel.eq(customLevel).or(qMktDiscountGiftDO.customLevel.isNull()));
        }
        return jpaQuery;
    }

    /**
     * 根据赠品code获取买赠赠品信息
     */
    public JPAQuery<BipGiftVO> getGiftListByCode(List<Long> giftCodes, List<Long> itemIds){
        LocalDateTime date = LocalDateTime.now();
        val jpaQuery = jpaQueryFactory.select(Projections.bean(BipGiftVO.class,
                bipItemSku.bipItemId.as("itemId"),
                bipItemSku.id.as("skuId"),
                bipItemSku.skuCode,

                qMktDiscountGiftDDO.limitAmt,
                qMktDiscountGiftDDO.limitNum,

                qMktGiftDO.id,
                qMktGiftDO.itemId.as("giftId"),
                qMktGiftDO.itemCode.as("giftCode"),
                qMktGiftDO.itemName.as("giftName"),
                qMktGiftDO.stockNum,
                qMktGiftDO.stockNowNum,

                qMktDiscountGiftDO.type,
                qMktDiscountGiftDO.status,
                qMktDiscountGiftDO.code.as("mktCode"),
                qMktDiscountGiftDO.name.as("mktName")
        )).from(bipItemSku);
        jpaQuery.leftJoin(qMktDiscountGiftDDO).on(bipItemSku.id.eq(qMktDiscountGiftDDO.itemId));
        jpaQuery.leftJoin(qMktDiscountGiftDO).on(qMktDiscountGiftDO.id.eq(qMktDiscountGiftDDO.mktDiscountGiftId));
        jpaQuery.leftJoin(qMktGiftDO).on(qMktGiftDO.discountGiftCode.eq(qMktDiscountGiftDO.code).and(qMktGiftDO.discountGiftItemId.eq(bipItemSku.id)));
        jpaQuery.where(qMktGiftDO.isNotNull());
        if(giftCodes != null && giftCodes.size()>0){
            //根据赠品表id获取信息
            jpaQuery.where(qMktGiftDO.id.in(giftCodes));
        }
        if(itemIds != null && itemIds.size()>0){
            //根据商品id获取赠品-需考虑买赠规则的有效期和是否启用
            jpaQuery.where(bipItemSku.bipItemId.in(itemIds))
                    .where(qMktDiscountGiftDO.validStime.before(date).or(qMktDiscountGiftDO.validStime.eq(date)))
                    .where(qMktDiscountGiftDO.validEtime.after(date).or(qMktDiscountGiftDO.validEtime.eq(date)))
                    .where(qMktGiftDO.stockNowNum.isNotNull().and(qMktGiftDO.stockNowNum.ne(0L)))
                    .where(qMktDiscountGiftDO.status.eq("ENABLE"));
        }
        jpaQuery.groupBy(qMktGiftDO.id);
        return jpaQuery;
    }

    public List<String> findDiscountCode(String itemCode){
        if (itemCode==null){
            return new ArrayList<>();
        }
        return jpaQueryFactory.select(qMktDiscountGiftDDO.discountGiftCode)
                .from(qMktDiscountGiftDDO)
                .where(qMktDiscountGiftDDO.itemCode.like("%"+itemCode+"%"))
                .fetch();
    }

}
