package com.elitesland.yst.production.sale.controller.shop;

import cn.hutool.core.lang.Assert;
import com.el.coordinator.boot.fsm.service.FileService;
import com.elitesland.yst.production.sale.api.service.shop.BipItemMngService;
import com.elitesland.yst.production.sale.api.vo.param.shop.*;
import com.elitesland.yst.production.sale.api.vo.resp.shop.*;
import com.elitesland.yst.production.sale.api.vo.save.shop.*;
import com.elitesland.yst.production.sale.controller.BaseController;
import com.elitescloud.cloudt.common.base.ApiResult;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitescloud.boot.exception.BusinessException;
import com.github.xiaoymin.knife4j.annotations.ApiOperationSupport;
import com.github.xiaoymin.knife4j.annotations.ApiSupport;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiImplicitParam;
import io.swagger.annotations.ApiImplicitParams;
import io.swagger.annotations.ApiOperation;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import net.lingala.zip4j.ZipFile;
import org.springframework.http.HttpEntity;
import org.springframework.http.MediaType;
import org.springframework.web.bind.annotation.*;
import org.springframework.web.multipart.MultipartFile;
import org.springframework.web.servlet.mvc.method.annotation.StreamingResponseBody;

import javax.validation.Valid;
import javax.validation.constraints.NotEmpty;
import java.io.File;
import java.io.FileOutputStream;
import java.nio.charset.Charset;
import java.util.List;
import java.util.Objects;

/**
 * <p>
 * 功能说明
 * </p>
 *
 * @author Shadow
 * @since 2021-08-10 17:07:37
 */
@Slf4j
@Api(tags = {"商品管理"})
@ApiSupport(author = "Kaiser（wang shao）")
@RestController
@RequiredArgsConstructor
@RequestMapping(value = "/bip/item", produces = MediaType.APPLICATION_JSON_VALUE)
public class BipItemMngController extends BaseController {

    private final BipItemMngService itemMngService;

    private final FileService fileService;

    @ApiOperation(value = "商品保存")
    @ApiOperationSupport(order = 1)
    @PostMapping(value = "")
    public ApiResult<Long> save(@RequestBody @Valid BipItemSaveVO saveVO) {
        return itemMngService.save(saveVO);
    }

    @ApiOperation(value = "商品修改")
    @ApiOperationSupport(order = 2)
    @ApiImplicitParams({
            @ApiImplicitParam(name = "id", value = "商品ID")
    })
    @PutMapping(value = "/{id}")
    public ApiResult<Long> update(@PathVariable Long id, @RequestBody @Valid BipItemUpdateVO updateVO) {
        return itemMngService.update(id, updateVO);
    }

    @ApiOperation(value = "商品获取")
    @ApiOperationSupport(order = 3)
    @ApiImplicitParams({
            @ApiImplicitParam(name = "id", value = "商品ID")
    })
    @GetMapping(value = "/{id}")
    public ApiResult<BipItemDetailRespVO> get(@PathVariable Long id) {
        return itemMngService.get(id);
    }

    @ApiOperation(value = "商品删除")
    @ApiOperationSupport(order = 4)
    @ApiImplicitParams({
            @ApiImplicitParam(name = "id", value = "商品ID")
    })
    @DeleteMapping(value = "/{id}")
    public ApiResult<Long> delete(@PathVariable Long id) {
        return itemMngService.delete(id);
    }

    @ApiOperation(value = "商品批量删除")
    @ApiOperationSupport(order = 4)
    @ApiImplicitParams({
            @ApiImplicitParam(name = "ids", value = "商品ID", allowMultiple = true, paramType = "body", dataType = "int")
    })
    @DeleteMapping(value = "/batch")
    public ApiResult<List<Long>> deleteBatch(@RequestBody List<Long> ids) {
        return itemMngService.delete(ids);
    }

    @ApiOperation(value = "批量立即上架")
    @ApiOperationSupport(order = 5)
    @ApiImplicitParams({
            @ApiImplicitParam(name = "ids", value = "商品ID", allowMultiple = true, paramType = "body", dataType = "int")
    })
    @PostMapping(value = "/batch/onShelf")
    public ApiResult<List<Long>> onShelf(@RequestBody @NotEmpty(message = "商品ID为空") List<Long> ids) {
        return itemMngService.updateShelf(ids, true);
    }

    @ApiOperation(value = "批量立即下架")
    @ApiOperationSupport(order = 5)
    @ApiImplicitParams({
            @ApiImplicitParam(name = "ids", value = "商品ID", allowMultiple = true, paramType = "body", dataType = "int")
    })
    @PostMapping(value = "/batch/offShelf")
    public ApiResult<List<Long>> offShelf(@RequestBody @NotEmpty(message = "商品ID为空") List<Long> ids) {
        return itemMngService.updateShelf(ids, false);
    }

    @ApiOperation(value = "审批商品")
    @ApiOperationSupport(order = 6)
    @PostMapping(value = "/approve")
    public ApiResult<List<Long>> approve(@RequestBody @Valid BipItemApproveSaveVO approveSaveVO) {
        return itemMngService.updateApprove(approveSaveVO);
    }

    @ApiOperation(value = "批量定时上架")
    @ApiOperationSupport(order = 7)
    @PostMapping(value = "/timing/onShelf")
    public ApiResult<List<Long>> timingOnShelf(@RequestBody @Valid BipItemShelfOnSaveVO shelfSaveVO) {
        return itemMngService.updateOnShelfTiming(shelfSaveVO);
    }

    @ApiOperation(value = "批量定时下架")
    @ApiOperationSupport(order = 7)
    @PostMapping(value = "/timing/offShelf")
    public ApiResult<List<Long>> timingOffShelf(@RequestBody @Valid BipItemShelfOffSaveVO shelfSaveVO) {
        return itemMngService.updateOffShelfTiming(shelfSaveVO);
    }

    @ApiOperation(value = "调整最小起订量")
    @ApiOperationSupport(order = 8)
    @PostMapping(value = "/limitBuy")
    public ApiResult<List<Long>> updateLimitBuy(@RequestBody @Valid BipItemLimitBuySaveVO limitBuySaveVO) {
        return itemMngService.updateLimitBuy(limitBuySaveVO);
    }

    @ApiOperation(value = "获取商品的SKU信息")
    @ApiOperationSupport(order = 9)
    @ApiImplicitParams({
            @ApiImplicitParam(name = "id", value = "商品ID")
    })
    @GetMapping(value = "/{id}/sku")
    public ApiResult<List<BipItemSkuRespVO>> getSku(@PathVariable Long id) {
        return itemMngService.getSku(id);
    }

    @ApiOperation(value = "调整商品的库存")
    @ApiOperationSupport(order = 10)
    @ApiImplicitParams({
            @ApiImplicitParam(name = "id", value = "商品ID")
    })
    @PostMapping(value = "/{id}/stock")
    public ApiResult<Long> updateSkuStock(@PathVariable Long id, @RequestBody @Valid List<BipItemStockSaveVO> stockSaveVOList) {
        return itemMngService.updateSkuStock(id, stockSaveVOList);
    }

    @ApiOperation(value = "获取商品计量单位的转换关系")
    @ApiOperationSupport(order = 20)
    @ApiImplicitParams({
            @ApiImplicitParam(name = "ids", value = "商品中心的商品ID", required = true, allowMultiple = true, paramType = "query", dataType = "int")
    })
    @GetMapping(value = "/uom/conversion")
    public ApiResult<List<BipItemUomConvertRespVO>> uomConversion(@RequestParam(name = "ids", required = false) List<Long> ids) {
        return itemMngService.queryItemUomConvert(ids);
    }

    @ApiOperation(value = "待上架商品列表")
    @ApiOperationSupport(order = 21)
    @PostMapping(value = "/search/toShelf")
    public ApiResult<PagingVO<BipItemToShelfRespVO>> queryToShelf(@RequestBody BipItemToShelfQueryParamVO queryParamVO) {
        return itemMngService.queryForToShelf(queryParamVO);
    }

    @ApiOperation(value = "待审批商品列表")
    @ApiOperationSupport(order = 22)
    @PostMapping(value = "/search/toApprove")
    public ApiResult<PagingVO<BipItemApproveRespVO>> queryToApprove(@RequestBody BipItemApproveQueryParamVO queryParamVO) {
        return itemMngService.queryForApprove(queryParamVO);
    }

    @ApiOperation(value = "已上架商品列表")
    @ApiOperationSupport(order = 23)
    @PostMapping(value = "/search/onShelf")
    public ApiResult<PagingVO<BipItemOnShelfRespVO>> queryOnShelf(@RequestBody BipItemOnShelfQueryParamVO queryParamVO) {
        return itemMngService.queryForOnShelf(queryParamVO);
    }

    @ApiOperation(value = "已下架商品列表")
    @ApiOperationSupport(order = 23)
    @PostMapping(value = "/search/offShelf")
    public ApiResult<PagingVO<BipItemOffShelfRespVO>> queryOffShelf(@RequestBody BipItemOffShelfQueryParamVO queryParamVO) {
        return itemMngService.queryForOffShelf(queryParamVO);
    }

    @ApiOperation(value = "查询商品（SKU级别）")
    @ApiOperationSupport(order = 31)
    @PostMapping(value = "/search/sku")
    public ApiResult<PagingVO<BipItemQueryRespVO>> query(@RequestBody BipItemQueryParamVO queryParamVO) {
        return itemMngService.query(queryParamVO);
    }

    @ApiOperation(value = "根据商品分类查询商品（SKU级别）")
    @ApiOperationSupport(order = 32)
    @ApiImplicitParams({
            @ApiImplicitParam(name = "categoryCode", value = "商品分类编码", paramType = "query"),
            @ApiImplicitParam(name = "ouId", value = "公司ID", paramType = "query"),
            @ApiImplicitParam(name = "onShelf", value = "是否已上架", paramType = "query"),
    })
    @GetMapping(value = "/searchByCat/sku")
    public ApiResult<List<BipItemQueryRespVO>> queryByCategory(String categoryCode, Long ouId, Boolean onShelf) {
        return itemMngService.queryByCategory(ouId, categoryCode, onShelf);
    }

    @PostMapping("/uploadBatch")
    @ApiOperationSupport(order = 33)
    @ApiOperation("批量上传商品图片")
    public ApiResult<String> fileUpload(@RequestBody @Valid ItemImageParamVO imageParamVO) throws Exception {
        HttpEntity<StreamingResponseBody> download = fileService.download(imageParamVO.getFileCode(), null);
        StreamingResponseBody body = download.getBody();
        Assert.notNull(body, "无法下载文件【"+imageParamVO.getFileCode()+"】！");

        File inFile = new File("/item-image-cache/" + imageParamVO.getFileCode());
        File outFile = new File("/item-image-cache/" + imageParamVO.getFileCode() + ".zip");

        if (!inFile.exists()){
            inFile.mkdirs();
        }
        if (!outFile.exists()) {
            outFile.createNewFile();
        }

        try {
            body.writeTo(new FileOutputStream(outFile));

            ZipFile zFile = new ZipFile(outFile.getPath());
            zFile.setCharset(Charset.forName("GBK"));
            zFile.extractAll(inFile.getPath());

            itemMngService.handleImage(new File(inFile.getPath()+"/"+imageParamVO.getOuCode()), imageParamVO.getOuCode());
        } catch (Exception e) {
            log.error("商品图片处理异常：{}", e.getMessage());
            throw new BusinessException(e.getMessage());
        } finally {
            outFile.delete();
            deleteAllFilesOfDir(inFile);
        }

        return ApiResult.ok();
    }

    private void deleteAllFilesOfDir(File path) {
        if (!path.exists())
            return;
        if (path.isFile()) {
            path.delete();
            return;
        }
        File[] files = path.listFiles();
        for (int i = 0; i < files.length; i++) {
            deleteAllFilesOfDir(files[i]);
        }
        path.delete();
    }

}