package com.elitesland.yst.production.sale.core.service;

import cn.hutool.core.util.ObjectUtil;
import cn.hutool.core.util.StrUtil;
import com.alibaba.cloud.commons.lang.StringUtils;
import com.el.coordinator.boot.fsm.model.vo.FileObjRespVO;
import com.el.coordinator.boot.fsm.service.FileService;
import com.elitescloud.boot.auth.util.SecurityContextUtil;
import com.elitescloud.boot.core.base.UdcProvider;
import com.elitesland.yst.production.sale.common.constant.UdcEnum;
import com.elitesland.yst.production.sale.core.config.SysConfigProperties;
import com.elitescloud.cloudt.common.base.ApiResult;
import com.elitescloud.boot.exception.BusinessException;
import com.elitescloud.cloudt.security.entity.GeneralUserDetails;
import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.querydsl.core.types.Order;
import com.querydsl.core.types.OrderSpecifier;
import com.querydsl.core.types.dsl.BooleanExpression;
import com.querydsl.core.types.dsl.EntityPathBase;
import com.querydsl.core.types.dsl.PathBuilder;
import com.querydsl.jpa.impl.JPAQuery;
import com.querydsl.jpa.impl.JPAQueryFactory;
import lombok.NonNull;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.MapUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.ApplicationEvent;
import org.springframework.context.ApplicationEventPublisher;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Sort;
import org.springframework.transaction.PlatformTransactionManager;

import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.concurrent.CompletableFuture;

/**
 * 服务实现类的基类.
 *
 * @author Kaiser（wang shao）
 * @date 2021-05-10
 */
@Slf4j
public abstract class BaseServiceImpl {

    /**
     * 判断文件是否存在
     *
     * @param fileCode 文件的唯一编码
     * @return 文件是否存在
     */
    protected boolean isExistsFile(@NonNull String fileCode) {
        com.el.coordinator.core.common.api.ApiResult<Boolean> queryResult = fileService.existsAll(fileCode);
        if (!queryResult.isSuccess()) {
            log.error("判断文件【{}】是否存在失败：{}", fileCode, queryResult.getMsg());
            throw new BusinessException("文件服务器异常，请稍后再试");
        }
        return queryResult.getData();
    }

    /**
     * 获取文件信息
     *
     * @param fileCode 文件的唯一编码
     * @return 文件信息
     */
    protected FileObjRespVO<?> getFile(@NonNull String fileCode) {
        com.el.coordinator.core.common.api.ApiResult<FileObjRespVO<?>> getResult = fileService.get(fileCode);
        if (getResult.isSuccess()) {
            return getResult.getData();
        }
        log.debug("获取文件【{}】信息失败：{}", fileCode, getResult.getMsg());
        return null;
    }

    /**
     * 获取文件信息
     *
     * @param fileCodes 文件的唯一编码
     * @return 文件信息
     */
    protected List<FileObjRespVO<?>> getFile(@NonNull List<String> fileCodes) {
        if (fileCodes.isEmpty()) {
            return Collections.emptyList();
        }
        com.el.coordinator.core.common.api.ApiResult<List<FileObjRespVO<?>>> getResult = fileService.query(fileCodes);
        if (getResult.isSuccess()) {
            return getResult.getData();
        }
        log.debug("获取文件【{}】信息失败：{}", fileCodes, getResult.getMsg());
        return Collections.emptyList();
    }

    /**
     * 获取UDC的编码和值
     *
     * @param udc UDC
     * @return 编码和值的映射
     */
    protected Map<String, String> convertUdc(UdcEnum udc) {
        if (udc == null) {
            return Collections.emptyMap();
        }
        return udcProvider.getValueMapByUdcCode("yst-sale", udc.getCode());
    }

    /**
     * 处理分页参数
     *
     * @param pageRequest 查询请求
     * @param defaultSort 默认排序
     * @return 分页参数
     */
    protected PageRequest wrapperPageRequest(PageRequest pageRequest, Sort defaultSort) {
        if (pageRequest == null) {
            return null;
        }

        Sort sort = pageRequest.getSort();
        if (sort.isUnsorted()) {
            sort = defaultSort == null ? Sort.by(Sort.DEFAULT_DIRECTION, "id") :
                    defaultSort.and(Sort.by("id"));
        }
        return PageRequest.of(pageRequest.getPageNumber(), pageRequest.getPageSize(), sort);
    }

    /**
     * 获取排序
     *
     * @param pageRequest 分页查询参数
     * @param entityPath  qdo
     * @param <T>         类型
     * @return 排序
     */
    protected <T extends EntityPathBase> OrderSpecifier<?>[] obtainOrder(PageRequest pageRequest, T entityPath) {
        return obtainOrder(pageRequest, entityPath, new OrderSpecifier[0]);
    }

    /**
     * 获取排序
     *
     * @param pageRequest 分页查询参数
     * @param entityPath  qdo
     * @param <T>         类型
     * @return 排序
     */
    protected <T extends EntityPathBase> OrderSpecifier<?>[] obtainOrder(PageRequest pageRequest, T entityPath, OrderSpecifier<?>[] defaultOrder) {
        if (pageRequest == null) {
            return defaultOrder;
        }

        Sort sort = pageRequest.getSort();
        if (sort.isUnsorted()) {
            return defaultOrder;
        }
        PathBuilder<T> pathBuilder = new PathBuilder(entityPath.getClass(), entityPath.getMetadata());
        return sort.stream()
                .map(o -> new OrderSpecifier(o.getDirection().isAscending() ? Order.ASC : Order.DESC, pathBuilder.get(o.getProperty())))
                .toArray(OrderSpecifier<?>[]::new);
    }

    /**
     * 添加排序
     *
     * @param jpaQuery    jpaQuery
     * @param pageRequest 分页查询参数
     * @param entityPath  qdo
     * @param <T>         类型
     */
    protected <T extends EntityPathBase> void appendPageAndSort(JPAQuery<?> jpaQuery, PageRequest pageRequest, T entityPath) {
        jpaQuery.orderBy(obtainOrder(pageRequest, entityPath))
                .offset(pageRequest.getOffset())
                .limit(pageRequest.getPageSize());
    }

    /**
     * 追加并条件
     *
     * @param predicate 原条件
     * @param condition 新追加的条件
     * @return 最终条件
     */
    protected BooleanExpression appendAndCondition(BooleanExpression predicate, BooleanExpression condition) {
        return predicate == null ? condition : predicate.and(condition);
    }

    /**
     * 追加或条件
     *
     * @param predicate 原条件
     * @param condition 新追加的条件
     * @return 最终条件
     */
    protected BooleanExpression appendOrCondition(BooleanExpression predicate, BooleanExpression condition) {
        return predicate == null ? condition : predicate.or(condition);
    }

    /**
     * 对象转字符串
     *
     * @param object 待转换的对象
     * @return json字符串
     */
    protected String convertObj2Str(Object object) {
        if (object == null) {
            return "";
        }
        try {
            return objectMapper.writeValueAsString(object);
        } catch (JsonProcessingException e) {
            log.error("转换信息失败", e);
            throw new IllegalArgumentException("转换信息失败", e);
        }
    }

    /**
     * 字符串转对象
     *
     * @param jsonStr json字符串
     * @return 对象
     */
    protected <T> T convert2Obj(String jsonStr, Class<T> cls) {
        if (StrUtil.isBlank(jsonStr)) {
            return null;
        }
        try {
            return objectMapper.readValue(jsonStr, cls);
        } catch (JsonProcessingException e) {
            log.error("转换信息失败", e);
            throw new IllegalArgumentException("转换信息失败", e);
        }
    }

    /**
     * 字符串转字符串列表
     *
     * @param jsonStr json字符串
     * @return 对象列表
     */
    protected List<String> convert2List(String jsonStr) {
        if (StrUtil.isBlank(jsonStr)) {
            return Collections.emptyList();
        }
        try {
            return objectMapper.readValue(jsonStr, new TypeReference<>() {
            });
        } catch (JsonProcessingException e) {
            log.error("转换信息失败", e);
            throw new IllegalArgumentException("转换信息失败", e);
        }
    }

    /**
     * 字符串转对象列表
     *
     * @param jsonStr json字符串
     * @return 对象列表
     */
    protected <T> List<T> convert2List(String jsonStr, Class<T> cls) {
        if (StrUtil.isBlank(jsonStr)) {
            return Collections.emptyList();
        }
        try {
            return objectMapper.readValue(jsonStr, new TypeReference<>() {
            });
        } catch (JsonProcessingException e) {
            log.error("转换信息失败", e);
            throw new IllegalArgumentException("转换信息失败", e);
        }
    }

    /**
     * 断点处理业务结果
     *
     * @param apiResult 业务结果
     * @param defMsg    默认的错误消息
     */
    protected void assertApiResult(ApiResult<?> apiResult, String defMsg) {
        if (apiResult == null) {
            throw new BusinessException(ObjectUtil.defaultIfNull(defMsg, "操作失败"));
        }

        if (apiResult.isSuccess()) {
            return;
        }
        throw new BusinessException(ObjectUtil.defaultIfNull(apiResult.getMsg(), ObjectUtil.defaultIfNull(defMsg, "操作失败")));
    }

    /**
     * 打印失败时的结果
     *
     * @param apiResult 结果
     * @param errorMsg  消息前缀
     */
    protected void printApiResultOnFail(ApiResult<?> apiResult, String errorMsg, Object... param) {
        if (apiResult == null) {
            log.error(errorMsg, param);
            return;
        }
        if (!apiResult.isSuccess()) {
            log.error(errorMsg + "，" + apiResult.getMsg(), param);
        }
    }

    /**
     * 检查是否存在 udc value
     *      若value为null，返回true
     *
     * @param udcMap udcMap: 来源：var udcMap = sysUdcService.getCodeMap("", "");
     * @param value  检验值
     * @return 存在 true, 不存在 false
     */
    protected boolean checkUdcValueExit(Map<String, String> udcMap, String value) {
        if(value == null){
            return true;
        }
        if (StringUtils.isNotBlank(value) && !MapUtils.isEmpty(udcMap)) {
            return udcMap.keySet().stream().filter(key -> value.equals(key)).count() == 1;
        }
        return false;
    }

    /**
     * 同步发送事件
     *
     * @param event 事件
     */
    protected void publishEventSync(ApplicationEvent event) {
        eventPublisher.publishEvent(event);
    }

    /**
     * 异步发送事件
     *
     * @param event 事件
     */
    protected void publishEventAsync(ApplicationEvent event) {
        CompletableFuture.runAsync(() -> eventPublisher.publishEvent(event));
    }

    /**
     * 获取当前用户的ID
     *
     * @return 当前用户的ID
     */
    protected Long currentUserId() {
        var user = SecurityContextUtil.currentUser();
        return user == null ? null : user.getUser().getId();
    }

    /**
     * 获取当前用户的信息
     *
     * @return 当前用户的信息
     */
    protected GeneralUserDetails currentUser() {
        return SecurityContextUtil.currentUser();
    }

    @Autowired
    protected JPAQueryFactory jpaQueryFactory;
    @Autowired
    protected ObjectMapper objectMapper;
    @Autowired
    protected PlatformTransactionManager transactionManager;
    @Autowired(required = false)
    protected SysConfigProperties sysConfigProperties;
    @Autowired
    protected ApplicationEventPublisher eventPublisher;

    @Autowired
    private UdcProvider udcProvider;
    @Autowired
    protected FileService fileService;
}
