package com.elitesland.yst.production.sale.repo.shop;

import cn.hutool.core.util.ObjectUtil;
import com.elitesland.yst.production.sale.entity.BipOrderDDO;
import com.elitesland.yst.production.sale.entity.QBipOrderDDO;
import com.elitesland.yst.production.sale.entity.QBipOrderDO;
import com.querydsl.core.types.Predicate;
import com.querydsl.core.types.Projections;
import com.querydsl.jpa.impl.JPAQueryFactory;
import org.springframework.stereotype.Repository;

import java.time.LocalDateTime;
import java.util.List;

/**
 * .
 *
 * @author Kaiser（wang shao）
 * @date 2021/09/02
 */
@Repository
public class BipOrderDRepoProc {

    private final JPAQueryFactory jpaQueryFactory;

    private static final QBipOrderDDO DO = QBipOrderDDO.bipOrderDDO;


    public BipOrderDRepoProc(JPAQueryFactory jpaQueryFactory) {
        this.jpaQueryFactory = jpaQueryFactory;
    }

    /**
     * 统计商品销量
     *
     * @param itemId 商品ID
     * @return 销量
     */
    public Long countItemSaleNum(Long itemId) {
        return jpaQueryFactory.select(DO.buyQty.sum())
                .from(DO)
                .where(DO.itemId.eq(itemId).and(DO.payTime.isNotNull()))
                .fetchOne();
    }

    /**
     * 统计指定时间段内的商品销量
     *
     * @param itemId 商品ID
     * @param start  起始时间
     * @param end    截止时间
     * @return 销量
     */
    public Long countItemSaleNum(Long itemId, LocalDateTime start, LocalDateTime end) {
        var condition = DO.itemId.eq(itemId);
        if (start != null) {
            condition = condition.and(DO.payTime.after(start));
        }
        if (end != null) {
            condition = condition.and(DO.payTime.before(end));
        }
        return jpaQueryFactory.select(DO.buyQty.sum())
                .from(DO)
                .where(condition)
                .fetchOne();
    }

    /**
     * 查询最近订单相关商品ID
     *
     * @param userId        用户ID，不可为空
     * @param size          获取数量
     * @return 商品ID
     */
    public List<Long> queryRecentOrderItemId(Long userId, Integer size) {
        var order = QBipOrderDO.bipOrderDO;
        var condition = order.custAccountId.eq(userId);

        return jpaQueryFactory.select(DO.itemId)
                .from(DO)
                .leftJoin(order).on(DO.orderId.eq(order.id))
                .where(condition)
                .limit(ObjectUtil.defaultIfNull(size, 10))
                .fetch();
    }

    /**
     * 查询赠品 的商品id和skuId
     * @param orderId
     * @return
     */
    public BipOrderDDO findorderItemIdAndSkuIdByOrderId(Long orderId){
        return jpaQueryFactory.select(Projections.bean(BipOrderDDO.class,
                DO.itemId,
                DO.skuId,
                DO.itemCode,
                DO.itemBrand,
                DO.itemTitle,
                DO.relateItemId,
                DO.itemType,
                DO.itemSpec,
                DO.itemName
        ))
                .from(DO)
                .where(DO.id.eq(orderId))
                .fetchOne();
    }

    /**
     * 根据订单id 查询订单下的赠品订单id
     * @param orderId
     * @return
     */
    public List<Long> findGiftOrderId(Long orderId){
        return jpaQueryFactory.select(DO.id).from(DO).where(DO.pid.eq(orderId)).fetch();
    }


    /**
     * 订单签收
     * @param orderId
     * @param status
     * @return
     */
    public boolean updateStatus(Long orderId,String status){
        QBipOrderDDO qBipOrderDDO=QBipOrderDDO.bipOrderDDO;
        Predicate predicate = qBipOrderDDO.isNotNull();
        return jpaQueryFactory.update(qBipOrderDDO)
                .set(qBipOrderDDO.status,status)
                .where(qBipOrderDDO.orderId.eq(orderId))
                .execute()==1;
    }

    /**
     * 订单更新状态
     * @param orderId
     * @param status
     * @return
     */
    public boolean updateOrderDAllowStatus(Long orderId,String status){
        QBipOrderDDO qBipOrderDDO=QBipOrderDDO.bipOrderDDO;
        Predicate predicate = qBipOrderDDO.isNotNull();
        return jpaQueryFactory.update(qBipOrderDDO)
                .set(qBipOrderDDO.status,status)
                .where(qBipOrderDDO.orderId.eq(orderId))
                .execute()==1;
    }


    /**
     * 更新可退货数量
     * @param id
     * @param num
     * @return
     */
    public boolean updateOrderDAllowBackNum(Long id,Long num){
        QBipOrderDDO qBipOrderDDO=QBipOrderDDO.bipOrderDDO;
        Predicate predicate = qBipOrderDDO.isNotNull();
        return jpaQueryFactory.update(qBipOrderDDO)
                .set(qBipOrderDDO.allowBackNum,qBipOrderDDO.allowBackNum.add(num))
                .where(qBipOrderDDO.id.eq(id).and(qBipOrderDDO.allowBackNum.goe(num)))
                .execute()==1;
    }



    /**
     * 更新订单物流信息
     * @param id
     * @param orderId
     * @param logistNo
     * @return
     */
    public boolean updateOrderDLogist(Long id,Long orderId,String logistNo){
        QBipOrderDDO qBipOrderDDO=QBipOrderDDO.bipOrderDDO;
        Predicate predicate = qBipOrderDDO.isNotNull();
        return jpaQueryFactory.update(qBipOrderDDO)
                .set(qBipOrderDDO.logisNo,logistNo)
                .where(qBipOrderDDO.id.eq(id).and(qBipOrderDDO.orderId.eq(orderId)))
                .execute()==1;
    }


    /**
     * 查询明细skuID
     * @param orderIds
     *
     */
    public List<Long> findOrderDSkuIdByIds(List<Long> orderDIds){
        QBipOrderDDO qBipOrderDDO=QBipOrderDDO.bipOrderDDO;
        Predicate predicate = qBipOrderDDO.isNotNull();
        return jpaQueryFactory.select(qBipOrderDDO.skuId).from(qBipOrderDDO).where(qBipOrderDDO.id.in(orderDIds)).fetch();
    }



    /**
     * 查询明细id
     * @param skuId
     *
     */
    public List<Long> findOrderDIdBySkuId(List<Long> skuIds,Long orderId){
        QBipOrderDDO qBipOrderDDO=QBipOrderDDO.bipOrderDDO;
        Predicate predicate = qBipOrderDDO.isNotNull();
        return jpaQueryFactory.select(qBipOrderDDO.id).from(qBipOrderDDO).where(qBipOrderDDO.skuId.in(skuIds).and(qBipOrderDDO.orderId.eq(orderId))).fetch();
    }




}
