package com.elitesland.yst.production.sale.service.shop;

import cn.hutool.core.collection.CollectionUtil;
import cn.hutool.core.lang.Assert;
import cn.hutool.core.lang.Validator;
import cn.hutool.core.util.ObjectUtil;
import cn.hutool.core.util.RandomUtil;
import cn.hutool.core.util.StrUtil;
import cn.hutool.crypto.digest.MD5;
import com.alibaba.cloud.commons.lang.StringUtils;
import com.alibaba.fastjson.JSON;
import com.elitescloud.boot.core.base.UdcProvider;
import com.elitesland.yst.production.sale.api.dto.CustDTO;
import com.elitesland.yst.production.sale.api.service.CrmCustService;
import com.elitesland.yst.production.sale.api.service.shop.BipCompanyManageService;
import com.elitesland.yst.production.sale.api.service.shop.BipCustUserBindService;
import com.elitesland.yst.production.sale.api.vo.param.com.SmsVerifyCodeParamVO;
import com.elitesland.yst.production.sale.api.vo.param.shop.BipCustUserBindPageQueryVO;
import com.elitesland.yst.production.sale.api.vo.resp.crm.CrmCustDetailRespVO;
import com.elitesland.yst.production.sale.api.vo.resp.shop.BipCompanyManageRespVO;
import com.elitesland.yst.production.sale.api.vo.resp.shop.BipCustUserBindCheckRespVO;
import com.elitesland.yst.production.sale.api.vo.resp.shop.BipCustUserBindPageRespVO;
import com.elitesland.yst.production.sale.api.vo.save.CrmCustSaveVO;
import com.elitesland.yst.production.sale.api.vo.save.shop.BipCustUserBindSaveVO;
import com.elitesland.yst.production.sale.api.vo.save.shop.BipCustUserRegisterSaveVO;
import com.elitesland.yst.production.sale.common.constant.ConstantsCache;
import com.elitesland.yst.production.sale.common.constant.ConstantsSale;
import com.elitesland.yst.production.sale.common.constant.UdcEnum;
import com.elitesland.yst.production.sale.common.model.CurrentUserDTO;
import com.elitesland.yst.production.sale.core.service.BaseServiceImpl;
import com.elitesland.yst.production.sale.core.service.UserService;
import com.elitesland.yst.production.sale.entity.BipCustUserBindDO;
import com.elitesland.yst.production.sale.entity.CrmCustDO;
import com.elitesland.yst.production.sale.entity.QBipCustUserBindDO;
import com.elitesland.yst.production.sale.repo.CrmCustRepo;
import com.elitesland.yst.production.sale.repo.CrmCustRepoProc;
import com.elitesland.yst.production.sale.repo.shop.BipCustUserBindRepo;
import com.elitesland.yst.production.sale.repo.shop.BipCustUserBindRepoProc;
import com.elitesland.yst.production.sale.rmi.ystauth.RmiOauthSystemRegisterRpcService;
import com.elitesland.yst.production.sale.rmi.ystsupport.RmiOrgEmpService;
import com.elitesland.yst.production.sale.rmi.ystsupport.RmiOrgOuService;
import com.elitesland.yst.production.sale.rmi.ystsystem.RmiSysUserService;
import com.elitescloud.cloudt.common.base.ApiCode;
import com.elitescloud.cloudt.common.base.ApiResult;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitescloud.boot.exception.BusinessException;
import com.elitescloud.boot.redis.util.RedisUtils;
import com.elitescloud.cloudt.security.entity.GeneralUserDetails;
import com.elitesland.yst.production.support.provider.org.dto.OrgEmpRpcDTO;
import com.elitesland.yst.production.support.provider.org.dto.OrgOuRpcDTO;
import com.elitesland.yst.production.support.provider.org.dto.OrgUserEmpInfoRpcDTO;
import com.elitesland.yst.production.support.provider.org.param.OrgEmpRpcDtoParam;
import com.elitesland.yst.production.support.provider.org.param.OrgOuRpcDtoParam;
import com.elitescloud.cloudt.system.vo.SysUserVO;
import lombok.AllArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.MapUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.cache.annotation.CacheEvict;
import org.springframework.cache.annotation.Caching;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;
import java.util.*;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

/**
 * .
 *
 * @author Kaiser（wang shao）
 * @date 2021/08/30
 */
@Service
@AllArgsConstructor
@Slf4j
public class BipCustUserBindServiceImpl extends BaseServiceImpl implements BipCustUserBindService {

    private final BipCustUserBindRepo bipCustUserBindRepo;
    private final BipCustUserBindRepoProc bipCustUserBindRepoProc;
    private final CrmCustRepo crmCustRepo;
    private final CrmCustRepoProc crmCustRepoProc;
    private final CrmCustService crmCustService;

    private final RmiOauthSystemRegisterRpcService systemRegisterRpcService;
    private final RmiSysUserService rmiSysUserService;
    private final RmiOrgOuService rmiOrgOuService;
    private final RmiOrgEmpService rmiOrgEmpService;
    private final RedisUtils redisUtils;
 //   private final RmiMsgCarrierDubboService rmiMsgCarrierDubboService;
    private final BipCompanyManageService bipCompanyManageService;

    @Autowired
    private UdcProvider udcProvider;

    private static final QBipCustUserBindDO bipCustUserBindDO = QBipCustUserBindDO.bipCustUserBindDO;

    private static final String CACHE_NAME = ConstantsCache.BIP_USER_BIND_CUST;
    private static final DateTimeFormatter FORMATTER = DateTimeFormatter.ofPattern("yyyyMMddHHmmssSSS");
    private static final MD5 MD_5 = MD5.create();

    @Transactional(rollbackFor = Exception.class)
    @Caching(
            evict = {
                    @CacheEvict(value = CACHE_NAME, key = "#result.data", condition = "#result.success")
            }
    )
    @Override
    public ApiResult<Long> saveRegister(BipCustUserRegisterSaveVO registerSaveVO) {
        // 先在本地保存信息
        var bindDO = saveRegisterInfo(registerSaveVO);

        // 调用系统域注册
        var userId = systemRegister(registerSaveVO, bindDO.getUsername());

        // 回写用户ID
        bipCustUserBindRepoProc.updateUserId(bindDO.getId(), userId);

        return ApiResult.ok(userId);
    }

    @Transactional(rollbackFor = Exception.class)
    @Caching(
            evict = {
                    @CacheEvict(value = CACHE_NAME, keyGenerator = ConstantsCache.KEY_GENERATOR_CURRENT_USER, condition = "#result.success")
            }
    )
    @Override
    public ApiResult<Long> saveBind(BipCustUserBindSaveVO saveVO) {
        var user = currentUser();
        if (user == null) {
            return ApiResult.fail("绑定失败，请重新登录");
        }

        // 判断验证码
        var codeKey = generateSmsKeyForBind(saveVO.getContactPhone());
        var code = (String) redisUtils.get(codeKey);
        if (StrUtil.isBlank(code)) {
            return ApiResult.fail("绑定失败，验证码已过期");
        }
        if (!StrUtil.equals(code, saveVO.getVerifyCode())) {
            return ApiResult.fail("绑定失败，验证码不正确");
        }

        // 判断客户是否存在
        var cust = getCrmCust(saveVO.getTaxRegNo());

        var bindDO = bipCustUserBindRepoProc.getByUserId(user.getUser().getId());
        if (bindDO == null) {
            // 如果没有绑定记录，则新增
            bindDO = addCustUserBind(user, saveVO, cust);

            return ApiResult.ok(bindDO.getId());
        }

        // 如果绑定过，则更新
        updateCustUserBind(user, saveVO, bindDO, cust);

        return ApiResult.ok(bindDO.getId());
    }

    @Override
    public ApiResult<Long> getCustIdByUserId(Long userId) {
        var custId = bipCustUserBindRepoProc.getCustIdByUserId(userId);
        return ApiResult.ok(custId);
    }


    @Override
    public ApiResult<CustDTO> getCustOfUserBind(Long userId) {
        if (userId==null){
            return ApiResult.ok(null);
        }
        var custId = bipCustUserBindRepoProc.getCustIdByUserId(userId);
        if (custId == null) {
            return ApiResult.ok(null);
        }

        var cust = crmCustRepoProc.getCustSimpleInfo(custId);

        return ApiResult.ok(cust);
    }

    @Override
    public ApiResult<Boolean> sendSmsVerifyCodeForBind(SmsVerifyCodeParamVO paramVO) {
        var cacheKey = generateSmsKeyForBind(paramVO.getMobile());

        // 先判断是否已发送过
        var codeOld = redisUtils.get(cacheKey);
        if (codeOld != null) {
            return ApiResult.fail("验证码发送过于频繁");
        }

        // 验证按存于缓存
        var verifyCode = RandomUtil.randomStringUpper(ObjectUtil.defaultIfNull(paramVO.getCodeLength(), 4));
        redisUtils.set(cacheKey, verifyCode, ObjectUtil.defaultIfNull(paramVO.getPeriod(), 60L), TimeUnit.SECONDS);

        // 发送验证码
//        List<GeneralDubboCarrier> carriers = new ArrayList<>(4);
//        GeneralDubboCarrier generalDubboCarrier = new GeneralDubboCarrier();
//        generalDubboCarrier.setTos(new String[]{paramVO.getMobile()});
//        generalDubboCarrier.setPureContent(generateSmsContentForBind(verifyCode));
//        generalDubboCarrier.setIsRichContent(false);
//        generalDubboCarrier.setTransType("SMS");
//        carriers.add(generalDubboCarrier);
//        try {
//            var sendResult = rmiMsgCarrierDubboService.generalSend(carriers);
//            if (sendResult) {
//                return ApiResult.ok(true);
//            }
//        } catch (Exception e) {
//            log.error("发送短信验证码失败", e);
//        }

        // 发送失败后清掉redis
        redisUtils.del(cacheKey);
        return ApiResult.ok(true);
    }

    @Override
    public PagingVO<BipCustUserBindPageRespVO> search(BipCustUserBindPageQueryVO queryParam) {
        var jpaQuery = bipCustUserBindRepoProc.select(queryParam);
        long total = jpaQuery.fetchCount();
        if (total == 0) {
            return PagingVO.<BipCustUserBindPageRespVO>builder().build();
        }

        var pageRequest = wrapperPageRequest(queryParam.getPageRequest(), null);
        appendPageAndSort(jpaQuery, pageRequest, bipCustUserBindDO);
        List<BipCustUserBindPageRespVO> pageResult = jpaQuery.fetch();
        // 设置 关联字段
        setDisplayFields(pageResult);
        // udc 翻译
        transUdc(pageResult);
        return PagingVO.<BipCustUserBindPageRespVO>builder().total(total).records(pageResult).build();
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void updateStateApply(List<Long> ids, String stateApply) {
        Assert.isFalse(CollectionUtil.isEmpty(ids), "[客户信息]操作数据为空");
        Map<String, String> stateApplyUdcMap = udcProvider.getValueMapByUdcCode("yst-sale", "CUST_USER_APPLY");
        Assert.isTrue(checkUdcValueExit(stateApplyUdcMap, stateApply), "[客户信息]审批状态数据异常");
        // 获取绑定信息
        List<BipCustUserBindDO> bipCustUserBindDOS = bipCustUserBindRepo.findAllById(ids);
        // 获取当前业务员信息
        CurrentUserDTO currentUser = getCurrentUser();
        // 根据userId获取用户员工信息
        OrgUserEmpInfoRpcDTO orgUserEmpInfoRpcDTO = rmiOrgEmpService.findUserEmpInfo(currentUser.getUserId());
        List<BipCompanyManageRespVO> manageByOuIds = bipCompanyManageService.findCompanyManageByOuIds(Arrays.asList(orgUserEmpInfoRpcDTO.getOuId()));
        if (CollectionUtils.isEmpty(manageByOuIds)) {
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "当前用户公司信息为空");
        }
        // 审批通过
        if (UdcEnum.BIP_CUST_USER_APPLY_PASSED.getValueCode().equals(stateApply)) {

            // 获取关联公司信息
            List<Long> ouIds = bipCustUserBindDOS.stream().map(BipCustUserBindDO::getOuId).filter(Objects::nonNull).distinct().collect(Collectors.toList());
            List<OrgOuRpcDTO> ouDtoList = getOrgOuRpcDTOList(ouIds);

            bipCustUserBindDOS.forEach(i -> {
                CrmCustDO crmCustDO;
                if (Validator.isNull(i.getCustId())) {
                    crmCustDO = crmCustRepoProc.getByTaxRegNo(i.getTaxRegNo());
                } else {
                    crmCustDO = crmCustRepo.findById(i.getCustId()).orElse(null);
                }
                if (Validator.isNull(crmCustDO) || ConstantsSale.COMMON_DELETE_YSE == crmCustDO.getDeleteFlag()) {
                    crmCustDO = null;
                }
                if (Validator.isNotNull(crmCustDO)) {
                    crmCustDO.setContactPhone(i.getContactPhone());
                    crmCustDO.setContactName(i.getContactName());
                    crmCustDO.setIsCreatedAccount(true);
                    crmCustDO.setSecOuId(orgUserEmpInfoRpcDTO.getOuId());
                    crmCustDO.setSecUserId(orgUserEmpInfoRpcDTO.getUserId());
                    crmCustDO.setSecBuId(manageByOuIds.get(0).getBuId());
                    crmCustRepo.save(crmCustDO); // 保存crmCust
                    i.setCustId(crmCustDO.getId());
                    i.setCustCode(crmCustDO.getCustCode());
                } else {
                    // 创建crmCustDO
                    CrmCustSaveVO saveVO = new CrmCustSaveVO();
                    saveVO.setSecOuId(orgUserEmpInfoRpcDTO.getOuId());
                    saveVO.setSecUserId(orgUserEmpInfoRpcDTO.getUserId());
                    saveVO.setSecBuId(manageByOuIds.get(0).getBuId());
                    saveVO.setCustType2(UdcEnum.CRM_CUST_TYPE2_A.getValueCode());
                    saveVO.setCustName(i.getCustName());
                    saveVO.setTaxRegNo(i.getTaxRegNo());
                    saveVO.setCustType(UdcEnum.CRM_CUST_TYPE_D.getValueCode());
                    saveVO.setInvType(UdcEnum.COM_INV_TYPE_VANA.getValueCode());
                    saveVO.setInvTitle(i.getCustName());
                    saveVO.setContactPhone(i.getContactPhone());
                    saveVO.setContactName(i.getContactName());
                    saveVO.setAgentEmpId(orgUserEmpInfoRpcDTO.getEmpId());
                    saveVO.setCustStatus("ACTIVE");

                    OrgOuRpcDTO orgOuRpcDTO = ouDtoList.stream().filter(o -> o.getId().equals(i.getOuId())).findAny().orElse(null);
                    saveVO.setOuId(i.getOuId());
                    if (Validator.isNotNull(orgOuRpcDTO)) {
                        saveVO.setOuName(orgOuRpcDTO.getOuName());
                        saveVO.setOuCode(orgOuRpcDTO.getOuCode());
                        saveVO.setRegion(orgOuRpcDTO.getRegion());
                    }
                    if (Validator.isNotNull(orgUserEmpInfoRpcDTO)) {
                        saveVO.setBuId(orgUserEmpInfoRpcDTO.getBuId());
                        saveVO.setBuCode(orgUserEmpInfoRpcDTO.getBuCode());
                        saveVO.setBuName(orgUserEmpInfoRpcDTO.getBuName());
                    }
                    ApiResult<CrmCustDetailRespVO> save = crmCustService.save(saveVO);
                    i.setCustId(save.getData().getId());
                    i.setCustCode(save.getData().getCustCode());
                }
                i.setState(UdcEnum.BIP_CUST_USER_STATE_ACTIVE.getValueCode());
                i.setStateApply(stateApply);
                i.setTimeApply(LocalDateTime.now());
                i.setAgentEmpId(orgUserEmpInfoRpcDTO.getEmpId());
                i.setSecOuId(orgUserEmpInfoRpcDTO.getOuId());
                i.setSecBuId(manageByOuIds.get(0).getBuId());
                i.setSecUserId(orgUserEmpInfoRpcDTO.getUserId());
                bipCustUserBindRepo.save(i);
            });
        } else {
            bipCustUserBindRepoProc.updateStateApplyBatch(ids, UdcEnum.BIP_CUST_USER_STATE_BIND_FAIL.getValueCode(), stateApply, orgUserEmpInfoRpcDTO.getEmpId(), LocalDateTime.now());
        }
    }


    @Override
    @Transactional(rollbackFor = Exception.class)
    public void updateState(List<Long> ids, String state) {
        Map<String, String> stateUdcMap = udcProvider.getValueMapByUdcCode("yst-sale", "CUST_USER_STATE");
        Assert.isFalse(CollectionUtil.isEmpty(ids), "[客户信息]操作数据为空");
        Assert.isTrue(checkUdcValueExit(stateUdcMap, state), "[客户信息]状态数据异常");

        CurrentUserDTO currentUser = getCurrentUser();
        OrgUserEmpInfoRpcDTO orgUserEmpInfoRpcDTO = rmiOrgEmpService.findUserEmpInfo(currentUser.getUserId());
        bipCustUserBindRepoProc.updateStateBatch(ids, orgUserEmpInfoRpcDTO.getEmpId(), state);
    }

    @Override
    public BipCustUserBindCheckRespVO checkCustUserBindInfo() {
        CurrentUserDTO currentUser = getCurrentUser();
        Long userId = currentUser.getUserId();
        Assert.isFalse(Validator.isNull(userId), "用户ID不能为空");
        BipCustUserBindCheckRespVO result = new BipCustUserBindCheckRespVO();

        // 获取绑定信息
        BipCustUserBindPageQueryVO queryParam = new BipCustUserBindPageQueryVO();
        queryParam.setUserId(userId);
        log.info("检验用户-绑定入参："+ JSON.toJSONString(queryParam));
        BipCustUserBindPageRespVO bipCustUserBindPageRespVO = bipCustUserBindRepoProc.select(queryParam).fetchFirst();
        log.info("检验用户-绑定信息："+ JSON.toJSONString(bipCustUserBindPageRespVO));

        // 补充该绑定用户的
        List<Long> ouIds = new ArrayList<>();
        ouIds.add(bipCustUserBindPageRespVO.getOuId());
        List<BipCompanyManageRespVO> companyManageByOuIds = null;
        if (ouIds.get(0)!=null){
             companyManageByOuIds = bipCompanyManageService.findCompanyManageByOuIds(ouIds);
        }

        log.info("检验用户-公司信息："+ JSON.toJSONString(companyManageByOuIds));
        if (!CollectionUtils.isEmpty(companyManageByOuIds)){
            bipCustUserBindPageRespVO.setViewOuName(companyManageByOuIds.get(0).getViewOuName());
            bipCustUserBindPageRespVO.setSource(companyManageByOuIds.get(0).getCustServiceSource());
            bipCustUserBindPageRespVO.setServiceTel(companyManageByOuIds.get(0).getServiceTel());
        }
        if (Validator.isNotNull(bipCustUserBindPageRespVO)) {
            // 设置 关联字段
            setDisplayFields(Collections.singletonList(bipCustUserBindPageRespVO));
            // udc 翻译
            transUdc(Collections.singletonList(bipCustUserBindPageRespVO));
        }

        // 设置绑定信息
        result.setBipCustUserBindPageRespVO(bipCustUserBindPageRespVO);
        if (Validator.isNull(bipCustUserBindPageRespVO) || bipCustUserBindPageRespVO.getOuId()==null || UdcEnum.BIP_CUST_USER_STATE_UNBIND.getValueCode().equals(bipCustUserBindPageRespVO.getState())) {
            result.setCode("201");
            result.setMsg("请绑定公司信息。");
        } else if (UdcEnum.BIP_CUST_USER_STATE_DISABLE.getValueCode().equals(bipCustUserBindPageRespVO.getState())) {
            result.setCode("202");
            result.setMsg("您的账号已冻结，请联系业务员。");
        } else {
            if (UdcEnum.BIP_CUST_USER_APPLY_TO_APPR.getValueCode().equals(bipCustUserBindPageRespVO.getStateApply())) {
                result.setCode("203");
                result.setMsg("请等待审核完成后下单或联系业务员。");
            } else if (UdcEnum.BIP_CUST_USER_APPLY_REFUSE.getValueCode().equals(bipCustUserBindPageRespVO.getStateApply())) {
                result.setCode("204");
                result.setMsg("请重新绑定公司信息。");
            } else {
                result.setCode("200");
                result.setMsg("用户状态正常。");
            }
        }
        return result;
    }

    private Long systemRegister(BipCustUserRegisterSaveVO registerSaveVO, String username) {
//        var registerParam = new OauthSysUserNewRpcParam();
//        registerParam.setUsername(username);
//        registerParam.setMobile(registerSaveVO.getMobile());
//        registerParam.setKaptcha(registerSaveVO.getCaptcha());
//        registerParam.setSourceType(ConstantsSale.USER_SOURCE_WEC);
//        registerParam.setPassword(registerSaveVO.getPassword());
//
//        Set<Terminal> terminals = new HashSet<>(4);
//        terminals.add(Terminal.APPLET);
      //  registerParam.setTerminals(terminals);

//        return systemRegisterRpcService.sysUserCreate(registerParam);
        return null;
    }

    private BipCustUserBindDO saveRegisterInfo(BipCustUserRegisterSaveVO registerSaveVO) {
        // 随机生成登录号
        var nowTime = LocalDateTime.now();
        var userName = MD_5.digestHex16(nowTime.format(FORMATTER)) + "_" + RandomUtil.randomString(6);

        var userBindDO = new BipCustUserBindDO();
        userBindDO.setUsername(userName);
        userBindDO.setState(UdcEnum.BIP_CUST_USER_STATE_UNBIND.getValueCode());
        userBindDO.setTimeRegister(LocalDateTime.now());

        bipCustUserBindRepo.saveAndFlush(userBindDO);

        return userBindDO;
    }

    private void updateCustUserBind(GeneralUserDetails user, BipCustUserBindSaveVO bindInfo, BipCustUserBindDO bindDO, CrmCustDO crmCust) {
        if (StrUtil.equals(UdcEnum.BIP_CUST_USER_STATE_TO_APPR.getValueCode(), bindDO.getState())) {
            // 已提交过
            throw new BusinessException("信息已提交，请等待审核");
        }
        if (StrUtil.equals(UdcEnum.BIP_CUST_USER_STATE_ACTIVE.getValueCode(), bindDO.getState())) {
            // 已激活
            throw new BusinessException("已绑定成功，无需再次绑定");
        }
        if (StrUtil.equals(UdcEnum.BIP_CUST_USER_STATE_DISABLE.getValueCode(), bindDO.getState())) {
            // 已禁用
            throw new BusinessException("您的账号已被公司禁用");
        }

        // 更新绑定信息
        initBindInfo(bindDO, user, bindInfo, crmCust);
        bipCustUserBindRepo.save(bindDO);
    }

    private BipCustUserBindDO addCustUserBind(GeneralUserDetails user, BipCustUserBindSaveVO bindInfo, CrmCustDO crmCust) {
        var bindDO = new BipCustUserBindDO();

        initBindInfo(bindDO, user, bindInfo, crmCust);

        bipCustUserBindRepo.save(bindDO);
        return bindDO;
    }

    private void initBindInfo(BipCustUserBindDO bindDO, GeneralUserDetails user, BipCustUserBindSaveVO bindInfo, CrmCustDO crmCust) {
        // 权限赋值
        List<BipCompanyManageRespVO> manageByOuIds = bipCompanyManageService.findCompanyManageByOuIds(Arrays.asList(bindInfo.getOuId()));
        if (CollectionUtils.isEmpty(manageByOuIds)) {
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "当前用户公司信息为空");
        }
        bindDO.setSecOuId(bindInfo.getOuId());
        bindDO.setSecBuId(manageByOuIds.get(0).getBuId());

        bindDO.setUserId(user.getUser().getId());
        bindDO.setUsername(user.getUsername());

        bindDO.setOuId(bindInfo.getOuId());
        if (crmCust == null) {
            // 客户不存在时
            bindDO.setCustId(null);
            bindDO.setAgentEmpId(null);
            bindDO.setCustCode(null);
            bindDO.setCustName(bindInfo.getCustName());
        } else {
            throw new BusinessException("企业已存在，请联系管理员");
        }

        bindDO.setTaxRegNo(bindInfo.getTaxRegNo());
        if (StrUtil.isNotBlank(bindInfo.getCertFileCode())) {
            // 判断营业执照是否存在
            if (!isExistsFile(bindInfo.getCertFileCode())) {
                throw new BusinessException("请重新上传营业执照");
            }
            bindDO.setCertFileCode(bindInfo.getCertFileCode());
        }

        bindDO.setContactName(bindInfo.getContactName());
        bindDO.setContactPhone(bindInfo.getContactPhone());
        bindDO.setTimeApply(LocalDateTime.now());
        bindDO.setState(UdcEnum.BIP_CUST_USER_STATE_TO_APPR.getValueCode());
        bindDO.setStateApply(UdcEnum.BIP_CUST_USER_APPLY_TO_APPR.getValueCode());
        // 再次申请也需要将之前的原因给清空
        bindDO.setApproveReason("");
        bindDO.setTimeApprove(null);
    }

    private CrmCustDO getCrmCust(String taxRegNo) {
        return crmCustRepoProc.getByTaxRegNo(taxRegNo);
    }

    /**
     * 根据ID 跨域查找关联字段
     *
     * @param source 数据源
     */
    private void setDisplayFields(List<BipCustUserBindPageRespVO> source) {
        if (CollectionUtil.isNotEmpty(source)) {
            List<Long> userIds = source.stream().map(BipCustUserBindPageRespVO::getUserId).filter(Objects::nonNull).distinct().collect(Collectors.toList());
            List<Long> ouIds = source.stream().map(BipCustUserBindPageRespVO::getOuId).filter(Objects::nonNull).distinct().collect(Collectors.toList());
            List<Long> agentEmpIdIds = source.stream().map(BipCustUserBindPageRespVO::getAgentEmpId).filter(Objects::nonNull).distinct().collect(Collectors.toList());

            log.info("检验用户-入参：userIds-"+ JSON.toJSONString(userIds));
            log.info("检验用户-入参：ouIds-"+ JSON.toJSONString(ouIds));
            log.info("检验用户-入参：agentEmpIdIds-"+ JSON.toJSONString(agentEmpIdIds));


            List<OrgOuRpcDTO> ouList = null;
            if (ouIds.size()>0 && ouIds.get(0)!=null){
                ouList = rmiOrgOuService.findOuDtoListByParam(OrgOuRpcDtoParam.builder().ouIds(ouIds).build());
            }
            List<SysUserVO> userList = rmiSysUserService.getUsers(userIds);
            List<OrgEmpRpcDTO> agentEmpList = getOrgEmpRpcDTOList(agentEmpIdIds);
            // 设置 用户登陆手机号
            if (CollectionUtil.isNotEmpty(userList)) {
                source.stream().filter(i -> i.getUserId() != null).forEach(i ->
                        userList.stream().filter(v -> v.getId().equals(i.getUserId())).findAny().ifPresent(v -> i.setMobile(v.getMobile()))
                );
            }
            // 设置 所属公司名称
            if (CollectionUtil.isNotEmpty(ouList)) {
                List<OrgOuRpcDTO> finalOuList = ouList;
                source.stream().filter(i -> i.getOuId() != null).forEach(i ->
                        finalOuList.stream().filter(v -> v.getId().equals(i.getOuId())).findAny().ifPresent(v -> i.setOuName(v.getOuName()))
                );
            }
            // 设置 业务员名称
            if (CollectionUtil.isNotEmpty(agentEmpList)) {
                source.stream().filter(i -> i.getAgentEmpId() != null).forEach(i ->
                        agentEmpList.stream().filter(v -> v.getId().equals(i.getAgentEmpId())).findAny().ifPresent(v -> i.setAgentEmpName(v.getEmpName()))
                );
            }
        }
    }

    /**
     * udc 翻译
     *
     * @param source 数据源
     */
    private void transUdc(List<BipCustUserBindPageRespVO> source) {
        Map<String, String> stateUdcMap = udcProvider.getValueMapByUdcCode("yst-sale", "CUST_USER_STATE");
        Map<String, String> stateApplyUdcMap = udcProvider.getValueMapByUdcCode("yst-sale", "CUST_USER_APPLY");


        source.forEach(i -> {
            if (StringUtils.isNotBlank(i.getState()) && !MapUtils.isEmpty(stateUdcMap)) {
                i.setStateName(stateUdcMap.get(i.getState()));
            }
            if (StringUtils.isNotBlank(i.getStateApply()) && !MapUtils.isEmpty(stateApplyUdcMap)) {
                i.setStateApplyName(stateApplyUdcMap.get(i.getStateApply()));
            }
        });
    }

    private String generateSmsKeyForBind(String mobile) {
        return "yst_sale_cust_user_bind_sms_" + mobile;
    }

    private String generateSmsContentForBind(String code) {
        return StrUtil.format(sysConfigProperties.getSmsContentOfBindOu(), code);
    }


    /**
     * 获取员工DTO信息
     *
     * @param empIds 员工ID
     * @return 员工DTO信息
     */
    private List<OrgEmpRpcDTO> getOrgEmpRpcDTOList(List<Long> empIds) {
        List<OrgEmpRpcDTO> empRpcDTOList = Collections.emptyList();
        if (CollectionUtil.isNotEmpty(empIds)) {
            OrgEmpRpcDtoParam param = new OrgEmpRpcDtoParam();
            param.setEmpIds(empIds);
            empRpcDTOList = rmiOrgEmpService.findEmpListByParam(param);
        }
        return empRpcDTOList;
    }

    /**
     * 获取公司DTO数据
     *
     * @param ouIds 公司ID集合
     * @return 公司DTO数据
     */
    private List<OrgOuRpcDTO> getOrgOuRpcDTOList(List<Long> ouIds) {
        List<OrgOuRpcDTO> ouDtoList = Collections.emptyList();
        if (CollectionUtil.isNotEmpty(ouIds)) {
            ouDtoList = rmiOrgOuService.findOuDtoList(ouIds, null);
        }
        return ouDtoList;
    }

    /**
     * 获取当前登陆人信息
     *
     * @return 当前登陆人信息
     */
    private CurrentUserDTO getCurrentUser() {
        CurrentUserDTO currentUser = UserService.currentUser();
        if (currentUser == null) {
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "当前用户已失效,请重登录重试");
        }
        return currentUser;
    }
}
