package com.elitesland.yst.production.sale.repo.shop;

import com.elitesland.yst.production.sale.api.vo.param.shop.BipCustUserBindPageQueryVO;
import com.elitesland.yst.production.sale.api.vo.resp.shop.BipCustUserBindPageRespVO;
import com.elitesland.yst.production.sale.api.vo.resp.shop.BipCustUserBineSimpleRespVO;
import com.elitesland.yst.production.sale.common.constant.ConstantsSale;
import com.elitesland.yst.production.sale.entity.BipCustUserBindDO;
import com.elitesland.yst.production.sale.entity.QBipCustUserBindDO;
import com.querydsl.core.types.ExpressionUtils;
import com.querydsl.core.types.Predicate;
import com.querydsl.core.types.Projections;
import com.querydsl.jpa.impl.JPAQuery;
import com.querydsl.jpa.impl.JPAQueryFactory;
import org.springframework.stereotype.Repository;
import org.springframework.util.StringUtils;

import java.time.LocalDateTime;
import java.time.LocalTime;
import java.util.List;
import java.util.stream.Collectors;

/**
 * .
 *
 * @author Kaiser（wang shao）
 * @date 2021/08/30
 */
@Repository
public class BipCustUserBindRepoProc {

    private final JPAQueryFactory jpaQueryFactory;

    private static final QBipCustUserBindDO DO = QBipCustUserBindDO.bipCustUserBindDO;

    public BipCustUserBindRepoProc(JPAQueryFactory jpaQueryFactory) {
        this.jpaQueryFactory = jpaQueryFactory;
    }

    /**
     * 更新用户ID
     *
     * @param id     ID
     * @param userId 用户ID
     */
    public void updateUserId(Long id, Long userId) {
        jpaQueryFactory.update(DO)
                .set(DO.userId, userId)
                .where(DO.id.eq(id))
                .execute();
    }

    /**
     * 根据用户ID获取绑定记录
     *
     * @param userId 用户ID
     * @return 绑定记录
     */
    public BipCustUserBindDO getByUserId(Long userId) {
        return jpaQueryFactory.select(DO)
                .from(DO)
                .where(DO.userId.eq(userId))
                .limit(1)
                .fetchOne();
    }

    /**
     * 根据用户ID集合获取绑定记录
     *
     * @param userIds 用户ID集合
     * @return 绑定记录
     */
    public List<BipCustUserBindDO> getAllByUserIds(List<Long> userIds) {
        return jpaQueryFactory.select(DO)
                .from(DO)
                .where(DO.userId.in(userIds))
                .fetch();
    }

    /**
     * 根据员工ID集合获取绑定记录
     *
     * @param empIds 员工ID集合
     * @return 绑定记录
     */
    public List<BipCustUserBindDO> getAllByEmpIds(List<Long> empIds) {
        return jpaQueryFactory.select(DO)
                .from(DO)
                .where(DO.custId.in(empIds))
                .where(DO.deleteFlag.eq(ConstantsSale.COMMON_DELETE_NO))
                .fetch();
    }
    /**
     * 根据用户ID获取所属客户ID
     *
     * @param userId 用户ID
     * @return 客户ID
     */
    public Long getCustIdByUserId(Long userId) {
        return jpaQueryFactory.select(DO.custId)
                .from(DO)
                .where(DO.userId.eq(userId))
                .limit(1)
                .fetchOne();
    }


    /**
     * 查询主表
     *
     * @param queryParam 查询条件,可为空
     * @return jpaQuery
     */
    public JPAQuery<BipCustUserBindPageRespVO> select(BipCustUserBindPageQueryVO queryParam) {
        JPAQuery<BipCustUserBindPageRespVO> jpaQuery = jpaQueryFactory.select(Projections.bean(BipCustUserBindPageRespVO.class,
                DO.id, DO.userId, DO.username, DO.custId,
                DO.custCode, DO.custName, DO.taxRegNo, DO.contactName,
                DO.contactPhone, DO.ouId, DO.agentEmpId, DO.state,
                DO.stateApply, DO.timeRegister, DO.certFileCode
        ))
                .from(DO)
                .where(DO.deleteFlag.eq(ConstantsSale.COMMON_DELETE_NO))
                // .where(DataAuthJpaUtil.dataAuthJpaPredicate(DO.getMetadata())) //数据权限
                ;
        if (queryParam != null) {
            jpaQuery.where(buildCondition(queryParam));
        }
        return jpaQuery;
    }

    /**
     * 查询条件
     *
     * @param queryParam 查询条件
     * @return predicate
     */
    private Predicate buildCondition(BipCustUserBindPageQueryVO queryParam) {
        Predicate predicate = DO.isNotNull();
        if (queryParam.getUserId() != null) {
            predicate = ExpressionUtils.and(predicate, DO.userId.eq(queryParam.getUserId()));
        }
        if (!StringUtils.isEmpty(queryParam.getUsername())) {
            predicate = ExpressionUtils.and(predicate, DO.username.like("%" + queryParam.getUsername() + "%"));
        }
        if (!StringUtils.isEmpty(queryParam.getCustCode())) {
            predicate = ExpressionUtils.and(predicate, DO.custCode.like("%" + queryParam.getCustCode() + "%"));
        }
        if (!StringUtils.isEmpty(queryParam.getCustName())) {
            predicate = ExpressionUtils.and(predicate, DO.custName.like("%"+queryParam.getCustName()+"%"));
        }
        if (!StringUtils.isEmpty(queryParam.getState())) {
            predicate = ExpressionUtils.and(predicate, DO.state.eq(queryParam.getState()));
        }
        if (!StringUtils.isEmpty(queryParam.getStateApply())) {
            predicate = ExpressionUtils.and(predicate, DO.stateApply.eq(queryParam.getStateApply()));
        }
        if (!StringUtils.isEmpty(queryParam.getTaxRegNo())) {
            predicate = ExpressionUtils.and(predicate, DO.taxRegNo.like("%" + queryParam.getTaxRegNo() + "%"));
        }
        if (queryParam.getTimeRegister() != null) {
            predicate = ExpressionUtils.and(predicate, DO.timeRegister.between(queryParam.getTimeRegister().toLocalDate().atTime(LocalTime.MIN), queryParam.getTimeRegister().toLocalDate().atTime(LocalTime.MAX)));
        }
        if (!StringUtils.isEmpty(queryParam.getContactName())) {
            predicate = ExpressionUtils.and(predicate, DO.contactName.like("%" + queryParam.getContactName() + "%"));
        }
        if (!StringUtils.isEmpty(queryParam.getContactPhone())) {
            predicate = ExpressionUtils.and(predicate, DO.contactPhone.like("%" + queryParam.getContactPhone() + "%"));
        }
        if (queryParam.getOuId() != null) {
            predicate = ExpressionUtils.and(predicate, DO.ouId.eq(queryParam.getOuId()));
        }
        if (queryParam.getAgentEmpId() != null) {
            predicate = ExpressionUtils.and(predicate, DO.agentEmpId.eq(queryParam.getAgentEmpId()));
        }

        return predicate;
    }

    /**
     * 更新审批状态 标记
     *
     * @param ids        id集合
     * @param state 状态
     * @param stateApply 审批状态
     * @param agentEmpId 审批人ID
     * @param timeApply  审批时间
     */
    public void updateStateApplyBatch(List<Long> ids, String state, String stateApply, Long agentEmpId, LocalDateTime timeApply) {
        jpaQueryFactory.update(DO)
                .set(DO.state, state)
                .set(DO.stateApply, stateApply)
                .set(DO.timeApply, timeApply)
                .set(DO.agentEmpId, agentEmpId)
                .where(DO.id.in(ids))
                // .where(DataAuthJpaUtil.dataAuthJpaPredicate(DO.getMetadata()))
                .execute();
    }

    /**
     * 更新状态 标记
     *
     * @param ids   id集合
     * @param agentEmpId 审批人ID
     * @param state 审批状态
     */
    public void updateStateBatch(List<Long> ids, Long agentEmpId, String state) {
        jpaQueryFactory.update(DO)
                .set(DO.state, state)
                .set(DO.agentEmpId, agentEmpId)
                .where(DO.id.in(ids))
                // .where(DataAuthJpaUtil.dataAuthJpaPredicate(DO.getMetadata()))
                .execute();
    }

    /**
     * 获取客户id对应的用户id
     * @param custIds 客户id集合
     * @return 客户id和用户id
     */
    public List<BipCustUserBineSimpleRespVO> getUserIdByCustId(List<Long> custIds) {
        return jpaQueryFactory.select(DO.userId,DO.custId)
                .from(DO)
                .where(DO.custId.in(custIds).and(DO.deleteFlag.ne(ConstantsSale.COMMON_DELETE_YSE)))
                .fetch()
                .parallelStream()
                .map(tuple ->{
                    BipCustUserBineSimpleRespVO bipCustUserBineSimpleRespVO = new BipCustUserBineSimpleRespVO();
                    bipCustUserBineSimpleRespVO.setUserId(tuple.get(DO.userId));
                    bipCustUserBineSimpleRespVO.setCustId(tuple.get(DO.custId));
                    return bipCustUserBineSimpleRespVO;
                }).collect(Collectors.toList());
    }

}
