package com.elitesland.yst.production.sale.service;

import cn.hutool.core.collection.CollectionUtil;
import com.alibaba.fastjson.JSONObject;
import com.alibaba.fastjson.JSON;
import com.elitescloud.boot.auth.util.SecurityContextUtil;
import com.elitescloud.boot.exception.BusinessException;
import com.elitescloud.cloudt.common.annotation.SysCodeProc;
import com.elitescloud.cloudt.common.base.ApiCode;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitescloud.cloudt.security.entity.GeneralUserDetails;
import com.elitescloud.cloudt.system.dto.SysEmployeeBasicDTO;
import com.elitescloud.cloudt.system.dto.resp.EmployeeUnderlingDTO;
import com.elitescloud.cloudt.system.vo.SysUserDTO;
import com.elitesland.yst.production.sale.api.service.CrmCustService;
import com.elitesland.yst.production.sale.api.service.SalesmanInfoService;
import com.elitesland.yst.production.sale.api.service.StatisticsDealerDtlService;
import com.elitesland.yst.production.sale.api.service.StatisticsDealerService;
import com.elitesland.yst.production.sale.api.vo.param.crm.CustAddrFindParam;
import com.elitesland.yst.production.sale.api.vo.param.taskinfo.SaleStatisticsDealerDtlQueryVO;
import com.elitesland.yst.production.sale.api.vo.param.taskinfo.SaleStatisticsDealerQueryVO;
import com.elitesland.yst.production.sale.api.vo.resp.crm.LmSaveCustRespVO;
import com.elitesland.yst.production.sale.api.vo.resp.taskinfo.*;
import com.elitesland.yst.production.sale.api.vo.save.SaleStatisticsDealerDtlSaveVO;
import com.elitesland.yst.production.sale.api.vo.save.SaleStatisticsDealerSaveVO;
import com.elitesland.yst.production.sale.common.constant.ConstantsSale;
import com.elitesland.yst.production.sale.convert.StatisticsDealerConvert;
import com.elitesland.yst.production.sale.dto.SalesmanLevelInfoDTO;
import com.elitesland.yst.production.sale.entity.SaleStatisticsDealerDO;
import com.elitesland.yst.production.sale.repo.SaleStatisticsDealerDtlRepo;
import com.elitesland.yst.production.sale.repo.SaleStatisticsDealerRepo;
import com.elitesland.yst.production.sale.repo.SaleStatisticsDealerRepoProc;
import com.elitesland.yst.production.sale.repo.SaleStatisticsDealerRepoSqlProc;
import com.elitesland.yst.production.sale.rmi.ystsystem.RmiEmployeeRpcService;
import lombok.Data;
import lombok.extern.slf4j.Slf4j;
import lombok.val;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

import java.math.BigDecimal;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.LocalTime;
import java.time.format.DateTimeFormatter;
import java.time.temporal.TemporalAdjusters;
import java.util.*;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import java.util.stream.StreamSupport;

/**
 * <p>
 * 功能说明:销售业绩统计-经销商
 * </p>
 *
 * @Author Darren
 * @Date 2023/06/01
 * @Version 1.0
 * @Content:
 */
@Slf4j
@Service
public class StatisticsDealerServiceImpl implements StatisticsDealerService {

    @Autowired
    private SaleStatisticsDealerRepo statisticsDealerRepo;
    @Autowired
    private SaleStatisticsDealerRepoProc statisticsDealerRepoProc;
    @Autowired
    private SaleStatisticsDealerRepoSqlProc saleStatisticsDealerRepoSqlProc;
    @Autowired
    private SaleStatisticsDealerDtlRepo statisticsDealerDtlRepo;

    @Autowired
    private StatisticsDealerDtlService statisticsDealerDtlService;
    @Autowired
    private CrmCustService crmCustService;
    @Autowired
    private RmiEmployeeRpcService rmiEmployeeRpcService;
    @Autowired
    private SalesmanInfoService salesmanInfoService;

    /**
     * 经销商销售业绩拉取订单数据并保存
     *
     * @param queryVO 入参
     * @return
     */
    @Override
    public List<SaleStatisticsDealerRespVO> pullOrder(SaleStatisticsDealerQueryVO queryVO) {
        if (Objects.isNull(queryVO.getDocTime())) {
            queryVO.setDocTime(LocalDateTime.now());
        }

        //按照日期查询订单域的接口，在立马项目里写
        //1.先查询订单数据
        List<SaleStatisticsDealerSaveVO> dealerSaveVOList = selectOrder(queryVO);
        if (CollectionUtils.isEmpty(dealerSaveVOList)) {
            log.info("经销商销售业绩未查询到订单数据,时间：{},信息：{}", queryVO);
            refreshHistory(queryVO);
            return Collections.EMPTY_LIST;
        }
        //2.组装客户类型信息
        assembleStatisticsDealerSaveVO(dealerSaveVOList, false);
        //3.删除该日期对应的数据
        deleteOldData(queryVO);
        //4.保存数据
        List<SaleStatisticsDealerRespVO> dealerRespVOList = createBatch(dealerSaveVOList);
        refreshHistory(queryVO);
        return dealerRespVOList;
    }

    private void refreshHistory(SaleStatisticsDealerQueryVO queryVO) {
        // 每天2点拉历史10天的数据
        if (queryVO.getDocTime() != null) {
            int hour = LocalDateTime.now().getHour();
            if (hour == 3 || (queryVO.getRefreshHistory() != null && queryVO.getRefreshHistory())) {
                LocalDateTime time = queryVO.getDocTime();
                for (int i = 0; i < 10; i++) {
                    LocalDateTime localDateTime = time.plusDays(-(i + 1));
                    queryVO.setDocTime(localDateTime);
                    List<SaleStatisticsDealerSaveVO> current = selectOrder(queryVO);
                    assembleStatisticsDealerSaveVO(current, i == 0);
                    deleteOldData(queryVO);
                    createBatch(current);
                }
            }
        }
    }

    /**
     * 经销商销售业绩分页查询-我的团队全-APP端
     *
     * @param pageParam 入参
     * @return 经销商销售业绩信息集合
     */
    @Override
    public PagingVO<SaleStatisticsDealerRespVO> leaderPage(SaleStatisticsDealerQueryVO pageParam) {
        //根据统计月份的日期拼装入参里的统计时间起始、统计时间截至
        assembleDocTime(pageParam);

        if (StringUtils.isBlank(pageParam.getAgentEmp())) {
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "业务员编码为空!");
        }
//        //检查业务员编码是否是当前登录人自己以及所属下级
//        if(!checkOneselfAndSub(pageParam.getAgentEmp())){
//            return PagingVO.<SaleStatisticsDealerRespVO>builder().total(0L).records(Collections.EMPTY_LIST).build();
//        }
        //明细入参查询
        if (StringUtils.isNotBlank(pageParam.getAgentEmp())) {
            SaleStatisticsDealerDtlQueryVO dealerDtlQueryVO = new SaleStatisticsDealerDtlQueryVO();
            dealerDtlQueryVO.setCode(pageParam.getAgentEmp());
            //此业务员的个人业绩列表，根据level='0'，匹配code
            //dealerDtlQueryVO.setLevel(ConstantsSale.STATISTICS_DEALER_DTL_LEVEL_0);
            if (Objects.isNull(pageParam.getSeeSelf())) {
                pageParam.setSeeSelf(true);
            }
            if (pageParam.getSeeSelf()) {
                //此业务员的个人业绩列表，根据level='0'，匹配code
                dealerDtlQueryVO.setLevel(ConstantsSale.STATISTICS_DEALER_DTL_LEVEL_0);
            }

            List<Long> masIdList = selectMasIdByDtlParam(dealerDtlQueryVO);
            if (CollectionUtils.isEmpty(masIdList)) {
                return PagingVO.<SaleStatisticsDealerRespVO>builder().total(0L).records(Collections.EMPTY_LIST).build();
            }
            List<Long> ids = CollectionUtils.isEmpty(pageParam.getIds()) ? Collections.EMPTY_LIST : pageParam.getIds();
            List<Long> idList = Stream.of(ids, masIdList).flatMap(Collection::stream).filter(Objects::nonNull).distinct().collect(Collectors.toList());
            pageParam.setIds(idList);

        }

//        if (Objects.nonNull(pageParam.getDocTimeStart())) {
//            String timeStrStart = pageParam.getDocTimeStart().format(DateTimeFormatter.ofPattern("yyyyMM"));
//            Integer integerStart = Integer.valueOf(timeStrStart);
//            pageParam.setDocTimeItStart(integerStart);
//        }
//        if (Objects.nonNull(pageParam.getDocTimeEnd())) {
//            String timeStrEnd = pageParam.getDocTimeEnd().format(DateTimeFormatter.ofPattern("yyyyMM"));
//            Integer integerEnd = Integer.valueOf(timeStrEnd);
//            pageParam.setDocTimeItEnd(integerEnd);
//
//        }

        PagingVO<SaleStatisticsDealerRespVO> pagingVO = statisticsDealerRepoProc.appPage(pageParam);
        if (CollectionUtils.isEmpty(pagingVO.getRecords())) {
            return PagingVO.<SaleStatisticsDealerRespVO>builder().total(0L).records(Collections.EMPTY_LIST).build();
        }
        List<SaleStatisticsDealerRespVO> respVOList = pagingVO.getRecords();

        //组装填充经销商销售业绩的相关信息
        translateDealer(respVOList);

        return PagingVO.<SaleStatisticsDealerRespVO>builder()
                .total(pagingVO.getTotal())
                .records(respVOList)
                .build();
    }

    /**
     * 经销商销售业绩统计汇总-我的团队全-APP端
     *
     * @param pageParam 入参
     * @return 统计汇总信息
     */
    @Override
    public StatisticsDealerSumRespVO leaderPageSum(SaleStatisticsDealerQueryVO pageParam) {
        StatisticsDealerSumRespVO sumRespVO = new StatisticsDealerSumRespVO();

        //根据统计月份的日期拼装入参里的统计时间起始、统计时间截至
        assembleDocTime(pageParam);
        if (StringUtils.isBlank(pageParam.getAgentEmp())) {
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "业务员编码为空!");
        }
//        //检查业务员编码是否是当前登录人自己以及所属下级
//        if(!checkOneselfAndSub(pageParam.getAgentEmp())){
//            return sumRespVO;
//        }
        //明细入参查询
        if (StringUtils.isNotBlank(pageParam.getAgentEmp())) {
            SaleStatisticsDealerDtlQueryVO dealerDtlQueryVO = new SaleStatisticsDealerDtlQueryVO();
            dealerDtlQueryVO.setCode(pageParam.getAgentEmp());
            //此业务员的个人业绩列表，根据level='0'，匹配code
            //dealerDtlQueryVO.setLevel(ConstantsSale.STATISTICS_DEALER_DTL_LEVEL_0);
            if (Objects.isNull(pageParam.getSeeSelf())) {
                pageParam.setSeeSelf(true);
            }
            if (pageParam.getSeeSelf()) {
                //此业务员的个人业绩列表，根据level='0'，匹配code
                dealerDtlQueryVO.setLevel(ConstantsSale.STATISTICS_DEALER_DTL_LEVEL_0);
            }

            List<Long> masIdList = selectMasIdByDtlParam(dealerDtlQueryVO);
            if (CollectionUtils.isEmpty(masIdList)) {
                return sumRespVO;
            }
            List<Long> ids = CollectionUtils.isEmpty(pageParam.getIds()) ? Collections.EMPTY_LIST : pageParam.getIds();
            List<Long> idList = Stream.of(ids, masIdList).flatMap(Collection::stream).filter(Objects::nonNull).distinct().collect(Collectors.toList());
            pageParam.setIds(idList);

        }
//        if (Objects.nonNull(pageParam.getDocTimeStart())) {
//            String timeStrStart = pageParam.getDocTimeStart().format(DateTimeFormatter.ofPattern("yyyyMM"));
//            Integer integerStart = Integer.valueOf(timeStrStart);
//            pageParam.setDocTimeItStart(integerStart);
//        }
//        if (Objects.nonNull(pageParam.getDocTimeEnd())) {
//            String timeStrEnd = pageParam.getDocTimeEnd().format(DateTimeFormatter.ofPattern("yyyyMM"));
//            Integer integerEnd = Integer.valueOf(timeStrEnd);
//            pageParam.setDocTimeItEnd(integerEnd);
//
//        }

        StatisticsDealerSumRespVO dealerSumRespVO = saleStatisticsDealerRepoSqlProc.pageSum(pageParam);
        if (Objects.isNull(dealerSumRespVO)) {
            return sumRespVO;
        }

        return dealerSumRespVO;
    }

    private void assembleStatisticsDealerSaveVO(List<SaleStatisticsDealerSaveVO> dealerSaveVOList, boolean refreshSaleManPath) {
        List<String> dealerCodeList = dealerSaveVOList.stream().map(SaleStatisticsDealerSaveVO::getDealerCode).filter(Objects::nonNull).distinct().collect(Collectors.toList());
        List<LmSaveCustRespVO> dealerList = crmCustService.getCustInfoByCustCodes(dealerCodeList);

        if (refreshSaleManPath) {
            List<SalesmanLevelInfoDTO> salesmanLevelInfoDTOS = selectSalesmanLevelByIds(dealerList);
            Map<Long, List<SalesmanLevelInfoDTO.path>> saleManMap = salesmanLevelInfoDTOS.stream().filter(i -> i.getSalesman_path() != null).collect(Collectors.toMap(i -> i.getSalesmanId(), i -> i.getSalesman_path(), (o, n) -> n));
            Map<String, List<SalesmanLevelInfoDTO.path>> custSalePathMap = new HashMap<>();
            for (Long agentId : saleManMap.keySet()) {
                String custCode = dealerList.stream().filter(i -> Objects.equals(i.getAgentEmpId(), agentId)).map(i -> i.getCustCode())
                        .findFirst().orElseGet(() -> null);
                if (StringUtils.isNotBlank(custCode)) {
                    custSalePathMap.put(custCode, saleManMap.get(agentId));
                }
            }
            for (SaleStatisticsDealerSaveVO saleStatisticsDealerSaveVO : dealerSaveVOList) {
                List<SalesmanLevelInfoDTO.path> paths = saleManMap.get(saleStatisticsDealerSaveVO.getDealerCode());
                if (!CollectionUtils.isEmpty(paths)) {
                    saleStatisticsDealerSaveVO.setSalesmanPath(JSON.toJSONString(paths));
                }
            }
        }

        dealerSaveVOList.forEach(saveVO -> {
            //客户类型
            val dealerOptional = dealerList.stream().filter(dealerVO -> Objects.equals(dealerVO.getCustCode(), saveVO.getDealerCode())).findFirst();
            dealerOptional.ifPresent(dealerVO -> {
                saveVO.setCustType(dealerVO.getCustType());
                saveVO.setRegion(dealerVO.getRegion());
            });
            if (Objects.isNull(saveVO.getShipQty())) {
                saveVO.setShipQty(BigDecimal.ZERO);
            }
            if (Objects.isNull(saveVO.getOrderQty())) {
                saveVO.setOrderQty(BigDecimal.ZERO);
            }
            if (StringUtils.isBlank(saveVO.getSalesmanPath())){
                //处理传过来的值是空字符串""，导致salesmanPath查询is null的逻辑不对，故设置为NULL
                saveVO.setSalesmanPath(null);
            }
            if (StringUtils.isBlank(saveVO.getRegion())){
                //处理传过来的值是空字符串""，导致region查询is null的逻辑不对，故设置为NULL
                saveVO.setRegion(null);
            }
        });
    }

    /**
     * 经销商销售业绩-查询订单数据
     *
     * @param queryVO 入参
     * @return
     */
    public List<SaleStatisticsDealerSaveVO> selectOrder(SaleStatisticsDealerQueryVO queryVO) {

        return Collections.EMPTY_LIST;
    }

    /**
     * 经销商销售业绩-删除旧数据
     *
     * @param queryVO 入参
     * @return
     */
    @Transactional(rollbackFor = {Exception.class})
    public void deleteOldData(SaleStatisticsDealerQueryVO queryVO) {
        //1.先按照日期查询是否存在已经有该日期的数据，有则全删，
       /* SaleStatisticsDealerRespVO dealerRespVO = statisticsDealerRepoProc.selectOneByQueryVO(queryVO);
        if (Objects.nonNull(dealerRespVO) && Objects.nonNull(queryVO.getDocTime())) {
            //不为空时，需要删除对应日期的数据,逻辑删
            statisticsDealerRepoProc.updateDeleteFlagByDocTime(1, dealerRespVO.getDocTime().toLocalDate());
        }*/
//        List<SaleStatisticsDealerRespVO> dealerRespVOList = statisticsDealerRepoProc.selectOldByQueryVO(queryVO);
//
//        if (!CollectionUtils.isEmpty(dealerRespVOList) && Objects.nonNull(queryVO.getDocTime())) {
//            List<Long> idList = dealerRespVOList.stream().map(SaleStatisticsDealerRespVO::getId).filter(Objects::nonNull).distinct().collect(Collectors.toList());
//            statisticsDealerDtlRepo.deleteByMasIdIn(idList);
//            statisticsDealerRepo.deleteAllByIdInBatch(idList);
//        }
        if(Objects.nonNull(queryVO.getDocTime())){
            LocalDateTime docTimeStart = LocalDateTime.of(queryVO.getDocTime().toLocalDate(), LocalTime.MIN);
            LocalDateTime docTimeEnd = LocalDateTime.of(queryVO.getDocTime().toLocalDate(), ConstantsSale.LOCAL_TIME_MAX);
            statisticsDealerDtlRepo.deleteByDocTime(docTimeStart, docTimeEnd);
            statisticsDealerRepo.deleteByDocTime(docTimeStart, docTimeEnd);
        }else if(Objects.nonNull(queryVO.getDocTimeStart())&& Objects.nonNull(queryVO.getDocTimeEnd())){
            statisticsDealerDtlRepo.deleteByDocTime(queryVO.getDocTimeStart(), queryVO.getDocTimeEnd());
            statisticsDealerRepo.deleteByDocTime(queryVO.getDocTimeStart(), queryVO.getDocTimeEnd());
        }

    }

    /**
     * 经销商销售业绩-批量新增
     *
     * @param saveVOList 入参
     * @return 出参对象
     */
    @Override
    public List<SaleStatisticsDealerRespVO> createBatch(List<SaleStatisticsDealerSaveVO> saveVOList) {
        if (CollectionUtils.isEmpty(saveVOList)) {
            return Collections.EMPTY_LIST;
        }

        List<SaleStatisticsDealerSaveVO> errorSaveVOList = new ArrayList<>();
        List<SaleStatisticsDealerSaveVO> successSaveVOList = new ArrayList<>();
        //区分出校验成功和失败的集合数据,分别对应处理
        for (SaleStatisticsDealerSaveVO saveVO : saveVOList) {
            //校验保存入参的必填项
            if (checkMandatoryField(saveVO)) {
                errorSaveVOList.add(saveVO);
            } else {
                successSaveVOList.add(saveVO);
            }
        }

        //先保存主表,再自行分拆保存附表
        List<SaleStatisticsDealerRespVO> respVOList = saveBatch(successSaveVOList);

        //抛出异常
        if (!CollectionUtils.isEmpty(errorSaveVOList)) {
            log.info("经销商销售业绩校验失败数据,时间：{},信息：{}", LocalDateTime.now(), JSON.toJSONString(errorSaveVOList));
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "经销商销售业绩校验失败:" + JSON.toJSONString(errorSaveVOList));
        }

        return respVOList;
    }

    /**
     * 销售业绩统计-经销商批量新增
     *
     * @param saveVOList 入参
     * @return 出参对象
     */
    @Transactional(rollbackFor = {Exception.class})
    public List<SaleStatisticsDealerRespVO> saveBatch(List<SaleStatisticsDealerSaveVO> saveVOList) {

        if (CollectionUtils.isEmpty(saveVOList)) {
            return Collections.EMPTY_LIST;
        }

        List<SaleStatisticsDealerDO> statisticsDealerDOList = saveVOList.stream().map(saveVO -> {
            saveVO.setType(ConstantsSale.SALESMAN_STATISTICS_DEALER_CUST);
            return StatisticsDealerConvert.INSTANCE.saveVoToDo(saveVO);
        }).collect(Collectors.toList());

        statisticsDealerRepo.saveAll(statisticsDealerDOList);

        List<SaleStatisticsDealerDtlSaveVO> dealerDtlSaveVOList = new ArrayList<>();
        statisticsDealerDOList.forEach(saveVO -> {
            if (StringUtils.isNotBlank(saveVO.getSalesmanPath())) {
                List<SaleStatisticsDealerDtlSaveVO> dtlSaveVOList = JSONObject.parseArray(saveVO.getSalesmanPath(), SaleStatisticsDealerDtlSaveVO.class);
                if (!CollectionUtils.isEmpty(dtlSaveVOList)) {
                    dtlSaveVOList.forEach(dtlSaveVO -> {
                        dtlSaveVO.setMasId(saveVO.getId());
                    });
                    dealerDtlSaveVOList.addAll(dtlSaveVOList);
                }
            }
        });

        //保存业务员对应的用户ID,用于数据权限
        if (!CollectionUtils.isEmpty(dealerDtlSaveVOList)) {
            Set<String> employeeCodes = dealerDtlSaveVOList.stream().map(SaleStatisticsDealerDtlSaveVO::getCode).filter(Objects::nonNull).collect(Collectors.toSet());
            List<SysEmployeeBasicDTO> basicDTOList = rmiEmployeeRpcService.findEmployeeByCodes(employeeCodes);
            dealerDtlSaveVOList.forEach(saveVO -> {
                Optional<SysEmployeeBasicDTO> optional = basicDTOList.stream().filter(employeeBasicDTO -> Objects.equals(saveVO.getCode(), employeeBasicDTO.getCode())).findFirst();
                optional.ifPresent(employeeBasicDTO -> saveVO.setUserId(employeeBasicDTO.getUserId()));
            });

            statisticsDealerDtlService.saveStatisticsDealerDtl(dealerDtlSaveVOList);
        }
        //目前不确定是否需要出参信息，故返回空
        return Collections.EMPTY_LIST;
    }


    /**
     * 校验保存入参的必填项：缺少必填返回 true 否则返回 false
     *
     * @param saveVO 入参
     * @return 缺少必填返回 true 否则返回 false
     */
    private boolean checkMandatoryField(SaleStatisticsDealerSaveVO saveVO) {
        if (Objects.isNull(saveVO.getDocTime())) {
            return true;
        }
        if (StringUtils.isBlank(saveVO.getDealerCode())) {
            return true;
        }
        if (StringUtils.isBlank(saveVO.getDealerName())) {
            return true;
        }
        if (StringUtils.isBlank(saveVO.getDealerSerialNo())) {
            return true;
        }
        /*if (StringUtils.isBlank(saveVO.getRegion())) {
            return true;
        }
        if (StringUtils.isBlank(saveVO.getSalesmanPath())) {
            return true;
        }*/
        if (StringUtils.isBlank(saveVO.getVehicleType())) {
            return true;
        }
        if (Objects.isNull(saveVO.getShipQty())) {
            return true;
        }
        if (Objects.isNull(saveVO.getOrderQty())) {
            return true;
        }
        /*if (StringUtils.isBlank(saveVO.getCustType())) {
            return true;
        }*/
        if (StringUtils.isBlank(saveVO.getItemType3())) {
            return true;
        }
        return false;
    }


    /**
     * 经销商销售业绩分页查询
     *
     * @param pageParam 入参
     * @return 经销商销售业绩信息集合
     */
    @Override
    @SysCodeProc
    public PagingVO<SaleStatisticsDealerRespVO> page(SaleStatisticsDealerQueryVO pageParam) {
        //根据统计月份的日期拼装入参里的统计时间起始、统计时间截至
        assembleDocTime(pageParam);
        //明细入参查询
        if (StringUtils.isNotBlank(pageParam.getAgentEmp())) {
            SaleStatisticsDealerDtlQueryVO dealerDtlQueryVO = new SaleStatisticsDealerDtlQueryVO();
            dealerDtlQueryVO.setCode(pageParam.getAgentEmp());
            List<Long> masIdList = selectMasIdByDtlParam(dealerDtlQueryVO);
            if (CollectionUtils.isEmpty(masIdList)) {
                return PagingVO.<SaleStatisticsDealerRespVO>builder().total(0L).records(Collections.EMPTY_LIST).build();
            }
            List<Long> ids = CollectionUtils.isEmpty(pageParam.getIds()) ? Collections.EMPTY_LIST : pageParam.getIds();
            List<Long> idList = Stream.of(ids, masIdList).flatMap(Collection::stream).filter(Objects::nonNull).distinct().collect(Collectors.toList());
            pageParam.setIds(idList);

        }

//        if (Objects.nonNull(pageParam.getDocTimeStart())) {
//            String timeStrStart = pageParam.getDocTimeStart().format(DateTimeFormatter.ofPattern("yyyyMM"));
//            Integer integerStart = Integer.valueOf(timeStrStart);
//            pageParam.setDocTimeItStart(integerStart);
//        }
//        if (Objects.nonNull(pageParam.getDocTimeEnd())) {
//            String timeStrEnd = pageParam.getDocTimeEnd().format(DateTimeFormatter.ofPattern("yyyyMM"));
//            Integer integerEnd = Integer.valueOf(timeStrEnd);
//            pageParam.setDocTimeItEnd(integerEnd);
//
//        }

        PagingVO<SaleStatisticsDealerRespVO> pagingVO = statisticsDealerRepoProc.page(pageParam);
        if (CollectionUtils.isEmpty(pagingVO.getRecords())) {
            return PagingVO.<SaleStatisticsDealerRespVO>builder().total(0L).records(Collections.EMPTY_LIST).build();
        }
        List<SaleStatisticsDealerRespVO> respVOList = pagingVO.getRecords();

        //组装填充经销商销售业绩的相关信息
        translateDealer(respVOList);

        return PagingVO.<SaleStatisticsDealerRespVO>builder()
                .total(pagingVO.getTotal())
                .records(respVOList)
                .build();
    }

    /**
     * 经销商销售业绩分页查询-APP端
     *
     * @param pageParam 入参
     * @return 经销商销售业绩信息集合
     */
    @Override
    @SysCodeProc
    public PagingVO<SaleStatisticsDealerRespVO> appPage(SaleStatisticsDealerQueryVO pageParam) {
        //根据统计月份的日期拼装入参里的统计时间起始、统计时间截至
        assembleDocTime(pageParam);

        if (StringUtils.isBlank(pageParam.getAgentEmp())) {
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "业务员编码为空!");
        }
        //检查业务员编码是否是当前登录人自己以及所属下级
        if (!checkOneselfAndSub(pageParam.getAgentEmp())) {
            return PagingVO.<SaleStatisticsDealerRespVO>builder().total(0L).records(Collections.EMPTY_LIST).build();
        }
        //明细入参查询
        if (StringUtils.isNotBlank(pageParam.getAgentEmp())) {
            SaleStatisticsDealerDtlQueryVO dealerDtlQueryVO = new SaleStatisticsDealerDtlQueryVO();
            dealerDtlQueryVO.setCode(pageParam.getAgentEmp());
            //此业务员的个人业绩列表，根据level='0'，匹配code
            //dealerDtlQueryVO.setLevel(ConstantsSale.STATISTICS_DEALER_DTL_LEVEL_0);
            if (Objects.isNull(pageParam.getSeeSelf())) {
                pageParam.setSeeSelf(true);
            }
            if (pageParam.getSeeSelf()) {
                //此业务员的个人业绩列表，根据level='0'，匹配code
                dealerDtlQueryVO.setLevel(ConstantsSale.STATISTICS_DEALER_DTL_LEVEL_0);
            }

            List<Long> masIdList = selectMasIdByDtlParam(dealerDtlQueryVO);
            if (CollectionUtils.isEmpty(masIdList)) {
                return PagingVO.<SaleStatisticsDealerRespVO>builder().total(0L).records(Collections.EMPTY_LIST).build();
            }
            List<Long> ids = CollectionUtils.isEmpty(pageParam.getIds()) ? Collections.EMPTY_LIST : pageParam.getIds();
            List<Long> idList = Stream.of(ids, masIdList).flatMap(Collection::stream).filter(Objects::nonNull).distinct().collect(Collectors.toList());
            pageParam.setIds(idList);

        }

//        if (Objects.nonNull(pageParam.getDocTimeStart())) {
//            String timeStrStart = pageParam.getDocTimeStart().format(DateTimeFormatter.ofPattern("yyyyMM"));
//            Integer integerStart = Integer.valueOf(timeStrStart);
//            pageParam.setDocTimeItStart(integerStart);
//        }
//        if (Objects.nonNull(pageParam.getDocTimeEnd())) {
//            String timeStrEnd = pageParam.getDocTimeEnd().format(DateTimeFormatter.ofPattern("yyyyMM"));
//            Integer integerEnd = Integer.valueOf(timeStrEnd);
//            pageParam.setDocTimeItEnd(integerEnd);
//
//        }

        PagingVO<SaleStatisticsDealerRespVO> pagingVO = statisticsDealerRepoProc.appPage(pageParam);
        if (CollectionUtils.isEmpty(pagingVO.getRecords())) {
            return PagingVO.<SaleStatisticsDealerRespVO>builder().total(0L).records(Collections.EMPTY_LIST).build();
        }
        List<SaleStatisticsDealerRespVO> respVOList = pagingVO.getRecords();

        //组装填充经销商销售业绩的相关信息
        translateDealer(respVOList);

        return PagingVO.<SaleStatisticsDealerRespVO>builder()
                .total(pagingVO.getTotal())
                .records(respVOList)
                .build();
    }

    /**
     * 检查业务员编码是否是当前登录人自己以及所属下级
     *
     * @param agentEmp 所属业务员编码
     * @return true 是 false 否
     */
    private boolean checkOneselfAndSub(String agentEmp) {
        //获取当前用户信息
        SysUserDTO sysUserDTO = this.getSysUser();
        //根据当前用户查询所属员工信息
        SysEmployeeBasicDTO employeeBasicDTO = this.getEmployeeByUser(sysUserDTO);
        //获取下属员工
        List<String> subEmployeeCodeList = this.queryAllSubEmployeeByCode(employeeBasicDTO.getCode());
        if (CollectionUtils.isEmpty(subEmployeeCodeList)) {
            return false;
        }

        return subEmployeeCodeList.contains(agentEmp);
    }

    /**
     * 查询符合查询条件的已发生交易的经销商编码
     *
     * @param pageParam 入参
     * @return 经销商编码
     */
    @Override
    public List<String> statisticDealerCode(SaleStatisticsDealerQueryVO pageParam) {
        //根据统计月份的日期拼装入参里的统计时间起始、统计时间截至
        assembleDocTime(pageParam);
        //明细入参查询
        if (StringUtils.isNotBlank(pageParam.getAgentEmp())) {
            SaleStatisticsDealerDtlQueryVO dealerDtlQueryVO = new SaleStatisticsDealerDtlQueryVO();
            dealerDtlQueryVO.setCode(pageParam.getAgentEmp());
            List<Long> masIdList = selectMasIdByDtlParam(dealerDtlQueryVO);
            if (CollectionUtils.isEmpty(masIdList)) {
                return Collections.EMPTY_LIST;
            }
            List<Long> ids = CollectionUtils.isEmpty(pageParam.getIds()) ? Collections.EMPTY_LIST : pageParam.getIds();
            List<Long> idList = Stream.of(ids, masIdList).flatMap(Collection::stream).filter(Objects::nonNull).distinct().collect(Collectors.toList());
            pageParam.setIds(idList);

        }

//        if (Objects.nonNull(pageParam.getDocTimeStart())) {
//            String timeStrStart = pageParam.getDocTimeStart().format(DateTimeFormatter.ofPattern("yyyyMM"));
//            Integer integerStart = Integer.valueOf(timeStrStart);
//            pageParam.setDocTimeItStart(integerStart);
//        }
//        if (Objects.nonNull(pageParam.getDocTimeEnd())) {
//            String timeStrEnd = pageParam.getDocTimeEnd().format(DateTimeFormatter.ofPattern("yyyyMM"));
//            Integer integerEnd = Integer.valueOf(timeStrEnd);
//            pageParam.setDocTimeItEnd(integerEnd);
//
//        }

        List<SaleStatisticsDealerRespVO> dealerRespVOList = statisticsDealerRepoProc.dealerQuery(pageParam);
        if (CollectionUtils.isEmpty(dealerRespVOList)) {
            return Collections.EMPTY_LIST;
        }

        List<String> dealerCodes = dealerRespVOList.stream().map(SaleStatisticsDealerRespVO::getDealerCode).filter(Objects::nonNull).distinct().collect(Collectors.toList());
        if (CollectionUtils.isEmpty(dealerCodes)) {
            return Collections.EMPTY_LIST;
        }

        return dealerCodes;
    }

    /**
     * 经销商销售业绩统计汇总
     *
     * @param pageParam 入参
     * @return 统计汇总信息
     */
    @Override
    public StatisticsDealerSumRespVO pageSum(SaleStatisticsDealerQueryVO pageParam) {
        StatisticsDealerSumRespVO sumRespVO = new StatisticsDealerSumRespVO();

        //根据统计月份的日期拼装入参里的统计时间起始、统计时间截至
        assembleDocTime(pageParam);
        //明细入参查询
        if (StringUtils.isNotBlank(pageParam.getAgentEmp())) {
            SaleStatisticsDealerDtlQueryVO dealerDtlQueryVO = new SaleStatisticsDealerDtlQueryVO();
            dealerDtlQueryVO.setCode(pageParam.getAgentEmp());
            List<Long> masIdList = selectMasIdByDtlParam(dealerDtlQueryVO);
            if (CollectionUtils.isEmpty(masIdList)) {
                return sumRespVO;
            }
            List<Long> ids = CollectionUtils.isEmpty(pageParam.getIds()) ? Collections.EMPTY_LIST : pageParam.getIds();
            List<Long> idList = Stream.of(ids, masIdList).flatMap(Collection::stream).filter(Objects::nonNull).distinct().collect(Collectors.toList());
            pageParam.setIds(idList);

        }

//        if (Objects.nonNull(pageParam.getDocTimeStart())) {
//            String timeStrStart = pageParam.getDocTimeStart().format(DateTimeFormatter.ofPattern("yyyyMM"));
//            Integer integerStart = Integer.valueOf(timeStrStart);
//            pageParam.setDocTimeItStart(integerStart);
//        }
//        if (Objects.nonNull(pageParam.getDocTimeEnd())) {
//            String timeStrEnd = pageParam.getDocTimeEnd().format(DateTimeFormatter.ofPattern("yyyyMM"));
//            Integer integerEnd = Integer.valueOf(timeStrEnd);
//            pageParam.setDocTimeItEnd(integerEnd);
//
//        }

        StatisticsDealerSumRespVO dealerSumRespVO = saleStatisticsDealerRepoSqlProc.pageSum(pageParam);
        if (Objects.isNull(dealerSumRespVO)) {
            return sumRespVO;
        }

        return dealerSumRespVO;
    }

    /**
     * 经销商销售业绩统计汇总-APP端
     *
     * @param pageParam 入参
     * @return 统计汇总信息
     */
    @Override
    public StatisticsDealerSumRespVO appPageSum(SaleStatisticsDealerQueryVO pageParam) {
        StatisticsDealerSumRespVO sumRespVO = new StatisticsDealerSumRespVO();

        //根据统计月份的日期拼装入参里的统计时间起始、统计时间截至
        assembleDocTime(pageParam);
        if (StringUtils.isBlank(pageParam.getAgentEmp())) {
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "业务员编码为空!");
        }
        //检查业务员编码是否是当前登录人自己以及所属下级
        if (!checkOneselfAndSub(pageParam.getAgentEmp())) {
            return sumRespVO;
        }
        //明细入参查询
        if (StringUtils.isNotBlank(pageParam.getAgentEmp())) {
            SaleStatisticsDealerDtlQueryVO dealerDtlQueryVO = new SaleStatisticsDealerDtlQueryVO();
            dealerDtlQueryVO.setCode(pageParam.getAgentEmp());
            //此业务员的个人业绩列表，根据level='0'，匹配code
            //dealerDtlQueryVO.setLevel(ConstantsSale.STATISTICS_DEALER_DTL_LEVEL_0);
            if (Objects.isNull(pageParam.getSeeSelf())) {
                pageParam.setSeeSelf(true);
            }
            if (pageParam.getSeeSelf()) {
                //此业务员的个人业绩列表，根据level='0'，匹配code
                dealerDtlQueryVO.setLevel(ConstantsSale.STATISTICS_DEALER_DTL_LEVEL_0);
            }

            List<Long> masIdList = selectMasIdByDtlParam(dealerDtlQueryVO);
            if (CollectionUtils.isEmpty(masIdList)) {
                return sumRespVO;
            }
            List<Long> ids = CollectionUtils.isEmpty(pageParam.getIds()) ? Collections.EMPTY_LIST : pageParam.getIds();
            List<Long> idList = Stream.of(ids, masIdList).flatMap(Collection::stream).filter(Objects::nonNull).distinct().collect(Collectors.toList());
            pageParam.setIds(idList);

        }
//        if (Objects.nonNull(pageParam.getDocTimeStart())) {
//            String timeStrStart = pageParam.getDocTimeStart().format(DateTimeFormatter.ofPattern("yyyyMM"));
//            Integer integerStart = Integer.valueOf(timeStrStart);
//            pageParam.setDocTimeItStart(integerStart);
//        }
//        if (Objects.nonNull(pageParam.getDocTimeEnd())) {
//            String timeStrEnd = pageParam.getDocTimeEnd().format(DateTimeFormatter.ofPattern("yyyyMM"));
//            Integer integerEnd = Integer.valueOf(timeStrEnd);
//            pageParam.setDocTimeItEnd(integerEnd);
//
//        }

        StatisticsDealerSumRespVO dealerSumRespVO = saleStatisticsDealerRepoSqlProc.pageSum(pageParam);
        if (Objects.isNull(dealerSumRespVO)) {
            return sumRespVO;
        }

        return dealerSumRespVO;
    }

    /**
     * 根据明细入参查询数据并返回主表ID集合
     *
     * @param dtlQueryVO 明细入参
     * @return 主表ID集合
     */
    private List<Long> selectMasIdByDtlParam(SaleStatisticsDealerDtlQueryVO dtlQueryVO) {
        //List<SaleStatisticsDealerDtlRespVO> dtlRespVOList = statisticsDealerDtlService.selectByParam(dtlQueryVO);
        List<SaleStatisticsDealerDtlRespVO> dtlRespVOList = statisticsDealerDtlService.selectPidByQueryVO(dtlQueryVO);

        if (CollectionUtils.isEmpty(dtlRespVOList)) {
            return Collections.EMPTY_LIST;
        }
        List<Long> masIdList = dtlRespVOList.stream().map(SaleStatisticsDealerDtlRespVO::getPid).filter(Objects::nonNull).distinct().collect(Collectors.toList());
        if (CollectionUtils.isEmpty(masIdList)) {
            return Collections.EMPTY_LIST;
        }
        return masIdList;
    }

    /**
     * 组装填充经销商销售业绩的相关信息
     *
     * @param respVOList 主表数据信息
     * @return
     */
    private void translateDealer(List<SaleStatisticsDealerRespVO> respVOList) {
        //根据主表ID查询附表信息
        List<Long> idList = respVOList.stream().map(SaleStatisticsDealerRespVO::getId).filter(Objects::nonNull).distinct().collect(Collectors.toList());
        List<SaleStatisticsDealerDtlRespVO> dtlRespVOList = selectDtlByMasId(idList);
        respVOList.forEach(respVO -> {
            //查询附件信息
            List<SaleStatisticsDealerDtlRespVO> dtlRespVOS = dtlRespVOList.stream().filter(dtlRespVO -> Objects.equals(dtlRespVO.getMasId(), respVO.getId())).collect(Collectors.toList());
            respVO.setDtlRespVOS(dtlRespVOS);
            String docMonth = respVO.getDocTime().format(DateTimeFormatter.ofPattern("yyyy-MM"));
            respVO.setDocMonth(docMonth);
            respVO.setDocDay(respVO.getDocTime().toLocalDate().toString());

            dtlRespVOS.stream().forEach(dtlRespVO -> {
                StringJoiner spliceName = new StringJoiner(" ").add(dtlRespVO.getTypeName()).add(dtlRespVO.getName());
                if (Objects.equals(ConstantsSale.STATISTICS_DEALER_DTL_LEVEL_0, dtlRespVO.getLevel())) {
                    respVO.setAgentEmp(dtlRespVO.getCode());
                    respVO.setAgentEmpName(spliceName.toString());
                } else if (Objects.equals(ConstantsSale.STATISTICS_DEALER_DTL_LEVEL_1, dtlRespVO.getLevel())) {
                    respVO.setLevelOne(spliceName.toString());
                } else if (Objects.equals(ConstantsSale.STATISTICS_DEALER_DTL_LEVEL_2, dtlRespVO.getLevel())) {
                    respVO.setLevelTwo(spliceName.toString());
                } else if (Objects.equals(ConstantsSale.STATISTICS_DEALER_DTL_LEVEL_3, dtlRespVO.getLevel())) {
                    respVO.setLevelThree(spliceName.toString());
                } else if (Objects.equals(ConstantsSale.STATISTICS_DEALER_DTL_LEVEL_4, dtlRespVO.getLevel())) {
                    respVO.setLevelFour(spliceName.toString());
                } else if (Objects.equals(ConstantsSale.STATISTICS_DEALER_DTL_LEVEL_5, dtlRespVO.getLevel())) {
                    respVO.setLevelFive(spliceName.toString());
                }
            });

        });
    }

    private List<SaleStatisticsDealerDtlRespVO> selectDtlByMasId(List<Long> masIds) {
        if (CollectionUtils.isEmpty(masIds)) {
            return Collections.EMPTY_LIST;
        }
        SaleStatisticsDealerDtlQueryVO dtlQueryVO = new SaleStatisticsDealerDtlQueryVO();
        dtlQueryVO.setMasIds(masIds);
        List<SaleStatisticsDealerDtlRespVO> dtlRespVOList = statisticsDealerDtlService.selectByParam(dtlQueryVO);
        if (CollectionUtils.isEmpty(dtlRespVOList)) {
            return Collections.EMPTY_LIST;
        }
        return dtlRespVOList;
    }

    /**
     * 根据统计月份的日期拼装入参里的统计时间起始、统计时间截至
     *
     * @param pageParam 查询入参
     */
    public void assembleDocTime(SaleStatisticsDealerQueryVO pageParam) {
        if (Objects.isNull(pageParam.getDocTimeStart()) || Objects.isNull(pageParam.getDocTimeEnd())) {
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "统计日期为空!");
        }
        if (Objects.nonNull(pageParam.getDocTimeStart())) {
            // 获取第一天的最小时间
            LocalDate localDateStart = pageParam.getDocTimeStart().toLocalDate();
            LocalDateTime minTime = LocalDateTime.of(localDateStart, LocalTime.MIN);
            pageParam.setDocTimeStart(minTime);
        }
        if (Objects.nonNull(pageParam.getDocTimeEnd())) {
            // 获取最后一天的最大时间
            LocalDate localDateEnd = pageParam.getDocTimeEnd().toLocalDate();
            LocalDateTime maxTime = LocalDateTime.of(localDateEnd, ConstantsSale.LOCAL_TIME_MAX);
            pageParam.setDocTimeEnd(maxTime);
        }
    }


    /**
     * 经销商销售业绩导出分页查询
     *
     * @param pageParam 入参
     * @return 经销商销售业绩信息集合
     */
    /*@Override
    @SysCodeProc
    public PagingVO<SaleStatisticsDealerRespVO> exportPage(SaleStatisticsDealerQueryVO pageParam) {

    }*/

    /**
     * 经销商销售业绩详情-头部信息和车型分页明细数据
     *
     * @param pageParam 入参
     * @return 经销商销售业绩详情信息
     */
    @Override
    @SysCodeProc
    public PagingVO<SaleStatisticsDealerRespVO> appDetails(SaleStatisticsDealerQueryVO pageParam) {
        /*StatisticsDealerAppDetailsRespVO appDetailsRespVO = new StatisticsDealerAppDetailsRespVO();
        appDetailsRespVO.setDealerCode();
        appDetailsRespVO.setDealerName();
        appDetailsRespVO.setDealerSerialNo();*/
        //校验入参必填项
        appDetailsCheck(pageParam);
        //明细入参查询
        if (StringUtils.isNotBlank(pageParam.getAgentEmp())) {
            SaleStatisticsDealerDtlQueryVO dealerDtlQueryVO = new SaleStatisticsDealerDtlQueryVO();
            dealerDtlQueryVO.setCode(pageParam.getAgentEmp());
            //此业务员的个人业绩列表，根据level='0'，匹配code
            // dealerDtlQueryVO.setLevel(ConstantsSale.STATISTICS_DEALER_DTL_LEVEL_0);
            if (Objects.isNull(pageParam.getSeeSelf())) {
                pageParam.setSeeSelf(true);
            }
            if (pageParam.getSeeSelf()) {
                //此业务员的个人业绩列表，根据level='0'，匹配code
                dealerDtlQueryVO.setLevel(ConstantsSale.STATISTICS_DEALER_DTL_LEVEL_0);
            }
            List<Long> masIdList = selectMasIdByDtlParam(dealerDtlQueryVO);
            if (CollectionUtils.isEmpty(masIdList)) {
                return PagingVO.<SaleStatisticsDealerRespVO>builder().total(0L).records(Collections.EMPTY_LIST).build();
                //appDetailsRespVO.setPagingVO(dealerRespVOPagingVO);
                // return appDetailsRespVO;
            }
            List<Long> ids = CollectionUtils.isEmpty(pageParam.getIds()) ? Collections.EMPTY_LIST : pageParam.getIds();
            List<Long> idList = Stream.of(ids, masIdList).flatMap(Collection::stream).filter(Objects::nonNull).distinct().collect(Collectors.toList());
            pageParam.setIds(idList);

        }

        if (StringUtils.isBlank(pageParam.getRegion())) {
            pageParam.setRegionFlag(1);
        }
        if (StringUtils.isBlank(pageParam.getAgentEmp())) {
            pageParam.setSalesmanPathFlag(1);
        }

        PagingVO<SaleStatisticsDealerRespVO> pagingVO = statisticsDealerRepoProc.appDetailsPage(pageParam);
        if (CollectionUtils.isEmpty(pagingVO.getRecords())) {
            return PagingVO.<SaleStatisticsDealerRespVO>builder().total(0L).records(Collections.EMPTY_LIST).build();
            //appDetailsRespVO.setPagingVO(appPagingVO);
            //return appDetailsRespVO;
        }

        // appDetailsRespVO.setPagingVO(pagingVO);
        return pagingVO;
    }

    private void appDetailsCheck(SaleStatisticsDealerQueryVO pageParam) {
        if (StringUtils.isBlank(pageParam.getDealerCode())) {
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "客户编码为空!");
        }
//        if (Objects.isNull(pageParam.getDocMonth())) {
//            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "统计月份为空!");
//        }
        if (Objects.isNull(pageParam.getDocTimeStart())||Objects.isNull(pageParam.getDocTimeEnd())) {
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "统计时间为空!");
        }
//        if (StringUtils.isBlank(pageParam.getRegion())) {
//            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "区域为空!");
//        }
        if (StringUtils.isBlank(pageParam.getAgentEmp())) {
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "业务员编码为空!");
        }
        /*if (StringUtils.isBlank(pageParam.getVehicleType())) {
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "车型为空!");
        }*/
    }


    /**
     * 经销商销售业绩根据业务员编码/年月查询其当月的团队业绩和我的业绩
     *
     * @param pageParam 业务员编码/年月
     * @return 信息集合
     */
    @Override
    public List<StatisticsDealerAppSumRespVO> appSumByEmpCode(SaleStatisticsDealerQueryVO pageParam) {
        List<String> empCodes = pageParam.getAgentEmpList();
        String docMonth = pageParam.getDocMonth();
        if (CollectionUtils.isEmpty(empCodes)) {
            return Collections.EMPTY_LIST;
        }

        if(pageParam.getDocTimeStart() != null){
            pageParam.setDocTimeStart(LocalDateTime.of(pageParam.getDocTimeStart().toLocalDate(),LocalTime.MIN));
        }
        if(pageParam.getDocTimeEnd() != null){
            pageParam.setDocTimeEnd(LocalDateTime.of(pageParam.getDocTimeEnd().toLocalDate(),ConstantsSale.LOCAL_TIME_MAX));
        }
        //默认为当月
        if(pageParam.getDocTimeStart() == null && pageParam.getDocTimeEnd() == null && StringUtils.isBlank(docMonth)){
//            docMonth = LocalDate.now().format(DateTimeFormatter.ofPattern("yyyy-MM"));
            pageParam.setDocTimeStart(LocalDateTime.of(LocalDate.now(),LocalTime.MIN));
            pageParam.setDocTimeEnd(LocalDateTime.of(LocalDate.now(),ConstantsSale.LOCAL_TIME_MAX));
        }
        SaleStatisticsDealerQueryVO myQueryVO = new SaleStatisticsDealerQueryVO();
        myQueryVO.setDocMonth(docMonth);
        myQueryVO.setDocTimeStart(pageParam.getDocTimeStart());
        myQueryVO.setDocTimeEnd(pageParam.getDocTimeEnd());
        myQueryVO.setAgentEmpList(empCodes);
        myQueryVO.setLevel(ConstantsSale.STATISTICS_DEALER_DTL_LEVEL_0);
        List<StatisticsDealerAppSumRespVO> myRespVOList = selectAppSumByEmpCode(myQueryVO);

        SaleStatisticsDealerQueryVO teamQueryVO = new SaleStatisticsDealerQueryVO();
        teamQueryVO.setDocMonth(docMonth);
        teamQueryVO.setDocTimeStart(pageParam.getDocTimeStart());
        teamQueryVO.setDocTimeEnd(pageParam.getDocTimeEnd());
        teamQueryVO.setAgentEmpList(empCodes);
        List<StatisticsDealerAppSumRespVO> teamRespVOList = selectAppSumByEmpCode(teamQueryVO);

        List<StatisticsDealerAppSumRespVO> respVOList = new ArrayList<>();
        for (String code : empCodes) {
            StatisticsDealerAppSumRespVO appSumRespVO = new StatisticsDealerAppSumRespVO();
            appSumRespVO.setAgentEmpCode(code);
            appSumRespVO.setMyShipTotalQty(BigDecimal.ZERO);
            appSumRespVO.setMyOrderTotalQty(BigDecimal.ZERO);
            appSumRespVO.setTeamShipTotalQty(BigDecimal.ZERO);
            appSumRespVO.setTeamOrderTotalQty(BigDecimal.ZERO);
            appSumRespVO.setDealerTotalQty(0L);
            Optional<StatisticsDealerAppSumRespVO> myOptional = myRespVOList.stream().filter(respVO -> Objects.equals(respVO.getAgentEmpCode(), code)).findFirst();
            if (myOptional.isPresent()) {
                appSumRespVO.setMyShipTotalQty(myOptional.get().getShipTotalQty());
                appSumRespVO.setMyOrderTotalQty(myOptional.get().getOrderTotalQty());
                appSumRespVO.setDealerTotalQty(myOptional.get().getDealerTotalQty());
            }
            Optional<StatisticsDealerAppSumRespVO> teamOptional = teamRespVOList.stream().filter(respVO -> Objects.equals(respVO.getAgentEmpCode(), code)).findFirst();
            if (teamOptional.isPresent()) {
                appSumRespVO.setTeamShipTotalQty(teamOptional.get().getShipTotalQty());
                appSumRespVO.setTeamOrderTotalQty(teamOptional.get().getOrderTotalQty());
                appSumRespVO.setDealerTotalQty(teamOptional.get().getDealerTotalQty());
            }
            respVOList.add(appSumRespVO);
        }


        return respVOList;
    }

    private List<StatisticsDealerAppSumRespVO> selectAppSumByEmpCode(SaleStatisticsDealerQueryVO queryVO) {
        List<StatisticsDealerAppSumRespVO> respVOList = statisticsDealerRepoProc.appSumByEmpCode(queryVO);
        if (CollectionUtils.isEmpty(respVOList)) {
            return Collections.EMPTY_LIST;
        }

        return respVOList;
    }

    private List<SaleStatisticsDealerRespVO> selectByQueryVO(SaleStatisticsDealerQueryVO queryVO) {
        List<SaleStatisticsDealerRespVO> respVOList = statisticsDealerRepoProc.selectByQueryVO(queryVO);
        if (CollectionUtils.isEmpty(respVOList)) {
            return Collections.EMPTY_LIST;
        }
        return respVOList;
    }

    private List<SaleStatisticsDealerDtlRespVO> selectDtlByParam(SaleStatisticsDealerDtlQueryVO dtlQueryVO) {
        List<SaleStatisticsDealerDtlRespVO> dtlRespVOList = statisticsDealerDtlService.selectByParam(dtlQueryVO);
        if (CollectionUtils.isEmpty(dtlRespVOList)) {
            return Collections.EMPTY_LIST;
        }
        return dtlRespVOList;
    }

    /**
     * 获取当前用户
     *
     * @param
     * @return 当前用户信息
     */
    private SysUserDTO getSysUser() {
        //获取当前用户
        GeneralUserDetails userDetails = SecurityContextUtil.currentUser();
        if (Objects.isNull(userDetails) || Objects.isNull(userDetails.getUser())) {
            throw new BusinessException(ApiCode.UNAUTHORIZED, "获取当前用户信息失败");
        }
        SysUserDTO sysUser = userDetails.getUser();
        if (ObjectUtils.isEmpty(sysUser)) {
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "用户信息获取失败");
        }

        return sysUser;
    }

    /**
     * 根据当前用户查询所属员工信息
     *
     * @param sysUserDTO 当前用户信息
     * @return 所属员工信息
     */
    private SysEmployeeBasicDTO getEmployeeByUser(SysUserDTO sysUserDTO) {
        if (Objects.isNull(sysUserDTO)) {
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "当前用户信息为空!");
        }
        //根据用户ID获取所属员工
        SysEmployeeBasicDTO employeeBasicDTO = rmiEmployeeRpcService.findEmployeeByUserId(sysUserDTO.getId());
        if (Objects.isNull(employeeBasicDTO)) {
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "当前用户所属员工信息为空!");
        }
        return employeeBasicDTO;
    }

    private List<String> queryAllSubEmployeeByCode(String salesmanNo) {
        Set<String> codeList = new HashSet<>();
        //包含当前登陆人
        codeList.add(salesmanNo);
        List<EmployeeUnderlingDTO> resultData = rmiEmployeeRpcService.getUnderlingByCode(salesmanNo, Boolean.TRUE, Boolean.TRUE);
        if (CollectionUtils.isEmpty(resultData)) {
            return codeList.stream().filter(Objects::nonNull).collect(Collectors.toList());
        }

        if (!CollectionUtils.isEmpty(resultData)) {
            List<EmployeeUnderlingDTO> listEmp = new ArrayList<>();
            List<EmployeeUnderlingDTO> res = this.queryAllSubTreeToList(resultData, listEmp);
            codeList.addAll(res.stream().map(EmployeeUnderlingDTO::getCode).collect(Collectors.toList()));
        }

        return codeList.stream().filter(Objects::nonNull).collect(Collectors.toList());
    }

    private List<EmployeeUnderlingDTO> queryAllSubTreeToList(List<EmployeeUnderlingDTO> source, List<EmployeeUnderlingDTO> obj) {
        source.stream().forEach(v -> {
            EmployeeUnderlingDTO t = new EmployeeUnderlingDTO();
            BeanUtils.copyProperties(v, t);
            t.setUnderlingList(new ArrayList<>());
            obj.add(t);
            if (CollectionUtil.isNotEmpty(v.getUnderlingList())) {
                queryAllSubTreeToList(v.getUnderlingList(), obj);
            }
        });
        return obj;
    }

    /**
     * 更新区域
     *
     * @param ids 入参
     * @return 更新数据ID
     */
    @Override
    @Transactional(rollbackFor = {Exception.class})
    public List<Long> updateRegion(List<Long> ids) {
        if (CollectionUtils.isEmpty(ids)) {
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "数据ID为空,请检查!");
        }

        SaleStatisticsDealerQueryVO queryVO = new SaleStatisticsDealerQueryVO();
        queryVO.setIds(ids);
        //1.先根据ID查询出选择的数据
        //校验勾选的数据区域字段是否为空，有值则报错：请勾选没有区域的数据进行更新！
        //2.根据选择的数据的分组字段，再去查该分组下的所有数据
        //3.1再根据客户编码，获取客户表里对应的区域存表
        List<SaleStatisticsDealerRespVO> respVOList = selectByQueryVO(queryVO);
        if (CollectionUtils.isEmpty(respVOList)) {
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "未查询到选择数据信息!");
        }
//        boolean errorFlag = respVOList.stream().anyMatch(respVO -> StringUtils.isNotBlank(respVO.getRegion()));
//        if (errorFlag) {
//            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "请勾选没有区域的数据进行更新!");
//        }

        //客户信息
        List<String> dealerCodes = respVOList.stream().map(SaleStatisticsDealerRespVO::getDealerCode).filter(Objects::nonNull).distinct().collect(Collectors.toList());
        List<LmSaveCustRespVO> custRespVOList = selectCustByCodes(dealerCodes);

        List<SaleStatisticsDealerDO> updateStatisticsDealerDOList = new ArrayList<>();
        for (SaleStatisticsDealerRespVO dealerRespVO : respVOList) {
            SaleStatisticsDealerQueryVO dealerQueryVO = new SaleStatisticsDealerQueryVO();
            dealerQueryVO.setDealerCode(dealerRespVO.getDealerCode());
            dealerQueryVO.setRegion(null);
            dealerQueryVO.setSalesmanPath(dealerRespVO.getSalesmanPath());
            dealerQueryVO.setVehicleType(dealerRespVO.getVehicleType());
            dealerQueryVO.setDocTime(dealerRespVO.getDocTime());
//            String docMonth = dealerRespVO.getDocTime().format(DateTimeFormatter.ofPattern("yyyy-MM"));
//            dealerQueryVO.setDocMonth(docMonth);
            //这样查询所有字段,以防后续表新增字段,此处更新时导致漏掉字段
            Iterable<SaleStatisticsDealerDO> dealerDOIterable = statisticsDealerRepo.findAll(statisticsDealerRepoProc.selectRegionOrSalesmanPathWhere(dealerQueryVO));
            List<SaleStatisticsDealerDO> statisticsDealerDOList = StreamSupport.stream(dealerDOIterable.spliterator(), false).collect(Collectors.toList());
            if (!CollectionUtils.isEmpty(statisticsDealerDOList)) {
                assembleStatisticsDealerRegion(statisticsDealerDOList, custRespVOList);
                updateStatisticsDealerDOList.addAll(statisticsDealerDOList);
            }

        }
        if (!CollectionUtils.isEmpty(updateStatisticsDealerDOList)) {
            //更新主表区域
            return statisticsDealerRepo.saveAll(updateStatisticsDealerDOList).stream().map(SaleStatisticsDealerDO::getId).collect(Collectors.toList());
        }


        return Collections.EMPTY_LIST;
    }

    private void assembleStatisticsDealerRegion(List<SaleStatisticsDealerDO> statisticsDealerDOList, List<LmSaveCustRespVO> custRespVOList) {
        statisticsDealerDOList.forEach(saleStatisticsDealerDO -> {
            if (CollectionUtils.isEmpty(custRespVOList)) {
                saleStatisticsDealerDO.setUpdateRegionFailureReason("未查询到客户数据");
            } else {
                Optional<LmSaveCustRespVO> first = custRespVOList.stream().filter(cust -> Objects.equals(saleStatisticsDealerDO.getDealerCode(), cust.getCustCode())).findFirst();
                if (first.isPresent()) {
                    if (StringUtils.isBlank(first.get().getRegion())) {
                        saleStatisticsDealerDO.setUpdateRegionFailureReason("此客户的区域信息为空");
                    } else {
                        saleStatisticsDealerDO.setRegion(first.get().getRegion());
                        saleStatisticsDealerDO.setUpdateRegionFailureReason(null);

                    }
                } else {
                    saleStatisticsDealerDO.setUpdateRegionFailureReason("未查询到此客户主信息");
                }
            }
        });
    }

    /**
     * 更新业务员
     *
     * @param ids 入参
     * @return 更新数据ID
     */
    @Override
    @Transactional(rollbackFor = {Exception.class})
    public List<Long> updateSalesmanPath(List<Long> ids) {
        if (CollectionUtils.isEmpty(ids)) {
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "数据ID为空,请检查!");
        }

        SaleStatisticsDealerQueryVO queryVO = new SaleStatisticsDealerQueryVO();
        queryVO.setIds(ids);
        //1.先根据ID查询出选择的数据
        //校验勾选的数据业务员关系路径字段是否为空，有值则报错：请勾选没有业务员的数据进行更新！
        //2.根据选择的数据的分组字段，再去查该分组下的所有数据
        //3.2根据客户编码，获取客户表里对应的业务员ID，然后根据业务员ID调用王天昊的接口，获取业务员关系路径存表
        List<SaleStatisticsDealerRespVO> respVOList = selectByQueryVO(queryVO);
        if (CollectionUtils.isEmpty(respVOList)) {
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "未查询到选择数据信息!");
        }
//        boolean errorFlag = respVOList.stream().anyMatch(respVO -> StringUtils.isNotBlank(respVO.getSalesmanPath()));
//        if (errorFlag) {
//            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "请勾选没有业务员的数据进行更新!");
//        }

        //客户信息
        List<String> dealerCodes = respVOList.stream().map(SaleStatisticsDealerRespVO::getDealerCode).filter(Objects::nonNull).distinct().collect(Collectors.toList());
        List<LmSaveCustRespVO> custRespVOList = selectCustByCodes(dealerCodes);
        Map<String, Long> custCodeAndAgentEmpIdMap = new HashMap<>();
        if (!CollectionUtils.isEmpty(custRespVOList)){
            custRespVOList.stream().forEach(t -> {
                custCodeAndAgentEmpIdMap.put(t.getCustCode(),t.getAgentEmpId());
            });
        }
        //业务员关系路径信息
        List<SalesmanLevelInfoDTO> salesmanLevelInfoDTOList = selectSalesmanLevelByIds(custRespVOList);

        List<SaleStatisticsDealerDO> updateStatisticsDealerDOList = new ArrayList<>();
        for (SaleStatisticsDealerRespVO dealerRespVO : respVOList) {
            SaleStatisticsDealerQueryVO dealerQueryVO = new SaleStatisticsDealerQueryVO();
            dealerQueryVO.setDealerCode(dealerRespVO.getDealerCode());
            dealerQueryVO.setRegion(dealerRespVO.getRegion());
            dealerQueryVO.setSalesmanPath(null);
            dealerQueryVO.setVehicleType(dealerRespVO.getVehicleType());
            dealerQueryVO.setDocTime(dealerRespVO.getDocTime());
//            String docMonth = dealerRespVO.getDocTime().format(DateTimeFormatter.ofPattern("yyyy-MM"));
//            dealerQueryVO.setDocMonth(docMonth);
            //这样查询所有字段,以防后续表新增字段,此处更新时导致漏掉字段
            Iterable<SaleStatisticsDealerDO> dealerDOIterable = statisticsDealerRepo.findAll(statisticsDealerRepoProc.selectRegionOrSalesmanPathWhere(dealerQueryVO));
            List<SaleStatisticsDealerDO> statisticsDealerDOList = StreamSupport.stream(dealerDOIterable.spliterator(), false).collect(Collectors.toList());
            if (!CollectionUtils.isEmpty(statisticsDealerDOList)) {
                assembleStatisticsDealerSalesmanPath(statisticsDealerDOList, custRespVOList,custCodeAndAgentEmpIdMap,salesmanLevelInfoDTOList);
                updateStatisticsDealerDOList.addAll(statisticsDealerDOList);
            }

        }
        if (!CollectionUtils.isEmpty(updateStatisticsDealerDOList)) {
            //更新主表业务员关系路径
            List<Long> statisticsDealerIds = statisticsDealerRepo.saveAll(updateStatisticsDealerDOList).stream().map(SaleStatisticsDealerDO::getId).collect(Collectors.toList());
            //新增附表信息
            assembleDtlSalesmanPath(updateStatisticsDealerDOList);
            return statisticsDealerIds;
        }


        return Collections.EMPTY_LIST;
    }

    private void assembleDtlSalesmanPath(List<SaleStatisticsDealerDO> updateStatisticsDealerDOList){
        //先删除明细
        List<Long> idList = updateStatisticsDealerDOList.stream().map(SaleStatisticsDealerDO::getId).filter(Objects::nonNull).distinct().collect(Collectors.toList());
        statisticsDealerDtlRepo.deleteByMasIdIn(idList);

        List<SaleStatisticsDealerDtlSaveVO> dealerDtlSaveVOList = new ArrayList<>();
        updateStatisticsDealerDOList.forEach(saveVO -> {
            if (StringUtils.isNotBlank(saveVO.getSalesmanPath())) {
                List<SaleStatisticsDealerDtlSaveVO> dtlSaveVOList = JSONObject.parseArray(saveVO.getSalesmanPath(), SaleStatisticsDealerDtlSaveVO.class);
                if (!CollectionUtils.isEmpty(dtlSaveVOList)) {
                    dtlSaveVOList.forEach(dtlSaveVO -> {
                        dtlSaveVO.setMasId(saveVO.getId());
                    });
                    dealerDtlSaveVOList.addAll(dtlSaveVOList);
                }
            }
        });

        //保存业务员对应的用户ID,用于数据权限
        if (!CollectionUtils.isEmpty(dealerDtlSaveVOList)) {
            Set<String> employeeCodes = dealerDtlSaveVOList.stream().map(SaleStatisticsDealerDtlSaveVO::getCode).filter(Objects::nonNull).collect(Collectors.toSet());
            List<SysEmployeeBasicDTO> basicDTOList = rmiEmployeeRpcService.findEmployeeByCodes(employeeCodes);
            dealerDtlSaveVOList.forEach(saveVO -> {
                Optional<SysEmployeeBasicDTO> optional = basicDTOList.stream().filter(employeeBasicDTO -> Objects.equals(saveVO.getCode(), employeeBasicDTO.getCode())).findFirst();
                optional.ifPresent(employeeBasicDTO -> saveVO.setUserId(employeeBasicDTO.getUserId()));
            });

            statisticsDealerDtlService.saveStatisticsDealerDtl(dealerDtlSaveVOList);
        }
    }

    private void assembleStatisticsDealerSalesmanPath(List<SaleStatisticsDealerDO> statisticsDealerDOList,List<LmSaveCustRespVO> custRespVOList,
                                                      Map<String, Long> custCodeAndAgentEmpIdMap,List<SalesmanLevelInfoDTO> salesmanLevelInfoDTOList) {
        statisticsDealerDOList.forEach(saleStatisticsDealerDO -> {
            if (CollectionUtils.isEmpty(custRespVOList)) {
                saleStatisticsDealerDO.setUpdateSalesmanFailureReason("未查询到客户数据");
            } else {
                Optional<LmSaveCustRespVO> custOptional = custRespVOList.stream().filter(custRespVO -> Objects.equals(saleStatisticsDealerDO.getDealerCode(), custRespVO.getCustCode())).findFirst();
                if (custOptional.isPresent()) {
                    if (CollectionUtils.isEmpty(custCodeAndAgentEmpIdMap) || Objects.isNull(custCodeAndAgentEmpIdMap.get(custOptional.get().getCustCode()))){
                        saleStatisticsDealerDO.setUpdateSalesmanFailureReason("此客户的业务员ID信息为空");
                    }else {
                        if (CollectionUtils.isEmpty(salesmanLevelInfoDTOList)){
                            saleStatisticsDealerDO.setUpdateSalesmanFailureReason("未查询到业务员数据");
                        }else {
                            Long agentEmpId =  custCodeAndAgentEmpIdMap.get(custOptional.get().getCustCode());
                            Optional<SalesmanLevelInfoDTO> salesmanOptional = salesmanLevelInfoDTOList.stream().filter(salesmanLevelInfo -> Objects.equals(agentEmpId,salesmanLevelInfo.getId())).findFirst();
                            if (salesmanOptional.isPresent()){
                               if (CollectionUtils.isEmpty(salesmanOptional.get().getSalesman_path())){
                                   saleStatisticsDealerDO.setUpdateSalesmanFailureReason("此客户的业务员的业务员路径信息为空");
                               }else {
                                   saleStatisticsDealerDO.setSalesmanPath(JSON.toJSONString(salesmanOptional.get().getSalesman_path()));
                                   saleStatisticsDealerDO.setUpdateSalesmanFailureReason(null);
                               }

                            }else {
                                saleStatisticsDealerDO.setUpdateSalesmanFailureReason("未查询到此客户的业务员主信息");
                            }
                        }
                    }
                } else {
                    saleStatisticsDealerDO.setUpdateSalesmanFailureReason("未查询到此客户主信息");
                }
            }
        });
    }

    private List<LmSaveCustRespVO> selectCustByCodes(List<String> custCodes) {
        if (CollectionUtils.isEmpty(custCodes)) {
            return Collections.EMPTY_LIST;
        }
        CustAddrFindParam custAddrFindParam = new CustAddrFindParam();
        custAddrFindParam.setCustCodes(custCodes);
        List<LmSaveCustRespVO> custRespVOList = crmCustService.getCustInfoByParam(custAddrFindParam);
        if (CollectionUtils.isEmpty(custRespVOList)) {
            return Collections.EMPTY_LIST;
        }
        return custRespVOList;
    }

    private List<SalesmanLevelInfoDTO> selectSalesmanLevelByIds(List<LmSaveCustRespVO> custRespVOList) {
        if (CollectionUtils.isEmpty(custRespVOList)) {
            return Collections.EMPTY_LIST;
        }
        List<Long> agentEmpIds = custRespVOList.stream().map(LmSaveCustRespVO::getAgentEmpId).filter(Objects::nonNull).distinct().collect(Collectors.toList());
        if (CollectionUtils.isEmpty(agentEmpIds)) {
            return Collections.EMPTY_LIST;
        }

       /* SalesmanInfoSimpleQueryVO salesmanQuery1VO = new SalesmanInfoSimpleQueryVO();
        salesmanQuery1VO.setIds(agentEmpIds);
        List<SalesmanInfoSimpleRespVO> simpleRespVOList = salesmanInfoService.simpleQuery(salesmanQuery1VO);
        if (CollectionUtils.isEmpty(simpleRespVOList)) {
            return Collections.EMPTY_LIST;
        }*/
        //查询业务员关系路径
        List<SalesmanLevelInfoDTO> salesmanLevelInfoDTOList = agentEmpIds.stream().map(t -> salesmanInfoService.queryLevelInfo(t)).filter(Objects::nonNull).collect(Collectors.toList());
        if (CollectionUtils.isEmpty(salesmanLevelInfoDTOList)) {
            return Collections.EMPTY_LIST;
        }
        return salesmanLevelInfoDTOList;
    }
}
