package com.elitesland.yst.production.sale.core.util.excel;

import com.alibaba.excel.EasyExcelFactory;
import com.alibaba.excel.read.builder.ExcelReaderBuilder;
import com.elitesland.yst.production.sale.core.util.excel.support.DataReadListener;
import com.elitesland.yst.production.sale.core.util.excel.support.ImportDataModel;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.InputStream;
import java.util.Collections;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.function.Consumer;

/**
 * 数据导入工具类.
 *
 * @author Kaiser（wang shao）
 * @date 2021-03-16
 */
public class ExcelImportUtil {

    private final Logger logger = LoggerFactory.getLogger(ExcelImportUtil.class);

    private final ExcelReaderBuilder excelReaderBuilder;
    private DataReadListener readListener;

    private ExcelImportUtil(InputStream inputStream) {
        excelReaderBuilder = EasyExcelFactory.read(inputStream)
                .autoCloseStream(true)
                .autoTrim(true)
                .ignoreEmptyRow(true)
        ;
    }

    public static ExcelImportUtil instance(InputStream inputStream) {
        return new ExcelImportUtil(inputStream);
    }

    /**
     * 头部数据行数
     *
     * @param headRow 头部数据行数
     * @return 类
     */
    public ExcelImportUtil headRow(Integer headRow) {
        excelReaderBuilder.headRowNumber(headRow);
        return this;
    }

    /**
     * 设置数据类型
     *
     * @param dataType     数据类型
     * @param dataTypeAttr 数据类型类的属性，按excel中列的顺序
     * @return 类
     */
    public ExcelImportUtil dataType(Class<?> dataType, List<String> dataTypeAttr) {
        readListener = new DataReadListener(dataTypeAttr);
        readListener.setDataType(dataType);
        return this;
    }

    /**
     * 设置数据类型
     *
     * @param dataType     数据类型
     * @param dataTypeAttr 数据类型类的属性所在行，必须在headRow内
     * @return 类
     */
    public ExcelImportUtil dataType(Class<?> dataType, Integer dataTypeAttr) {
        readListener = new DataReadListener(dataTypeAttr);
        readListener.setDataType(dataType);
        return this;
    }

    /**
     * 同步读取所有数据
     * <p>
     * 数据量大时不建议
     *
     * @return 所有数据
     */
    public List<Object> readAllSync() throws Exception {
        if (readListener == null) {
            return excelReaderBuilder.doReadAllSync();
        }
        excelReaderBuilder.registerReadListener(readListener);

        CompletableFuture<Boolean> completableFuture = CompletableFuture.supplyAsync(() -> {
            excelReaderBuilder.doReadAll();
            return readListener.isFinish();
        });
        if (completableFuture.get()) {
            return readListener.getDataList();
        }

        return Collections.emptyList();
    }

    /**
     * 异步读取数据
     *
     * @param dataConsumer 数据消费者
     * @throws Exception 异常
     */
    public void readAllAsync(Consumer<ImportDataModel> dataConsumer) throws Exception {
        readAllAsync(dataConsumer, null);
    }

    /**
     * 异步读取数据
     *
     * @param dataConsumer 数据消费者
     * @param batchSize    每次读取的数据量，如果为空或不大于1，则表示无限制，一次性获取
     * @throws Exception 异常
     */
    public void readAllAsync(Consumer<ImportDataModel> dataConsumer, Integer batchSize) throws Exception {
        if (readListener == null) {
            readListener = new DataReadListener();
        }
        readListener.setDataSize(batchSize);
        readListener.setDataConsumer(dataConsumer);
        excelReaderBuilder.registerReadListener(readListener);

        excelReaderBuilder.doReadAll();
    }
}
