package com.elitesland.yst.production.sale.service.shop;

import cn.hutool.core.collection.CollUtil;
import cn.hutool.core.lang.Assert;
import cn.hutool.core.text.CharSequenceUtil;
import cn.hutool.core.util.BooleanUtil;
import cn.hutool.core.util.NumberUtil;
import cn.hutool.core.util.ObjectUtil;
import cn.hutool.core.util.StrUtil;
import com.alibaba.fastjson.JSON;
import com.el.coordinator.boot.fsm.model.vo.FileObjRespVO;
import com.elitescloud.boot.core.base.UdcProvider;
import com.elitescloud.boot.exception.BusinessException;
import com.elitescloud.boot.task.delay.common.DelayTaskSender;
import com.elitescloud.cloudt.common.base.ApiResult;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitesland.yst.production.sale.api.service.shop.BipItemMngService;
import com.elitesland.yst.production.sale.api.service.shop.BipItemService;
import com.elitesland.yst.production.sale.api.vo.param.shop.*;
import com.elitesland.yst.production.sale.api.vo.resp.shop.*;
import com.elitesland.yst.production.sale.api.vo.save.shop.*;
import com.elitesland.yst.production.sale.common.constant.ConstantsSale;
import com.elitesland.yst.production.sale.common.constant.UdcEnum;
import com.elitesland.yst.production.sale.convert.shop.BipItemConvert;
import com.elitesland.yst.production.sale.core.service.BaseServiceImpl;
import com.elitesland.yst.production.sale.entity.*;
import com.elitesland.yst.production.sale.event.ItemShelfEvent;
import com.elitesland.yst.production.sale.repo.shop.*;
import com.elitesland.yst.production.sale.rmi.ystsystem.RmiSysNextNumberService;
import com.elitesland.yst.production.sale.task.delay.BipItemTimingOffShelfTask;
import com.elitesland.yst.production.sale.task.delay.BipItemTimingOnShelfTask;
import com.querydsl.core.types.*;
import com.querydsl.core.types.dsl.BooleanExpression;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.PageRequest;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.StringUtils;

import java.io.File;
import java.math.BigDecimal;
import java.time.LocalDateTime;
import java.util.*;
import java.util.stream.Collectors;

/**
 * .
 *
 * @author Kaiser（wang shao）
 * @date 2021/08/27
 */
@Service
@RequiredArgsConstructor
@Slf4j
public class BipItemMngServiceImpl extends BaseServiceImpl implements BipItemMngService {

    private final BipItemRepo itemRepo;
    private final BipItemRepoProc itemRepoProc;
    private final BipItemExtRepo itemExtRepo;
    private final BipItemExtRepoProc itemExtRepoProc;
    private final BipItemPicRepo itemPicRepo;
    private final BipItemPicRepoProc itemPicRepoProc;
    private final BipItemSkuRepo itemSkuRepo;
    private final BipItemSkuRepoProc itemSkuRepoProc;
    private final BipItemSkuRepoProc bipItemSkuRepoProc;
    private final BipItemCategoryRepoProc itemCategoryRepoProc;
    private final BipCompanyManageRepoProc bipCompanyManageRepoProc;

    private final DelayTaskSender delayTaskSender;
    private final BipItemService bipItemService;
//    private final RmiItemService rmiItemService;
    private final RmiSysNextNumberService sysNextNumberService;
    @Autowired
    private UdcProvider udcProvider;

    private static final BipItemConvert CONVERT = BipItemConvert.INSTANCE;
    private static final QBipItemDO ITEM_DO = QBipItemDO.bipItemDO;
    /**
     * 可进行申请上架的状态
     */
    private static final Set<String> STATE_APPLY_SHELF_ON = new HashSet<>(8);

    static {
        STATE_APPLY_SHELF_ON.add(UdcEnum.BIP_ITEM_STATE_DRAFT.getValueCode());
        STATE_APPLY_SHELF_ON.add(UdcEnum.BIP_ITEM_STATE_OFF.getValueCode());
        STATE_APPLY_SHELF_ON.add(UdcEnum.BIP_ITEM_STATE_REJECT.getValueCode());
    }

    /**
     * 可进行申请下架的状态
     */
    private static final Set<String> STATE_APPLY_SHELF_OFF = new HashSet<>(8);

    static {
        STATE_APPLY_SHELF_OFF.add(UdcEnum.BIP_ITEM_STATE_SHELF.getValueCode());
        STATE_APPLY_SHELF_OFF.add(UdcEnum.BIP_ITEM_STATE_REJECT.getValueCode());
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public ApiResult<Long> save(BipItemSaveVO saveVO) {
        try {
            validateForSave(saveVO);
        } catch (IllegalArgumentException e) {
            log.info("商品保存校验不通过", e);
            return ApiResult.fail("商品保存失败，" + e.getMessage());
        }

        // 先保存主表信息
        BipItemDO itemDO = CONVERT.saveVO2DO(saveVO);
        // 初始化草稿状态
        itemDO.setState(UdcEnum.BIP_ITEM_STATE_DRAFT.getValueCode());
        itemRepo.save(itemDO);
        // 保存sku
        saveSku(itemDO, saveVO.getSkuList());
        // 保存商品图片
        saveItemPic(itemDO, saveVO.getPicList());
        // 保存扩展信息
        saveItemExt(itemDO, saveVO);
        // 生成编号
        itemDO.setShelfCode(generateCode());
        // 最后更新商品信息
        itemRepo.save(itemDO);

        return ApiResult.ok(itemDO.getId());
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public ApiResult<Long> update(Long id, BipItemUpdateVO updateVO) {
        try {
            validateForUpdate(id, updateVO);
        } catch (IllegalArgumentException e) {
            log.info("商品修改校验不通过", e);
            return ApiResult.fail("商品保存失败，" + e.getMessage());
        }

        // 先保存主表信息
        BipItemDO itemDO = itemRepo.findById(id).orElseThrow(new BusinessException("修改的商品不存在"));
        CONVERT.copyUpdateVO2DO(updateVO, itemDO);
        itemRepo.save(itemDO);

        // 保存sku
        updateSku(itemDO, updateVO.getSkuList());
        // 保存商品图片
        itemPicRepoProc.deleteByBipItemId(id);
        saveItemPic(itemDO, updateVO.getPicList());
        // 保存扩展信息
        updateItemExt(itemDO, updateVO);

        // 最后更新商品信息
        itemRepo.save(itemDO);

        return ApiResult.ok(itemDO.getId());
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public ApiResult<List<Long>> updateShelf(List<Long> ids, boolean onShelf) {
        var stateMap = itemRepoProc.getState(ids);
        if (stateMap.isEmpty()) {
            return ApiResult.ok(ids);
        }

        // 更新状态为待审批
        // 无论是手动上架还是手动下架，都需要审批，筛选出需要更新的商品ID
        var idsTemp = stateMap.entrySet().stream()
                .filter(t -> onShelf ? filterForOnShelf(t.getValue()) : filterForOffShelf(t.getValue()))
                .map(Map.Entry::getKey).collect(Collectors.toList());
        if (idsTemp.isEmpty()) {
            return ApiResult.fail("没有可更新状态的记录");
        }

        // 全部sku冻结的商品不能上架
        if (onShelf){
            checkFreezeOnShelf(idsTemp);
            // 校验图片信息
            checkItemImage(idsTemp);
        }

        itemRepoProc.updateState(idsTemp, UdcEnum.BIP_ITEM_STATE_APPR.getValueCode());
        itemExtRepoProc.updateToApprove(ids);

        return ApiResult.ok(ids);
    }

    private void checkItemImage(List<Long> idsTemp) {
        List<BipItemDO> items = itemRepo.findAllById(idsTemp);

        for (BipItemDO item : items) {
            BipItemPicDO mainImage = itemPicRepo.findFirstByBipItemIdAndPicAndMain(item.getId(), true, true);

            if (mainImage==null){
                throw new BusinessException("商品【"+item.getItemCode()+"】缺少主图");
            }

            var itemExtDO = itemExtRepo.findById(item.getExtId()).orElseThrow(new BusinessException("商品信息异常"));
            List<BipItemPicContentRespVO> contentRespVOS = convert2List(itemExtDO.getContent(), BipItemPicContentRespVO.class);
            Assert.isTrue(contentRespVOS.size()>0, "商品【"+item.getItemCode()+"】缺少图文详情");
        }
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public ApiResult<List<Long>> updateOnShelfTiming(BipItemShelfOnSaveVO shelfSaveVO) {
        var stateMap = itemRepoProc.getState(shelfSaveVO.getIdList());
        // 过滤出可更新的记录
        var ids = stateMap.entrySet().stream().filter(t -> STATE_APPLY_SHELF_ON.contains(t.getValue())).map(Map.Entry::getKey).collect(Collectors.toList());
        if (ids.isEmpty()) {
            return ApiResult.fail("没有可更新的记录");
        }

        // 校验图片信息
        checkItemImage(ids);

        // 更新为待审批状态
        itemRepoProc.updateState(ids, UdcEnum.BIP_ITEM_STATE_APPR.getValueCode());
        itemExtRepoProc.updateToApprove(ids, shelfSaveVO.getTimeOnShelf(), shelfSaveVO.getTimeOffShelf());

        return ApiResult.ok(ids);
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public ApiResult<List<Long>> updateOffShelfTiming(BipItemShelfOffSaveVO shelfSaveVO) {
        var stateMap = itemRepoProc.getState(shelfSaveVO.getIdList());
        // 过滤出可更新的记录
        var ids = stateMap.entrySet().stream().filter(t -> STATE_APPLY_SHELF_OFF.contains(t.getValue())).map(Map.Entry::getKey).collect(Collectors.toList());
        if (ids.isEmpty()) {
            return ApiResult.fail("没有可更新的记录");
        }

        // 更新为待审批状态
        itemRepoProc.updateState(ids, UdcEnum.BIP_ITEM_STATE_APPR.getValueCode());
        itemExtRepoProc.updateToApprove(ids, shelfSaveVO.getTimeOffShelf());

        return ApiResult.ok(ids);
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public ApiResult<Long> updateOnShelfByTask(Long id) {
//        var item = itemRepo.findById(id).orElse(null);
//        if (item == null) {
//            return ApiResult.fail("商品不存在");
//        }
//
//        // 只有待上架的才可以上架
//        if (!StrUtil.equals(item.getState(), UdcEnum.BIP_ITEM_STATE_TO_SHELF.getValueCode())) {
//            return ApiResult.fail("当前状态不可上架");
//        }
//
//        var nowTime = LocalDateTime.now();
//        var planTime = itemExtRepoProc.getTimeOnShelfPlan(item.getExtId());
//        if (nowTime.isBefore(planTime)) {
//            return ApiResult.fail("未到上架时间");
//        }
//
//        // 冻结场景判断
//        ItmItemStatusConfigRpcParam param = new ItmItemStatusConfigRpcParam();
//        param.setBusinessCodes(Collections.singletonList("S002"));
//        List<ItmItemStatusConfigRpcDTO> config = rmiItemService.findItmItemStatusConfigByParam(param);
//        if (config==null || config.size()!=1) {
//            throw new BusinessException("支撑域冻结场景获取异常！");
//        }
//
//        // 冻结场景为true
//        ItmItemStatusConfigRpcDTO dto = config.get(0);
//        boolean flag = dto.getConfigStatus() && dto.getBusinExcuSignStatus2();
//        if (flag) {
//            // 校验冻结
//            Map<Long, List<BipItemSkuDO>> itemListMap = itemSkuRepoProc.findAllByItemIDS(Collections.singletonList(id));
//            // 全部sku冻结的商品不能上架
//            for (Long aLong : itemListMap.keySet()) {
//                List<BipItemSkuDO> skuDOS = itemListMap.get(aLong);
//                long count = skuDOS.stream().filter(BipItemSkuDO::getFreeze).count();
//                if (count == (long) skuDOS.size()) {
//                    // 更新商品的下架信息
//                    item.setState(UdcEnum.BIP_ITEM_STATE_OFF.getValueCode());
//                    item.setOnShelf(false);
//                    item.setTimeOnShelf(null);
//                    item.setTimeOffShelf(nowTime);
//                    item.setOffShelfReason(UdcEnum.BIP_ITEM_OFF_SHELF_FREEZE.getValueCode());
//                    itemRepo.save(item);
//                    return ApiResult.ok(item.getId());
//                }
//            }
//        }
//
//        // 更新商品的上架信息
//        item.setState(UdcEnum.BIP_ITEM_STATE_SHELF.getValueCode());
//        item.setOnShelf(true);
//        item.setShelf(true);
//        item.setTimeOnShelf(nowTime);
//        item.setTimeOffShelf(null);
//        item.setOffShelfReason(null);
//        itemRepo.save(item);
//
//        // 更新扩展信息
//        itemExtRepoProc.updateToOnShelf(item.getExtId());
//
//        return ApiResult.ok(id);
        return null;
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public ApiResult<Long> updateOffShelfByTask(Long id) {
        var item = itemRepo.findById(id).orElse(null);
        if (item == null) {
            return ApiResult.fail("商品不存在");
        }

        // 只有待下架的才可以下架
        if (!StrUtil.equals(item.getState(), UdcEnum.BIP_ITEM_STATE_TO_OFF.getValueCode())) {
            return ApiResult.fail("当前状态不可下架");
        }

        var nowTime = LocalDateTime.now();
        var planTime = itemExtRepoProc.getTimeOffShelfPlan(item.getExtId());
        if (nowTime.isBefore(planTime)) {
            return ApiResult.fail("未到下架时间");
        }

        // 更新商品的下架信息
        item.setState(UdcEnum.BIP_ITEM_STATE_OFF.getValueCode());
        item.setOnShelf(false);
        item.setTimeOnShelf(null);
        item.setTimeOffShelf(nowTime);
        item.setOffShelfReason(UdcEnum.BIP_ITEM_OFF_SHELF_AUTO.getValueCode());
        itemRepo.save(item);

        // 更新扩展信息
        itemExtRepoProc.updateToOffShelf(item.getExtId());
        // 发布下架事件
        publishEventAsync(new ItemShelfEvent(this, false, id));

        return ApiResult.ok(id);
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public ApiResult<List<Long>> updateApprove(BipItemApproveSaveVO approveSaveVO) {
        var itemList = itemRepo.findAllById(approveSaveVO.getIdList());
        if (itemList.isEmpty()) {
            return ApiResult.ok(approveSaveVO.getIdList());
        }

        // 如果是审批通过，则更新为已上架（或已下架）
        if (Boolean.TRUE.equals(approveSaveVO.getPass())) {
            updateOnShelfByApprove(itemList, approveSaveVO);
            updateOffShelfByApprove(itemList, approveSaveVO);
            return ApiResult.ok(approveSaveVO.getIdList());
        }

        // 更新为审批拒绝
        itemRepoProc.updateState(approveSaveVO.getIdList(), UdcEnum.BIP_ITEM_STATE_REJECT.getValueCode());
        var extIds = itemList.stream().map(BipItemDO::getExtId).collect(Collectors.toList());
        var itemExtList = itemExtRepo.findAllById(extIds);
        var nowTime = LocalDateTime.now();
        for (var ext : itemExtList) {
            ext.setTimeApproved(nowTime);
            ext.setApproverId(approveSaveVO.getApproverId());
            ext.setApproverName(approveSaveVO.getApproverName());
            ext.setApproveOpinion(approveSaveVO.getOpinion());
            ext.setApproved(approveSaveVO.getPass());
        }
        itemExtRepo.saveAll(itemExtList);

        return ApiResult.ok(approveSaveVO.getIdList());
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public ApiResult<List<Long>> updateLimitBuy(BipItemLimitBuySaveVO limitBuySaveVO) {
        itemSkuRepoProc.updateLimitBuy(limitBuySaveVO.getIdList(), limitBuySaveVO.getLimitBuy());
        return ApiResult.ok(limitBuySaveVO.getIdList());
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public ApiResult<Long> updateSkuStock(Long id, List<BipItemStockSaveVO> stockSaveVOList) {
        for (var itemStock : stockSaveVOList) {
            bipItemService.updateStock(itemStock.getSkuId(), itemStock.getStock());
        }
        return ApiResult.ok(id);
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public ApiResult<Long> updateSale(Long id, Long num, Long num90) {
        if (num == null || num90 == null) {
            return ApiResult.fail("商品销量为空");
        }

        itemRepoProc.updateSale(id, num, num90);
        return ApiResult.ok(id);
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public ApiResult<Long> updateEval(Long id, Long numTotal, Long numGood) {
        if (numTotal == null || numGood == null) {
            return ApiResult.fail("商品评价数量为空");
        }

        BigDecimal evalRate = BigDecimal.ZERO;
        if (numTotal != 0 && numGood != 0) {
            evalRate = BigDecimal.valueOf(numGood * 1.0 / numTotal);
        }
        itemRepoProc.updateRateEvalGood(id, numTotal, numGood, evalRate);

        return ApiResult.ok(id);
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public ApiResult<Long> updateEvalScore(Long id, BigDecimal evalScore) {
        if (evalScore == null) {
            return ApiResult.fail("评分为空");
        }

        itemRepoProc.updateScoreEval(id, evalScore);
        return ApiResult.ok(id);
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public ApiResult<Boolean> executeCheckStock(Long id) {
        var itemStocks = itemSkuRepoProc.getStockByBipItemId(id);
        if (itemStocks.isEmpty()) {
            return ApiResult.ok(true);
        }

        // 判断库存为0的个数
        var zeroNum = itemStocks.values().stream().filter(t -> t != null && t == 0).count();
        if (zeroNum == 0) {
            // 没有缺库存的
            return ApiResult.ok(true);
        }

        if (zeroNum != itemStocks.size()) {
            // 更新为部分库存不足
            itemRepoProc.updateLackStock(id, true);
            return ApiResult.ok(false);
        }

        // 更新商品状态为下架
        var item = itemRepo.findById(id).orElseThrow(new BusinessException("商品不存在"));
        if (BooleanUtil.isFalse(item.getOnShelf())) {
            // 如果商品已下架，则忽略，不再更新
            return ApiResult.ok(false);
        }
        item.setState(UdcEnum.BIP_ITEM_STATE_OFF.getValueCode());
        item.setOnShelf(false);
        item.setTimeOnShelf(null);
        item.setTimeOffShelf(LocalDateTime.now());
        item.setOffShelfReason(UdcEnum.BIP_ITEM_OFF_SHELF_STOCK.getValueCode());
        itemRepo.save(item);

        return ApiResult.ok(false);
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public ApiResult<Long> delete(Long id) {
        var exists = itemRepoProc.existsById(id);
        if (!exists) {
            // 不存在则直接返回
            return ApiResult.fail("数据不存在");
        }

        var shelf = itemRepoProc.getShelf(id);
        if (Boolean.TRUE.equals(shelf)) {
            // 如果已上架过，则做假删除
//            itemRepoProc.deleteFake(id);
//            return ApiResult.ok(id);
            throw new BusinessException("上架过的商品不可删除");
        }

        // 删除商品及扩展信息
        itemRepoProc.delete(id);
        itemExtRepoProc.deleteByBipItemId(id);
        itemPicRepoProc.deleteByBipItemId(id);
        itemSkuRepoProc.deleteByBipItemId(id);

        return ApiResult.ok(id);
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public ApiResult<List<Long>> delete(List<Long> ids) {
        var shelfMap = itemRepoProc.getShelf(ids);
        if (shelfMap.isEmpty()) {
            return ApiResult.ok(ids);
        }

        List<Long> idsDel = new ArrayList<>(ids.size());
        List<Long> idsDelFake = new ArrayList<>(ids.size());
        for (var entry : shelfMap.entrySet()) {
            if (entry.getValue()) {
                // 已上架过，则假删除
                idsDelFake.add(entry.getKey());
                continue;
            }

            idsDel.add(entry.getKey());
        }

        if (!idsDel.isEmpty()) {
            // 删除商品及扩展信息
            itemRepoProc.delete(idsDel);
            itemExtRepoProc.deleteByBipItemId(idsDel);
            itemPicRepoProc.deleteByBipItemId(idsDel);
            itemSkuRepoProc.deleteByBipItemId(idsDel);
        }

        if (!idsDelFake.isEmpty()) {
//            itemRepoProc.deleteFake(idsDelFake);
            throw new BusinessException("上架过的商品不可删除");
        }

        return ApiResult.ok(ids);
    }

    @Override
    public ApiResult<BipItemDetailRespVO> get(Long id) {
        log.info("商品id:---------"+id);
        var itemDO = itemRepo.findById(id).orElse(null);
        if (itemDO == null) {
            log.info("商品信息不存在--------------------");
            return ApiResult.fail("商品信息不存在");
        }
        var itemExtDO = itemExtRepo.findById(itemDO.getExtId()).orElseThrow(new BusinessException("商品信息异常"));
        log.info("商品信息：--------"+ JSON.toJSONString(itemExtDO));

        var respVO = CONVERT.itemDO2DetailVO(itemDO);
        respVO.setSkuList(querySkuDetail(id));

        respVO.setSpecList(convert2List(itemExtDO.getSpec(), BipItemSpecRespVO.class));
        respVO.setPicList(queryItemPic(id));
        respVO.setContentPicList(convert2List(itemExtDO.getContent(), BipItemPicContentRespVO.class));
        if (StrUtil.isNotBlank(itemExtDO.getUnit())) {
            respVO.setUnit(itemExtDO.getUnit());
            respVO.setUnitName(convertUdc(UdcEnum.COM_UOM_BX).get(itemExtDO.getUnit()));
        }
        log.info("商品信息:-------"+JSON.toJSONString(respVO));
        return ApiResult.ok(respVO);
    }

    @Override
    public ApiResult<List<BipItemSkuRespVO>> getSku(Long id) {
        var skuList = querySkuDetail(id);
        return ApiResult.ok(skuList);
    }

    @Override
    public ApiResult<Integer> getStockBySku(Long skuId) {
        var stock = itemSkuRepoProc.getStock(skuId);
        return ApiResult.ok(stock);
    }

    @Override
    public ApiResult<Long> getIdBySkuId(Long skuId) {
        var id = itemSkuRepoProc.getBipItemId(skuId);
        return ApiResult.ok(id);
    }

    @Override
    public ApiResult<List<BipItemUomConvertRespVO>> queryItemUomConvert(List<Long> itemIds) {
//        if (itemIds.isEmpty()) {
//            return ApiResult.ok(Collections.emptyList());
//        }
//        var converts = rmiItemService.findItemUomConvDtoList(itemIds);
//        if (CollUtil.isEmpty(converts)) {
//            return ApiResult.ok(Collections.emptyList());
//        }
//
//        return ApiResult.ok(CONVERT.itemConvertDTO2VO(converts));
        return null;
    }

    @Override
    public ApiResult<PagingVO<BipItemToShelfRespVO>> queryForToShelf(BipItemToShelfQueryParamVO queryParamVO) {
        var pagingDO = queryItems(queryParamVO);
        if (CollUtil.isEmpty(pagingDO.getRecords())) {
            return ApiResult.ok(PagingVO.<BipItemToShelfRespVO>builder().total(pagingDO.getTotal()).records(Collections.emptyList()).build());
        }

        // do list转vo
        var stateNameMap = convertUdc(UdcEnum.BIP_ITEM_STATE_DRAFT);
        var dataList = pagingDO.getRecords().stream().map(t -> {
            var vo = CONVERT.itemDO2ToShelfVO(t);
            vo.setStateName(stateNameMap.get(t.getState()));
            return vo;
        }).collect(Collectors.toList());
        return ApiResult.ok(PagingVO.<BipItemToShelfRespVO>builder().total(pagingDO.getTotal()).records(dataList).build());
    }

    @Override
    public ApiResult<PagingVO<BipItemApproveRespVO>> queryForApprove(BipItemApproveQueryParamVO queryParamVO) {
        var pagingDO = queryItems(queryParamVO);
        if (CollUtil.isEmpty(pagingDO.getRecords())) {
            return ApiResult.ok(PagingVO.<BipItemApproveRespVO>builder().total(pagingDO.getTotal()).records(Collections.emptyList()).build());
        }

        var extIds = pagingDO.getRecords().stream().map(BipItemDO::getExtId).collect(Collectors.toList());
        var extMap = itemExtRepo.findAllById(extIds).stream().collect(Collectors.toMap(BipItemExtDO::getId, t -> t, (t1, t2) -> t1));

        // do list转vo
        var stateNameMap = convertUdc(UdcEnum.BIP_ITEM_STATE_DRAFT);
        var dataList = pagingDO.getRecords().stream().map(t -> {
            var vo = CONVERT.itemDO2ToApproveVO(t);
            vo.setStateName(stateNameMap.get(t.getState()));
            vo.setShelf(!t.getOnShelf());

            // 处理扩展信息
            CONVERT.do2ApproveVO(extMap.get(t.getExtId()), vo);

            return vo;
        }).collect(Collectors.toList());
        return ApiResult.ok(PagingVO.<BipItemApproveRespVO>builder().total(pagingDO.getTotal()).records(dataList).build());
    }

    @Override
    public ApiResult<PagingVO<BipItemOnShelfRespVO>> queryForOnShelf(BipItemOnShelfQueryParamVO queryParamVO) {
        var pagingDO = queryItems(queryParamVO);

        if (CollUtil.isEmpty(pagingDO.getRecords())) {
            return ApiResult.ok(PagingVO.<BipItemOnShelfRespVO>builder()
                    .total(pagingDO.getTotal())
                    .records(Collections.emptyList())
                    .build());
        }

        // do list转vo
        var stateNameMap = convertUdc(UdcEnum.BIP_ITEM_STATE_DRAFT);
        var dataList = pagingDO.getRecords().stream().map(t -> {
            var vo = CONVERT.itemDO2OnShelfVO(t);
            vo.setStateName(stateNameMap.get(t.getState()));

            return vo;
        }).collect(Collectors.toList());

        return ApiResult.ok(PagingVO.<BipItemOnShelfRespVO>builder()
                .total(pagingDO.getTotal())
                .records(dataList)
                .build());
    }

    @Override
    public ApiResult<PagingVO<BipItemOffShelfRespVO>> queryForOffShelf(BipItemOffShelfQueryParamVO queryParamVO) {
        var pagingDO = queryItems(queryParamVO);
        if (CollUtil.isEmpty(pagingDO.getRecords())) {
            return ApiResult.ok(PagingVO.<BipItemOffShelfRespVO>builder().total(pagingDO.getTotal()).records(Collections.emptyList()).build());
        }

        // do list转vo
        var stateNameMap = convertUdc(UdcEnum.BIP_ITEM_STATE_DRAFT);
        var offShelfReasonNameMap = convertUdc(UdcEnum.BIP_ITEM_OFF_SHELF_MANUAL);
        var dataList = pagingDO.getRecords().stream().map(t -> {
            var vo = CONVERT.itemDO2OffShelfVO(t);
            vo.setStateName(stateNameMap.get(t.getState()));
            vo.setOffShelfReasonName(offShelfReasonNameMap.get(t.getOffShelfReason()));

            return vo;
        }).collect(Collectors.toList());
        return ApiResult.ok(PagingVO.<BipItemOffShelfRespVO>builder().total(pagingDO.getTotal()).records(dataList).build());
    }

    @Override
    public ApiResult<PagingVO<BipItemQueryRespVO>> query(BipItemQueryParamVO queryParam) {
        var result = querySku(queryParam);
        return ApiResult.ok(result);
    }

    @Override
    public ApiResult<List<BipItemQueryRespVO>> queryByCategory(Long ouId, String categoryCode, Boolean onShelf) {
        if (CharSequenceUtil.isBlank(categoryCode)) {
            return ApiResult.ok(Collections.emptyList());
        }

        var result = querySkuByCategoryCode(ouId, categoryCode, onShelf);
        return ApiResult.ok(result.getRecords());
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void updatePrice(Long id, BigDecimal price, BigDecimal priceOld) {
        bipItemSkuRepoProc.updatePrice(id, price, priceOld);
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void handleImage(File file, String ouCode) {
        BipCompanyManageDO ou = bipCompanyManageRepoProc.findByOuCode(ouCode);
        Assert.notNull(ou,"公司【"+ouCode+"】不存在!");

        for (File first : file.listFiles()) {
            if (first.getName().startsWith("主")) {
                for (File spu : first.listFiles()) {
                    String itemCode = spu.getName();
                    BipItemDO itemDO = itemRepoProc.getItemForImage(itemCode, ouCode);

                    Assert.notNull(itemDO, "公司【"+ouCode+"】不存在商品【"+itemCode+"】");
                    Assert.isTrue((itemDO.getOnShelf()==null || Boolean.FALSE.equals(itemDO.getOnShelf()))
                            && StringUtils.isEmpty(itemDO.getOffShelfReason()),
                            "商品【"+itemDO.getItemCode()+"】不在待上架列表中，不能上传图片！");

                    List<BipItemPicSaveVO> picSaveVOS = new ArrayList<>();
                    for (File image : spu.listFiles()) {
                        com.el.coordinator.core.common.api.ApiResult result = fileService.upload(image);
                        FileObjRespVO data = (FileObjRespVO)result.getData();

                        BipItemPicSaveVO picSaveVO = new BipItemPicSaveVO();
                        picSaveVO.setFileCode(data.getFileCode());
                        picSaveVO.setFileName(data.getOriginalName());
                        picSaveVO.setMimeType(data.getMimeType());
                        picSaveVO.setUrl(data.getUrl());
                        picSaveVO.setFileSize(data.getFileSize());
                        picSaveVO.setMain(false);

                        // 主图
                        if (image.getName().split("\\.")[0].equals(itemCode)) picSaveVO.setMain(true);
                        picSaveVOS.add(picSaveVO);

                    }
                    itemPicRepoProc.deleteByBipItemId(itemDO.getId());
                    saveItemPic(itemDO, picSaveVOS);
                    itemRepo.save(itemDO);

                }
            }
            if (first.getName().startsWith("详情")){
                for (File spu : first.listFiles()) {
                    String itemCode = spu.getName();
                    BipItemDO itemDO = itemRepoProc.getItemForImage(itemCode, ouCode);

                    Assert.notNull(itemDO, "公司【"+ouCode+"】不存在商品【"+itemCode+"】");
                    Assert.isTrue((itemDO.getOnShelf()==null || Boolean.FALSE.equals(itemDO.getOnShelf()))
                            && StringUtils.isEmpty(itemDO.getOffShelfReason()),
                            "商品【"+itemDO.getItemCode()+"】不在待上架列表中，不能上传图片！");

                    List<BipItemPicContentSaveVO> contentPicList = new ArrayList<>();
                    for (File image : spu.listFiles()) {
                        com.el.coordinator.core.common.api.ApiResult result = fileService.upload(image);
                        FileObjRespVO data = (FileObjRespVO)result.getData();

                        BipItemPicContentSaveVO picSaveVO = new BipItemPicContentSaveVO();
                        picSaveVO.setFileCode(data.getFileCode());
                        picSaveVO.setFileName(data.getOriginalName());
                        picSaveVO.setMimeType(data.getMimeType());
                        picSaveVO.setUrl(data.getUrl());

                        contentPicList.add(picSaveVO);

                    }

                    itemExtRepo.findById(itemDO.getExtId())
                            .ifPresentOrElse(extDO->{
                                    extDO.setContent(convertObj2Str(contentPicList));
                                    itemExtRepo.save(extDO);
                                }, ()-> {throw new BusinessException("商品【"+itemCode+"】不存在扩展信息"); });

                }
            }
        }
    }

    private String generateCode() {
        var code = sysNextNumberService.generateCode("yst-sale","SAL_ITEM_SHELF_CODE", new ArrayList<>());
        if (itemRepoProc.existsByShelfCode(code)) {
            throw new BusinessException("保存失败，请稍后重试");
        }
        return code;
    }

    /**
     * 是否可提交上架
     * <p>
     * 草稿的、已下架的、已拒绝的可提交上架
     *
     * @param state 商品状态
     * @return 是否可
     */
    private boolean filterForOnShelf(String state) {
        return state != null && STATE_APPLY_SHELF_ON.contains(state);
    }

    /**
     * 是否可提交下架
     * <p>
     * 已上架的可提交下架
     *
     * @param state 商品状态
     * @return 是否可
     */
    private boolean filterForOffShelf(String state) {
        return state != null && STATE_APPLY_SHELF_OFF.contains(state);
    }

    /**
     * 同意审批上架
     *
     * @param itemList      商品列表
     * @param approveSaveVO 审批结果
     */
    private void updateOnShelfByApprove(List<BipItemDO> itemList, BipItemApproveSaveVO approveSaveVO) {
        // 待审批 且 未上架  才可以上架
        var items = itemList.stream()
                .filter(t -> UdcEnum.BIP_ITEM_STATE_APPR.getValueCode().equals(t.getState()) && Boolean.FALSE.equals(t.getOnShelf()))
                .collect(Collectors.toList());
        if (items.isEmpty()) {
            return;
        }

        // 查询扩展信息
        var extIds = items.stream().map(BipItemDO::getExtId).collect(Collectors.toList());
        var itemExtList = itemExtRepo.findAllById(extIds);
        var itemExtMap = itemExtList.stream().collect(Collectors.toMap(BipItemExtDO::getId, t -> t, (t1, t2) -> t1));

        var nowTime = LocalDateTime.now();
        // 需要定时上架的商品
        Set<BipItemExtDO> timingItems = new HashSet<>(64);
        Set<Long> onShelfNowIds = new HashSet<>(64);
        for (var item : items) {
            var ext = itemExtMap.get(item.getExtId());
            if (ext.getTimeOnShelfPlan() != null) {
                // 定时上架
                item.setState(UdcEnum.BIP_ITEM_STATE_TO_SHELF.getValueCode());
                timingItems.add(ext);
            } else {
                // 立即上架
                item.setState(UdcEnum.BIP_ITEM_STATE_SHELF.getValueCode());
                item.setShelf(true);
                item.setOnShelf(true);
                item.setTimeOnShelf(nowTime);
                item.setTimeOffShelf(null);
                item.setOffShelfReason(null);

                onShelfNowIds.add(item.getId());
            }
        }
        // 只校验立即上架
        checkFreezeOnShelf(new ArrayList<>(onShelfNowIds));

        // 更新商品状态
        itemRepo.saveAll(items);

        // 更新审批人
        if (approveSaveVO != null) {
            for (var ext : itemExtList) {
                ext.setTimeApproved(nowTime);
                ext.setApproverId(approveSaveVO.getApproverId());
                ext.setApproverName(approveSaveVO.getApproverName());
                ext.setApproveOpinion(approveSaveVO.getOpinion());
                ext.setApproved(approveSaveVO.getPass());
            }
            itemExtRepo.saveAll(itemExtList);
        }

        // 需要定时上架的加定时任务
        if (!timingItems.isEmpty()) {
            for (var ext : timingItems) {
                var id = ext.getBipItemId().toString();
                delayTaskSender.send(BipItemTimingOnShelfTask.TASK_TYPE, id, ext.getBipItemId(), ext.getTimeOnShelfPlan());
                if (ext.getTimeOffShelfPlan() != null) {
                    // 还需要定时下线
                    delayTaskSender.send(BipItemTimingOffShelfTask.TASK_TYPE, id, ext.getBipItemId(), ext.getTimeOffShelfPlan());
                }
            }
        }

        // 发布上架事件
        if (!onShelfNowIds.isEmpty()) {
            for (var bipItemId : onShelfNowIds) {
                publishEventSync(new ItemShelfEvent(this, true, bipItemId));
            }
        }
    }

    /**
     * 上架时校验冻结
     * @param idList 商品主表id
     */
    private void checkFreezeOnShelf(List<Long> idList) {
//        Map<Long, List<BipItemSkuDO>> itemListMap = itemSkuRepoProc.findAllByItemIDS(idList);
//
//        /**
//         * 如果配置状态是true  那么就需要判断销售冻结
//         *                 销售冻结 = true ： 不控制
//         *                               销售冻结 = false ： 控制（检查商品的冻结）
//         * 如果配置状态是false,就不需要进行下面的判断了，直接不控制
//         */
//
//        // 冻结场景判断
//        ItmItemStatusConfigRpcParam param = new ItmItemStatusConfigRpcParam();
//        param.setBusinessCodes(Collections.singletonList("S002"));
//        List<ItmItemStatusConfigRpcDTO> config = rmiItemService.findItmItemStatusConfigByParam(param);
//
//        if (config==null || config.size()!=1) throw new BusinessException("支撑域冻结场景获取异常！");
//
//        ItmItemStatusConfigRpcDTO dto = config.get(0);
//        boolean flag = dto.getConfigStatus() && !dto.getBusinExcuSignStatus2();
//        if (!flag) return;
//
//        // 全部sku冻结的商品不能上架
//        for (Long aLong : itemListMap.keySet()) {
//            List<BipItemSkuDO> skuDOS = itemListMap.get(aLong);
//            long count = skuDOS.stream().filter(BipItemSkuDO::getFreeze).count();
//            if (count == (long) skuDOS.size()) {
//                throw new BusinessException("商品：" + aLong + "已冻结");
//            }
//        }
    }

    /**
     * 同意审批下架
     *
     * @param itemList      商品列表
     * @param approveSaveVO 审批结果
     */
    private void updateOffShelfByApprove(List<BipItemDO> itemList, BipItemApproveSaveVO approveSaveVO) {
        // 待审批 且 已上架  才可以下架
        var items = itemList.stream()
                .filter(t -> UdcEnum.BIP_ITEM_STATE_APPR.getValueCode().equals(t.getState()) && Boolean.TRUE.equals(t.getOnShelf()))
                .collect(Collectors.toList());
        if (items.isEmpty()) {
            return;
        }

        // 查询扩展信息
        var extIds = items.stream().map(BipItemDO::getExtId).collect(Collectors.toList());
        var itemExtList = itemExtRepo.findAllById(extIds);
        var itemExtMap = itemExtList.stream().collect(Collectors.toMap(BipItemExtDO::getId, t -> t, (t1, t2) -> t1));

        var nowTime = LocalDateTime.now();
        // 需要定时下架的商品
        Set<BipItemExtDO> timingItems = new HashSet<>(64);
        Set<Long> offShelfNowIds = new HashSet<>(64);
        for (var item : items) {
            var ext = itemExtMap.get(item.getExtId());
            if (ext.getTimeOffShelfPlan() != null) {
                // 定时下架
                item.setState(UdcEnum.BIP_ITEM_STATE_TO_OFF.getValueCode());
                timingItems.add(ext);
            } else {
                // 立即下架
                item.setState(UdcEnum.BIP_ITEM_STATE_OFF.getValueCode());
                item.setOnShelf(false);
                item.setTimeOnShelf(null);
                item.setTimeOffShelf(nowTime);
                item.setOffShelfReason(UdcEnum.BIP_ITEM_OFF_SHELF_MANUAL.getValueCode());
                offShelfNowIds.add(item.getId());
            }
        }
        // 更新商品状态
        itemRepo.saveAll(items);

        // 更新审批人
        if (approveSaveVO != null) {
            for (var ext : itemExtList) {
                ext.setTimeApproved(nowTime);
                ext.setApproverId(approveSaveVO.getApproverId());
                ext.setApproverName(approveSaveVO.getApproverName());
                ext.setApproveOpinion(approveSaveVO.getOpinion());
                ext.setApproved(approveSaveVO.getPass());
            }
            itemExtRepo.saveAll(itemExtList);
        }

        // 需要定时下架的加定时任务
        if (!timingItems.isEmpty()) {
            for (var ext : timingItems) {
                var id = ext.getBipItemId().toString();
                delayTaskSender.send(BipItemTimingOffShelfTask.TASK_TYPE, id, ext.getBipItemId(), ext.getTimeOffShelfPlan());
            }
        }

        // 发布下架事件
        if (!offShelfNowIds.isEmpty()) {
            for (var bipItemId : offShelfNowIds) {
                publishEventSync(new ItemShelfEvent(this, false, bipItemId));
            }
        }
    }

    private PagingVO<BipItemDO> queryItems(BipItemToShelfQueryParamVO queryParamVO) {
        Predicate condition = ITEM_DO.isNotNull();
        if (queryParamVO.getCategoryId1() != null) {
            condition = ExpressionUtils.and(condition, ITEM_DO.categoryId1.eq(queryParamVO.getCategoryId1()));
        }
        if (queryParamVO.getCategoryId2() != null) {
            condition = ExpressionUtils.and(condition, ITEM_DO.categoryId2.eq(queryParamVO.getCategoryId2()));
        }
        if (queryParamVO.getCategoryId3() != null) {
            condition = ExpressionUtils.and(condition, ITEM_DO.categoryId3.eq(queryParamVO.getCategoryId3()));
        }
        if (queryParamVO.getOuId() != null) {
            condition = ExpressionUtils.and(condition, ITEM_DO.ouId.eq(queryParamVO.getOuId()));
        }
        if (StrUtil.isNotBlank(queryParamVO.getState())) {
            condition = ExpressionUtils.and(condition, ITEM_DO.state.eq(queryParamVO.getState()));
        }
        if (StrUtil.isNotBlank(queryParamVO.getShelfCode())) {
            condition = ExpressionUtils.and(condition, ITEM_DO.shelfCode.eq(queryParamVO.getShelfCode()));
        }
        if (queryParamVO.getCreateUserId() != null) {
            condition = ExpressionUtils.and(condition, ITEM_DO.createUserId.eq(queryParamVO.getCreateUserId()));
        }
        if (StrUtil.isNotBlank(queryParamVO.getItemCode())) {
            condition = ExpressionUtils.and(condition, ITEM_DO.itemCode.eq(queryParamVO.getItemCode()));
        }
        if (queryParamVO.getFareFree() != null) {
            condition = ExpressionUtils.and(condition, ITEM_DO.fareFree.eq(queryParamVO.getFareFree()));
        }
        if (StrUtil.isNotBlank(queryParamVO.getItemName())) {
            condition = ExpressionUtils.and(condition, ITEM_DO.itemName.like("%" + queryParamVO.getItemName() + "%"));
        }
        if (StrUtil.isNotBlank(queryParamVO.getCreator())) {
            condition = ExpressionUtils.and(condition, ITEM_DO.creator.like("%" + queryParamVO.getCreator() + "%"));
        }
        // 未上架的
        condition = ExpressionUtils.and(condition, ITEM_DO.onShelf.eq(false).and(ITEM_DO.offShelfReason.isNull())
                .and(ITEM_DO.deleteFlag.eq(ConstantsSale.COMMON_DELETE_NO)));

        var pageRequest = queryParamVO.getPageRequest();
        var jpaQuery = jpaQueryFactory.select(ITEM_DO)
                .from(ITEM_DO)
                .where(condition);
        var count = jpaQuery.fetchCount();
        var dataList = jpaQuery
                .orderBy(obtainOrder(pageRequest, ITEM_DO, new OrderSpecifier[]{new OrderSpecifier(Order.DESC, ITEM_DO.createTime)}))
                .offset(pageRequest.getOffset())
                .limit(pageRequest.getPageSize())
                .fetch();
        return PagingVO.<BipItemDO>builder().total(count).records(dataList).build();
    }

    private PagingVO<BipItemDO> queryItems(BipItemApproveQueryParamVO queryParamVO) {
        Predicate condition = ITEM_DO.isNotNull();
        if (queryParamVO.getCategoryId1() != null) {
            condition = ExpressionUtils.and(condition, ITEM_DO.categoryId1.eq(queryParamVO.getCategoryId1()));
        }
        if (queryParamVO.getCategoryId2() != null) {
            condition = ExpressionUtils.and(condition, ITEM_DO.categoryId2.eq(queryParamVO.getCategoryId2()));
        }
        if (queryParamVO.getCategoryId3() != null) {
            condition = ExpressionUtils.and(condition, ITEM_DO.categoryId3.eq(queryParamVO.getCategoryId3()));
        }
        if (queryParamVO.getOuId() != null) {
            condition = ExpressionUtils.and(condition, ITEM_DO.ouId.eq(queryParamVO.getOuId()));
        }
//        if (StrUtil.isNotBlank(queryParamVO.getState())) {
//            condition = ExpressionUtils.and(condition, ITEM_DO.state.eq(queryParamVO.getState()));
//        }
        if (StrUtil.isNotBlank(queryParamVO.getShelfCode())) {
            condition = ExpressionUtils.and(condition, ITEM_DO.shelfCode.eq(queryParamVO.getShelfCode()));
        }
        if (queryParamVO.getCreateUserId() != null) {
            condition = ExpressionUtils.and(condition, ITEM_DO.createUserId.eq(queryParamVO.getCreateUserId()));
        }
        if (StrUtil.isNotBlank(queryParamVO.getItemCode())) {
            condition = ExpressionUtils.and(condition, ITEM_DO.itemCode.eq(queryParamVO.getItemCode()));
        }
        if (queryParamVO.getFareFree() != null) {
            condition = ExpressionUtils.and(condition, ITEM_DO.fareFree.eq(queryParamVO.getFareFree()));
        }
        if (StrUtil.isNotBlank(queryParamVO.getItemName())) {
            condition = ExpressionUtils.and(condition, ITEM_DO.itemName.like("%" + queryParamVO.getItemName() + "%"));
        }
        if (queryParamVO.getShelf() != null) {
            condition = ExpressionUtils.and(condition, ITEM_DO.onShelf.eq(!queryParamVO.getShelf()));
        }
        // 需要审批的
        condition = ExpressionUtils.and(condition, ITEM_DO.state.eq(UdcEnum.BIP_ITEM_STATE_APPR.getValueCode()).and(ITEM_DO.deleteFlag.eq(ConstantsSale.COMMON_DELETE_NO)));

        var pageRequest = queryParamVO.getPageRequest();
        var jpaQuery = jpaQueryFactory.select(ITEM_DO)
                .from(ITEM_DO)
                .where(condition);
        var count = jpaQuery.fetchCount();
        var dataList = jpaQuery
                .orderBy(obtainOrder(pageRequest, ITEM_DO, new OrderSpecifier[]{new OrderSpecifier(Order.DESC, ITEM_DO.createTime)}))
                .offset(pageRequest.getOffset())
                .limit(pageRequest.getPageSize())
                .fetch();
        return PagingVO.<BipItemDO>builder().total(count).records(dataList).build();
    }

    private PagingVO<BipItemDO> queryItems(BipItemOnShelfQueryParamVO queryParamVO) {
        Predicate condition = ITEM_DO.isNotNull();
        if (queryParamVO.getCategoryId1() != null) {
            condition = ExpressionUtils.and(condition, ITEM_DO.categoryId1.eq(queryParamVO.getCategoryId1()));
        }
        if (queryParamVO.getCategoryId2() != null) {
            condition = ExpressionUtils.and(condition, ITEM_DO.categoryId2.eq(queryParamVO.getCategoryId2()));
        }
        if (queryParamVO.getCategoryId3() != null) {
            condition = ExpressionUtils.and(condition, ITEM_DO.categoryId3.eq(queryParamVO.getCategoryId3()));
        }
        if (queryParamVO.getOuId() != null) {
            condition = ExpressionUtils.and(condition, ITEM_DO.ouId.eq(queryParamVO.getOuId()));
        }
        if (StrUtil.isNotBlank(queryParamVO.getState())) {
            condition = ExpressionUtils.and(condition, ITEM_DO.state.eq(queryParamVO.getState()));
        }
        if (StrUtil.isNotBlank(queryParamVO.getShelfCode())) {
            condition = ExpressionUtils.and(condition, ITEM_DO.shelfCode.eq(queryParamVO.getShelfCode()));
        }
        if (queryParamVO.getCreateUserId() != null) {
            condition = ExpressionUtils.and(condition, ITEM_DO.createUserId.eq(queryParamVO.getCreateUserId()));
        }
        if (StrUtil.isNotBlank(queryParamVO.getItemCode())) {
            condition = ExpressionUtils.and(condition, ITEM_DO.itemCode.eq(queryParamVO.getItemCode()));
        }
        if (queryParamVO.getFareFree() != null) {
            condition = ExpressionUtils.and(condition, ITEM_DO.fareFree.eq(queryParamVO.getFareFree()));
        }
        if (StrUtil.isNotBlank(queryParamVO.getItemName())) {
            condition = ExpressionUtils.and(condition, ITEM_DO.itemName.like("%" + queryParamVO.getItemName() + "%"));
        }
        if (StrUtil.isNotBlank(queryParamVO.getCreator())){
            condition = ExpressionUtils.and(condition, ITEM_DO.creator.like("%" + queryParamVO.getCreator() + "%"));
        }
        // 已上架的
        condition = ExpressionUtils.and(condition, ITEM_DO.onShelf.eq(true).and(ITEM_DO.deleteFlag.eq(ConstantsSale.COMMON_DELETE_NO)));

        var pageRequest = queryParamVO.getPageRequest();
        var jpaQuery = jpaQueryFactory.select(ITEM_DO)
                .from(ITEM_DO)
                .where(condition);
        var count = jpaQuery.fetchCount();
        var dataList = jpaQuery
                .orderBy(obtainOrder(pageRequest, ITEM_DO, new OrderSpecifier[]{new OrderSpecifier(Order.DESC, ITEM_DO.createTime)}))
                .offset(pageRequest.getOffset())
                .limit(pageRequest.getPageSize())
                .fetch();

        return PagingVO.<BipItemDO>builder()
                .total(count)
                .records(dataList)
                .build();
    }

    private PagingVO<BipItemDO> queryItems(BipItemOffShelfQueryParamVO queryParamVO) {
        Predicate condition = ITEM_DO.isNotNull();
        if (queryParamVO.getOuId() != null) {
            condition = ExpressionUtils.and(condition, ITEM_DO.ouId.eq(queryParamVO.getOuId()));
        }
        if (StrUtil.isNotBlank(queryParamVO.getState())) {
            condition = ExpressionUtils.and(condition, ITEM_DO.state.eq(queryParamVO.getState()));
        }
        if (StrUtil.isNotBlank(queryParamVO.getShelfCode())) {
            condition = ExpressionUtils.and(condition, ITEM_DO.shelfCode.eq(queryParamVO.getShelfCode()));
        }
        if (queryParamVO.getCreateUserId() != null) {
            condition = ExpressionUtils.and(condition, ITEM_DO.createUserId.eq(queryParamVO.getCreateUserId()));
        }
        if (queryParamVO.getOffShelfReason() != null) {
            condition = ExpressionUtils.and(condition, ITEM_DO.offShelfReason.eq(queryParamVO.getOffShelfReason()));
        }
        if (StrUtil.isNotBlank(queryParamVO.getCreator())){
            condition = ExpressionUtils.and(condition, ITEM_DO.creator.like("%" + queryParamVO.getCreator() + "%"));
        }

        // 已下架的
        condition = ExpressionUtils.and(condition, ITEM_DO.onShelf.eq(false).and(ITEM_DO.offShelfReason.isNotNull())
                .and(ITEM_DO.deleteFlag.eq(ConstantsSale.COMMON_DELETE_NO)));

        var pageRequest = queryParamVO.getPageRequest();
        var jpaQuery = jpaQueryFactory.select(ITEM_DO)
                .from(ITEM_DO)
                .where(condition);
        var count = jpaQuery.fetchCount();
        var dataList = jpaQuery
                .orderBy(obtainOrder(pageRequest, ITEM_DO, new OrderSpecifier[]{new OrderSpecifier(Order.DESC, ITEM_DO.createTime)}))
                .offset(pageRequest.getOffset())
                .limit(pageRequest.getPageSize())
                .fetch();
        return PagingVO.<BipItemDO>builder().total(count).records(dataList).build();
    }

    private PagingVO<BipItemQueryRespVO> querySkuByCategoryCode(Long ouId, String bipCategoryCode, Boolean onShelf) {
        // 先查询分类信息
        var category = itemCategoryRepoProc.getByCode(bipCategoryCode);
        if (category == null) {
            // 没有查询到分类
            return PagingVO.<BipItemQueryRespVO>builder().total(0L).records(Collections.emptyList()).build();
        }

        var itemCatPath = itemRepoProc.getCategory(category.getLevel());

        var condition = itemCatPath.eq(category.getId()).and(ITEM_DO.deleteFlag.ne(ConstantsSale.COMMON_DELETE_YSE));
        if (ouId != null) {
            condition = appendAndCondition(condition, ITEM_DO.ouId.eq(ouId));
        }
        if (onShelf != null) {
            condition = appendAndCondition(condition, ITEM_DO.onShelf.eq(onShelf));
        }

        return queryItemAndSku(condition, PageRequest.of(0, Integer.MAX_VALUE));
    }

    private PagingVO<BipItemQueryRespVO> querySku(BipItemQueryParamVO queryParamVO) {
        var qSku = QBipItemSkuDO.bipItemSkuDO;

        BooleanExpression condition = null;
        if (queryParamVO.getOuId() != null) {
            condition = appendAndCondition(condition, ITEM_DO.ouId.eq(queryParamVO.getOuId()));
        }
        if (queryParamVO.getOnShelf() != null) {
            condition = appendAndCondition(condition, ITEM_DO.onShelf.eq(queryParamVO.getOnShelf()));
        }
        condition = appendAndCondition(condition, ITEM_DO.deleteFlag.ne(ConstantsSale.COMMON_DELETE_YSE));
        if (CharSequenceUtil.isNotBlank(queryParamVO.getItemCode())) {
            condition = appendAndCondition(condition, ITEM_DO.itemCode.like("%" + queryParamVO.getItemCode() + "%"));
        }
        if (CharSequenceUtil.isNotBlank(queryParamVO.getItemName())) {
            condition = appendAndCondition(condition, ITEM_DO.itemName.like("%" + queryParamVO.getItemName() + "%"));
        }
        if (CharSequenceUtil.isNotBlank(queryParamVO.getSkuCode())) {
            condition = appendAndCondition(condition, qSku.skuCode.like("%" + queryParamVO.getSkuCode() + "%"));
        }
        if (CharSequenceUtil.isNotBlank(queryParamVO.getSkuName())) {
            condition = appendAndCondition(condition, qSku.itemName.like("%" + queryParamVO.getSkuName() + "%"));
        }


       return queryItemAndSku(condition, queryParamVO.getPageRequest());
    }

    private PagingVO<BipItemQueryRespVO> queryItemAndSku(Predicate condition, PageRequest pageRequest) {
        var qSku = QBipItemSkuDO.bipItemSkuDO;
        var qExt = QBipItemExtDO.bipItemExtDO;

        var jpaQuery = jpaQueryFactory.select(Projections.bean(BipItemQueryRespVO.class,
                        qSku.bipItemId,
                        qSku.id.as("skuId"),
                        qSku.skuCode,
                        qSku.itemName.as("skuName"),
                        ITEM_DO.categoryId1,
                        ITEM_DO.categoryName1,
                        ITEM_DO.categoryId2,
                        ITEM_DO.categoryName2,
                        ITEM_DO.categoryId3,
                        ITEM_DO.categoryName3,
                        ITEM_DO.itemCode,
                        ITEM_DO.itemName,
                        ITEM_DO.itemId,
                        qSku.price,
                        qExt.unit
                )).from(qSku)
                .leftJoin(ITEM_DO).on(qSku.bipItemId.eq(ITEM_DO.id))
                .leftJoin(qExt).on(qExt.id.eq(qSku.bipItemId))
                .where(condition);
        var count = jpaQuery.fetchCount();
        if (count == 0) {
            // 没有查询到有效记录
            return PagingVO.<BipItemQueryRespVO>builder().total(count).records(Collections.emptyList()).build();
        }

        var defaultOrders = new OrderSpecifier[]{ITEM_DO.createTime.desc()};
        var records = jpaQuery.orderBy(obtainOrder(pageRequest, qSku, defaultOrders))
                .offset(pageRequest.getOffset())
                .limit(pageRequest.getPageSize())
                .fetch();
        if (records.isEmpty()) {
            return PagingVO.<BipItemQueryRespVO>builder().total(count).records(Collections.emptyList()).build();
        }

        var unitMap = convertUdc(UdcEnum.COM_UOM_BX);
        // 设置其它信息
        for (var item :records) {
            item.setUnitName(unitMap.get(item.getUnit()));
        }
        return PagingVO.<BipItemQueryRespVO>builder().total(count).records(records).build();
    }

    private List<BipItemPicRespVO> queryItemPic(Long bipItemId) {
        var picDOList = itemPicRepoProc.queryByBipItemId(bipItemId);
        if (picDOList.isEmpty()) {
            return Collections.emptyList();
        }
        return picDOList.stream().map(CONVERT::itemPic2DetailRespVO).collect(Collectors.toList());
    }

    private List<BipItemSkuRespVO> querySkuDetail(Long bipItemId) {
        var skuDOList = itemSkuRepoProc.queryByBipItemId(bipItemId);
        if (skuDOList.isEmpty()) {
            return Collections.emptyList();
        }

        Map<String, String> itemTypeUdcMap = udcProvider.getValueMapByUdcCode("yst-sale", "ITEM_TYPE");
        Map<String, String> itemType2UdcMap = udcProvider.getValueMapByUdcCode("yst-sale", "ITEM_TYPE2");


        return skuDOList.stream().map(t -> {
            var vo = CONVERT.sku2DetailRespVO(t);
            vo.setAttrList(convert2List(t.getAttr(), BipItemSkuAttrRespVO.class));

            if (StringUtils.hasLength(vo.getType())){
                vo.setTypeName(itemTypeUdcMap.get(vo.getType()));
            }
            if (StringUtils.hasLength(vo.getMateriel())){
                vo.setMaterielName(itemType2UdcMap.get(vo.getMateriel()));
            }

            return vo;
        }).collect(Collectors.toList());
    }

    private void validateForSave(BipItemSaveVO saveVO) {
        var exists = false;

        // 判断商品分类
        exists = itemCategoryRepoProc.existsById(saveVO.getCategoryId3());
        Assert.isTrue(exists, "商品分类不存在");

        // 判断商品是否已存在
        exists = itemRepoProc.existsByItemId(saveVO.getItemId());
        Assert.isFalse(exists, "商品已存在");

        // 校验sku
        validateForSaveSku(saveVO.getSkuList());

        // 校验运费收费方式
        if (Boolean.FALSE.equals(saveVO.getFareFree())) {
            Assert.notNull(saveVO.getFareTmplId(), "运费模板为空");
        }
    }

    private void validateForUpdate(Long id, BipItemUpdateVO updateVO) {
        var exists = itemRepoProc.existsById(id);
        Assert.isTrue(exists, "修改的商品不存在");

        // 判断状态是否可以修改商品，只有草稿状态的可以修改商品
        String state = itemRepoProc.getState(id);
        if (!StrUtil.equals(state, UdcEnum.BIP_ITEM_STATE_DRAFT.getValueCode()) && !StrUtil.equals(state,UdcEnum.BIP_ITEM_STATE_OFF.getValueCode())) {
            Long itemIdOld = itemRepoProc.getItemId(id);
            Assert.isTrue(updateVO.getItemId().longValue() == itemIdOld.longValue(), "商品已不可更改");
        }

        // 判断商品分类
        exists = itemCategoryRepoProc.existsById(updateVO.getCategoryId3());
        Assert.isTrue(exists, "商品分类不存在");

        // 校验sku
        validateForUpdateSku(id, updateVO.getSkuList());

        // 校验运费收费方式
        if (Boolean.FALSE.equals(updateVO.getFareFree())) {
            Assert.notNull(updateVO.getFareTmplId(), "运费模板为空");
        }
    }

    private void validateForSaveSku(List<BipItemSkuSaveVO> skuSaveVOList) {
        var skuNum = skuSaveVOList.size();
        Set<String> skuCodes = new HashSet<>(skuNum);
        for (var sku : skuSaveVOList) {
            Assert.isFalse(skuCodes.contains(sku.getSkuCode()), "SKU编码【{}】重复", sku.getSkuCode());
            skuCodes.add(sku.getSkuCode());

            sku.setShipOne(ObjectUtil.defaultIfNull(sku.getShipOne(), false));
            sku.setStock(ObjectUtil.defaultIfNull(sku.getStock(), 0));
            sku.setLimitBuy(ObjectUtil.defaultIfNull(sku.getLimitBuy(), 1));

            if (skuNum > 1) {
                Assert.notEmpty(sku.getAttrList(), "超过一个SKU时，销售属性不能为空");
            }
        }
    }

    private void validateForUpdateSku(Long id, List<BipItemSkuUpdateVO> skuUpdateVOList) {
        var existsCodeMap = itemSkuRepoProc.getSkuCode(id);

        var skuNum = skuUpdateVOList.size();
        Set<String> skuCodes = new HashSet<>(skuNum);
        for (var sku : skuUpdateVOList) {
            Assert.isFalse(skuCodes.contains(sku.getSkuCode()), "SKU编码【{}】重复", sku.getSkuCode());
            skuCodes.add(sku.getSkuCode());

            // 判断与已存在的是否重复
            Assert.isTrue(ObjectUtil.equal(existsCodeMap.get(sku.getSkuCode()), sku.getId()), "SKU编码【{}】已存在", sku.getSkuCode());

            sku.setShipOne(ObjectUtil.defaultIfNull(sku.getShipOne(), false));
            sku.setStock(ObjectUtil.defaultIfNull(sku.getStock(), 0));
            sku.setLimitBuy(ObjectUtil.defaultIfNull(sku.getLimitBuy(), 1));

            if (skuNum > 1) {
                Assert.notEmpty(sku.getAttrList(), "超过一个SKU时，销售属性不能为空");
            }
        }
    }

    private void updateSku(BipItemDO itemDO, List<BipItemSkuUpdateVO> skuUpdateVOList) {
        var oldSkuMap = itemSkuRepoProc.queryByBipItemId(itemDO.getId()).stream().collect(Collectors.toMap(BipItemSkuDO::getId, t -> t, (t1, t2) -> t1));

        var i = 0;
        List<BipItemSkuDO> doList = new ArrayList<>(skuUpdateVOList.size());
        BigDecimal minPrice = null;
        BipItemSkuDO skuDO = null;
        for (var vo : skuUpdateVOList) {
            if (vo.getId() == null) {
                skuDO = CONVERT.skuUpdateVO2DO(vo);
            } else {
                skuDO = oldSkuMap.get(vo.getId());
                CONVERT.copySkuVO2DO(vo, skuDO);
            }
            skuDO.setBipItemId(itemDO.getId());
            skuDO.setSortNo(i++);
            skuDO.setAttr(CollUtil.isEmpty(vo.getAttrList()) ? "" : convertObj2Str(vo.getAttrList()));
            minPrice = minPrice == null ? vo.getPrice() : NumberUtil.min(minPrice, vo.getPrice());

            doList.add(skuDO);
        }
        itemSkuRepo.saveAll(doList);

        itemDO.setPrice(minPrice);
    }

    private void saveSku(BipItemDO itemDO, List<BipItemSkuSaveVO> skuSaveVOList) {
        var i = 0;
        List<BipItemSkuDO> doList = new ArrayList<>(skuSaveVOList.size());
        BigDecimal minPrice = null;
        BipItemSkuDO skuDO = null;
        for (var vo : skuSaveVOList) {
            skuDO = CONVERT.skuSaveVO2DO(vo);
            skuDO.setBipItemId(itemDO.getId());
            skuDO.setSortNo(i++);
            skuDO.setAttr(CollUtil.isEmpty(vo.getAttrList()) ? "" : convertObj2Str(vo.getAttrList()));
            minPrice = minPrice == null ? vo.getPrice() : NumberUtil.min(minPrice, vo.getPrice());

            doList.add(skuDO);
        }
        itemSkuRepo.saveAll(doList);

        itemDO.setPrice(minPrice);
    }

    private void saveItemPic(BipItemDO itemDO, List<BipItemPicSaveVO> picSaveVOList) {
        var i = 1;
        List<BipItemPicDO> doList = new ArrayList<>(picSaveVOList.size());
        BipItemPicDO picDO = null;
        BipItemPicDO mainPicDO = null;
        for (var vo : picSaveVOList) {
            picDO = CONVERT.itemPicSaveVO2DO(vo);
            picDO.setBipItemId(itemDO.getId());
            picDO.setSortNo(i++);
            picDO.setPic(isImage(vo.getMimeType()));
            if (Boolean.TRUE.equals(vo.getMain())) {
                mainPicDO = picDO;
                mainPicDO.setSortNo(0);
            }

            doList.add(picDO);
        }
        if (mainPicDO != null) {
            // 更新主图信息
            itemDO.setMainPicId(mainPicDO.getId());
            itemDO.setMainPicFileCode(mainPicDO.getFileCode());
            itemDO.setMainPicUrl(mainPicDO.getUrl());
        }
        itemPicRepo.saveAll(doList);

    }

    private void saveItemExt(BipItemDO itemDO, BipItemSaveVO itemSaveVO) {
        var extDO = new BipItemExtDO();
        extDO.setBipItemId(itemDO.getId());
        extDO.setContent(convertObj2Str(itemSaveVO.getContentPicList()));
        extDO.setSpec(CollUtil.isEmpty(itemSaveVO.getSpecList()) ? "" : convertObj2Str(itemSaveVO.getSpecList()));
        extDO.setUnit(itemSaveVO.getUnit());
        extDO.setTimeOnShelfPlan(null);
        extDO.setTimeOffShelfPlan(null);
        itemExtRepo.save(extDO);

        // 回写扩展信息的id
        itemDO.setExtId(extDO.getId());
    }

    private void updateItemExt(BipItemDO itemDO, BipItemUpdateVO itemUpdateVO) {
        var extId = itemDO.getExtId();
        var extDO = itemExtRepo.findById(extId).orElse(new BipItemExtDO());
        extDO.setId(extId);
        extDO.setContent(convertObj2Str(itemUpdateVO.getContentPicList()));
        extDO.setSpec(CollUtil.isEmpty(itemUpdateVO.getSpecList()) ? "" : convertObj2Str(itemUpdateVO.getSpecList()));
        extDO.setUnit(itemUpdateVO.getUnit());
        extDO.setTimeOnShelfPlan(null);
        extDO.setTimeOffShelfPlan(null);
        itemExtRepo.save(extDO);
    }

    private boolean isImage(String mimeType) {
        return mimeType != null && mimeType.toLowerCase().startsWith("image/");
    }
}
