package com.elitesland.yst.production.sale.service.shop;

import cn.hutool.core.lang.Assert;
import com.alibaba.fastjson.JSON;
import com.elitesland.yst.production.sale.common.constant.UdcEnum;
import com.elitesland.yst.production.sale.core.service.BaseServiceImpl;
import com.elitesland.yst.production.sale.dto.param.ItemFreezeParam;
import com.elitesland.yst.production.sale.entity.BipItemDO;
import com.elitesland.yst.production.sale.entity.BipItemSkuDO;
import com.elitesland.yst.production.sale.event.ItemShelfEvent;
import com.elitesland.yst.production.sale.repo.shop.BipItemRepo;
import com.elitesland.yst.production.sale.repo.shop.BipItemRepoProc;
import com.elitesland.yst.production.sale.repo.shop.BipItemSkuRepo;
import com.elitesland.yst.production.sale.repo.shop.BipItemSkuRepoProc;
import com.elitesland.yst.production.sale.service.BipItemRpcService;
import com.elitescloud.cloudt.common.base.ApiResult;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RestController;

import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.List;
import java.util.stream.Collectors;

/**
 * @author Shadow（li qun）
 * @since 2021-11-12 15:40
 */
@Slf4j
@RequiredArgsConstructor
@Service
@RestController
@RequestMapping(BipItemRpcService.URI)
public class BipItemRpcServiceImpl extends BaseServiceImpl implements BipItemRpcService {

    private final BipItemRepo itemRepo;

    private final BipItemSkuRepo bipItemSkuRepo;

    private final BipItemRepoProc bipItemRepoProc;

    private final BipItemSkuRepoProc bipItemSkuRepoProc;

    @Override
    @Transactional(rollbackFor = Exception.class)
    public ApiResult<String> updateItemFreeze(List<ItemFreezeParam> paramList) {

        log.info("冻结接口入参：{}", JSON.toJSONString(paramList));

        List<Long> itemOffShelfIds = new ArrayList<>();

        for (ItemFreezeParam param : paramList) {

            Assert.notNull(param.getFreeze(),"冻结状态不能为空");
            Assert.notEmpty(param.getSkuCode(),"SKU编码不能为空");
            Assert.notNull(param.getBuCode(),"BuCode不能为空");

            List<BipItemSkuDO> skuByCode = bipItemSkuRepoProc.findSkuByCode(param.getSkuCode());
            List<Long> itemIds = skuByCode.stream().map(BipItemSkuDO::getBipItemId).distinct().collect(Collectors.toList());

            List<Long> itemShouldIds = bipItemRepoProc.findByIdAndOuCode(itemIds, param.getBuCode());
            if (itemShouldIds==null || itemShouldIds.size()==0){
                continue;
            }
            List<BipItemSkuDO> collect = new ArrayList<>();

            for (BipItemSkuDO sku : skuByCode) {
                if (itemShouldIds.contains(sku.getBipItemId())) {
                    sku.setFreeze(param.getFreeze());
                    collect.add(sku);
                    itemOffShelfIds.add(sku.getBipItemId());
                }
            }

            log.info("冻结SKU信息：{}", JSON.toJSONString(collect));

            if (collect.size()==0){
                continue;
            }
            bipItemSkuRepo.saveAll(collect);
        }

        List<Long> offShelf = filterOffShelf(itemOffShelfIds);
        updateOffShelfByApprove(offShelf);

        return ApiResult.ok();
    }

    /**
     * 过滤出应该下架的spuId，一个spu下的全部sku冻结则下架
     * @param itemOffShelfIds spuId
     * @return 下架的spuId
     */
    private List<Long> filterOffShelf(List<Long> itemOffShelfIds){
        List<Long> offShelfIds = new ArrayList<>();
        for (Long shelfId : itemOffShelfIds) {
            List<BipItemSkuDO> skuDOS = bipItemSkuRepoProc.queryByBipItemId(shelfId);
            List<BipItemSkuDO> collect = skuDOS.stream().filter(BipItemSkuDO::getFreeze).collect(Collectors.toList());
            if (skuDOS.size() == collect.size()){
                offShelfIds.add(shelfId);
            }
        }
        return offShelfIds;
    }

    /**
     * 下架spu
     * @param offShelfIds spu
     */
    private void updateOffShelfByApprove(List<Long> offShelfIds) {

        List<BipItemDO> itemList = itemRepo.findAllById(offShelfIds);
        log.info("冻结自动下架信息：{}", JSON.toJSONString(itemList));

        // 待审批 且 已上架  才可以下架
        var items = itemList.stream()
                .filter(t -> UdcEnum.BIP_ITEM_STATE_SHELF.getValueCode().equals(t.getState()) && Boolean.TRUE.equals(t.getOnShelf()))
                .collect(Collectors.toList());
        if (items.isEmpty()) {
            return;
        }

        for (var item : items) {
            // 立即下架
            item.setState(UdcEnum.BIP_ITEM_STATE_OFF.getValueCode());
            item.setOnShelf(false);
            item.setTimeOnShelf(null);
            item.setTimeOffShelf(LocalDateTime.now());
            item.setOffShelfReason(UdcEnum.BIP_ITEM_OFF_SHELF_FREEZE.getValueCode());
        }

        log.info("冻结自动下架商品信息：{}", JSON.toJSONString(items));
        // 更新商品状态
        itemRepo.saveAll(items);

        // 发布下架事件
        for (var item : items) {
            publishEventSync(new ItemShelfEvent(this, false, item.getId()));
        }

    }
}
