package com.elitesland.yst.production.sale.event.listener;

import cn.hutool.core.lang.Assert;
import cn.hutool.core.util.ObjectUtil;
import com.elitesland.yst.production.sale.api.service.shop.BipItemMngService;
import com.elitesland.yst.production.sale.api.service.shop.BipOrderDService;
import com.elitesland.yst.production.sale.event.ItemSaleChangeEvent;
import lombok.extern.slf4j.Slf4j;
import org.springframework.context.ApplicationListener;
import org.springframework.stereotype.Component;

import java.time.LocalDateTime;
import java.util.concurrent.CompletableFuture;

/**
 * 商品销量监听器.
 * <p>
 * 当监听到有订单支付成功时就更新商品的销量
 *
 * @author Kaiser（wang shao）
 * @date 2021/09/02
 */
@Component
@Slf4j
public class ItemSaleNumListener implements ApplicationListener<ItemSaleChangeEvent> {

    private final BipOrderDService orderDService;
    private final BipItemMngService itemMngService;

    public ItemSaleNumListener(BipOrderDService orderDService, BipItemMngService itemMngService) {
        this.orderDService = orderDService;
        this.itemMngService = itemMngService;
    }

    @Override
    public void onApplicationEvent(ItemSaleChangeEvent event) {
        if (event.getBipItemId() == null) {
            log.error("监听到商品销量变更事件，但商品ID为空");
            return;
        }

        log.debug("监听到商品【{}】的销量变更", event.getBipItemId());
        CompletableFuture.runAsync(() -> updateItemSaleNum(event.getBipItemId()))
                .exceptionally(e -> {
                    log.error("更新商品【" + event.getBipItemId() + "】销量发生异常", e);
                    return null;
                });
    }

    private void updateItemSaleNum(Long bipItemId) {
        // 总销量
        var saleResult = orderDService.countItemSaleNum(bipItemId, null, null);
        Assert.isTrue(saleResult.isSuccess(), saleResult.getMsg());

        // 90天的销量
        var sale90Result = orderDService.countItemSaleNum(bipItemId, LocalDateTime.now().minusDays(90), null);
        Assert.isTrue(sale90Result.isSuccess(), sale90Result.getMsg());

        itemMngService.updateSale(bipItemId, ObjectUtil.defaultIfNull(saleResult.getData(), 0L), ObjectUtil.defaultIfNull(sale90Result.getData(), 0L));
    }
}
