package com.elitesland.yst.production.sale.repo;

import cn.hutool.core.collection.CollUtil;
import cn.hutool.core.collection.CollectionUtil;
import cn.hutool.core.collection.IterUtil;
import com.elitescloud.boot.jpa.common.BaseRepoProc;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitesland.yst.production.sale.api.vo.param.salesman.*;
import com.elitesland.yst.production.sale.api.vo.resp.salesman.SalesmanDetailInfoRespVO;
import com.elitesland.yst.production.sale.api.vo.resp.salesman.SalesmanInfoRespVO;
import com.elitesland.yst.production.sale.api.vo.resp.salesman.SalesmanInfoSimpleRespVO;
import com.elitesland.yst.production.sale.api.vo.resp.salesman.SalesmanSuperiorRespVO;
import com.elitesland.yst.production.sale.dto.save.SalesmanUpdateDTO;
import com.elitesland.yst.production.sale.entity.QSalesmanInfoDO;
import com.elitesland.yst.production.sale.entity.QSalesmanRegionDO;
import com.elitesland.yst.production.sale.entity.SalesmanInfoDO;
import com.google.common.collect.Lists;
import com.querydsl.core.types.ExpressionUtils;
import com.querydsl.core.types.Predicate;
import com.querydsl.core.types.Projections;
import com.querydsl.core.types.dsl.BooleanExpression;
import com.querydsl.core.types.dsl.Expressions;
import com.querydsl.jpa.JPAExpressions;
import com.querydsl.jpa.impl.JPAQuery;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Repository;

import java.util.ArrayList;
import java.util.List;
import java.util.Set;

/**
 * @author Danny.Wang
 * @title SalesmanInfoRepoProc
 * @Date2023/2/21 16:09
 */
@Repository
public class SalesmanInfoRepoProc extends BaseRepoProc<SalesmanInfoDO> {

    @Autowired
    private SalesmanInfoRepo salesmanInfoRepo;

    private static final QSalesmanInfoDO Q_SALESMAN_INFO_DO = QSalesmanInfoDO.salesmanInfoDO;

    protected SalesmanInfoRepoProc() {
        super(Q_SALESMAN_INFO_DO);
    }


    public JPAQuery<SalesmanInfoRespVO> findSalesmanInfo(SalesmanInfoQueryVO salesmanInfoQueryVO){
        JPAQuery<SalesmanInfoRespVO> jpaQuery = jpaQueryFactory.select(Projections.bean(SalesmanInfoRespVO.class,
                        Q_SALESMAN_INFO_DO.id,
                        Q_SALESMAN_INFO_DO.salesmanNo,
                        Q_SALESMAN_INFO_DO.salesmanType,
                        Q_SALESMAN_INFO_DO.ouCode,
                        Q_SALESMAN_INFO_DO.ouName,
                        Q_SALESMAN_INFO_DO.name,
                        Q_SALESMAN_INFO_DO.enableStatus))
                .from(Q_SALESMAN_INFO_DO);

        List<Predicate> predicates = new ArrayList<>();
        where(salesmanInfoQueryVO, predicates);
        jpaQuery.where(ExpressionUtils.allOf(predicates));
        salesmanInfoQueryVO.fillOrders(jpaQuery, Q_SALESMAN_INFO_DO);
        if(StringUtils.isNotEmpty(salesmanInfoQueryVO.getJurisdiction())){
            QSalesmanRegionDO qSalesmanRegionDO = QSalesmanRegionDO.salesmanRegionDO;
            BooleanExpression booleanExpression = JPAExpressions.select(Expressions.ONE)
                    .from(qSalesmanRegionDO).where(qSalesmanRegionDO.masId.eq(Q_SALESMAN_INFO_DO.id))
                    .where(qSalesmanRegionDO.regionCode.eq(salesmanInfoQueryVO.getJurisdiction())).exists();
            jpaQuery.where(booleanExpression);
        }
        if(CollectionUtil.isNotEmpty(salesmanInfoQueryVO.getJurisdictions())){
            QSalesmanRegionDO qSalesmanRegionDO = QSalesmanRegionDO.salesmanRegionDO;
            BooleanExpression booleanExpression = JPAExpressions.select(Expressions.ONE)
                    .from(qSalesmanRegionDO).where(qSalesmanRegionDO.masId.eq(Q_SALESMAN_INFO_DO.id))
                    .where(qSalesmanRegionDO.regionCode.in(salesmanInfoQueryVO.getJurisdictions())).exists();
            jpaQuery.where(booleanExpression);
        }

        return jpaQuery;
    }

    public JPAQuery<SalesmanDetailInfoRespVO> findSalesmanInfoDetail(SalesmanInfoQueryVO salesmanInfoQueryVO){
        JPAQuery<SalesmanDetailInfoRespVO> jpaQuery = jpaQueryFactory.select(Projections.bean(SalesmanDetailInfoRespVO.class,
                        Q_SALESMAN_INFO_DO.id,
                        Q_SALESMAN_INFO_DO.salesmanNo,
                        Q_SALESMAN_INFO_DO.ouName,
                        Q_SALESMAN_INFO_DO.salesmanType,
                        Q_SALESMAN_INFO_DO.enableStatus))
                .from(Q_SALESMAN_INFO_DO);

        List<Predicate> predicates = new ArrayList<>();

        where(salesmanInfoQueryVO, predicates);
        if(StringUtils.isNotEmpty(salesmanInfoQueryVO.getJurisdiction())){
            QSalesmanRegionDO qSalesmanRegionDO = QSalesmanRegionDO.salesmanRegionDO;
            BooleanExpression booleanExpression = JPAExpressions.select(Expressions.ONE)
                    .from(qSalesmanRegionDO).where(qSalesmanRegionDO.masId.eq(Q_SALESMAN_INFO_DO.id))
                    .where(qSalesmanRegionDO.regionCode.eq(salesmanInfoQueryVO.getJurisdiction())).exists();
            jpaQuery.where(booleanExpression);
        }
        jpaQuery.where(ExpressionUtils.allOf(predicates));

        salesmanInfoQueryVO.fillOrders(jpaQuery, Q_SALESMAN_INFO_DO);
        salesmanInfoQueryVO.setPaging(jpaQuery);

        return jpaQuery;
    }

    public List<SalesmanInfoSimpleRespVO> simpleQuery(SalesmanInfoSimpleQueryVO param){
        Predicate expression = Expressions.ONE.eq(Expressions.ONE);
        if (param.getEnableStatus() != null){
            expression =  ExpressionUtils.and(expression, Q_SALESMAN_INFO_DO.enableStatus.eq(param.getEnableStatus()));
        }
        if (CollUtil.isNotEmpty(param.getCodes())){
            expression =  ExpressionUtils.and(expression, Q_SALESMAN_INFO_DO.salesmanNo.in(param.getCodes()));
        }
        if (CollUtil.isNotEmpty(param.getIds())){
            expression =  ExpressionUtils.and(expression, Q_SALESMAN_INFO_DO.id.in(param.getIds()));
        }

        return jpaQueryFactory.select(Projections.bean(SalesmanInfoSimpleRespVO.class,
                Q_SALESMAN_INFO_DO.salesmanNo,
                Q_SALESMAN_INFO_DO.id,
                Q_SALESMAN_INFO_DO.enableStatus,
                Q_SALESMAN_INFO_DO.salesmanType))
                .from(Q_SALESMAN_INFO_DO)
                .where(expression).fetch();
    }

    public List<SalesmanInfoDO> getSalesmanByCodes(Set<String> salesmanCodes){
        BooleanExpression e = Q_SALESMAN_INFO_DO.salesmanNo.in(salesmanCodes).and(Q_SALESMAN_INFO_DO.deleteFlag.eq(0));
        Iterable<SalesmanInfoDO> iterable = salesmanInfoRepo.findAll(e);
        if(IterUtil.isEmpty(iterable)){
            return new ArrayList<>();
        }
        return Lists.newArrayList(iterable);
    }

    public List<SalesmanInfoDO> findSalesmanAllInfo(SalesmanInfoQueryVO salesmanInfoQueryVO){
        List<SalesmanInfoDO> salesmanInfoDOS = jpaQueryFactory.select(Q_SALESMAN_INFO_DO)
                .from(Q_SALESMAN_INFO_DO).fetch();

        List<Predicate> predicates = new ArrayList<>();
        where(salesmanInfoQueryVO, predicates);
        return salesmanInfoDOS;
    }

    private static void where(SalesmanInfoQueryVO salesmanInfoQueryVO, List<Predicate> predicates) {
        if(StringUtils.isNotBlank(salesmanInfoQueryVO.getSalesman())){
            predicates.add(Q_SALESMAN_INFO_DO.salesmanNo.contains(salesmanInfoQueryVO.getSalesman()).or(Q_SALESMAN_INFO_DO.name.contains(salesmanInfoQueryVO.getSalesman())));
        }
        if(StringUtils.isNotBlank(salesmanInfoQueryVO.getSalesmanCode())){
            predicates.add(Q_SALESMAN_INFO_DO.salesmanNo.contains(salesmanInfoQueryVO.getSalesmanCode()));
        }
        if (StringUtils.isNotBlank(salesmanInfoQueryVO.getOuCode())) {
            predicates.add(Q_SALESMAN_INFO_DO.ouCode.eq(salesmanInfoQueryVO.getOuCode()));
        }
        if (salesmanInfoQueryVO.getEnableStatus() != null) {
            predicates.add(Q_SALESMAN_INFO_DO.enableStatus.eq(salesmanInfoQueryVO.getEnableStatus()));
        }

        if (StringUtils.isNotBlank(salesmanInfoQueryVO.getSalesmanType())){
            predicates.add(Q_SALESMAN_INFO_DO.salesmanType.eq(salesmanInfoQueryVO.getSalesmanType()));
        }
        if (CollUtil.isNotEmpty(salesmanInfoQueryVO.getCodes())){
            predicates.add(Q_SALESMAN_INFO_DO.salesmanNo.in(salesmanInfoQueryVO.getCodes()));
        }
    }

    public void changeEnableStatus(Long id, Integer enableStatus){

        jpaQueryFactory.update(Q_SALESMAN_INFO_DO)
                .set(Q_SALESMAN_INFO_DO.enableStatus, enableStatus)
                .where(Q_SALESMAN_INFO_DO.id.eq(id))
                .execute();
    }

    public JPAQuery<SalesmanSuperiorRespVO> findBySalesmanType (SalesmanSuperiorQueryVO param){
        JPAQuery<SalesmanSuperiorRespVO> jpaQuery = jpaQueryFactory.select(Projections.bean(SalesmanSuperiorRespVO.class
                        , Q_SALESMAN_INFO_DO.id
                        , Q_SALESMAN_INFO_DO.salesmanNo
                        , Q_SALESMAN_INFO_DO.salesmanType.as("type")))
                .from(Q_SALESMAN_INFO_DO);

        List<Predicate> predicates = new ArrayList<>();
        if (StringUtils.isNotBlank(param.getUdc())) {
            predicates.add(Q_SALESMAN_INFO_DO.salesmanType.eq(param.getUdc()));
        }
        if(StringUtils.isNotBlank(param.getSalesmanNo())){
            String like = "%" + param.getSalesmanNo() + "%";
            predicates.add(Q_SALESMAN_INFO_DO.salesmanNo.like(like));
        }
        jpaQuery.where(ExpressionUtils.allOf(predicates));
        param.setPaging(jpaQuery);
        param.fillOrders(jpaQuery,Q_SALESMAN_INFO_DO);


        return jpaQuery;
    }

    public JPAQuery<SalesmanInfoRespVO> findByRelateCode(List<String> codes, SalesmanInfoQueryVO salesmanInfoQueryVO){
        JPAQuery<SalesmanInfoRespVO> jpaQuery = jpaQueryFactory.select(Projections.bean(SalesmanInfoRespVO.class,
                        Q_SALESMAN_INFO_DO.id,
                        Q_SALESMAN_INFO_DO.salesmanType,
                        Q_SALESMAN_INFO_DO.ouName,
                        Q_SALESMAN_INFO_DO.ouCode,
                        Q_SALESMAN_INFO_DO.salesmanNo,
                        Q_SALESMAN_INFO_DO.enableStatus))
                .from(Q_SALESMAN_INFO_DO)
                .where(Q_SALESMAN_INFO_DO.salesmanNo.in(codes));

        salesmanInfoQueryVO.fillOrders(jpaQuery, Q_SALESMAN_INFO_DO);
        salesmanInfoQueryVO.setPaging(jpaQuery);
        return jpaQuery;
    }

    public JPAQuery<SalesmanInfoRespVO> findByRelateCode(List<String> codes, SalesmanSubordinateQueryVO salesmanSubordinateQueryVO){
        JPAQuery<SalesmanInfoRespVO> jpaQuery = jpaQueryFactory.select(Projections.bean(SalesmanInfoRespVO.class,
                        Q_SALESMAN_INFO_DO.id,
                        Q_SALESMAN_INFO_DO.salesmanType,
                        Q_SALESMAN_INFO_DO.ouName,
                        Q_SALESMAN_INFO_DO.salesmanNo,
                        Q_SALESMAN_INFO_DO.enableStatus))
                .from(Q_SALESMAN_INFO_DO)
                .where(Q_SALESMAN_INFO_DO.salesmanNo.in(codes));

        salesmanSubordinateQueryVO.fillOrders(jpaQuery, Q_SALESMAN_INFO_DO);
        salesmanSubordinateQueryVO.setPaging(jpaQuery);
        return jpaQuery;
    }

    public Long updateSalesmanInfo(SalesmanUpdateDTO salesmanUpdateDTO){
        return jpaQueryFactory.update(Q_SALESMAN_INFO_DO)
                .set(Q_SALESMAN_INFO_DO.fileCode, salesmanUpdateDTO.getFileCode())
                .where(Q_SALESMAN_INFO_DO.id.eq(salesmanUpdateDTO.getSalesmanId()))
                .execute();
    }

    public List<SalesmanInfoDO> getByIdByCodes(List<String> empCodes){
        BooleanExpression expression = Q_SALESMAN_INFO_DO.salesmanNo.in(empCodes);
        Iterable<SalesmanInfoDO> iterable = salesmanInfoRepo.findAll(expression);
        if(!IterUtil.isEmpty(iterable)){
            return Lists.newArrayList(iterable);
        }
        return new ArrayList<>();
    }

    public List<SalesmanInfoDO> getByIds(List<Long> ids){
        BooleanExpression expression = Q_SALESMAN_INFO_DO.id.in(ids);
        Iterable<SalesmanInfoDO> iterable = salesmanInfoRepo.findAll(expression);
        if(!IterUtil.isEmpty(iterable)){
            return Lists.newArrayList(iterable);
        }
        return new ArrayList<>();
    }


    public PagingVO<SalesmanInfoDO> getPageByQueryVO(SalesmanUnderLingQueryVO queryVO) {
        BooleanExpression expression = Q_SALESMAN_INFO_DO.salesmanNo.in(queryVO.getUnderlingCodes()).and(Q_SALESMAN_INFO_DO.deleteFlag.eq(0));
        JPAQuery<SalesmanInfoDO> query = jpaQueryFactory.select(Q_SALESMAN_INFO_DO)
                .from(Q_SALESMAN_INFO_DO)
                .where(expression);
        if(StringUtils.isNotEmpty(queryVO.getSalemanNoNameLike())){
            query.where(Q_SALESMAN_INFO_DO.salesmanNo.like(queryVO.getSalemanNoNameLike()).or(Q_SALESMAN_INFO_DO.name.like(queryVO.getSalemanNoNameLike())));
        }
        queryVO.fillOrders(query, Q_SALESMAN_INFO_DO);
        queryVO.setPaging(query);
        return PagingVO.<SalesmanInfoDO>builder()
                .total(query.fetchCount())
                .records(query.fetch())
                .build();
    }


    public PagingVO<SalesmanSuperiorRespVO> getSalesmanSelectPage(SalesmanSuperiorQueryVO queryVO){
        JPAQuery<SalesmanSuperiorRespVO> jpaQuery = jpaQueryFactory.select(Projections.bean(SalesmanSuperiorRespVO.class,
                Q_SALESMAN_INFO_DO.id,
                Q_SALESMAN_INFO_DO.salesmanNo,
                Q_SALESMAN_INFO_DO.salesmanType.as("type"),
                Q_SALESMAN_INFO_DO.name
                )).from(Q_SALESMAN_INFO_DO);
        BooleanExpression e = Q_SALESMAN_INFO_DO.deleteFlag.eq(0).and(Q_SALESMAN_INFO_DO.enableStatus.eq(1));
        if(StringUtils.isNotBlank(queryVO.getUdc())){
            e = e.and(Q_SALESMAN_INFO_DO.salesmanType.eq(queryVO.getUdc()));
        }
        if(StringUtils.isNotBlank(queryVO.getSalesmanNo())){
            e = e.and(Q_SALESMAN_INFO_DO.salesmanNo.contains(queryVO.getSalesmanNo()));
        }
        if(StringUtils.isNotBlank(queryVO.getName())){
            e = e.and(Q_SALESMAN_INFO_DO.name.contains(queryVO.getName()));
        }
        jpaQuery.where(e);
        queryVO.fillOrders(jpaQuery,Q_SALESMAN_INFO_DO);
        queryVO.setPaging(jpaQuery);
        return PagingVO.<SalesmanSuperiorRespVO>builder()
                .total(jpaQuery.fetchCount())
                .records(jpaQuery.fetch())
                .build();
    }

}
