package com.elitesland.yst.production.sale.service;

import cn.hutool.core.collection.CollectionUtil;
import cn.hutool.core.util.StrUtil;
//import com.elitesland.yst.production.sale.api.service.SalContractService;
import com.elitesland.yst.production.sale.api.service.SalPayDepositService;
import com.elitesland.yst.production.sale.api.vo.param.sal.SalPayDepositApproveParamVO;
import com.elitesland.yst.production.sale.api.vo.param.sal.SalPayDepositParamVO;
import com.elitesland.yst.production.sale.api.vo.resp.pro.SalContractSimpleRespVO;
import com.elitesland.yst.production.sale.api.vo.resp.sal.SalPayDepositRespVO;
import com.elitesland.yst.production.sale.api.vo.resp.sal.SalPayDepositReturnRespVO;
import com.elitesland.yst.production.sale.common.constant.UdcEnum;
import com.elitesland.yst.production.sale.convert.SalReceiptConvert;
import com.elitesland.yst.production.sale.core.service.BaseServiceImpl;
import com.elitesland.yst.production.sale.entity.QSalReceiptDO;
import com.elitesland.yst.production.sale.entity.SalReceiptDO;
import com.elitesland.yst.production.sale.repo.SalReceiptRepo;
import com.elitesland.yst.production.sale.rmi.ystsystem.RmiSysNextNumberService;
import com.elitescloud.cloudt.common.annotation.SysCodeProc;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitescloud.boot.exception.BusinessException;
import com.elitescloud.cloudt.core.security.util.DataAuthJpaUtil;
import com.querydsl.core.types.ExpressionUtils;
import com.querydsl.core.types.Predicate;
import com.querydsl.core.types.Projections;
import com.querydsl.jpa.impl.JPAQuery;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import lombok.val;
import org.springframework.data.domain.PageRequest;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;
import org.springframework.util.StringUtils;

import java.math.BigDecimal;
import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.stream.Collectors;

/**
 * <p>
 *
 * </p>
 *
 * @author zhao.zhi.hao
 * @since 2021/7/16 15:07
 */
@Service
@RequiredArgsConstructor
@Slf4j
public class SalPayDepositServiceImpl extends BaseServiceImpl implements SalPayDepositService {


    private static final QSalReceiptDO qSalReceiptDO = QSalReceiptDO.salReceiptDO;

    private final RmiSysNextNumberService rmiSysNextNumberService;

    private final SalReceiptRepo salReceiptRepo;

//    private final SalContractService salContractService;


    /**
     * 定义docType 为PD
     */
    private static final String DOC_TYPE_01 = "10";
    private static final String DOC_TYPE_02 = "20";
    //"查询范围 QueryRange --全部 all  已退押金 depositRefunded 未退押金 unreturnedDeposit"
    private static final String QUERYRANGE_ALL = "all";
    private static final String QUERYRANGE_DEPOSIT_REFUNDED = "depositRefunded";
    private static final String QUERYRANGE_UN_DEPOSIT = "unreturnedDeposit";

    @Override
    @SysCodeProc
    public PagingVO<SalPayDepositRespVO> search(SalPayDepositParamVO searchParam) {
        if (StrUtil.isNotBlank(searchParam.getContractNo()) ||
                StrUtil.isNotBlank(searchParam.getContractName()) || Objects.nonNull(searchParam.getCustId())) {
//            var contractQueryParam = new SalContractQueryParam();
//            contractQueryParam.setContractNo(searchParam.getContractNo());
//            contractQueryParam.setContractName(searchParam.getContractName());
//            contractQueryParam.setCustId(searchParam.getCustId());
//            List<SalContractSimpleRespVO> contractResult = salContractService.queryByParam(contractQueryParam);
//            List<String> contractNos = contractResult.stream().map(SalContractSimpleRespVO::getContractNo).collect(Collectors.toList());
//            searchParam.setContractNos(contractNos);
        }

        var jpaQuery = this.select(searchParam);
        long total = jpaQuery.fetchCount();
        if (total == 0) {
            return PagingVO.<SalPayDepositRespVO>builder().build();
        }
        // 添加分页和排序
        PageRequest pageRequest = wrapperPageRequest(searchParam.getPageRequest(), null);
        appendPageAndSort(jpaQuery, pageRequest, qSalReceiptDO);
        List<SalPayDepositRespVO> respVOS = jpaQuery.fetch();

        // 补充数据
        this.translatePage(respVOS);
        return PagingVO.<SalPayDepositRespVO>builder()
                .total(total)
                .records(respVOS).build();
    }


    private void translatePage(List<SalPayDepositRespVO> respVOS) {
        List<String> contractNos = respVOS.stream().map(SalPayDepositRespVO::getCustCode).filter(Objects::nonNull).distinct().collect(Collectors.toList());
        List<SalContractSimpleRespVO> contractResult = new ArrayList<>();
        if (!CollectionUtils.isEmpty(contractNos)) {
//            SalContractQueryParam contractQueryParam = new SalContractQueryParam();
//            contractQueryParam.setContractNos(contractNos);
//            contractResult = salContractService.queryByParam(contractQueryParam);
        }
        List<SalContractSimpleRespVO> finalContractResult = contractResult;
        respVOS.forEach(deposit -> {
            Optional<SalContractSimpleRespVO> contractSimpleRespVO = finalContractResult.stream().filter(con -> con.getContractNo().equals(deposit.getContractCode()))
                    .findFirst();

            if (contractSimpleRespVO.isPresent()) {
                SalContractSimpleRespVO simpleRespVO = contractSimpleRespVO.get();
                deposit.setCustName(simpleRespVO.getCustName());
                deposit.setContractName(simpleRespVO.getContractName());
                deposit.setCustContactTel(simpleRespVO.getCustPicTel());
                deposit.setCustContactName(simpleRespVO.getCustPic());
            }
        });
    }

    @Override
    @SysCodeProc
    public SalPayDepositRespVO findSalPayDepositById(Long id) {
        List<SalPayDepositRespVO> salPayDepositRespVoS = this.select2(id).fetch();
        if (!CollectionUtils.isEmpty(salPayDepositRespVoS)) {
            this.translatePage(salPayDepositRespVoS);
            return salPayDepositRespVoS.get(0);
        } else {
            throw new BusinessException("该支付押金单不存在,请检查");
        }
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public Long createSalPayDeposit(SalPayDepositRespVO salPayDepositRespVO) {
        SalReceiptConvert salReceiptConvert = SalReceiptConvert.INSTANCE;
        /*
         * 第一步
         *
         *  1.保存只保存为待提交草稿状态，不更新销售合同。单据类型保存为PD。
         */
        if (StringUtils.isEmpty(salPayDepositRespVO.getDocNo())) {
            String salQuoDocNo = rmiSysNextNumberService.generateCode("yst-sale","PD", null);
            salPayDepositRespVO.setDocNo(salQuoDocNo);
        }
        /*
         * 押金金额不可大于合同的质保金
         * */
        //checkForSave(salPayDepositRespVO);
        /*
         * 第二步
         *   2.数据保存
         */
        if (StringUtils.isEmpty(salPayDepositRespVO.getDocStatus())) {
            salPayDepositRespVO.setDocStatus(UdcEnum.SAL_QUOTATION_STATUS_DRAFT.getValueCode());
        }
        if (!salPayDepositRespVO.getDocStatus().equals(UdcEnum.SAL_QUOTATION_STATUS_APPROVING.getValueCode())) {
            salPayDepositRespVO.setDocStatus(UdcEnum.SAL_QUOTATION_STATUS_DRAFT.getValueCode());
        }

        salPayDepositRespVO.setDocType(DOC_TYPE_01);
        SalReceiptDO salReceiptDO = salReceiptConvert.detailRespVOToDo(salPayDepositRespVO);
        salReceiptDO.setOpenAmt(salPayDepositRespVO.getAmt());
        salReceiptDO.setReturningAmt(new BigDecimal(0));
        salReceiptDO.setReturnAmt(new BigDecimal(0));
        //初始化版本号
        salReceiptDO.setAuditDataVersion(0);
        //添加销售组织权限管理字段
        salReceiptDO.setSecBuId(salPayDepositRespVO.getBuId());
        SalReceiptDO save = salReceiptRepo.save(salReceiptDO);
        //处理集合
        return save.getId();
    }

    private void checkForSave(SalPayDepositRespVO salPayDepositRespVO) {
        Optional<SalReceiptDO> byId = salReceiptRepo.findById(salPayDepositRespVO.getId());
        if (byId.isPresent()) {
//
//            SalReceiptDO salReceiptDO = byId.get();
//            SalContractDetailRespVO salContractDetailRespVO = salContractService.queryOneDetail(salReceiptDO.getContractId());
//            //押金金额不可大于合同的质保金  compareTo<=时  结果小于1
//            if (salPayDepositRespVO.getAmt().compareTo(salContractDetailRespVO.getGuaranteeAmt()) < 1) {
//                throw new BusinessException("押金金额不可大于合同的质保金");
//            }
        }
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public Long approveSalPayDeposit(SalPayDepositApproveParamVO param) {
        salReceiptRepo.approveSalPayDeposit(
                param.getId(),
                param.getDocStatus(),
                param.getApprComment(),
                LocalDateTime.now(),
                param.getApprUserId(),
                param.getApprUserName()
        );
        return param.getId();
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public Long submitSalPayDeposit(SalPayDepositRespVO salPayDepositRespVO) {

        //校验
        String contractCode = salPayDepositRespVO.getContractCode();
        JPAQuery<SalPayDepositRespVO> jpaQuery = this.select5(contractCode);
        List<SalPayDepositRespVO> fetch = jpaQuery.fetch();
        if (CollectionUtil.isEmpty(fetch)||StringUtils.isEmpty(salPayDepositRespVO.getContractCode())) {
            //如果为空则说明这是直接提交没走保存
            if (StringUtils.isEmpty(salPayDepositRespVO.getDocStatus())) {
                salPayDepositRespVO.setDocStatus(UdcEnum.SAL_QUOTATION_STATUS_APPROVING.getValueCode());
                return this.createSalPayDeposit(salPayDepositRespVO);
            }
            salPayDepositRespVO.setDocStatus(UdcEnum.SAL_QUOTATION_STATUS_APPROVING.getValueCode());
            return this.createSalPayDeposit(salPayDepositRespVO);
        }
        throw new BusinessException("此合同已支付押金，请勿重复申请");
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public Long deleteSalPayDeposit(Long id) {
        Optional<SalReceiptDO> byId = salReceiptRepo.findById(id);
        if (byId.isPresent()) {
            SalReceiptDO salReceiptDO = byId.get();
            if (salReceiptDO.getDocStatus().equals(UdcEnum.SAL_QUOTATION_STATUS_DRAFT.getValueCode())) {
                salReceiptRepo.deleteSalPayDeposit(id, "1");
                return id;
            } else {
                throw new BusinessException("只能删除待提交状态订单");
            }
        }
        throw new BusinessException("没有查出订单");
    }



    //退押金申请==========================================================================================================


    @Override
    @Transactional(rollbackFor = Exception.class)
    public Long refundSalQuotationSubmit(SalPayDepositReturnRespVO param) {
        if (StringUtils.isEmpty(param.getBuName())) {

            throw new BusinessException("组织名称为空");
        }
        if (param.getId()!=null) {
            Optional<SalReceiptDO> result = salReceiptRepo.findById(param.getId());
            if (result.isPresent()) {
                if (UdcEnum.SAL_QUOTATION_STATUS_APPROVING.getValueCode().equals(result.get().getDocStatus())) {
                    throw new BusinessException("单据状态已经为待审批的不能再次提交");
                }
            }else {
                throw new BusinessException("单据状态不存在");
            }
        }


        //获得押金单单号并查出该押金单
        String relateDocNo = param.getRelateDocNo();
        SalReceiptDO salReceiptRelateDO = salReceiptRepo.findByDocNo(relateDocNo);
        if (salReceiptRelateDO.getDocStatus().equals(UdcEnum.SAL_QUOTATION_STATUS_APPROVED.getValueCode())
                && salReceiptRelateDO.getDocType().equals(DOC_TYPE_01)) {

            Integer auditDataVersion = salReceiptRelateDO.getAuditDataVersion();

            //获得本单退押金金额
            BigDecimal returnAmt = param.getReturnAmt();
            if (returnAmt.compareTo(salReceiptRelateDO.getOpenAmt()) > 0) {
                throw new BusinessException("退押金金额大于押金余额");
            }
            //计算退款中金额
            BigDecimal returningAmt = salReceiptRelateDO.getReturningAmt().add(param.getReturnAmt());
            //计算押金余额
            BigDecimal openAmt = salReceiptRelateDO.getOpenAmt().subtract(param.getReturnAmt());
            if (openAmt.compareTo(new BigDecimal(0)) < 0) {
                throw new BusinessException("押金余额不能为负数");
            }
            //更新原押金单   使用版本号防止并发出现问题
            int num = salReceiptRepo.updateByDocNoAndVersion(openAmt, returningAmt, relateDocNo, auditDataVersion, auditDataVersion + 1);
            if (num == 0) {
                throw new BusinessException("请勿同时操作");
            }


            SalReceiptConvert salReceiptConvert = SalReceiptConvert.INSTANCE;
            SalReceiptDO salReceiptDO = salReceiptConvert.detailReturnRespVOToDo(param);
            if (!StrUtil.isNotBlank(param.getDocNo())) {
                String salQuoDocNo = rmiSysNextNumberService.generateCode("yst-sale","PD", new ArrayList<>());
                salReceiptDO.setDocNo(salQuoDocNo);

            }
            //添加销售组织权限管理字段
            salReceiptDO.setSecBuId(param.getBuId());
            salReceiptDO.setAmt(returnAmt);
            salReceiptDO.setDocStatus(UdcEnum.SAL_QUOTATION_STATUS_APPROVING.getValueCode());
            return salReceiptRepo.save(salReceiptDO).getId();
        } else {
            throw new BusinessException("仅类型为支付押金的可申请，且状态必须为已审批");
        }
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public Long refundSalQuotationSave(SalPayDepositReturnRespVO param) {

        if (StringUtils.isEmpty(param.getBuName())) {
            throw new BusinessException("组织名称不能为空");
        }
        if (param.getId()!=null) {
            SalPayDepositReturnRespVO salPayDepositReturnRespVO = this.refundSalQuotationQueryById(param.getId());
            if (salPayDepositReturnRespVO.getDocStatus().equals(DOC_TYPE_01)) {
                throw new BusinessException("押金类型只能为退押金类型");
            }
        }
        //获得押金单单号并查出该押金单
        SalReceiptDO salReceiptRelateDO = salReceiptRepo.findByDocNo(param.getRelateDocNo());

        if (salReceiptRelateDO.getDocStatus().equals(UdcEnum.SAL_QUOTATION_STATUS_APPROVED.getValueCode()) && salReceiptRelateDO.getDocType().equals(DOC_TYPE_01)) {
            //获得本单退押金金额
            BigDecimal returnAmt = param.getReturnAmt();
            if (returnAmt.compareTo(salReceiptRelateDO.getOpenAmt()) >= 1) {
                throw new BusinessException("退押金金额大于押金余额");
            }
            SalReceiptConvert salReceiptConvert = SalReceiptConvert.INSTANCE;
            SalReceiptDO salReceiptDO = salReceiptConvert.detailReturnRespVOToDo(param);
            if (!StrUtil.isNotBlank(param.getDocNo())) {
                String salQuoDocNo = rmiSysNextNumberService.generateCode("yst-sale","PD", new ArrayList<>());
                salReceiptDO.setDocNo(salQuoDocNo);
            }
            salReceiptDO.setDocType(DOC_TYPE_02);
            salReceiptDO.setAmt(returnAmt);
            salReceiptDO.setRelateDocNo(param.getRelateDocNo());
            salReceiptDO.setDocStatus(UdcEnum.SAL_QUOTATION_STATUS_DRAFT.getValueCode());
            //添加销售组织权限管理字段
            salReceiptDO.setSecBuId(param.getBuId());
            return salReceiptRepo.save(salReceiptDO).getId();
        } else {
            throw new BusinessException("仅类型为支付押金的可申请，且状态必须为已审批");
        }
    }

    @Override
    @SysCodeProc
    public PagingVO<SalPayDepositReturnRespVO> findRefundByPayDepositDocNo(String docNo) {
        JPAQuery<SalPayDepositReturnRespVO> jpaQuery = this.select3(docNo);
        long total = jpaQuery.fetchCount();
        if (total < 1) {
            return PagingVO.<SalPayDepositReturnRespVO>builder().build();
        }
        List<SalPayDepositReturnRespVO> result = jpaQuery.fetch();
        return PagingVO.<SalPayDepositReturnRespVO>builder()
                .total(total)
                .records(result).build();
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public Long refundSalQuotationRefuse(SalPayDepositApproveParamVO param) {
        Optional<SalReceiptDO> result = salReceiptRepo.findById(param.getId());
        if (result.isEmpty()) {
            throw new BusinessException("没有找到此单据");
        }
        if (!UdcEnum.SAL_QUOTATION_STATUS_APPROVING.getValueCode().equals(result.get().getDocStatus())) {
            throw new BusinessException("单据状态必须为待审批的才能审批");
        }
        //获得原押金单
        SalReceiptDO salReceiptDO = salReceiptRepo.findByDocNo(result.get().getRelateDocNo());
        //获得原押金单的原来退款中金额
        BigDecimal returningAmt = salReceiptDO.getReturningAmt();
        //获得新的退款中金额得到批准后的退款中金额
        returningAmt = returningAmt.subtract(result.get().getReturnAmt());
        //获得原押金单押金余额
        BigDecimal openAmt = salReceiptDO.getOpenAmt().add(result.get().getReturnAmt());
        //更新原单的信息并获得影响总数
        int total = salReceiptRepo.updateReturningAmtByIdAndVersion(returningAmt,
                salReceiptDO.getId(),
                salReceiptDO.getAuditDataVersion(),
                salReceiptDO.getAuditDataVersion() + 1,
                openAmt
        );

        if (total < 1) {
            throw new BusinessException("请勿同时操作");
        }
        salReceiptRepo.approveSalPayDeposit(
                param.getId(),
                UdcEnum.SAL_QUOTATION_STATUS_REJECTED.getValueCode(),
                param.getApprComment(),
                LocalDateTime.now(),
                param.getApprUserId(),
                param.getApprUserName()
        );
        return  param.getId();

    }


    @Override
    @Transactional(rollbackFor = Exception.class)
    public Long refundSalQuotationApprove(SalPayDepositApproveParamVO param) {
        Optional<SalReceiptDO> result = salReceiptRepo.findById(param.getId());
        if (result.isEmpty()) {
            throw new BusinessException("没有找到此单据");
        }
        if (!UdcEnum.SAL_QUOTATION_STATUS_APPROVING.getValueCode().equals(result.get().getDocStatus())) {
            throw new BusinessException("单据状态必须为待审批的才能审批");
        }
        //获得原押金单
        SalReceiptDO salReceiptDO = salReceiptRepo.findByDocNo(result.get().getRelateDocNo());
        //获得原版本号
        Integer auditDataVersion = salReceiptDO.getAuditDataVersion();
        //获得原押金单的押金余额
        BigDecimal openAmt = salReceiptDO.getOpenAmt();

        //更新退款中金额
        BigDecimal returningAmt = salReceiptDO.getReturningAmt();
        returningAmt = returningAmt.subtract(result.get().getReturnAmt()) ;
        //更新原单的信息并获得影响总数
        int total = salReceiptRepo.updateOpenAmtByIdAndVersion(returningAmt,
                salReceiptDO.getId(),
                auditDataVersion,
                auditDataVersion + 1);
        if (total < 1) {
            throw new BusinessException("请勿同时操作");
        }
        salReceiptRepo.approveSalPayDeposit(
                param.getId(),
                UdcEnum.SAL_QUOTATION_STATUS_APPROVED.getValueCode(),
                param.getApprComment(),
                LocalDateTime.now(),
                param.getApprUserId(),
                param.getApprUserName()
        );
        return param.getId();
    }

    @Override
    @SysCodeProc
    public SalPayDepositReturnRespVO refundSalQuotationQueryById(Long id) {
        JPAQuery<SalPayDepositReturnRespVO> jpaQuery = this.select4(id);
        List<SalPayDepositReturnRespVO> result = jpaQuery.fetch();
        return result.get(0);
    }

    //查询条件============================================================================================================

    public JPAQuery<SalPayDepositRespVO> select(SalPayDepositParamVO searchParam) {

        val jpaQuery = jpaQueryFactory.select(Projections.bean(SalPayDepositRespVO.class,
                qSalReceiptDO.id,
                qSalReceiptDO.docNo,
                qSalReceiptDO.custId,
                qSalReceiptDO.custName,
                qSalReceiptDO.custCode,
                qSalReceiptDO.contractCode,
                qSalReceiptDO.contractName,
                qSalReceiptDO.amt,
                qSalReceiptDO.applyDate,
                qSalReceiptDO.demandPaymentDate,
                qSalReceiptDO.planReturnDate,
                qSalReceiptDO.returnDate,
                qSalReceiptDO.bankId,
                qSalReceiptDO.bankCode,
                qSalReceiptDO.bankName,
                qSalReceiptDO.bankAcc,
                qSalReceiptDO.custContactName,
                qSalReceiptDO.custContactTel,
                qSalReceiptDO.applyEmpId,
                qSalReceiptDO.applyName,
                qSalReceiptDO.docStatus,
                qSalReceiptDO.apprUserId,
                qSalReceiptDO.apprUserName,
                qSalReceiptDO.remark,
                qSalReceiptDO.apprComment,
                qSalReceiptDO.soNo,
                qSalReceiptDO.depositDescription,
                qSalReceiptDO.ouName,
                qSalReceiptDO.payMethod,
                qSalReceiptDO.relateDocNo,
                qSalReceiptDO.docType,
                qSalReceiptDO.saleRegion,
                qSalReceiptDO.buName,
                qSalReceiptDO.buId
        )).from(qSalReceiptDO);
        if (searchParam != null) {
            jpaQuery.where(where(searchParam));
        }
        return jpaQuery;
    }

    public JPAQuery<SalPayDepositRespVO> select2(Long id) {
        val jpaQuery = jpaQueryFactory.select(Projections.bean(SalPayDepositRespVO.class,
                qSalReceiptDO.id,
                qSalReceiptDO.docNo,
                qSalReceiptDO.custId,
                qSalReceiptDO.custName,
                qSalReceiptDO.custCode,
                qSalReceiptDO.contractCode,
                qSalReceiptDO.contractName,
                qSalReceiptDO.amt,
                qSalReceiptDO.applyDate,
                qSalReceiptDO.demandPaymentDate,
                qSalReceiptDO.planReturnDate,
                qSalReceiptDO.returnDate,
                qSalReceiptDO.bankId,
                qSalReceiptDO.bankCode,
                qSalReceiptDO.bankName,
                qSalReceiptDO.bankAcc,
                qSalReceiptDO.custContactName,
                qSalReceiptDO.custContactTel,
                qSalReceiptDO.applyEmpId,
                qSalReceiptDO.applyName,
                qSalReceiptDO.docStatus,
                qSalReceiptDO.apprUserId,
                qSalReceiptDO.apprUserName,
                qSalReceiptDO.remark,
                qSalReceiptDO.apprComment,
                qSalReceiptDO.soNo,
                qSalReceiptDO.depositDescription,
                qSalReceiptDO.ouName,
                qSalReceiptDO.ouId,
                qSalReceiptDO.payMethod,
                qSalReceiptDO.relateDocNo,
                qSalReceiptDO.docType,
                qSalReceiptDO.saleRegion,
                qSalReceiptDO.buName,
                qSalReceiptDO.openAmt,
                qSalReceiptDO.buId

        )).from(qSalReceiptDO);
        if (id != null) {
            jpaQuery.where(qSalReceiptDO.id.eq(id));
        }
        return jpaQuery;
    }

    public JPAQuery<SalPayDepositRespVO> select5(String  contractCode) {

        val jpaQuery = jpaQueryFactory.select(Projections.bean(SalPayDepositRespVO.class,
                qSalReceiptDO.id
        )).from(qSalReceiptDO);
        if (!StringUtils.isEmpty(contractCode)) {
            jpaQuery.where(qSalReceiptDO.contractCode
                    .eq(contractCode)
                    .and(qSalReceiptDO.docStatus.in("APPROVING","APPROVED"))
            );
        }
        return jpaQuery;
    }


    public JPAQuery<SalPayDepositReturnRespVO> select3(String docNo) {

        val jpaQuery = jpaQueryFactory.select(Projections.bean(SalPayDepositReturnRespVO.class,
                qSalReceiptDO.id,
                qSalReceiptDO.docNo,
                qSalReceiptDO.custId,
                qSalReceiptDO.custName,
                qSalReceiptDO.custCode,
                qSalReceiptDO.soNo,
                qSalReceiptDO.contractCode,
                qSalReceiptDO.contractName,
                qSalReceiptDO.applyDate,
                qSalReceiptDO.demandPaymentDate,
                qSalReceiptDO.returnDate,
                qSalReceiptDO.bankId,
                qSalReceiptDO.bankCode,
                qSalReceiptDO.bankName,
                qSalReceiptDO.bankAcc,
                qSalReceiptDO.custContactName,
                qSalReceiptDO.custContactTel,
                qSalReceiptDO.applyEmpId,
                qSalReceiptDO.applyName,
                qSalReceiptDO.docStatus,
                qSalReceiptDO.apprUserId,
                qSalReceiptDO.apprUserName,
                qSalReceiptDO.depositDescription,
                qSalReceiptDO.ouName,
                qSalReceiptDO.remark,
                qSalReceiptDO.apprComment,
                qSalReceiptDO.payMethod,
                qSalReceiptDO.relateDocNo,
                qSalReceiptDO.docType,
                qSalReceiptDO.saleRegion,
                qSalReceiptDO.returnAmt,
                qSalReceiptDO.openAmt,
                qSalReceiptDO.buName,
                qSalReceiptDO.buId

        )).from(qSalReceiptDO);
        if (!StringUtils.isEmpty(docNo)) {
            jpaQuery.where(qSalReceiptDO.relateDocNo.eq(docNo).and(qSalReceiptDO.docType.eq(DOC_TYPE_02)));
        }
        return jpaQuery;
    }

    public JPAQuery<SalPayDepositReturnRespVO> select4(Long  id) {

        val jpaQuery = jpaQueryFactory.select(Projections.bean(SalPayDepositReturnRespVO.class,
                qSalReceiptDO.id,
                qSalReceiptDO.docNo,
                qSalReceiptDO.docType,
                qSalReceiptDO.custId,
                qSalReceiptDO.custName,
                qSalReceiptDO.custCode,
                qSalReceiptDO.soNo,
                qSalReceiptDO.contractCode,
                qSalReceiptDO.contractName,
                qSalReceiptDO.amt,
                qSalReceiptDO.applyDate,
                qSalReceiptDO.demandPaymentDate,
                qSalReceiptDO.planReturnDate,
                qSalReceiptDO.returnDate,
                qSalReceiptDO.bankId,
                qSalReceiptDO.bankCode,
                qSalReceiptDO.bankAcc,
                qSalReceiptDO.bankName,
                qSalReceiptDO.custContactName,
                qSalReceiptDO.custContactTel,
                qSalReceiptDO.applyEmpId,
                qSalReceiptDO.applyName,
                qSalReceiptDO.docStatus,
                qSalReceiptDO.apprUserId,
                qSalReceiptDO.apprUserName,
                qSalReceiptDO.depositDescription,
                qSalReceiptDO.remark,
                qSalReceiptDO.apprComment,
                qSalReceiptDO.ouName,
                qSalReceiptDO.ouId,
                qSalReceiptDO.payMethod,
                qSalReceiptDO.relateDocNo,
                qSalReceiptDO.saleRegion,
                qSalReceiptDO.returnAmt,
                qSalReceiptDO.openAmt,
                qSalReceiptDO.buName,
                qSalReceiptDO.buId

        )).from(qSalReceiptDO);
        if (!StringUtils.isEmpty(id)) {
            jpaQuery.where(qSalReceiptDO.id.eq(id).and(qSalReceiptDO.docType.eq(DOC_TYPE_02)));
        }
        return jpaQuery;
    }





    /**
     * 条件查询
     *
     * @param searchParam 查询条件
     */
    public Predicate where(SalPayDepositParamVO searchParam) {


        Predicate predicate = qSalReceiptDO.isNotNull();

        //押金单单号
        if (!StringUtils.isEmpty(searchParam.getDocNo())) {
            predicate = ExpressionUtils.and(predicate, qSalReceiptDO.docNo.like("%"+searchParam.getDocNo()+"%"));
        }
        //客户编码模糊查询
        if (!StringUtils.isEmpty(searchParam.getCustCode())) {
            predicate = ExpressionUtils.and(predicate, qSalReceiptDO.custCode.like("%"+searchParam.getCustCode()+"%"));
        }

        //客户名称模糊查询
        if (!StringUtils.isEmpty(searchParam.getCustName())) {
            predicate = ExpressionUtils.and(predicate, qSalReceiptDO.custName.like("%"+searchParam.getCustName()+"%"));
        }
        //申请人ID
        if (!StringUtils.isEmpty(searchParam.getApplyEmpId())) {
            predicate = ExpressionUtils.and(predicate, qSalReceiptDO.applyEmpId.eq(searchParam.getApplyEmpId()));
        }
        //申请日期
        if (!StringUtils.isEmpty(searchParam.getApplyDateBegin())) {
            predicate = ExpressionUtils.and(predicate, qSalReceiptDO.applyDate.eq(searchParam.getApplyDateBegin()).or(qSalReceiptDO.applyDate.after(searchParam.getApplyDateBegin())));
        }
        if (!StringUtils.isEmpty(searchParam.getApplyDateEnd())) {
            predicate = ExpressionUtils.and(predicate, qSalReceiptDO.applyDate.eq(searchParam.getApplyDateEnd()).or(qSalReceiptDO.applyDate.before(searchParam.getApplyDateEnd())));
        }
        //合同号
        if (!StringUtils.isEmpty(searchParam.getContractNo())) {
            predicate = ExpressionUtils.and(predicate, qSalReceiptDO.contractCode.like("%"+searchParam.getContractNo()+"%"));
        }
        //合同名称
        if (!StringUtils.isEmpty(searchParam.getContractName())) {
            predicate = ExpressionUtils.and(predicate, qSalReceiptDO.contractName.like("%"+searchParam.getContractName()+"%"));
        }

        //销售公司名称
        if (!StringUtils.isEmpty(searchParam.getOuId())) {
            predicate = ExpressionUtils.and(predicate, qSalReceiptDO.ouId.eq(searchParam.getOuId()));
        }

        //根据单据类型
        if (!StringUtils.isEmpty(searchParam.getDocType())) {
            predicate = ExpressionUtils.and(predicate, qSalReceiptDO.docType.eq(searchParam.getDocType()));
        }else {
            predicate = ExpressionUtils.and(predicate, qSalReceiptDO.docType.in(DOC_TYPE_01,DOC_TYPE_02));
        }
        //"查询范围 QueryRange --全部 all  已退押金 depositRefunded 未退押金 unreturnedDeposit"
        if (!StringUtils.isEmpty(searchParam.getQueryRange()) && QUERYRANGE_DEPOSIT_REFUNDED.equals(searchParam.getQueryRange())) {
            //既然根据退款状态查询了那么一定是支付押金单  即doctype=10
            if (StringUtils.isEmpty(searchParam.getDocType())) {
                predicate = ExpressionUtils.and(predicate, qSalReceiptDO.docType.eq(DOC_TYPE_01));
            }
            //openAmt押金余额为0为已退押金
            predicate = ExpressionUtils.and(predicate, qSalReceiptDO.openAmt.eq(new BigDecimal("0.0000")));
        }
        if (!StringUtils.isEmpty(searchParam.getQueryRange()) && QUERYRANGE_UN_DEPOSIT.equals(searchParam.getQueryRange())) {
            //既然根据退款状态查询了那么一定是支付押金单  即doctype=10
            if (StringUtils.isEmpty(searchParam.getDocType())) {
                predicate = ExpressionUtils.and(predicate, qSalReceiptDO.docType.eq(DOC_TYPE_01));
            }
            //openAmt押金余额不为0为未退押金
            predicate = ExpressionUtils.and(predicate, qSalReceiptDO.openAmt.ne(new BigDecimal("0.0000")));
        }

        //添加权限处理
        predicate = ExpressionUtils.and(predicate,DataAuthJpaUtil.dataAuthJpaPredicate(qSalReceiptDO.getMetadata()));

        return predicate;
    }
}
