package com.elitesland.yst.production.sale.event.listener;

import cn.hutool.core.util.BooleanUtil;
import com.elitesland.yst.production.sale.api.service.shop.BipItemMngService;
import com.elitesland.yst.production.sale.event.ItemStockUpdateEvent;
import lombok.extern.slf4j.Slf4j;
import org.springframework.context.ApplicationListener;
import org.springframework.lang.NonNull;
import org.springframework.stereotype.Component;

import java.util.concurrent.CompletableFuture;

/**
 * 商品自动下架监听器.
 * <p>
 * 监听库存变更，当商品下的所有SKU的库存都不足时，将商品自动下架
 *
 * @author Kaiser（wang shao）
 * @date 2021/09/01
 */
@Component
@Slf4j
public class AutoOffShelfListener implements ApplicationListener<ItemStockUpdateEvent> {

    private final BipItemMngService itemMngService;

    public AutoOffShelfListener(BipItemMngService itemMngService) {
        this.itemMngService = itemMngService;
    }

    @Override
    public void onApplicationEvent(@NonNull ItemStockUpdateEvent event) {
        if (BooleanUtil.isTrue(event.getIncrement())) {
            // 新增库存时不考虑
            return;
        }
        log.debug("监听到sku【{}】变更，检查库存是否充足来对商品自动下架", event.getBipItemSkuId());
        CompletableFuture.runAsync(() -> checkStock(event.getBipItemSkuId()))
                .exceptionally(e -> {
                    log.error("检查库存是否充足来对商品自动下架发生异常", e);
                    return null;
                });
    }

    private void checkStock(Long skuId) {
        var itemId = itemMngService.getIdBySkuId(skuId);
        itemMngService.executeCheckStock(itemId.getData());
    }
}
