package com.elitesland.yst.production.sale.repo.shop;

import cn.hutool.core.util.ObjectUtil;
import com.elitesland.yst.production.sale.entity.BipItemDO;
import com.elitesland.yst.production.sale.entity.BipItemSkuDO;
import com.elitesland.yst.production.sale.entity.QBipItemDO;
import com.elitesland.yst.production.sale.entity.QBipItemSkuDO;
import com.querydsl.jpa.JPAExpressions;
import com.querydsl.jpa.impl.JPAQueryFactory;
import org.springframework.stereotype.Repository;

import java.math.BigDecimal;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

/**
 * .
 *
 * @author Kaiser（wang shao）
 * @date 2021/08/19
 */
@Repository
public class BipItemSkuRepoProc {

    private final JPAQueryFactory jpaQueryFactory;

    private static final QBipItemSkuDO DO = QBipItemSkuDO.bipItemSkuDO;

    public BipItemSkuRepoProc(JPAQueryFactory jpaQueryFactory) {
        this.jpaQueryFactory = jpaQueryFactory;
    }

    /**
     * 修改最小起订量
     *
     * @param bipItemIds 商品ID
     * @param limitBuy   最小起订量
     */
    public void updateLimitBuy(List<Long> bipItemIds, Integer limitBuy) {
        jpaQueryFactory.update(DO)
                .set(DO.limitBuy, limitBuy)
                .where(DO.bipItemId.in(bipItemIds))
                .execute();
    }

    /**
     * 更新库存
     *
     * @param id    SKU ID
     * @param stock 库存
     */
    public void updateStock(Long id, Integer stock) {
        jpaQueryFactory.update(DO)
                .set(DO.stock, stock)
                .where(DO.id.eq(id))
                .execute();
    }

    /**
     * 更新价格
     *
     * @param id       SKU ID
     * @param price    现价
     * @param priceOld 之前的价格
     */
    public void updatePrice(Long id, BigDecimal price, BigDecimal priceOld) {
        jpaQueryFactory.update(DO)
                .set(DO.price, price)
                .set(DO.priceOld, priceOld)
                .where(DO.id.eq(id))
                .execute();
    }

    /**
     * 更新库存
     *
     * @param id        SKU ID
     * @param increment 库存增量
     */
    public void incrementStock(Long id, Integer increment) {
        jpaQueryFactory.update(DO)
                .set(DO.stock, DO.stock.add(increment))
                .where(DO.id.eq(id))
                .execute();
    }

    /**
     * 删除商品
     *
     * @param bipItemId 商品ID
     */
    public void deleteByBipItemId(Long bipItemId) {
        jpaQueryFactory.delete(DO)
                .where(DO.bipItemId.eq(bipItemId))
                .execute();
    }

    /**
     * 删除商品
     *
     * @param bipItemIds 商品ID
     */
    public void deleteByBipItemId(List<Long> bipItemIds) {
        jpaQueryFactory.delete(DO)
                .where(DO.bipItemId.in(bipItemIds))
                .execute();
    }

    /**
     * 获取商品ID
     *
     * @param id ID
     * @return 商品ID
     */
    public Long getBipItemId(Long id) {
        return jpaQueryFactory.select(DO.bipItemId)
                .from(DO)
                .where(DO.id.eq(id))
                .limit(1)
                .fetchOne();
    }

    /**
     * 获取商品sku 编码
     *
     * @param bipItemId 商品ID
     * @return SKU code 与 id
     */
    public Map<String, Long> getSkuCode(Long bipItemId) {
        return jpaQueryFactory.select(DO.id, DO.skuCode)
                .from(DO)
                .where(DO.bipItemId.eq(bipItemId))
                .fetch()
                .stream()
                .collect(Collectors.toMap(t -> t.get(DO.skuCode), t -> t.get(DO.id), (t1, t2) -> t1));
    }

    /**
     * 获取商品库存
     *
     * @param id SKU id
     * @return 库存量
     */
    public Integer getStock(Long id) {
        return jpaQueryFactory.select(DO.stock)
                .from(DO)
                .where(DO.id.eq(id))
                .limit(1)
                .fetchOne();
    }

    /**
     * 获取商品库存
     *
     * @param ids SKU id
     * @return skuCodes
     */
    public List<String> getskuCodes(List<Long> ids) {

        return jpaQueryFactory.select(DO.skuCode)
                .from(DO)
                .where(DO.id.in(ids))
                .fetch();
    }

    /**
     * 获取商品库存
     *
     * @param ids SKU id
     * @return SKU id 与 库存量
     */
    public Map<Long, Integer> getStock(List<Long> ids) {
        return jpaQueryFactory.select(DO.id, DO.stock)
                .from(DO)
                .where(DO.id.in(ids))
                .fetch()
                .stream()
                .collect(Collectors.toMap(t -> t.get(DO.id), t -> t.get(DO.stock), (t1, t2) -> t1));
    }

    public List<Long> getSpuIds(List<Long> ids) {
        return jpaQueryFactory.select(DO.bipItemId)
                .from(DO)
                .where(DO.id.in(ids))
                .fetch();
    }

    /**
     * 获取商品库存
     *
     * @param bipItemId 商品ID
     * @return SKU id 与 库存量
     */
    public Map<Long, Integer> getStockByBipItemId(Long bipItemId) {
        return jpaQueryFactory.select(DO.id, DO.stock)
                .from(DO)
                .where(DO.bipItemId.eq(bipItemId))
                .fetch()
                .stream()
                .collect(Collectors.toMap(t -> t.get(DO.id), t -> t.get(DO.stock), (t1, t2) -> t1));
    }

    /**
     * 判断是否存在库存不足的
     *
     * @param bipItemId 商品ID
     * @return 是否存在
     */
    public boolean existsLackStockByBipItemId(Long bipItemId) {
        return jpaQueryFactory.select(DO.id)
                .from(DO)
                .where(DO.bipItemId.eq(bipItemId).and(DO.stock.loe(0)))
                .limit(1)
                .fetchOne() != null;
    }

    /**
     * 判断是否存在库存不足的
     *
     * @param id SKU ID
     * @return 是否存在
     */
    public boolean existsLackStock(Long id) {
        return jpaQueryFactory.select(DO.id)
                .from(DO)
                .where(DO.bipItemId.in(JPAExpressions.select(DO.bipItemId).from(DO).where(DO.id.eq(id)).fetchOne())
                        .and(DO.stock.loe(0)))
                .limit(1)
                .fetchOne() != null;
    }

    /**
     * 根据商品ID查询SKU信息
     *
     * @param bipItemId 商品ID
     * @return SKU信息
     */
    public List<BipItemSkuDO> queryByBipItemId(Long bipItemId) {
        return jpaQueryFactory.select(DO)
                .from(DO)
                .where(DO.bipItemId.eq(bipItemId))
                .fetch();
    }

    public List<BipItemSkuDO> queryByBipItemIdFreeze(Long bipItemId) {
        return jpaQueryFactory.select(DO)
                .from(DO)
                .where(DO.bipItemId.eq(bipItemId).and(DO.freeze.eq(false)))
                .fetch();
    }

    /**
     * 根据商品ID查询SKU信息
     *
     * @param bipItemIds 商品ID
     * @return SKU信息
     */
    public List<BipItemSkuDO> queryByBipItemId(List<Long> bipItemIds) {
        return jpaQueryFactory.select(DO)
                .from(DO)
                .where(DO.bipItemId.in(bipItemIds))
                .fetch();
    }

    /**
     * 获取商品SKU的价格
     *
     * @param ids SKU的ID
     * @return SKU的ID与价格
     */
    public Map<Long, BigDecimal> getPrice(List<Long> ids) {
        return jpaQueryFactory.select(DO.id, DO.price)
                .from(DO)
                .where(DO.id.in(ids))
                .fetch()
                .stream()
                .collect(Collectors.toMap(tuple -> tuple.get(DO.id), tuple -> ObjectUtil.defaultIfNull(tuple.get(DO.price), BigDecimal.ZERO), (t1, t2) -> t1));
    }

    public List<BipItemSkuDO> findSkuByCode(String skuCode){
        return jpaQueryFactory.select(DO)
                .from(DO)
                .where(DO.skuCode.eq(skuCode))
                .fetch();
    }

    public Boolean findFreezeStatus(Long skuId){
        Boolean freeze = jpaQueryFactory.select(DO.freeze)
                .from(DO)
                .where(DO.id.eq(skuId))
                .limit(1)
                .fetchOne();

        return freeze == null || freeze;
    }

    public Map<Long, List<BipItemSkuDO>> findAllByItemIDS(List<Long> bipItemIds){

        return jpaQueryFactory.select(DO)
                .from(DO)
                .where(DO.bipItemId.in(bipItemIds))
                .fetch()
                .stream()
                .collect(Collectors.groupingBy(BipItemSkuDO::getBipItemId));
    }

    public BipItemSkuDO findSkuOnOuId(Long ouId, String skuCode){
        QBipItemDO qBipItemDO = QBipItemDO.bipItemDO;
        QBipItemSkuDO qBipItemSkuDO = QBipItemSkuDO.bipItemSkuDO;
        return jpaQueryFactory.select(qBipItemSkuDO)
                .from(qBipItemDO)
                .leftJoin(qBipItemSkuDO).on(qBipItemDO.id.eq(qBipItemSkuDO.bipItemId))
                .where(qBipItemDO.ouId.eq(ouId).and(qBipItemSkuDO.skuCode.eq(skuCode)))
                .limit(1)
                .fetchOne();
    }

    public List<BipItemSkuDO> findSkuOnMenu(Long ouId, String spuCode){
        QBipItemDO qBipItemDO = QBipItemDO.bipItemDO;
        QBipItemSkuDO qBipItemSkuDO = QBipItemSkuDO.bipItemSkuDO;
        return jpaQueryFactory.select(qBipItemSkuDO)
                .from(qBipItemDO)
                .leftJoin(qBipItemSkuDO).on(qBipItemDO.id.eq(qBipItemSkuDO.bipItemId))
                .where(qBipItemDO.ouId.eq(ouId).and(qBipItemDO.itemCode.eq(spuCode)).and(qBipItemSkuDO.freeze.eq(false)))
                .fetch();
    }

    public List<BipItemDO> findSkuOnRotation(Long ouId, List<String> skuCodes){
        QBipItemDO qBipItemDO = QBipItemDO.bipItemDO;
        QBipItemSkuDO qBipItemSkuDO = QBipItemSkuDO.bipItemSkuDO;
        return jpaQueryFactory.select(qBipItemDO)
                .from(qBipItemDO)
                .leftJoin(qBipItemSkuDO).on(qBipItemDO.id.eq(qBipItemSkuDO.bipItemId))
                .where(qBipItemDO.ouId.eq(ouId).and(qBipItemSkuDO.skuCode.in(skuCodes)).and(qBipItemSkuDO.freeze.eq(false)))
                .fetch();
    }

}

