package com.elitesland.yst.production.sale.service.shop;

import cn.hutool.core.lang.Assert;
import com.elitesland.yst.production.sale.api.service.shop.BipFavoritesService;
import com.elitesland.yst.production.sale.api.service.shop.BipItemAppService;
import com.elitesland.yst.production.sale.api.vo.param.shop.BipFavoritesQueryParamVO;
import com.elitesland.yst.production.sale.api.vo.param.shop.BipFavoritesSaveParamVO;
import com.elitesland.yst.production.sale.api.vo.resp.shop.BipFavoritesRespVO;
import com.elitesland.yst.production.sale.api.vo.resp.shop.app.BipItemDetailAppRespVO;
import com.elitesland.yst.production.sale.api.vo.resp.shop.app.BipItemPicAppRespVO;
import com.elitesland.yst.production.sale.common.model.CurrentUserDTO;
import com.elitesland.yst.production.sale.core.service.BaseServiceImpl;
import com.elitesland.yst.production.sale.core.service.UserService;
import com.elitesland.yst.production.sale.entity.BipCustUserBindDO;
import com.elitesland.yst.production.sale.entity.BipFavoritesDo;
import com.elitesland.yst.production.sale.entity.QBipFavoritesDo;
import com.elitesland.yst.production.sale.repo.shop.BipCustUserBindRepoProc;
import com.elitesland.yst.production.sale.repo.shop.BipFavoritesRepo;
import com.elitesland.yst.production.sale.repo.shop.BipFavoritesRepoProc;
import com.elitescloud.cloudt.common.base.ApiCode;
import com.elitescloud.cloudt.common.base.ApiResult;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitescloud.boot.exception.BusinessException;
import com.querydsl.core.types.ExpressionUtils;
import com.querydsl.core.types.Predicate;
import com.querydsl.core.types.Projections;
import com.querydsl.jpa.impl.JPAQuery;
import lombok.RequiredArgsConstructor;
import lombok.val;
import org.springframework.data.domain.PageRequest;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;
import org.springframework.util.ObjectUtils;

import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.List;

/**
 * <p>
 *
 * </p>
 *
 * @author zhao.zhi.hao
 * @since 2021/9/30 09:37
 */
@Service
@RequiredArgsConstructor
public class BipFavoritesServiceImpl extends BaseServiceImpl implements BipFavoritesService {

    private QBipFavoritesDo qBipFavoritesDo = QBipFavoritesDo.bipFavoritesDo;

    private final BipFavoritesRepo bipFavoritesRepo;

    private final BipItemAppService bipItemAppService;

    private final BipFavoritesRepoProc bipFavoritesRepoProc;

    private final BipCustUserBindRepoProc bipCustUserBindRepoProc;

    @Override
    public PagingVO<BipFavoritesRespVO> search(BipFavoritesQueryParamVO searchParam) {

        //获得当前登录账号ID
        CurrentUserDTO currentUserDTO = UserService.currentUser();
        if (ObjectUtils.isEmpty(currentUserDTO)) {
            throw new BusinessException("无法获得当前登录人请检查");
        }
        searchParam.setCustAccountId(currentUserDTO.getUserId());

        var jpaQuery = this.select(searchParam);
        long total = jpaQuery.fetchCount();
        if (total == 0) {
            return PagingVO.<BipFavoritesRespVO>builder().build();
        }
        // 添加分页和排序
        PageRequest pageRequest = wrapperPageRequest(searchParam.getPageRequest(), null);
        appendPageAndSort(jpaQuery, pageRequest, qBipFavoritesDo);
        List<BipFavoritesRespVO> respVOS = jpaQuery.fetch();

        // 补充数据(商品数据)
        this.translatePage(respVOS);
        return PagingVO.<BipFavoritesRespVO>builder()
                .total(total)
                .records(respVOS).build();
    }

    private void translatePage(List<BipFavoritesRespVO> bipFavoritesRespVOS) {

        bipFavoritesRespVOS.forEach(bip -> {
            ApiResult<BipItemDetailAppRespVO> detail = bipItemAppService.getDetail(bip.getItemId(),getCurrentOuIdInfo());
            Assert.isFalse(!detail.isSuccess(), "未查到商品id为:" + bip.getItemId() + "的数据");
            BipItemDetailAppRespVO detailData = detail.getData();
            bip.setAmt(detailData.getPrice());
            bip.setItemTitle(detailData.getTitle());
            List<BipItemPicAppRespVO> picList = detailData.getPicList();
            if (!CollectionUtils.isEmpty(picList)) {
                bip.setFileCode(picList.get(0).getFileCode());
            }
        });
    }

    private Long getCurrentOuIdInfo() {
        CurrentUserDTO currentUser = UserService.currentUser();
        if (currentUser == null) {
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "当前用户已失效,请重登录重试");
        }
        BipCustUserBindDO bipCustUserBindDO = bipCustUserBindRepoProc.getByUserId(currentUser.getUserId());

        if (bipCustUserBindDO == null || bipCustUserBindDO.getOuId() ==null ) {
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "当前用户未绑定公司");
        }


        return bipCustUserBindDO.getOuId();
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void collectBipFavorites(BipFavoritesSaveParamVO bipFavoritesSaveParamVO) {
        //获得当前登录账号ID
        CurrentUserDTO currentUserDTO = UserService.currentUser();
        if (ObjectUtils.isEmpty(currentUserDTO)) {
            throw new BusinessException("无法收藏");
        }
        bipFavoritesSaveParamVO.setCustAccountId(currentUserDTO.getUserId());
        //数据为空返回
        if (ObjectUtils.isEmpty(bipFavoritesSaveParamVO)) {
            return;
        }
        if (bipFavoritesSaveParamVO.getIfCollectFlag().equals(1)) {
            this.saveBipFavorites(bipFavoritesSaveParamVO);
        } else {
            bipFavoritesRepo.deleteBipFavorites(bipFavoritesSaveParamVO.getCustAccountId(), bipFavoritesSaveParamVO.getItemIds());
        }
    }

    @Override
    public Boolean ifCollected(Long itemId) {

        CurrentUserDTO currentUserDTO = UserService.currentUser();
        if (ObjectUtils.isEmpty(currentUserDTO)) {
            return Boolean.FALSE;
        }
        return bipFavoritesRepoProc.exists(currentUserDTO.getUserId(), itemId);
    }

    @Transactional(rollbackFor = Exception.class)
    public void saveBipFavorites(BipFavoritesSaveParamVO bipFavoritesSaveParamVO) {
        List<BipFavoritesDo> bipFavoritesDos = new ArrayList<>();
        BipFavoritesDo bipFavoritesDo = new BipFavoritesDo();
        bipFavoritesDo.setCustAccountId(bipFavoritesSaveParamVO.getCustAccountId());
        if (CollectionUtils.isEmpty(bipFavoritesSaveParamVO.getItemIds())) {
            return;
        }
        bipFavoritesSaveParamVO.getItemIds().forEach(bip -> {
            bipFavoritesDo.setItemId(bip);
            bipFavoritesDo.setCollectTime(LocalDateTime.now());
            bipFavoritesDos.add(bipFavoritesDo);
        });
        bipFavoritesRepo.saveAll(bipFavoritesDos);
    }

    public JPAQuery<BipFavoritesRespVO> select(BipFavoritesQueryParamVO searchParam) {

        val jpaQuery = jpaQueryFactory.select(Projections.bean(BipFavoritesRespVO.class,
                qBipFavoritesDo.id,
                qBipFavoritesDo.itemCode,
                qBipFavoritesDo.itemId,
                qBipFavoritesDo.collectTime,
                qBipFavoritesDo.custAccountId,
                qBipFavoritesDo.num,
                qBipFavoritesDo.putAmt
        )).from(qBipFavoritesDo);
        if (searchParam != null) {
            jpaQuery.where(where(searchParam));
        }
        return jpaQuery;
    }

    /**
     * 条件查询
     *
     * @param searchParam 查询条件
     */
    public Predicate where(BipFavoritesQueryParamVO searchParam) {

        Predicate predicate = qBipFavoritesDo.isNotNull();
        if (!ObjectUtils.isEmpty(searchParam.getCustAccountId())) {
            predicate = ExpressionUtils.and(predicate, qBipFavoritesDo.custAccountId.eq(searchParam.getCustAccountId()));
        }
        return predicate;
    }
}
