package com.elitesland.yst.production.sale.service;

import cn.hutool.core.collection.CollUtil;
import cn.hutool.core.util.StrUtil;
import com.elitescloud.boot.auth.util.SecurityContextUtil;
import com.elitescloud.boot.exception.BusinessException;
import com.elitescloud.cloudt.common.annotation.SysCodeProc;
import com.elitescloud.cloudt.common.base.ApiCode;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitescloud.cloudt.security.entity.GeneralUserDetails;
import com.elitescloud.cloudt.system.dto.SysEmployeeBasicDTO;
import com.elitescloud.cloudt.system.dto.resp.SysAreaRespDTO;
import com.elitescloud.cloudt.system.vo.SysUserDTO;
import com.elitesland.yst.production.sale.api.service.*;
import com.elitesland.yst.production.sale.api.vo.param.taskinfo.FileInfoQueryVO;
import com.elitesland.yst.production.sale.api.vo.param.taskinfo.TaskInfoDtlQueryVO;
import com.elitesland.yst.production.sale.api.vo.param.taskinfo.TaskInfoQueryVO;
import com.elitesland.yst.production.sale.api.vo.resp.crm.LmSaveCustRespVO;
import com.elitesland.yst.production.sale.api.vo.resp.taskinfo.*;
import com.elitesland.yst.production.sale.api.vo.save.FileInfoSaveVO;
import com.elitesland.yst.production.sale.api.vo.save.TaskInfoDtlSaveVO;
import com.elitesland.yst.production.sale.api.vo.save.TaskInfoSaveVO;
import com.elitesland.yst.production.sale.common.constant.ConstantsSale;
import com.elitesland.yst.production.sale.common.constant.UdcEnum;
import com.elitesland.yst.production.sale.convert.FileinfoConvert;
import com.elitesland.yst.production.sale.convert.TaskInfoConvert;
import com.elitesland.yst.production.sale.convert.TaskInfoDtlConvert;
import com.elitesland.yst.production.sale.entity.FileInfoDO;
import com.elitesland.yst.production.sale.entity.SalesmanInfoDO;
import com.elitesland.yst.production.sale.entity.SalesmanRegionDO;
import com.elitesland.yst.production.sale.entity.TaskInfoDO;
import com.elitesland.yst.production.sale.repo.*;
import com.elitesland.yst.production.sale.rmi.ystsupport.RmiOrgOuService;
import com.elitesland.yst.production.sale.rmi.ystsupport.RmiOrgStoreRpcService;
import com.elitesland.yst.production.sale.rmi.ystsystem.RmiEmployeeRpcService;
import com.elitesland.yst.production.sale.rmi.ystsystem.RmiSysAreaRpcService;
import com.elitesland.yst.production.sale.rmi.ystsystem.RmiSysNextNumberService;
import com.elitesland.yst.production.sale.rmi.ystsystem.RmiUdcService;
import com.elitesland.yst.production.support.provider.org.dto.OrgOuRpcDTO;
import com.elitesland.yst.production.support.provider.org.dto.OrgStoreDetailRpcDTO;
import com.google.common.collect.Lists;
import lombok.extern.slf4j.Slf4j;
import lombok.val;
import org.apache.commons.collections4.MapUtils;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Lazy;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

import java.math.BigDecimal;
import java.time.DayOfWeek;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.LocalTime;
import java.time.temporal.ChronoUnit;
import java.time.temporal.TemporalAdjusters;
import java.util.*;
import java.util.function.BiFunction;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;

/**
 * <p>
 * 功能说明:业务员任务管理
 * </p>
 *
 * @Author Darren
 * @Date 2023/04/10
 * @Version 1.0
 * @Content:
 */
@Slf4j
@Service
public class TaskInfoServiceImpl implements TaskInfoService {

    @Autowired
    private TaskInfoRepo taskInfoRepo;
    @Autowired
    private TaskInfoRepoProc taskInfoRepoProc;
    @Autowired
    private TaskInfoDtlService taskInfoDtlService;

    @Autowired
    private RmiSysNextNumberService rmiSysNextNumberService;
    @Autowired
    private RmiOrgOuService rmiOrgOuService;
    @Autowired
    private RmiUdcService rmiUdcService;
    @Autowired
    private RmiEmployeeRpcService rmiEmployeeRpcService;
    @Autowired
    private RmiOrgStoreRpcService rmiOrgStoreRpcService;
    @Autowired
    private RmiSysAreaRpcService rmiSysAreaRpcService;
    @Autowired
    private CrmCustService crmCustService;

    @Autowired
    private FileInfoRepoProc fileInfoRepoProc;
    @Autowired
    private FileInfoRepo fileInfoRepo;

    @Autowired
    private SalesmanInfoRepo salesmanInfoRepo;
    @Autowired
    private SalesmanRegionRepo salesmanRegionRepo;

    private ExectRecordService exectRecordService;

    @Autowired
    @Lazy
    public void setExectRecordService(ExectRecordService exectRecordService) {
        this.exectRecordService = exectRecordService;
    }

    private ExectRecordTempService exectRecordTempService;

    @Autowired
    @Lazy
    public void setExectRecordTempService(ExectRecordTempService exectRecordTempService) {
        this.exectRecordTempService = exectRecordTempService;
    }

    /**
     * 执行模板和执行记录
     * 根据执行模板编码查询任务列表的接口,主要是要状态字段，我这边要根据状态判断模板的停用启用
     */

    /**
     *发布人存ID、执行人存CODE
     * 公司、执行模板、执行记录存CODE
     * 门店，业务员存编码、
     *
     * 发布人 员工列表 系统中的员工列表信息；
     *
     * 分配执行人 调用基础域的接口，查询启用状态的业务员列表，支持单选修改勾选表单行的执行人。
     * 业务员表没有名称，要通过业务员编码查询员工的名称
     * 选择关联业务编码 根据业务类型，调用基础域的接口，查询门店、经销商和业务员列表，均为启用状态，支持多选添加。
     *
     * 省市区存编码，前端后续需要在弹出框里增加省市区信息，保存时并返回省市区编码传给后端
     */

    /**
     * 业务员任务新增时保存/发布
     *
     * @param saveVO 入参
     * @return 任务出参对象
     */
    @Override
    @Transactional(rollbackFor = {Exception.class})
    public TaskInfoRespVO createTask(TaskInfoSaveVO saveVO) {
        List<TaskInfoDtlSaveVO> dtlSaveVOList = saveVO.getDtlSaveVOList();
        //校验必填项
        checkMandatoryField(saveVO);
        checkMandatoryFieldDtlList(dtlSaveVOList);
        //校验业务员编码的员工信息并赋值员工ID
        //checkAndSaveDtlEmployeeId(dtlSaveVOList);
        //校验任务重复设置
        checkRepeatSet(saveVO);

        //新增创建时默认值
        saveDefaultValue(true, saveVO, null);
        //保存任务主表
        Long id = saveTaskInfo(saveVO);
        saveVO.setId(id);

        //保存任务重复设置,并计算执行时间
        repeatSetSave(saveVO);

        //明细新增创建时默认值
        saveDtlDefaultValue(dtlSaveVOList, saveVO);
        //保存任务明细
        if(org.apache.commons.collections4.CollectionUtils.isNotEmpty(dtlSaveVOList)){
            taskInfoDtlService.saveTaskInfoDtl(dtlSaveVOList);
        }

        if(saveVO.getReleaseSign()){
            taskInfoDtlService.releaseDtl(saveVO.getId(), UdcEnum.SALESMAN_TASK_COMPLETION_STATUS_WEE.getValueCode());
        }
        //保存附件信息
        saveFile(saveVO.getFileInfoSaveVOS(), id);


        TaskInfoRespVO taskInfoRespVO = new TaskInfoRespVO();
        taskInfoRespVO.setId(id);
        return taskInfoRespVO;
    }

    /**
     * 业务员任务修改时保存/发布
     *
     * @param saveVO 入参
     * @return 任务出参对象
     */
    @Override
    @Transactional(rollbackFor = {Exception.class})
    public TaskInfoRespVO updateTask(TaskInfoSaveVO saveVO) {
        List<TaskInfoDtlSaveVO> dtlSaveVOList = saveVO.getDtlSaveVOList();
        //校验必填项
        checkMandatoryField(saveVO);
        checkMandatoryFieldDtlList(dtlSaveVOList);
        //校验业务员编码的员工信息并赋值员工ID
        //checkAndSaveDtlEmployeeId(dtlSaveVOList);
        //校验任务重复设置
        checkRepeatSet(saveVO);

        //业务校验逻辑
        TaskInfoRespVO taskInfoRespVO = findById(saveVO.getId());
        if (Objects.isNull(taskInfoRespVO)) {
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "未查询到该任务数据信息!");
        }
        checkUpdateBusiness(taskInfoRespVO);

        //修改时默认值
        saveDefaultValue(false, saveVO, taskInfoRespVO);
        //保存任务主表
        Long id = saveTaskInfo(saveVO);

        //保存任务重复设置,并计算执行时间
        repeatSetSave(saveVO);

        //明细全删全插,先删除
//        taskInfoDtlService.deleteByMasId(saveVO.getId());
        //明细新增创建时默认值
        saveDtlDefaultValue(dtlSaveVOList, saveVO);
        //保存任务明细
        if(org.apache.commons.collections4.CollectionUtils.isNotEmpty(dtlSaveVOList)){
            taskInfoDtlService.saveTaskInfoDtl(dtlSaveVOList);
        }

        if(saveVO.getReleaseSign()){
            taskInfoDtlService.releaseDtl(saveVO.getId(), UdcEnum.SALESMAN_TASK_COMPLETION_STATUS_WEE.getValueCode());
        }
        //附件表全删全查
        deleteFile(id);
        //保存附件信息
        saveFile(saveVO.getFileInfoSaveVOS(), id);


        TaskInfoRespVO respVO = new TaskInfoRespVO();
        respVO.setId(id);
        return respVO;
    }

    /**
     * 保存附件信息
     *
     * @param fileInfoSaveVOS 附件信息入参
     * @param id              主表ID
     * @return
     */
    public void saveFile(List<FileInfoSaveVO> fileInfoSaveVOS, Long id) {
        if (!CollectionUtils.isEmpty(fileInfoSaveVOS)) {
            List<FileInfoDO> collect = fileInfoSaveVOS.stream().map(t -> {
                FileInfoDO fileInfoDO = FileinfoConvert.INSTANCE.voToDO(t);
                fileInfoDO.setSourceId(id);
                return fileInfoDO;
            }).collect(Collectors.toList());
            fileInfoRepo.saveAll(collect);
        }
    }

    /**
     * 根据主表ID删除附件信息
     *
     * @param id 主表ID
     * @return
     */
    public void deleteFile(Long id) {
        if (Objects.nonNull(id)) {
            //附件表全删全查
            fileInfoRepoProc.sourceDel(Arrays.asList(id));
        }
    }

    /**
     * 保存任务主表
     *
     * @param saveVO 入参
     * @return 任务ID
     */
    @Transactional(rollbackFor = {Exception.class})
    public Long saveTaskInfo(TaskInfoSaveVO saveVO) {
        TaskInfoDO taskInfoDO = TaskInfoConvert.INSTANCE.saveVoToDo(saveVO);
        return taskInfoRepo.save(taskInfoDO).getId();
    }

    /**
     * 新增/修改保存时默认值
     *
     * @param flag   新增/修改标识 true 新增 false 修改
     * @param saveVO 入参
     * @param respVO 数据库旧任务数据
     * @return
     */
    private void saveDefaultValue(Boolean flag, TaskInfoSaveVO saveVO, TaskInfoRespVO respVO) {

        if (flag) {
            //新增时-发号器新增编码
            saveVO.setCode(getTaskCode());
        } else {
            //修改
            if (StringUtils.isBlank(saveVO.getCode())) {
                saveVO.setCode(respVO.getCode());
            }
        }

        if (saveVO.getReleaseSign()) {
            //发布时
            //任务状态:待执行
            saveVO.setState(UdcEnum.SALESMAN_TASK_STATUS_WEE.getValueCode());
            //执行模板
            if (Objects.isNull(saveVO.getExecutTemplateCode())) {
                ExectRecordTempRespVO exectRecordTempRespVO = getExecuteTemplate(saveVO.getType(), saveVO.getOuCode());
                saveVO.setExecutTemplateCode(exectRecordTempRespVO.getTempCode());
                saveVO.setExecutTemplateName(exectRecordTempRespVO.getTempName());
                saveVO.setExecutTemplateId(exectRecordTempRespVO.getId());
            }
        } else {
            //保存时
            //任务状态:待发布
            saveVO.setState(UdcEnum.SALESMAN_TASK_STATUS_WRE.getValueCode());
        }

        //是否逾期:未逾期
        saveVO.setDelayFlag(verifyDelayFlag(flag,saveVO));
        //任务进度:默认为0%
        saveVO.setProgress(BigDecimal.ZERO);
        //附件编码数据库里以逗号分隔的字符串形式保存
        if (!CollectionUtils.isEmpty(saveVO.getFileCodes())) {
            String fileInfo = transitionCodesToStr(saveVO.getFileCodes());
            saveVO.setFileInfo(fileInfo);
        }

        //所属公司:当前登录人的所属公司信息
        if (StringUtils.isBlank(saveVO.getOuCode())) {
            //当前用户信息,当等于空再去查询,因为DT里调用这个方法报错，
            // DT里复制任务重复设置时是已经有公司了，就不会走这个方法
            SysUserDTO sysUserDTO = getSysUser();
            saveVO.setOuCode(sysUserDTO.getOuCode());
        }
        //重复设置:默认为不重复
        if (StringUtils.isBlank(saveVO.getRepeatSet())) {
            saveVO.setRepeatSet(ConstantsSale.SALESMAN_TASK_REPEAT_SET_N);
        }
        //签到范围:默认为零,当为空或者0时，表示不限制允许签到的范围
        if (Objects.isNull(saveVO.getSignInRange())) {
            saveVO.setSignInRange(BigDecimal.ZERO);
        }
        //签退范围:默认为零,当为空或者0时，表示不限制允许签退的范围
        if (Objects.isNull(saveVO.getSignOutRange())) {
            saveVO.setSignOutRange(BigDecimal.ZERO);
        }


    }

    /**
     * 校验任务主表是否逾期
     *
     * @param flag   新增/修改标识 true 新增 false 修改
     * @param saveVO 入参
     *
     * @return
     */
    private String verifyDelayFlag(Boolean flag, TaskInfoSaveVO saveVO){
        String delayFlag = UdcEnum.SALESMAN_TASK_DELAY_FLAG_N.getValueCode();
        //任务主表状态为‘待发布’、‘待执行’、‘进行中’的
        List<String> stateList = Lists.newArrayList(
                UdcEnum.SALESMAN_TASK_STATUS_WRE.getValueCode(),
                UdcEnum.SALESMAN_TASK_STATUS_WEE.getValueCode(),
                UdcEnum.SALESMAN_TASK_STATUS_IPS.getValueCode()
        );
        if (flag) {
            //新增时
            if (stateList.contains(saveVO.getState()) &&
                    LocalDateTime.of(LocalDate.now(),ConstantsSale.LOCAL_TIME_MAX).isAfter(saveVO.getEndTime())){
                delayFlag = UdcEnum.SALESMAN_TASK_DELAY_FLAG_Y.getValueCode();
            }
        } else {
            //修改时
            if (stateList.contains(saveVO.getState()) &&
                    LocalDateTime.of(LocalDate.now(),ConstantsSale.LOCAL_TIME_MAX).isAfter(saveVO.getEndTime())){
                delayFlag = UdcEnum.SALESMAN_TASK_DELAY_FLAG_Y.getValueCode();
            }
        }

        return delayFlag;
    }

    /**
     * 校验明细的执行人(业务员编码)是否有对应的员工信息，
     * 并对明细的执行人默认赋值保存对应的员工ID,业务员编码和员工编码是一样的
     * <p>
     * 用于数据权限
     *
     * @param dtlSaveVOList 明细入参
     * @return
     */
    private void checkAndSaveDtlEmployeeId(List<TaskInfoDtlSaveVO> dtlSaveVOList) {
        Set<String> codes = dtlSaveVOList.stream().map(TaskInfoDtlSaveVO::getExecutUserCode).distinct().collect(Collectors.toSet());
        List<SysEmployeeBasicDTO> employeeList = rmiEmployeeRpcService.findEmployeeByCodes(codes);

        List<TaskInfoDtlSaveVO> errorDtlSaveVOList = new ArrayList<>();
        for (TaskInfoDtlSaveVO dtlSaveVO : dtlSaveVOList) {
            Optional<SysEmployeeBasicDTO> employeeOptional = employeeList.stream().filter(employeeDTO -> Objects.equals(employeeDTO.getCode(), dtlSaveVO.getExecutUserCode())).findFirst();
            if (employeeOptional.isPresent() && Objects.nonNull(employeeOptional.get().getId())) {
                Long employeeId = employeeOptional.get().getId();
                dtlSaveVO.setEmployeeId(employeeId);
            } else {
                errorDtlSaveVOList.add(dtlSaveVO);
            }
        }
        if (!CollectionUtils.isEmpty(errorDtlSaveVOList)) {
            String checkResult = errorDtlSaveVOList.stream().map(vo -> vo.getExecutUserCode()
            ).collect(Collectors.joining(",", "未查询到业务员的员工信息,业务员编码:[", "].请检查!"));
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, checkResult);
        }
    }

    /**
     * 明细新增/修改时默认值
     *
     * @param dtlSaveVOList 明细入参
     * @param saveVO        任务主表信息
     * @return
     */
    private void saveDtlDefaultValue(List<TaskInfoDtlSaveVO> dtlSaveVOList, TaskInfoSaveVO saveVO) {
        if(org.apache.commons.collections4.CollectionUtils.isNotEmpty(dtlSaveVOList)){
            dtlSaveVOList.forEach(dtlSaveVO -> {
                dtlSaveVO.setMasId(saveVO.getId());
                if (saveVO.getReleaseSign()) {
                    //发布时
                    //完成状态:待执行
                    dtlSaveVO.setCompleteState(UdcEnum.SALESMAN_TASK_COMPLETION_STATUS_WEE.getValueCode());
                } else {
                    //保存时
                    //完成状态:待发布
                    dtlSaveVO.setCompleteState(UdcEnum.SALESMAN_TASK_COMPLETION_STATUS_WRE.getValueCode());
                }

                //是否逾期:未逾期
                dtlSaveVO.setDelayFlag(verifyDtlDelayFlag(saveVO,dtlSaveVO));
                //明细的关联业务编码数据库里以逗号分隔的字符串形式保存
                //目前一行任务明细的业务编码只有一个,前端传businessCode，不使用businessCodes了。
            /*if (!CollectionUtils.isEmpty(dtlSaveVO.getBusinessCodes())) {
                String businessCodeStr = transitionCodesToStr(dtlSaveVO.getBusinessCodes());
                dtlSaveVO.setBusinessCode(businessCodeStr);
            }*/

            });
        }

    }

    /**
     * 校验任务明细是否逾期
     *
     * @param saveVO        任务主表信息
     * @param dtlSaveVO 明细入参
     * @return
     */
    private String verifyDtlDelayFlag(TaskInfoSaveVO saveVO, TaskInfoDtlSaveVO dtlSaveVO){
        String delayFlag = UdcEnum.SALESMAN_TASK_DELAY_FLAG_N.getValueCode();

        //主表未逾期，那么明细也未逾期，主表已逾期，明细可能逾期
        if (Objects.equals(saveVO.getDelayFlag(),UdcEnum.SALESMAN_TASK_DELAY_FLAG_Y.getValueCode())){
            //所有明细的完成状态不是‘已完成’的均将是否逾期字段标记为已逾期，其余明细仍保持‘未逾期’
            if (!Objects.equals(dtlSaveVO.getCompleteState(),UdcEnum.SALESMAN_TASK_COMPLETION_STATUS_CPD.getValueCode())){
                delayFlag = UdcEnum.SALESMAN_TASK_DELAY_FLAG_Y.getValueCode();
            }
        }

        return delayFlag;
    }

    /**
     * 明细数据转换处理
     *
     * @param saveVOList 入参
     * @return
     */
    private void transitionDtlToStr(List<TaskInfoDtlSaveVO> saveVOList) {
        saveVOList.forEach(saveVO -> {
            //明细的关联业务编码数据库里以逗号分隔的字符串形式保存
            /*if (!CollectionUtils.isEmpty(saveVO.getBusinessCodes())) {
                String businessCodeStr = transitionCodesToStr(saveVO.getBusinessCodes());
                saveVO.setBusinessCode(businessCodeStr);
            }*/

        });
    }

    /**
     * 根据任务类型+所属公司,查询启用状态的执行模版,存模版编码
     *
     * @param type   任务类型
     * @param ouCode 所属公司
     * @return 版编码
     */
    private ExectRecordTempRespVO getExecuteTemplate(String type, String ouCode) {
        if (StringUtils.isBlank(ouCode)) {
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "所属公司不可为空!");
        }
        if (StringUtils.isBlank(type)) {
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "任务类型不可为空!");
        }
        //调用执行模板的接口查询
        List<ExectRecordTempRespVO> recordTempList = exectRecordTempService.queryByCodeAndTask(type, ouCode);
        if (CollectionUtils.isEmpty(recordTempList)) {
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "根据任务类型和所属公司未查询到执行模板!");
        }
        ExectRecordTempRespVO recordTempRespVO = recordTempList.get(0);
        if (StringUtils.isBlank(recordTempRespVO.getTempCode())) {
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "执行模板不可为空!");
        }

        return recordTempRespVO;
    }

    /**
     * 根据发号器规则自动生成任务编码
     *
     * @param
     * @return 任务编码
     */
    private String getTaskCode() {
        String taskCode = rmiSysNextNumberService.generateCode(ConstantsSale.YST_SALE, ConstantsSale.SALESMAN_TASK_CODE, Collections.emptyList());
        if (StringUtils.isBlank(taskCode)) {
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "生成任务编码失败!");
        }
        return taskCode;
    }

    /**
     * 任务修改时的业务校验逻辑
     *
     * @param respVO 数据库旧数据信息
     * @return
     */
    private void checkUpdateBusiness(TaskInfoRespVO respVO) {
        //针对‘待发布’和‘已取消’状态的任务（主表），支持编辑修改相关信息。
        if (!Objects.equals(respVO.getState(), UdcEnum.SALESMAN_TASK_STATUS_WRE.getValueCode()) &&
                !Objects.equals(respVO.getState(), UdcEnum.SALESMAN_TASK_STATUS_CCD.getValueCode())) {
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "只有待发布和已取消的任务才可编辑修改!");
        }

    }

    /**
     * 校验必填项:任务主表
     *
     * @param saveVO 入参
     */
    private void checkMandatoryField(TaskInfoSaveVO saveVO) {
        if (StringUtils.isBlank(saveVO.getName())) {
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "任务名称为空!");
        }
        if (StringUtils.isBlank(saveVO.getType())) {
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "任务类型为空!");
        }
        if (Objects.isNull(saveVO.getStartTime())) {
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "开始时间为空!");
        }
        if (Objects.isNull(saveVO.getEndTime())) {
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "结束时间为空!");
        }
        if (StringUtils.isBlank(saveVO.getForceSignFlag())) {
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "强制签到为空!");
        }
        if (Objects.isNull(saveVO.getPublishUserId())) {
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "发布人为空!");
        }
        if (Objects.isNull(saveVO.getReleaseSign())) {
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "发布标志为空!");
        }
    }

    /**
     * 校验必填项:任务明细
     *
     * @param saveVOList 入参
     */
    private void checkMandatoryFieldDtlList(List<TaskInfoDtlSaveVO> saveVOList) {
        if (!CollectionUtils.isEmpty(saveVOList)) {
//            throw new BusinessException(ApiCode.VALIDATE_FAILED, "明细数据为空!");
            saveVOList.forEach(saveVO -> {
                checkMandatoryFieldDtl(saveVO);
            });
        }


    }

    /**
     * 校验必填项:任务明细
     *
     * @param saveVO 入参
     */
    private void checkMandatoryFieldDtl(TaskInfoDtlSaveVO saveVO) {
        if (StringUtils.isBlank(saveVO.getBusinessType())) {
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "业务类型为空!");
        }
        if (CollectionUtils.isEmpty(saveVO.getBusinessCodes()) && StringUtils.isBlank(saveVO.getBusinessCode())) {
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "业务编码为空!");
        }
        if (Objects.isNull(saveVO.getExecutUserCode())) {
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "执行人为空!");
        }
    }

    /**
     * 校验重复设置
     *
     * @param saveVO 入参
     * @return
     */
    private void checkRepeatSet(TaskInfoSaveVO saveVO) {
        //重复设置为是时做校验
        if (Objects.equals(saveVO.getRepeatSet(), ConstantsSale.SALESMAN_TASK_REPEAT_SET_Y)) {
            if (!ConstantsSale.SALESMAN_TASK_REPEAT_TYPE_LIST.contains(saveVO.getRepeatType())) {
                throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "重复类型未匹配到!");
            }
            checkRepeatSet01(saveVO);
        }

    }

    /**
     * 校验重复设置必填项
     *
     * @param saveVO 入参
     */
    private void checkRepeatSet01(TaskInfoSaveVO saveVO) {
        if (StringUtils.isBlank(saveVO.getRepeatType())) {
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "重复类型为空!");
        }
        //当重复类型不是次时,重复间隔不为空
        if (!Objects.equals(saveVO.getRepeatType(), ConstantsSale.SALESMAN_TASK_REPEAT_TYPE_TIMES) &&
                Objects.isNull(saveVO.getRepeatInterval())) {
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "重复间隔为空!");
        }
        //重复类型为周时,需指定值
        if (Objects.equals(saveVO.getRepeatType(), ConstantsSale.SALESMAN_TASK_REPEAT_TYPE_WEEK) && StringUtils.isBlank(saveVO.getAppointDay())) {
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "重复类型为周时,指定值为空!");
        }
        //重复类型为月时,需指定值
        if (Objects.equals(saveVO.getRepeatType(), ConstantsSale.SALESMAN_TASK_REPEAT_TYPE_MONTH) && StringUtils.isBlank(saveVO.getAppointDay())) {
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "重复类型为月时,指定值为空!");
        }
        //当重复类型不是次时,重复开始时间不为空
        if (!Objects.equals(saveVO.getRepeatType(), ConstantsSale.SALESMAN_TASK_REPEAT_TYPE_TIMES) && Objects.isNull(saveVO.getRepeatStartTime())) {
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "重复开始时间为空!");
        }
        //当重复类型不是次时,重复结束时间不为空
        if (!Objects.equals(saveVO.getRepeatType(), ConstantsSale.SALESMAN_TASK_REPEAT_TYPE_TIMES) && Objects.isNull(saveVO.getRepeatEndTime())) {
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "重复结束时间为空!");
        }
        //重复类型为次时,执行时间不能为空
        if (Objects.equals(saveVO.getRepeatType(), ConstantsSale.SALESMAN_TASK_REPEAT_TYPE_TIMES) && Objects.isNull(saveVO.getExecutTime())) {
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "重复类型为次时,执行时间不能为空!");
        }
    }

    /**
     * 任务分页查询
     *
     * @param pageParam 入参
     * @return 任务信息集合
     */
    @Override
    @SysCodeProc
    public PagingVO<TaskInfoRespVO> page(TaskInfoQueryVO pageParam) {

        //查询明细的条件
        if (StringUtils.isNotBlank(pageParam.getBusinessKeyword()) || !CollectionUtils.isEmpty(pageParam.getExecutUserCodes())
                || StringUtils.isNotBlank(pageParam.getExecutRecordKeyword())) {
            TaskInfoDtlQueryVO taskInfoDtlQueryVO = new TaskInfoDtlQueryVO();
            taskInfoDtlQueryVO.setBusinessKeyword(pageParam.getBusinessKeyword());
            taskInfoDtlQueryVO.setExecutUserCodes(pageParam.getExecutUserCodes());
            taskInfoDtlQueryVO.setExecutRecordKeyword(pageParam.getExecutRecordKeyword());
            List<TaskInfoDtlRespVO> dtlRespVOList = taskInfoDtlService.selectByParam(taskInfoDtlQueryVO);
            List<Long> masIdList = dtlRespVOList.stream().map(TaskInfoDtlRespVO::getMasId).filter(Objects::nonNull).distinct().collect(Collectors.toList());
            if (!CollectionUtils.isEmpty(masIdList)) {
                List<Long> ids = CollectionUtils.isEmpty(pageParam.getIds()) ? Collections.EMPTY_LIST : pageParam.getIds();
                List<Long> idList = Stream.of(ids, masIdList).flatMap(Collection::stream).filter(Objects::nonNull).distinct().collect(Collectors.toList());
                pageParam.setIds(idList);
            }
        }

        PagingVO<TaskInfoRespVO> pagingVO = taskInfoRepoProc.page(pageParam);
        if (CollectionUtils.isEmpty(pagingVO.getRecords())) {
            return PagingVO.<TaskInfoRespVO>builder().total(0L).records(Collections.EMPTY_LIST).build();
        }
        List<TaskInfoRespVO> respVOList = pagingVO.getRecords();

        //组装填充任务的相关关联字段信息
        translateTask(respVOList);
        //组装填充任务的UDC
        translateTaskUdc(respVOList);

        return PagingVO.<TaskInfoRespVO>builder()
                .total(pagingVO.getTotal())
                .records(respVOList)
                .build();
    }

    public List<FileInfoRespVO> selectFileBySourceId(Long sourceId) {
        if (Objects.isNull(sourceId)) {
            return Collections.EMPTY_LIST;
        }
        FileInfoQueryVO fileInfoQueryVO = new FileInfoQueryVO();
        fileInfoQueryVO.setSourceId(sourceId);
        List<FileInfoRespVO> list = fileInfoRepoProc.getList(fileInfoQueryVO);
        if (CollectionUtils.isEmpty(list)) {
            return Collections.EMPTY_LIST;
        }
        return list;
    }

    public List<FileInfoRespVO> selectFileBySourceIds(List<Long> sourceIds) {
        if (CollectionUtils.isEmpty(sourceIds)) {
            return Collections.EMPTY_LIST;
        }
        FileInfoQueryVO fileInfoQueryVO = new FileInfoQueryVO();
        fileInfoQueryVO.setSourceIds(sourceIds);
        List<FileInfoRespVO> list = fileInfoRepoProc.getList(fileInfoQueryVO);
        if (CollectionUtils.isEmpty(list)) {
            return Collections.EMPTY_LIST;
        }
        return list;
    }

    /**
     * 组装填充任务的相关关联字段信息
     *
     * @param respVOList 任务主表数据信息
     * @return
     */
    private void translateTask(List<TaskInfoRespVO> respVOList) {
        //公司-编码
        List<String> ouCodeList = respVOList.stream().map(TaskInfoRespVO::getOuCode).filter(Objects::nonNull).distinct().collect(Collectors.toList());
        List<OrgOuRpcDTO> ouRpcDTOList = rmiOrgOuService.findOuDtoListByOuCodes(ouCodeList);
        //发布人-ID
        Set<Long> publishUserIdList = respVOList.stream().map(TaskInfoRespVO::getPublishUserId).filter(Objects::nonNull).distinct().collect(Collectors.toSet());
        List<SysEmployeeBasicDTO> employeeList = rmiEmployeeRpcService.findEmployeeByIds(publishUserIdList);
        //执行模板-编码
        List<String> templateCodeList = respVOList.stream().map(TaskInfoRespVO::getExecutTemplateCode).filter(Objects::nonNull).distinct().collect(Collectors.toList());
        //关联执行模板名称
        List<ExectRecordTempRespVO> recordTempVOList = exectRecordTempService.queryByCodes(templateCodeList);
        //任务ID查询附件信息
        List<Long> idList = respVOList.stream().map(TaskInfoRespVO::getId).filter(Objects::nonNull).distinct().collect(Collectors.toList());
        List<FileInfoRespVO> fileInfoRespVOList = selectFileBySourceIds(idList);


        respVOList.forEach(respVO -> {
            val ouRpcDTOOptional = ouRpcDTOList.stream().filter(ouRpcDTO -> Objects.equals(ouRpcDTO.getOuCode(), respVO.getOuCode())).findFirst();
            ouRpcDTOOptional.ifPresent(ouRpcDTO -> respVO.setOuName(ouRpcDTO.getOuName()));
            val employeeDTOOptional = employeeList.stream().filter(employeeDTO -> Objects.equals(employeeDTO.getId(), respVO.getPublishUserId())).findFirst();
            employeeDTOOptional.ifPresent(employeeDTO -> respVO.setPublishUser(employeeDTO.getFullName()));
            //附件编码字符串转换为编码集合,以逗号分隔的字符串形式
            List<String> fileCodes = transitionStrToCodes(respVO.getFileInfo());
            respVO.setFileCodes(fileCodes);
            //执行模板
            if (!CollectionUtils.isEmpty(recordTempVOList)) {
                val recordTempOptional = recordTempVOList.stream().filter(tempRespVO -> Objects.equals(tempRespVO.getTempCode(), respVO.getExecutTemplateCode())).findFirst();
                recordTempOptional.ifPresent(tempRespVO -> respVO.setExecutTemplateName(tempRespVO.getTempName()));
            }
            //查询附件信息
            List<FileInfoRespVO> fileInfoList = fileInfoRespVOList.stream().filter(fileInfoRespVO -> Objects.equals(fileInfoRespVO.getSourceId(), respVO.getId())).collect(Collectors.toList());
            respVO.setFileInfoRespVOS(fileInfoList);
        });
    }

    /**
     * APP端组装填充任务的相关关联字段信息
     *
     * @param respVOList 任务主表数据信息
     * @return
     */
    private void translateAppTask(List<TaskInfoRespVO> respVOList) {
        //发布人-ID
        Set<Long> publishUserIdList = respVOList.stream().map(TaskInfoRespVO::getPublishUserId).filter(Objects::nonNull).distinct().collect(Collectors.toSet());
        List<SysEmployeeBasicDTO> employeeList = rmiEmployeeRpcService.findEmployeeByIds(publishUserIdList);
        //任务ID查询附件信息
        List<Long> idList = respVOList.stream().map(TaskInfoRespVO::getId).filter(Objects::nonNull).distinct().collect(Collectors.toList());
        List<FileInfoRespVO> fileInfoRespVOList = selectFileBySourceIds(idList);

        respVOList.forEach(respVO -> {
            val employeeDTOOptional = employeeList.stream().filter(employeeDTO -> Objects.equals(employeeDTO.getId(), respVO.getPublishUserId())).findFirst();
            employeeDTOOptional.ifPresent(employeeDTO -> respVO.setPublishUser(employeeDTO.getFullName()));
            //附件编码字符串转换为编码集合,以逗号分隔的字符串形式
            List<String> fileCodes = transitionStrToCodes(respVO.getFileInfo());
            respVO.setFileCodes(fileCodes);

            //查询附件信息
            List<FileInfoRespVO> fileInfoList = fileInfoRespVOList.stream().filter(fileInfoRespVO -> Objects.equals(fileInfoRespVO.getSourceId(), respVO.getId())).collect(Collectors.toList());
            respVO.setFileInfoRespVOS(fileInfoList);
        });
    }

    /**
     * 组装填充任务的UDC
     *
     * @param respVOList 任务主表数据信息
     * @return
     */
    private void translateTaskUdc(List<TaskInfoRespVO> respVOList) {
        Map<String, String> taskRepeatSetWeekMap = rmiUdcService.getUdcMapByUdcCode(ConstantsSale.YST_SALE_UDC_MODEL, UdcEnum.TASK_REPEAT_SET_WEEK_0.getCode());
        Map<String, String> taskRepeatSetMonthMap = rmiUdcService.getUdcMapByUdcCode(ConstantsSale.YST_SALE_UDC_MODEL, UdcEnum.TASK_REPEAT_SET_MONTH_0.getCode());
        respVOList.forEach(respVO -> {
            //重复设置类型为周时
            if (StringUtils.isNotBlank(respVO.getAppointDay()) && Objects.equals(respVO.getRepeatType(), ConstantsSale.SALESMAN_TASK_REPEAT_TYPE_WEEK)
                    && !MapUtils.isEmpty(taskRepeatSetWeekMap)) {
                respVO.setAppointDayName(taskRepeatSetWeekMap.get(respVO.getAppointDay()));
            }
            //重复设置类型为月时
            if (StringUtils.isNotBlank(respVO.getAppointDay()) && Objects.equals(respVO.getRepeatType(), ConstantsSale.SALESMAN_TASK_REPEAT_TYPE_MONTH)
                    && !MapUtils.isEmpty(taskRepeatSetMonthMap)) {
                respVO.setAppointDayName(taskRepeatSetMonthMap.get(respVO.getAppointDay()));
            }
        });
    }

    /**
     * 根据任务ID查询任务详情数据
     *
     * @param id 任务ID
     * @return 任务详情数据
     */
    @Override
    @SysCodeProc
    public TaskInfoRespVO findIdOne(Long id) {
        //查询任务主表
        Optional<TaskInfoDO> optional = taskInfoRepo.findById(id);
        if (optional.isEmpty()) {
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "未查询到数据信息!");
        }
        TaskInfoRespVO taskInfoRespVO = TaskInfoConvert.INSTANCE.doToRespVo(optional.get());

        selectTaskInfo(taskInfoRespVO);

        return taskInfoRespVO;
    }

    /**
     * APP端-根据任务ID查询任务详情数据
     *
     * @param id 任务ID
     * @return 任务详情数据
     */
    @Override
    @SysCodeProc
    public TaskInfoRespVO findAppTaskById(Long id) {
        //查询任务主表
        Optional<TaskInfoDO> optional = taskInfoRepo.findById(id);
        if (optional.isEmpty()) {
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "未查询到数据信息!");
        }
        TaskInfoRespVO taskInfoRespVO = TaskInfoConvert.INSTANCE.doToRespVo(optional.get());

        //组装填充任务的相关关联字段信息
        translateAppTask(Collections.singletonList(taskInfoRespVO));
        //查询任务明细
        List<TaskInfoDtlRespVO> dtlRespVOList = taskInfoDtlService.selectByMasId(id);
        if (!CollectionUtils.isEmpty(dtlRespVOList)) {
            taskInfoRespVO.setDtlRespVOList(dtlRespVOList);
        }

        return taskInfoRespVO;
    }

    /**
     * 根据任务明细ID查询任务详情信息
     *
     * @param dtlId 任务明细ID
     * @return 任务详情数据
     */
    @Override
    @SysCodeProc
    public TaskInfoRespVO findTaskByDtlId(Long dtlId) {
        //1.先根据任务明细ID查询任务明细信息
        TaskInfoDtlRespVO taskInfoDtlRespVO = taskInfoDtlService.selectById(dtlId);
        if (Objects.isNull(taskInfoDtlRespVO)) {
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "未查询到任务明细数据信息!");
        }
        //2.查询任务主表
        Optional<TaskInfoDO> optional = taskInfoRepo.findById(taskInfoDtlRespVO.getMasId());
        if (optional.isEmpty()) {
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "未查询到任务数据信息!");
        }
        TaskInfoRespVO taskInfoRespVO = TaskInfoConvert.INSTANCE.doToRespVo(optional.get());

        //组装填充任务的相关关联字段信息
        translateAppTask(Collections.singletonList(taskInfoRespVO));
        //组装任务主表和明细信息
        taskInfoRespVO.setDtlRespVOList(Collections.singletonList(taskInfoDtlRespVO));
        //设置执行人管辖区域
        setJurisdictionName(List.of(taskInfoDtlRespVO));
        return taskInfoRespVO;
    }

    /**
     * 设置执行人管辖区域
     * @param records
     */
    private void setJurisdictionName(List<TaskInfoDtlRespVO> records) {
        if (!CollectionUtils.isEmpty(records)) {
            List<String> salesmanNos = records.stream().map(TaskInfoDtlRespVO::getExecutUserCode).filter(StrUtil::isNotEmpty).distinct().collect(Collectors.toList());
            if (CollectionUtils.isEmpty(salesmanNos)) {
                return;
            }
            List<SalesmanInfoDO> bySalesmanNoIn = salesmanInfoRepo.findBySalesmanNoIn(salesmanNos);
            if (CollectionUtils.isEmpty(bySalesmanNoIn)) {
                return;
            }
            List<Long> salesmanIds = bySalesmanNoIn.stream().map(SalesmanInfoDO::getId).collect(Collectors.toList());
            List<SalesmanRegionDO> regionDOS = salesmanRegionRepo.findByMasIdIn(salesmanIds);
            if (CollUtil.isNotEmpty(regionDOS)) {
                Map<Long, List<SalesmanRegionDO>> map = regionDOS.stream().collect(Collectors.groupingBy(SalesmanRegionDO::getMasId));
                Map<String, String> regionMap = rmiUdcService.getUdcMapByUdcCode("yst-supp", "REGION");

                records.forEach(i -> {
                    if (StringUtils.isNotEmpty(i.getExecutUserCode())) {
                        i.setExecutUserId(bySalesmanNoIn.stream().filter(salesmanInfoDO -> salesmanInfoDO.getSalesmanNo().equals(i.getExecutUserCode()))
                                .map(SalesmanInfoDO::getId)
                                .findAny().orElse(null));
                    }
                    if (!CollectionUtils.isEmpty(map.get(i.getExecutUserId()))) {
                        String jurisdictionName = map.get(i.getExecutUserId()).stream().map(SalesmanRegionDO::getRegionCode)
                                .filter(StringUtils::isNotEmpty)
                                .map(regionCode -> regionMap.get(regionCode))
                                .filter(StringUtils::isNotEmpty).collect(Collectors.joining(","));
                        i.setJurisdictionName(jurisdictionName);
                    }
                });
            }
        }
    }

    /**
     * 根据任务ID查询任务主表数据
     *
     * @param id 任务ID
     * @return 任务主表数据
     */
    @Override
    @SysCodeProc
    public TaskInfoRespVO findById(Long id) {
        if (Objects.isNull(id)) {
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "任务ID不能为空!");
        }
        TaskInfoQueryVO queryVO = new TaskInfoQueryVO();
        queryVO.setId(id);
        List<TaskInfoRespVO> respVOList = taskInfoRepoProc.selectByQueryVO(queryVO);
        if (CollectionUtils.isEmpty(respVOList)) {
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "未查询到数据信息!");
        }
        if (respVOList.size() > 1) {
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "查询到多条数据信息!");
        }
        TaskInfoRespVO taskInfoRespVO = respVOList.get(0);

        return taskInfoRespVO;
    }

    /**
     * 根据任务编码查询任务详情数据
     *
     * @param code 任务编码
     * @return 任务详情数据
     */
    @Override
    @SysCodeProc
    public TaskInfoRespVO findCodeOne(String code) {
        //查询任务主表
        TaskInfoQueryVO queryVO = new TaskInfoQueryVO();
        queryVO.setCode(code);
        List<TaskInfoRespVO> respVOList = taskInfoRepoProc.selectByQueryVO(queryVO);
        if (CollectionUtils.isEmpty(respVOList)) {
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "未查询到数据信息!");
        }
        if (respVOList.size() > 1) {
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "查询到多条数据信息!");
        }
        TaskInfoRespVO taskInfoRespVO = respVOList.get(0);

        selectTaskInfo(taskInfoRespVO);

        return taskInfoRespVO;
    }

    /**
     * 根据任务编码查询任务主表数据
     *
     * @param code 根据任务编码
     * @return 任务主表数据
     */
    @Override
    @SysCodeProc
    public TaskInfoRespVO findByCode(String code) {
        if (StringUtils.isBlank(code)) {
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "任务编码不能为空!");
        }
        TaskInfoQueryVO queryVO = new TaskInfoQueryVO();
        queryVO.setCode(code);
        List<TaskInfoRespVO> respVOList = taskInfoRepoProc.selectByQueryVO(queryVO);
        if (CollectionUtils.isEmpty(respVOList)) {
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "未查询到数据信息!");
        }
        if (respVOList.size() > 1) {
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "查询到多条数据信息!");
        }
        TaskInfoRespVO taskInfoRespVO = respVOList.get(0);

        return taskInfoRespVO;
    }

    /**
     * 根据入参查询任务主表数据
     *
     * @param queryVO 入参
     * @return 任务主表数据
     */
    @Override
    @SysCodeProc
    public List<TaskInfoRespVO> selectByParam(TaskInfoQueryVO queryVO) {
        List<TaskInfoRespVO> taskInfoRespVOList = taskInfoRepoProc.selectByQueryVO(queryVO);
        if (CollectionUtils.isEmpty(taskInfoRespVOList)) {
            return Collections.emptyList();
        }
        return taskInfoRespVOList;
    }

    /**
     * 根据入参查询任务主表数据-带分页
     *
     * @param queryVO 入参
     * @return 任务主表数据
     */
    @Override
    @SysCodeProc
    public List<TaskInfoRespVO> selectPageByQueryVO(TaskInfoQueryVO queryVO) {
        List<TaskInfoRespVO> taskInfoRespVOList = taskInfoRepoProc.selectPageByQueryVO(queryVO);
        if (CollectionUtils.isEmpty(taskInfoRespVOList)) {
            return Collections.emptyList();
        }
        return taskInfoRespVOList;
    }

    /**
     * 查询任务主表和明细信息
     *
     * @param taskInfoRespVO 任务主表信息
     * @return
     */
    private void selectTaskInfo(TaskInfoRespVO taskInfoRespVO) {
        Long id = taskInfoRespVO.getId();
        //附件编码字符串转换为编码集合,以逗号分隔的字符串形式
        List<String> fileCodes = transitionStrToCodes(taskInfoRespVO.getFileInfo());
        taskInfoRespVO.setFileCodes(fileCodes);

        //组装填充任务的相关关联字段信息
        translateTask(Collections.singletonList(taskInfoRespVO));
        //组装填充任务的UDC
        translateTaskUdc(Collections.singletonList(taskInfoRespVO));

        //查询任务明细
        List<TaskInfoDtlRespVO> dtlRespVOList = taskInfoDtlService.selectByMasId(id);
        if (!CollectionUtils.isEmpty(dtlRespVOList)) {
            taskInfoRespVO.setDtlRespVOList(dtlRespVOList);
        }
    }

    /**
     * 明细数据转换处理
     *
     * @param respVOList 入参
     * @return
     */
    private void transitionDtlToCodes(List<TaskInfoDtlRespVO> respVOList) {
        respVOList.forEach(respVO -> {
            //明细的关联业务编码数据库里以逗号分隔的字符串形式保存
            //List<String> businessCodes = transitionStrToCodes(respVO.getBusinessCode());
            //respVO.setBusinessCodes(businessCodes);
        });
    }

    /**
     * 根据任务ID批量删除
     *
     * @param ids 任务ID集合
     * @return
     */
    @Override
    @Transactional(rollbackFor = {Exception.class})
    public void deleteBatch(List<Long> ids) {
        TaskInfoQueryVO queryVO = new TaskInfoQueryVO();
        queryVO.setIds(ids);
        List<TaskInfoRespVO> respVOList = taskInfoRepoProc.selectByQueryVO(queryVO);
        if (CollectionUtils.isEmpty(respVOList)) {
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "未查询到数据信息!");
        }
        //针对‘待发布’状态的任务（主表）
        List<TaskInfoRespVO> voList = respVOList.stream().filter(respVO -> !Objects.equals(respVO.getState(), UdcEnum.SALESMAN_TASK_STATUS_WRE.getValueCode())).collect(Collectors.toList());
        if (!CollectionUtils.isEmpty(voList)) {
            String checkResult = voList.stream().map(vo ->
                    "任务名称:" + vo.getName() + ""
            ).collect(Collectors.joining(";", "只有待发布可删除,[", "], 请检查"));
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, checkResult);
        }

        //逻辑删除主表
        taskInfoRepoProc.updateDeleteFlagBatch(1, ids);
        //逻辑删除明细
        taskInfoDtlService.updateDeleteFlagBatch(1, ids);

    }

    /**
     * 根据任务ID取消
     *
     * @param id 任务ID
     * @return
     */
    @Override
    @Transactional(rollbackFor = {Exception.class})
    public void cancelTask(Long id) {
        /*TaskInfoQueryVO queryVO = new TaskInfoQueryVO();
        queryVO.setId(id);
        List<TaskInfoRespVO> respVOList = taskInfoRepoProc.selectByQueryVO(queryVO);
        if (CollectionUtils.isEmpty(respVOList)) {
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "未查询到数据信息!");
        }*/
        //TaskInfoRespVO respVO = respVOList.get(0);
        TaskInfoRespVO respVO = findById(id);

        //针对待发布和‘待执行’状态的任务（主表），支持取消操作，将任务主表和明细状态更新为‘已取消’
        if (!Objects.equals(respVO.getState(), UdcEnum.SALESMAN_TASK_STATUS_WRE.getValueCode()) &&
                !Objects.equals(respVO.getState(), UdcEnum.SALESMAN_TASK_STATUS_WEE.getValueCode())) {
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "待发布和待执行状态才可取消!");
        }

        // 根据任务ID更新任务状态
        this.updateStateById(UdcEnum.SALESMAN_TASK_STATUS_CCD.getValueCode(), id);
        //根据主表任务ID更新明细完成状态
        taskInfoDtlService.updateCompleteStateByMasId(UdcEnum.SALESMAN_TASK_COMPLETION_STATUS_CCD.getValueCode(), id);
    }

    /**
     * 根据任务ID发布
     *
     * @param id 任务ID
     * @return
     */
    @Override
    @Transactional(rollbackFor = {Exception.class})
    public void releaseTask(Long id) {
        TaskInfoRespVO respVO = findById(id);
        //针对‘待发布’状态的任务（主表），支持发布
        if (!Objects.equals(respVO.getState(), UdcEnum.SALESMAN_TASK_STATUS_WRE.getValueCode())) {
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "待发布状态才可发布!");
        }
        //校验：执行模版是否为空，为空时根据任务类型+所属公司去匹配启用状态的执行模版关联起来，
        // 如果还是没有匹配到执行模版，则提示发布任务失败。
        if (StringUtils.isBlank(respVO.getExecutTemplateCode())) {
            ExectRecordTempRespVO exectRecordTempRespVO = getExecuteTemplate(respVO.getType(), respVO.getOuCode());
            //更新维护执行模版
            taskInfoRepoProc.updateExecuteTemplateCodeById(exectRecordTempRespVO.getId(), exectRecordTempRespVO.getTempCode(),
                    exectRecordTempRespVO.getTempName(), id);
        }
        //任务主表和明细表的状态更新为‘待执行’
        // 根据任务ID更新任务状态
        this.updateStateById(UdcEnum.SALESMAN_TASK_STATUS_WEE.getValueCode(), id);
        //根据主表任务ID更新明细完成状态
        taskInfoDtlService.updateCompleteStateByMasId(UdcEnum.SALESMAN_TASK_COMPLETION_STATUS_WEE.getValueCode(), id);

    }

    /**
     * 根据任务ID完成前置校验
     *
     * @param id 任务ID
     * @return true 校验通过  false 校验未通过
     */
    @Override
    public Boolean accomplishCheckTask(Long id) {
        TaskInfoRespVO respVO = findById(id);
        //针对‘待执行’和‘进行中’状态的任务（主表），点击时需要校验
        if (!Objects.equals(respVO.getState(), UdcEnum.SALESMAN_TASK_STATUS_WEE.getValueCode()) &&
                !Objects.equals(respVO.getState(), UdcEnum.SALESMAN_TASK_STATUS_IPS.getValueCode())) {
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "待执行或进行中状态才可完成!");
        }
        //点击时需要校验：是否有任务明细状态为‘进行中’、‘待审核’、‘审核拒绝’的其关联的执行记录
        //如果有则弹窗提示操作人，此任务有在进行中的任务明细，标记完成会将任务明细状态统一标记为已完成，执行记录状态统一标记为审核通过，确定要标记完成嘛？
        TaskInfoDtlQueryVO dtlQueryVO = new TaskInfoDtlQueryVO();
        dtlQueryVO.setMasId(id);
        List<String> completeStates = Lists.newArrayList(
                UdcEnum.SALESMAN_TASK_COMPLETION_STATUS_IPS.getValueCode(),
                UdcEnum.SALESMAN_TASK_COMPLETION_STATUS_WAT.getValueCode(),
                UdcEnum.SALESMAN_TASK_COMPLETION_STATUS_ARN.getValueCode());
        dtlQueryVO.setCompleteStates(completeStates);
        List<TaskInfoDtlRespVO> dtlRespVOList = taskInfoDtlService.selectByParam(dtlQueryVO);
        //当没有任务明细状态为‘进行中’、‘待审核’、‘审核拒绝’的数据时,校验通过
        if (CollectionUtils.isEmpty(dtlRespVOList)) {
            return true;
        }
        //根据执行记录编码查询执行记录信息
        List<String> recordCode = dtlRespVOList.stream().map(TaskInfoDtlRespVO::getExecutRecordCode).filter(Objects::nonNull).distinct().collect(Collectors.toList());
        boolean recordFlag = selectExecutionRecordCheck(recordCode);
        if (recordFlag) {
            //有关联的执行记录,则校验不通过
            return false;
        }

        return true;
    }

    /**
     * 根据执行记录编码查询执行记录信息
     *
     * @param recordCode 执行记录编码
     * @return 有关联的执行记录为true, 否则为false
     */
    @Override
    public boolean selectExecutionRecordCheck(List<String> recordCode) {
        if (CollectionUtils.isEmpty(recordCode)) {
            return false;
        }
        //根据执行记录编码查询执行记录信息,有关联的执行记录为true,否则为false
        List<ExecutRecordRespVO> executeRecordList = exectRecordService.queryByCodes(recordCode);
        if (CollectionUtils.isEmpty(executeRecordList)) {
            return false;
        }
        return true;
    }

    /**
     * 根据任务ID完成
     *
     * @param id 任务ID
     * @return
     */
    @Override
    @Transactional(rollbackFor = {Exception.class})
    public void accomplishTask(Long id) {
        TaskInfoRespVO respVO = findById(id);
        //针对‘待执行’和‘进行中’状态的任务（主表），点击时需要校验
        if (!Objects.equals(respVO.getState(), UdcEnum.SALESMAN_TASK_STATUS_WEE.getValueCode()) &&
                !Objects.equals(respVO.getState(), UdcEnum.SALESMAN_TASK_STATUS_IPS.getValueCode())) {
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "待执行或进行中状态才可完成!");
        }

        //点击确认后，将任务主表状态更新为‘已完成’，完成时间更新为当前操作时间，
        // 根据任务ID更新任务状态、完成时间
        this.updateStateAndCompleteTime(UdcEnum.SALESMAN_TASK_STATUS_CPD.getValueCode(), LocalDateTime.now(), id);
        //查询所有任务明细
        TaskInfoDtlQueryVO dtlQueryVO = new TaskInfoDtlQueryVO();
        dtlQueryVO.setMasId(id);
        List<TaskInfoDtlRespVO> dtlRespVOList = taskInfoDtlService.selectByParam(dtlQueryVO);
        if (CollectionUtils.isEmpty(dtlRespVOList)) {
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "未查询到任务明细数据!");
        }
        //过滤出已完成的明细，主任务 操作 完成 或关闭 操作后  是将子任务除了已完成状态 都改为 已完成或已关闭
        List<TaskInfoDtlRespVO> infoDtlRespVOList = dtlRespVOList.stream().filter(dtlRespVO ->
                accomplishTaskDtlCheck(dtlRespVO.getCompleteState())).collect(Collectors.toList());
        List<Long> dtlIds = infoDtlRespVOList.stream().map(TaskInfoDtlRespVO::getId).filter(Objects::nonNull).collect(Collectors.toList());
        //根据主表任务ID更新明细完成状态
        //taskInfoDtlService.updateCompleteStateAndTimeByMasId(UdcEnum.SALESMAN_TASK_COMPLETION_STATUS_CPD.getValueCode(), LocalDateTime.now(), id);
        if (!CollectionUtils.isEmpty(dtlIds)) {
            taskInfoDtlService.updateCompleteStateAndTimeByIdBatch(UdcEnum.SALESMAN_TASK_COMPLETION_STATUS_CPD.getValueCode(), LocalDateTime.now(), dtlIds);
        }

        //根据任务主表ID计算更新任务进度
        taskInfoDtlService.updateProgressById(id);
        //同时将任务关联的执行记录状态统一更新为‘审核通过’（执行记录的审核情况记录完成按钮操作人和时间）。
        accomplishUpdateExecuteRecord(id);

    }

    private boolean accomplishTaskDtlCheck(String completeState) {
        if (Objects.equals(completeState, UdcEnum.SALESMAN_TASK_COMPLETION_STATUS_CPD.getValueCode())
                || Objects.equals(completeState, UdcEnum.SALESMAN_TASK_COMPLETION_STATUS_CSD.getValueCode())) {
            return false;
        }

        return true;
    }

    /**
     * 完成-将任务关联的执行记录状态统一更新为‘审核通过’（执行记录的审核情况记录完成按钮操作人和时间）
     *
     * @param id 任务ID
     * @return
     */
    private void accomplishUpdateExecuteRecord(Long id) {
        //更新执行记录

        //查询任务明细信息
        TaskInfoDtlQueryVO dtlQueryVO = new TaskInfoDtlQueryVO();
        dtlQueryVO.setMasId(id);
        List<TaskInfoDtlRespVO> dtlRespVOList = taskInfoDtlService.selectByParam(dtlQueryVO);
        //获取任务关联的执行记录编码
        List<String> recordCodes = dtlRespVOList.stream().map(TaskInfoDtlRespVO::getExecutRecordCode).filter(Objects::nonNull).distinct().collect(Collectors.toList());
        //当前用户信息
        //SysUserDTO sysUserDTO = getSysUser();
        //sysUserDTO.getId();
        if (!CollectionUtils.isEmpty(recordCodes)) {
            exectRecordService.complete(recordCodes);
        }

    }

    /**
     * 根据任务ID关闭前置校验
     *
     * @param id 任务ID
     * @return true 校验通过  false 校验未通过
     */
    @Override
    public Boolean closeCheckTask(Long id) {
        TaskInfoRespVO respVO = findById(id);
        //勾选‘待执行’和‘进行中’任务（主表），点击时需要校验
        if (!Objects.equals(respVO.getState(), UdcEnum.SALESMAN_TASK_STATUS_WEE.getValueCode()) &&
                !Objects.equals(respVO.getState(), UdcEnum.SALESMAN_TASK_STATUS_IPS.getValueCode())) {
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "待执行或进行中状态才可关闭!");
        }
        //点击时需要校验：是否有任务明细状态为‘进行中’、‘待审核’、‘审核拒绝’的其关联的执行记录，
        //如果有则弹窗提示操作人，此任务有在进行中的任务明细，标记关闭会将任务明细状态统一标记为已关闭，执行记录状态统一标记为已关闭，确定要标记关闭嘛？
        TaskInfoDtlQueryVO dtlQueryVO = new TaskInfoDtlQueryVO();
        dtlQueryVO.setMasId(id);
        List<String> completeStates = Lists.newArrayList(
                UdcEnum.SALESMAN_TASK_COMPLETION_STATUS_IPS.getValueCode(),
                UdcEnum.SALESMAN_TASK_COMPLETION_STATUS_WAT.getValueCode(),
                UdcEnum.SALESMAN_TASK_COMPLETION_STATUS_ARN.getValueCode());
        dtlQueryVO.setCompleteStates(completeStates);
        List<TaskInfoDtlRespVO> dtlRespVOList = taskInfoDtlService.selectByParam(dtlQueryVO);
        //当没有明细状态为‘进行中’、‘待审核’、‘审核拒绝’的数据时,校验通过
        if (CollectionUtils.isEmpty(dtlRespVOList)) {
            return true;
        }
        //根据执行记录编码查询执行记录信息
        List<String> recordCode = dtlRespVOList.stream().map(TaskInfoDtlRespVO::getExecutRecordCode).filter(Objects::nonNull).distinct().collect(Collectors.toList());
        boolean recordFlag = selectExecutionRecordCheck(recordCode);
        if (recordFlag) {
            //有关联的执行记录,则校验不通过
            return false;
        }

        return true;
    }

    /**
     * 根据任务ID关闭
     *
     * @param id 任务ID
     * @return
     */
    @Override
    @Transactional(rollbackFor = {Exception.class})
    public void closeTask(Long id) {
        TaskInfoRespVO respVO = findById(id);
        //勾选‘待执行’和‘进行中’任务（主表），点击时需要校验
        if (!Objects.equals(respVO.getState(), UdcEnum.SALESMAN_TASK_STATUS_WEE.getValueCode()) &&
                !Objects.equals(respVO.getState(), UdcEnum.SALESMAN_TASK_STATUS_IPS.getValueCode())) {
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "待执行或进行中状态才可关闭!");
        }
        //点击确认后，将任务明细状态非‘已完成’状态的全部更新为‘已关闭’状态，
        // 相关执行记录将状态非‘审核通过’的全部更新为‘已关闭’，此时任务主表状态判断：
        // 如果有明细状态为已完成，则主表状态为已完成，如果明细全部为已关闭，则主表状态更新为已关闭。

        //查询任务明细信息
        TaskInfoDtlQueryVO dtlQueryVO = new TaskInfoDtlQueryVO();
        dtlQueryVO.setMasId(id);
        List<TaskInfoDtlRespVO> dtlRespVOList = taskInfoDtlService.selectByParam(dtlQueryVO);
        //过滤出非‘已完成’状态的任务明细
        List<Long> dtlIds = dtlRespVOList.stream().filter(vo -> !Objects.equals(vo.getCompleteState(), UdcEnum.SALESMAN_TASK_COMPLETION_STATUS_CPD.getValueCode()))
                .map(TaskInfoDtlRespVO::getId).collect(Collectors.toList());
        //根据任务明细ID批量更新完成状态-已关闭
        taskInfoDtlService.updateCompleteStateByIdBatch(UdcEnum.SALESMAN_TASK_COMPLETION_STATUS_CSD.getValueCode(), dtlIds);
        //如果有明细状态为已完成，则主表状态为已完成
        boolean dtlFlag1 = dtlRespVOList.stream().anyMatch(vo -> Objects.equals(vo.getCompleteState(), UdcEnum.SALESMAN_TASK_COMPLETION_STATUS_CPD.getValueCode()));
        if (dtlFlag1) {
            //根据任务ID更新任务状态-已完成
            //this.updateStateById(UdcEnum.SALESMAN_TASK_STATUS_CPD.getValueCode(), id);
            this.updateStateAndCompleteTime(UdcEnum.SALESMAN_TASK_STATUS_CPD.getValueCode(), LocalDateTime.now(), id);
        } else {
            //此时明细状态就只有两种情况：1.全是已关闭,2.已完成和已关闭
            //根据任务ID更新任务状态-已关闭
            this.updateStateById(UdcEnum.SALESMAN_TASK_STATUS_CSD.getValueCode(), id);
        }

        //相关执行记录将状态非‘审核通过’的全部更新为‘已关闭’
        closeUpdateExecuteRecord(dtlRespVOList);
    }

    /**
     * 关闭-相关执行记录将状态非‘审核通过’的全部更新为‘已关闭’
     *
     * @param dtlRespVOList 任务明细信息
     * @return
     */
    private void closeUpdateExecuteRecord(List<TaskInfoDtlRespVO> dtlRespVOList) {
        //更新执行记录
        //获取任务关联的执行记录编码
        List<String> recordCodes = dtlRespVOList.stream().map(TaskInfoDtlRespVO::getExecutRecordCode).filter(Objects::nonNull).distinct().collect(Collectors.toList());
        if (!CollectionUtils.isEmpty(recordCodes)) {
            exectRecordService.filterClose(recordCodes);
        }

    }

    /**
     * 判断任务是否逾期,并更新是否逾期
     * <p>
     * 创建任务时默认为未逾期；当任务整体没有在规定的结束时间完成时（任务主表状态为‘待发布’、‘待执行’、‘进行中’的），所有明细的完成状态不是‘已完成’的均将是否逾期字段标记为已逾期，其余明细仍保持‘未逾期’，
     * 当任务整体没有在规定的结束时间完成时，所有明细的完成状态不是‘已完成’的均将是否逾期字段标记为已逾期，其余明细仍保持‘未逾期’，
     * 默认为未逾期，当整体任务没有在结束时间前完成所有明细时，将明细的完成状态非‘已完成’的标记为已逾期
     *
     * @param queryVO 入参
     * @return 已逾期的任务ID
     */
    @Override
    @Transactional(rollbackFor = {Exception.class})
    public List<Long> overdueTask(TaskInfoQueryVO queryVO) {
        //当任务整体没有在规定的结束时间完成时（任务主表状态为‘待发布’、‘待执行’、‘进行中’的），
        // 所有明细的完成状态不是‘已完成’的均将是否逾期字段标记为已逾期，其余明细仍保持‘未逾期’
        if (CollectionUtils.isEmpty(queryVO.getStateList())) {
            //任务主表状态为‘待发布’、‘待执行’、‘进行中’的
            List<String> stateList = Lists.newArrayList(
                    UdcEnum.SALESMAN_TASK_STATUS_WRE.getValueCode(),
                    UdcEnum.SALESMAN_TASK_STATUS_WEE.getValueCode(),
                    UdcEnum.SALESMAN_TASK_STATUS_IPS.getValueCode()
            );
            queryVO.setStateList(stateList);
        }
        if (Objects.isNull(queryVO.getEndTimeE())) {
            //结束时间默认当前日期
            queryVO.setEndTimeE(LocalDateTime.of(LocalDate.now(), LocalTime.MIN));
        }
        //查询未逾期
        queryVO.setDelayFlag(UdcEnum.SALESMAN_TASK_DELAY_FLAG_N.getValueCode());
        //通过分页分批查询任务主表数据-不是分页列表查询
        selectOverdueTask(queryVO, paramVO -> this.selectPageByQueryVO(queryVO));

        return Collections.emptyList();
    }

    private void selectOverdueTask(TaskInfoQueryVO queryParam, Function<TaskInfoQueryVO, List<TaskInfoRespVO>> dataProducer) {
        disposeOverdueTask((page, pageSize) -> {
            queryParam.setCurrent(page);
            queryParam.setSize(pageSize);
            return dataProducer.apply(queryParam);
        });
    }

    private void disposeOverdueTask(BiFunction<Integer, Integer, List<TaskInfoRespVO>> dataProducer) {
        int page = 1;
        int size = 100;
        List<TaskInfoRespVO> dataList = null;
        //分页查询数据
        while (true) {
            dataList = dataProducer.apply(page++, size);
            if (CollectionUtils.isEmpty(dataList)) {
                break;
            }
            //任务是否逾期计算处理逻辑
            disposeOverdue(dataList);
            if (dataList.size() < size) {
                //少于获取的数量，则说明已无数据
                break;
            }
        }
    }

    /**
     * 判断任务是否逾期计算处理逻辑
     *
     * @param respVOList 任务主表数据
     * @return
     */
    private void disposeOverdue(List<TaskInfoRespVO> respVOList) {
        //先更新任务主表为已逾期
        List<Long> ids = respVOList.stream().map(TaskInfoRespVO::getId).filter(Objects::nonNull).distinct().collect(Collectors.toList());
        taskInfoRepoProc.updateDelayFlagByIdBatch(UdcEnum.SALESMAN_TASK_DELAY_FLAG_Y.getValueCode(), ids);
        //所有明细的完成状态不是‘已完成’的均将是否逾期字段标记为已逾期，其余明细仍保持‘未逾期’
        TaskInfoDtlQueryVO dtlQueryVO = new TaskInfoDtlQueryVO();
        dtlQueryVO.setMasIds(ids);
        //明细的完成状态不是‘已完成’的
        dtlQueryVO.setNoCompleteState(UdcEnum.SALESMAN_TASK_COMPLETION_STATUS_CPD.getValueCode());
        List<TaskInfoDtlRespVO> dtlRespVOList = taskInfoDtlService.selectByParam(dtlQueryVO);
        if (!CollectionUtils.isEmpty(dtlRespVOList)) {
            List<Long> dtlIds = dtlRespVOList.stream().map(TaskInfoDtlRespVO::getId).filter(Objects::nonNull).distinct().collect(Collectors.toList());
            taskInfoDtlService.updateDelayFlagByIdBatch(UdcEnum.SALESMAN_TASK_DELAY_FLAG_Y.getValueCode(), dtlIds);
        }
    }

    /**
     * 任务重复设置保存
     *
     * @param saveVO 入参
     * @return 任务出参对象
     */
    @Override
    @Transactional(rollbackFor = {Exception.class})
    public TaskInfoRespVO repeatSetSave(TaskInfoSaveVO saveVO) {
        if (Objects.isNull(saveVO.getId())) {
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "任务ID为空!");
        }
        //校验任务重复设置
        checkRepeatSet(saveVO);
        //重复设置为是时
        if (Objects.equals(saveVO.getRepeatSet(), ConstantsSale.SALESMAN_TASK_REPEAT_SET_Y)) {
            switch (saveVO.getRepeatType()) {
                case ConstantsSale.SALESMAN_TASK_REPEAT_TYPE_TIMES:
                    repeatSetSaveTimes(saveVO);
                    break;
                case ConstantsSale.SALESMAN_TASK_REPEAT_TYPE_DAY:
                    repeatSetSaveDay(saveVO);
                    break;
                case ConstantsSale.SALESMAN_TASK_REPEAT_TYPE_WEEK:
                    repeatSetSaveWeek(saveVO);
                    break;
                case ConstantsSale.SALESMAN_TASK_REPEAT_TYPE_MONTH:
                    repeatSetSaveMonth(saveVO);
                    break;
            }
        } else {
            //重复设置为否时
            saveVO.setRepeatSet(ConstantsSale.SALESMAN_TASK_REPEAT_SET_N);
            saveVO.setRepeatType(null);
            saveVO.setRepeatInterval(null);
            saveVO.setAppointDay(null);
            saveVO.setRepeatStartTime(null);
            saveVO.setRepeatEndTime(null);
            saveVO.setExecutTime(null);
            updateRepeatSetById(saveVO);
        }

        TaskInfoRespVO taskInfoRespVO = new TaskInfoRespVO();
        taskInfoRespVO.setId(saveVO.getId());
        return taskInfoRespVO;
    }

    /**
     * 先保存重复设置的信息
     *
     * @param saveVO 任务主表信息
     * @return
     */
    private void repeatSetSaveTimes(TaskInfoSaveVO saveVO) {
        if (Objects.isNull(saveVO.getExecutTime())) {
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "重复设置的执行时间为空!");
        }
        saveVO.setAppointDay(null);
        saveVO.setRepeatStartTime(null);
        saveVO.setRepeatEndTime(null);
        //先保存重复设置的信息
        updateRepeatSetById(saveVO);
    }

    /**
     * 先保存重复设置的信息,勾选每天时，计算执行时间，并更新执行时间
     *
     * @param saveVO 任务主表信息
     * @return
     */
    private void repeatSetSaveDay(TaskInfoSaveVO saveVO) {
        if (Objects.isNull(saveVO.getRepeatInterval())) {
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "重复设置的重复间隔为空!");
        }
        if (Objects.isNull(saveVO.getRepeatStartTime())) {
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "重复设置的重复开始时间为空!");
        }
        if (Objects.isNull(saveVO.getRepeatEndTime())) {
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "重复设置的重复结束时间为空!");
        }
        saveVO.setAppointDay(null);
        saveVO.setExecutTime(null);
        //先保存重复设置的信息
        updateRepeatSetById(saveVO);
        //勾选每天时，计算执行时间，并更新执行时间
        disposeNextExecuteTimeDay(saveVO.getRepeatInterval(), saveVO.getRepeatStartTime(), saveVO.getRepeatEndTime(), saveVO.getExecutTime(), saveVO.getId());
    }

    /**
     * 先保存重复设置的信息,勾选每周时，计算执行时间，并更新执行时间
     *
     * @param saveVO 任务主表信息
     * @return
     */
    private void repeatSetSaveWeek(TaskInfoSaveVO saveVO) {
        if (Objects.isNull(saveVO.getRepeatInterval())) {
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "重复设置的重复间隔为空!");
        }
        if (StringUtils.isBlank(saveVO.getAppointDay())) {
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "重复设置的指定天为空!");
        }
        if (Objects.isNull(saveVO.getRepeatStartTime())) {
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "重复设置的重复开始时间为空!");
        }
        if (Objects.isNull(saveVO.getRepeatEndTime())) {
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "重复设置的重复结束时间为空!");
        }
        saveVO.setExecutTime(null);
        //先保存重复设置的信息
        updateRepeatSetById(saveVO);
        //勾选每周时，计算执行时间，并更新执行时间
        disposeNextExecuteTimeWeek(saveVO.getRepeatInterval(), saveVO.getAppointDay(), saveVO.getRepeatStartTime(), saveVO.getRepeatEndTime(), saveVO.getExecutTime(), saveVO.getId());
    }

    /**
     * 先保存重复设置的信息,勾选每月时，计算执行时间，并更新执行时间
     *
     * @param saveVO 任务主表信息
     * @return
     */
    private void repeatSetSaveMonth(TaskInfoSaveVO saveVO) {
        if (Objects.isNull(saveVO.getRepeatInterval())) {
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "重复设置的重复间隔为空!");
        }
        if (StringUtils.isBlank(saveVO.getAppointDay())) {
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "重复设置的指定天为空!");
        }
        if (Objects.isNull(saveVO.getRepeatStartTime())) {
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "重复设置的重复开始时间为空!");
        }
        if (Objects.isNull(saveVO.getRepeatEndTime())) {
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "重复设置的重复结束时间为空!");
        }
        saveVO.setExecutTime(null);
        //先保存重复设置的信息
        updateRepeatSetById(saveVO);
        //勾选每月时，计算执行时间，并更新执行时间
        disposeNextExecuteTimeMonth(saveVO.getRepeatInterval(), saveVO.getAppointDay(), saveVO.getRepeatStartTime(), saveVO.getRepeatEndTime(), saveVO.getExecutTime(), saveVO.getId());

    }

    /**
     * 任务重复设置-定时任务
     *
     * @param queryVO 入参
     * @return
     */
    @Override
    @Transactional(rollbackFor = {Exception.class})
    public void setTaskDuplication(TaskInfoQueryVO queryVO) {

        //1.查询重复设置的任务主表数据

        //查询重复设置的任务
        queryVO.setRepeatSet(ConstantsSale.SALESMAN_TASK_REPEAT_SET_Y);
        //查询非发布的任务
        List<String> stateList = Lists.newArrayList(
                UdcEnum.SALESMAN_TASK_STATUS_WEE.getValueCode(), UdcEnum.SALESMAN_TASK_STATUS_IPS.getValueCode(),
                UdcEnum.SALESMAN_TASK_STATUS_CPD.getValueCode(), UdcEnum.SALESMAN_TASK_STATUS_CCD.getValueCode(),
                UdcEnum.SALESMAN_TASK_STATUS_CSD.getValueCode()
        );
        queryVO.setStateList(stateList);
        //通过分页分批查询任务主表数据-不是分页列表查询
        selectTaskDuplicationSet(queryVO, paramVO -> this.selectPageByQueryVO(queryVO));
    }

    private void selectTaskDuplicationSet(TaskInfoQueryVO queryParam, Function<TaskInfoQueryVO, List<TaskInfoRespVO>> dataProducer) {
        disposeTaskDuplicationSet((page, pageSize) -> {
            queryParam.setCurrent(page);
            queryParam.setSize(pageSize);
            return dataProducer.apply(queryParam);
        });
    }

    private void disposeTaskDuplicationSet(BiFunction<Integer, Integer, List<TaskInfoRespVO>> dataProducer) {
        int page = 1;
        int size = 100;
        List<TaskInfoRespVO> dataList = null;
        //分页查询数据
        while (true) {
            dataList = dataProducer.apply(page++, size);
            if (CollectionUtils.isEmpty(dataList)) {
                break;
            }
            //判断任务是否重复设置并进行处理
            disposeDuplicationSet(dataList);
            if (dataList.size() < size) {
                //少于获取的数量，则说明已无数据
                break;
            }
        }
    }

    /**
     * 判断任务是否重复设置并进行处理
     *
     * @param respVOList 任务主表数据
     * @return
     */
    private void disposeDuplicationSet(List<TaskInfoRespVO> respVOList) {

        /**
         * 根据设置的重复间隔和重复开始时间计算，第一次定时任务执行的时间，记录在执行时间字段中，
         * 后面每次定时任务执行完后，自动计算下一次的执行时间更新覆盖在执行时间字段中（
         * 需要判断计算的下一次执行时间如果超过了重复的结束时间，则不更新数据）；
         * 定时任务根据任务表中的执行时间自动生成需要重复的任务，然后开始计算下一次的执行时间判断更新覆盖；
         * 生成任务的内容：复制原始任务的基本信息+明细信息；主表和明细的状态均为‘待发布’；
         * 执行模版取原始任务上的执行模版，如果执行模版为禁用状态的，则为空；
         * 新任务的设置任务重复字段默认为不重复；
         * 新任务的开始时间为执行时间、结束时间根据原始任务的时间跨度计算得出。。
         */
        for (TaskInfoRespVO respVO : respVOList) {
            log.info("任务重复设置执行,时间：{},信息：{}", LocalDateTime.now(), respVO);
            switch (respVO.getRepeatType()) {
                case ConstantsSale.SALESMAN_TASK_REPEAT_TYPE_TIMES:
                    disposeDuplicationSetTimes(respVO);
                    break;
                case ConstantsSale.SALESMAN_TASK_REPEAT_TYPE_DAY:
                    disposeDuplicationSetDay(respVO);
                    break;
                case ConstantsSale.SALESMAN_TASK_REPEAT_TYPE_WEEK:
                    disposeDuplicationSetWeek(respVO);
                    break;
                case ConstantsSale.SALESMAN_TASK_REPEAT_TYPE_MONTH:
                    disposeDuplicationSetMonth(respVO);
                    break;
            }
        }
    }

    /**
     * 勾选一次时，设定执行时间，定时任务根据执行时间生成任务；
     *
     * @param respVO 任务主表信息
     * @return
     */
    private void disposeDuplicationSetTimes(TaskInfoRespVO respVO) {
        //目前次数时,只有一次
        if (!Objects.isNull(respVO.getExecutTime()) && LocalDate.now().isEqual(respVO.getExecutTime().toLocalDate())) {
           /* 生成任务的内容：复制原始任务的基本信息+明细信息；主表和明细的状态均为‘待发布’；
            执行模版取原始任务上的执行模版，如果执行模版为禁用状态的，则为空；
            新任务的设置任务重复字段默认为不重复；
            新任务的开始时间为执行时间、结束时间根据原始任务的时间跨度计算得出。。*/
            copyTaskInfoSave(respVO);
        }
    }

    /**
     * 勾选每天时，固定为每隔0天（即为每天重复），可修改；第一次的执行时间就根据重复开始时间+每隔天数进行计算，
     * 后面就根据上一次的执行时间+每隔天数进行计算；
     *
     * @param respVO 任务主表信息
     * @return
     */
    private void disposeDuplicationSetDay(TaskInfoRespVO respVO) {
        //1.先执行生成新任务
        //当执行时间不为空,并且执行时间是当前日期时,执行生成新任务逻辑
        if (!Objects.isNull(respVO.getExecutTime()) && LocalDate.now().isEqual(respVO.getExecutTime().toLocalDate())) {
           /* 生成任务的内容：复制原始任务的基本信息+明细信息；主表和明细的状态均为‘待发布’；
            执行模版取原始任务上的执行模版，如果执行模版为禁用状态的，则为空；
            新任务的设置任务重复字段默认为不重复；
            新任务的开始时间为执行时间、结束时间根据原始任务的时间跨度计算得出。。*/
            copyTaskInfoSave(respVO);
        }
        //2.再计算下次执行时间
        //重复间隔
        BigDecimal repeatInterval = respVO.getRepeatInterval();
        //重复开始时间
        LocalDateTime repeatStartTime = respVO.getRepeatStartTime();
        //重复结束时间
        LocalDateTime repeatEndTime = respVO.getRepeatEndTime();
        //执行时间
        LocalDateTime executTime = respVO.getExecutTime();

        if (!Objects.isNull(repeatInterval) && !Objects.isNull(repeatStartTime) && !Objects.isNull(repeatEndTime)) {
            //勾选每天时，计算下次执行时间，并更新执行时间
            disposeNextExecuteTimeDay(repeatInterval, repeatStartTime, repeatEndTime, executTime, respVO.getId());
        } else {
            log.error("每天时任务重复设置执行,时间：{},信息：{}", LocalDateTime.now(), respVO);

        }

    }

    /**
     * 勾选每天时，计算下次执行时间，并更新执行时间
     *
     * @param repeatInterval  重复间隔
     * @param repeatStartTime 重复开始时间
     * @param repeatEndTime   重复结束时间
     * @param executTime      执行时间
     * @param id              任务ID
     * @return
     */
    private void disposeNextExecuteTimeDay(BigDecimal repeatInterval, LocalDateTime repeatStartTime,
                                           LocalDateTime repeatEndTime, LocalDateTime executTime, Long id) {
        //执行时间为空时则默认为第一次,基准时间取重复开始时间,否则取执行时间
        //基准时间
        LocalDateTime referenceTime = Objects.isNull(executTime) ? repeatStartTime : executTime;
        //当重复间隔为0时,表示每天重复,那么其执行时间为下一天
        if (repeatInterval.compareTo(BigDecimal.ONE) == -1) {
            if(Objects.isNull(executTime)) {
                executTime = judgmentExecuteTime(repeatStartTime);
            }else {
                executTime = referenceTime.plusDays(1);
            }
        } else {
            //执行时间就根据基准时间+每隔天数进行计算，
            long afterDays = repeatInterval.longValue();
            executTime = referenceTime.plusDays(afterDays);
        }
        //3.需要判断计算的下一次执行时间如果超过了重复的结束时间，则不更新数据
        if (executTime.isBefore(repeatEndTime)) {
            //更新覆盖在执行时间字段
            updateExecutTimeById(executTime, id);
        }
    }

    /**
     *举例辅助逻辑：
     *1.假如7.6新建，开始时间为7.6，那么执行时间为开始时间+1变为7.7号
     *2.假如7.5新建，开始时间为7.6，那么执行时间为开始时间为7.6
     *3.假如7.6新建，开始时间为7.1，那么执行时间为当前时间+1变为7.7号
     *
     * @param localDateTime 执行时间
     * @return
     */
    private LocalDateTime judgmentExecuteTime(LocalDateTime localDateTime){
       LocalDate localDate =  LocalDate.now();
       LocalDate executeDate = localDateTime.toLocalDate();
       LocalTime executeTime = localDateTime.toLocalTime();
        LocalDateTime newDateTime = localDateTime;
       if (localDate.isEqual(executeDate)){
           newDateTime = LocalDateTime.of(executeDate,executeTime).plusDays(1);
       }else if (localDate.isBefore(executeDate)){
           newDateTime = localDateTime;
       }else if (localDate.isAfter(executeDate)){
           newDateTime = LocalDateTime.of(localDate,executeTime).plusDays(1);
       }

       return newDateTime;
    }

    /**
     * 勾选每周时，固定为每隔0周（即为每周重复），可修改；第一次的执行时间就根据重复开始时间+每隔周数进行计算，
     * 后面就根据上一次的执行时间+每隔周数进行计算；
     * 那如果指定了周几重复，则第一次的执行时间就根据重复开始时间+每隔周数+制定星期进行计算；
     *
     * @param respVO 任务主表信息
     * @return
     */
    private void disposeDuplicationSetWeek(TaskInfoRespVO respVO) {
        //1.先执行生成新任务
        //当执行时间不为空,并且执行时间是当前日期时,执行生成新任务逻辑
        if (!Objects.isNull(respVO.getExecutTime()) && LocalDate.now().isEqual(respVO.getExecutTime().toLocalDate())) {
           /* 生成任务的内容：复制原始任务的基本信息+明细信息；主表和明细的状态均为‘待发布’；
            执行模版取原始任务上的执行模版，如果执行模版为禁用状态的，则为空；
            新任务的设置任务重复字段默认为不重复；
            新任务的开始时间为执行时间、结束时间根据原始任务的时间跨度计算得出。。*/
            copyTaskInfoSave(respVO);
        }
        //2.再计算下次执行时间
        //重复间隔
        BigDecimal repeatInterval = respVO.getRepeatInterval();
        //间隔指定值
        String appointDay = respVO.getAppointDay();
        //重复开始时间
        LocalDateTime repeatStartTime = respVO.getRepeatStartTime();
        //重复结束时间
        LocalDateTime repeatEndTime = respVO.getRepeatEndTime();
        //执行时间
        LocalDateTime executTime = respVO.getExecutTime();

        if (!Objects.isNull(repeatInterval) && StringUtils.isNotBlank(appointDay) && !Objects.isNull(repeatStartTime) && !Objects.isNull(repeatEndTime)) {
            //勾选每周时，计算下次执行时间，并更新执行时间
            disposeNextExecuteTimeWeek(repeatInterval, appointDay, repeatStartTime, repeatEndTime, executTime, respVO.getId());
        } else {
            log.error("每周时任务重复设置执行,时间：{},信息：{}", LocalDateTime.now(), respVO);

        }


       /* //每周重复或每月重复时，第一次执行时间为重复开始时间？可以试一下往后推0周
        LocalDate nextDays = LocalDate.now().plus(1, ChronoUnit.DAYS);
        LocalDate nextDays2 = LocalDate.now().plus(1, ChronoUnit.WEEKS);
        LocalDate nextDays3 = LocalDate.now().plus(1, ChronoUnit.MONTHS);
        //获取周一
        LocalDate week = nextDays2.with(TemporalAdjusters.previousOrSame(DayOfWeek.MONDAY));
        //获取月的指定几号
        LocalDate nextDays4 = nextDays3.withDayOfMonth(Integer.valueOf(1));*/


    }

    /**
     * 勾选每周时，计算下次执行时间，并更新执行时间
     *
     * @param repeatInterval  重复间隔
     * @param appointDay      间隔指定值
     * @param repeatStartTime 重复开始时间
     * @param repeatEndTime   重复结束时间
     * @param executTime      执行时间
     * @param id              任务ID
     * @return
     */
    private void disposeNextExecuteTimeWeek(BigDecimal repeatInterval, String appointDay, LocalDateTime repeatStartTime,
                                            LocalDateTime repeatEndTime, LocalDateTime executTime, Long id) {
        //执行时间为空时则默认为第一次,基准时间取重复开始时间,否则取执行时间
        //基准时间
        LocalDateTime referenceTime = Objects.isNull(executTime) ? repeatStartTime : executTime;
        //当重复间隔为0时,表示每周重复,那么其执行时间为当周
        if (repeatInterval.compareTo(BigDecimal.ONE) == -1) {
            executTime = getAppointDayOfWeek(appointDay, referenceTime);
        } else {
            //执行时间就根据基准时间+每隔周数+制定星期进行计算
            long afterWeek = repeatInterval.longValue();
            //先推出每隔周数后的时间，再推指定周几的时间
            LocalDateTime weekTime = referenceTime.plus(afterWeek, ChronoUnit.WEEKS);
            executTime = getAppointDayOfWeek(appointDay, weekTime);
        }
        //3.需要判断计算的下一次执行时间如果超过了重复的结束时间，则不更新数据
        if (!Objects.isNull(executTime) && executTime.isBefore(repeatEndTime)) {
            //更新覆盖在执行时间字段
            updateExecutTimeById(executTime, id);
        }

    }

    /**
     * 根据计算后的周时间、指定的周几，再重新计算时间
     *
     * @param appointDay 间隔指定值
     * @param weekTime   根据重复间隔处理后的时间
     * @return
     */
    private LocalDateTime getAppointDayOfWeek(String appointDay, LocalDateTime weekTime) {
        LocalDateTime localDateTime = null;
        if (Objects.equals(appointDay, UdcEnum.TASK_REPEAT_SET_WEEK_0.getValueCode())) {
            localDateTime = weekTime;
        } else if (Objects.equals(appointDay, UdcEnum.TASK_REPEAT_SET_WEEK_1.getValueCode())) {
            //获取周一
            localDateTime = weekTime.with(TemporalAdjusters.nextOrSame(DayOfWeek.MONDAY));
        } else if (Objects.equals(appointDay, UdcEnum.TASK_REPEAT_SET_WEEK_2.getValueCode())) {
            localDateTime = weekTime.with(TemporalAdjusters.nextOrSame(DayOfWeek.TUESDAY));
        } else if (Objects.equals(appointDay, UdcEnum.TASK_REPEAT_SET_WEEK_3.getValueCode())) {
            localDateTime = weekTime.with(TemporalAdjusters.nextOrSame(DayOfWeek.WEDNESDAY));
        } else if (Objects.equals(appointDay, UdcEnum.TASK_REPEAT_SET_WEEK_4.getValueCode())) {
            localDateTime = weekTime.with(TemporalAdjusters.nextOrSame(DayOfWeek.THURSDAY));
        } else if (Objects.equals(appointDay, UdcEnum.TASK_REPEAT_SET_WEEK_5.getValueCode())) {
            localDateTime = weekTime.with(TemporalAdjusters.nextOrSame(DayOfWeek.FRIDAY));
        } else if (Objects.equals(appointDay, UdcEnum.TASK_REPEAT_SET_WEEK_6.getValueCode())) {
            localDateTime = weekTime.with(TemporalAdjusters.nextOrSame(DayOfWeek.SATURDAY));
        } else if (Objects.equals(appointDay, UdcEnum.TASK_REPEAT_SET_WEEK_7.getValueCode())) {
            localDateTime = weekTime.with(TemporalAdjusters.nextOrSame(DayOfWeek.SUNDAY));
        }

        return localDateTime;
    }

    /**
     * 勾选每月时，固定为每隔0月（即为每月重复），可修改；第一次的执行时间就根据重复开始时间+每隔月数进行计算，
     * 后面就根据上一次的执行时间+每隔月数进行计算；
     * 那如果指定了日期重复，则第一次的执行时间就根据重复开始时间+每隔周数+制定日期进行计算；
     *
     * @param respVO 任务主表信息
     * @return
     */
    private void disposeDuplicationSetMonth(TaskInfoRespVO respVO) {
        //1.先执行生成新任务
        //当执行时间不为空,并且执行时间是当前日期时,执行生成新任务逻辑
        if (!Objects.isNull(respVO.getExecutTime()) && LocalDate.now().isEqual(respVO.getExecutTime().toLocalDate())) {
           /* 生成任务的内容：复制原始任务的基本信息+明细信息；主表和明细的状态均为‘待发布’；
            执行模版取原始任务上的执行模版，如果执行模版为禁用状态的，则为空；
            新任务的设置任务重复字段默认为不重复；
            新任务的开始时间为执行时间、结束时间根据原始任务的时间跨度计算得出。。*/
            copyTaskInfoSave(respVO);
        }
        //2.再计算下次执行时间
        //重复间隔
        BigDecimal repeatInterval = respVO.getRepeatInterval();
        //间隔指定值
        String appointDay = respVO.getAppointDay();
        //重复开始时间
        LocalDateTime repeatStartTime = respVO.getRepeatStartTime();
        //重复结束时间
        LocalDateTime repeatEndTime = respVO.getRepeatEndTime();
        //执行时间
        LocalDateTime executTime = respVO.getExecutTime();

        if (!Objects.isNull(repeatInterval) && StringUtils.isNotBlank(appointDay) && !Objects.isNull(repeatStartTime) && !Objects.isNull(repeatEndTime)) {
            //勾选每月时，计算下次执行时间，并更新执行时间
            disposeNextExecuteTimeMonth(repeatInterval, appointDay, repeatStartTime, repeatEndTime, executTime, respVO.getId());
        } else {
            log.error("每月时任务重复设置执行,时间：{},信息：{}", LocalDateTime.now(), respVO);

        }


    }

    /**
     * 勾选每月时，计算下次执行时间，并更新执行时间
     *
     * @param repeatInterval  重复间隔
     * @param appointDay      间隔指定值
     * @param repeatStartTime 重复开始时间
     * @param repeatEndTime   重复结束时间
     * @param executTime      执行时间
     * @param id              任务ID
     * @return
     */
    private void disposeNextExecuteTimeMonth(BigDecimal repeatInterval, String appointDay, LocalDateTime repeatStartTime,
                                             LocalDateTime repeatEndTime, LocalDateTime executTime, Long id) {
        //执行时间为空时则默认为第一次,基准时间取重复开始时间,否则取执行时间
        //基准时间
        LocalDateTime referenceTime = Objects.isNull(executTime) ? repeatStartTime : executTime;
        //当重复间隔为0时,表示每月重复,那么其执行时间为当月
        if (repeatInterval.compareTo(BigDecimal.ONE) == -1) {
            executTime = getAppointDayOfMonth(appointDay, referenceTime);
        } else {
            //执行时间就根据基准时间+每隔月数+制定几号进行计算
            long afterMonth = repeatInterval.longValue();
            //先推出每隔月数后的时间，再推指定几号的时间
            LocalDateTime monthTime = referenceTime.plus(afterMonth, ChronoUnit.MONTHS);
            executTime = getAppointDayOfMonth(appointDay, monthTime);
        }
        //3.需要判断计算的下一次执行时间如果超过了重复的结束时间，则不更新数据
        if (!Objects.isNull(executTime) && executTime.isBefore(repeatEndTime)) {
            //更新覆盖在执行时间字段
            updateExecutTimeById(executTime, id);
        }
    }

    /**
     * 根据计算后的月时间、指定的几号，再重新计算时间
     *
     * @param appointDay 间隔指定值
     * @param monthTime  根据重复间隔处理后的时间
     * @return
     */
    private LocalDateTime getAppointDayOfMonth(String appointDay, LocalDateTime monthTime) {
        LocalDateTime localDateTime = null;
        if (Objects.equals(appointDay, UdcEnum.TASK_REPEAT_SET_MONTH_0.getValueCode())) {
            localDateTime = monthTime;
        } else if (!Objects.isNull(appointDay)) {
            Integer appointDayOfMonth = Integer.valueOf(appointDay);

            //这个月有多少天
            Integer monthLength = monthTime.toLocalDate().lengthOfMonth();
            if (appointDayOfMonth > monthLength) {
                //当指定的几号值大于所在月份最大天数时默认为最后一天
                //取本月最后一天，再也不用计算是28，29，30还是31
                LocalDateTime lastDayOfThisMonth = monthTime.with(TemporalAdjusters.lastDayOfMonth());
                localDateTime = lastDayOfThisMonth;
            } else {
                //获取月的指定几号
                localDateTime = monthTime.withDayOfMonth(appointDayOfMonth);
            }
        }

        return localDateTime;
    }

    /**
     * 复制原始任务生成新任务
     * <p>
     * 生成任务的内容：复制原始任务的基本信息+明细信息；主表和明细的状态均为‘待发布’；
     * 执行模版取原始任务上的执行模版，如果执行模版为禁用状态的，则为空；
     * 新任务的设置任务重复字段默认为不重复；
     * 新任务的开始时间为执行时间、结束时间根据原始任务的时间跨度计算得出
     *
     * @param respVO 任务主表信息
     * @return
     */
    private void copyTaskInfoSave(TaskInfoRespVO respVO) {
        TaskInfoSaveVO saveVO = TaskInfoConvert.INSTANCE.respVoToSaveVO(respVO);
        //保存,待发布
        saveVO.setId(null);
        saveVO.setReleaseSign(false);
        saveVO.setCompleteTime(null);
        //附件编码字符串转换为编码集合,以逗号分隔的字符串形式
        List<String> fileCodes = transitionStrToCodes(respVO.getFileInfo());
        saveVO.setFileCodes(fileCodes);
        //执行模版取原始任务上的执行模版，如果执行模版为禁用状态的，则为空
        //查询执行模板状态
        if (!Objects.isNull(saveVO.getExecutTemplateCode())) {
            List<ExectRecordTempRespVO> recordTempList = exectRecordTempService.queryByCodes(Collections.singletonList(saveVO.getExecutTemplateCode()));
            if (!CollectionUtils.isEmpty(recordTempList)) {
                ExectRecordTempRespVO recordTempRespVO = recordTempList.get(0);
                if (Objects.nonNull(recordTempRespVO)
                        && Objects.equals(recordTempRespVO.getState(), ConstantsSale.EXECTE_RECORD_TEMP_STATE_1)) {
                    saveVO.setExecutTemplateCode(null);
                }
            }
        }

        //新任务的设置任务重复字段默认为不重复
        saveVO.setRepeatSet(ConstantsSale.SALESMAN_TASK_REPEAT_SET_N);
        saveVO.setRepeatType(null);
        saveVO.setRepeatInterval(null);
        saveVO.setAppointDay(null);
        saveVO.setRepeatStartTime(null);
        saveVO.setRepeatEndTime(null);
        saveVO.setExecutTime(null);
        //新任务的开始时间为执行时间、结束时间根据原始任务的时间跨度计算得出
        saveVO.setStartTime(LocalDateTime.of(respVO.getExecutTime().toLocalDate(),
                LocalTime.MIN));
        //计算两个日期相差多少天:
        long differ = getChronoUnitBetween(respVO.getStartTime(), respVO.getEndTime(), ChronoUnit.DAYS);
        LocalDateTime endTime = respVO.getExecutTime().plusDays(differ);
        saveVO.setEndTime(LocalDateTime.of(endTime.toLocalDate(),ConstantsSale.LOCAL_TIME_MAX));

        saveVO.setCreateTime(LocalDateTime.now());
        saveVO.setModifyTime(LocalDateTime.now());

        //明细
        List<TaskInfoDtlRespVO> dtlRespVOList = taskInfoDtlService.selectByMasId(respVO.getId());
        List<TaskInfoDtlSaveVO> dtlSaveVOList = dtlRespVOList.stream().map(TaskInfoDtlConvert.INSTANCE::respVoToSaveVO).collect(Collectors.toList());
        dtlSaveVOList.forEach(dtlSaveVO -> {
            //明细的关联业务编码数据库里以逗号分隔的字符串形式保存
            //目前一行任务明细的业务编码只有一个,前端传businessCode，不使用businessCodes了。
            //List<String> businessCodes = transitionStrToCodes(dtlSaveVO.getBusinessCode());
            //dtlSaveVO.setBusinessCodes(businessCodes);
            dtlSaveVO.setId(null);
            dtlSaveVO.setMasId(null);
            dtlSaveVO.setCompleteTime(null);
            dtlSaveVO.setExecutRecordId(null);
            dtlSaveVO.setExecutRecordCode(null);
            dtlSaveVO.setExecutRecordName(null);
            dtlSaveVO.setCreateTime(LocalDateTime.now());
            dtlSaveVO.setModifyTime(LocalDateTime.now());
        });

        saveVO.setDtlSaveVOList(dtlSaveVOList);

        //附件信息
        List<FileInfoRespVO> fileInfoRespVOList = selectFileBySourceId(respVO.getId());
        if (!CollectionUtils.isEmpty(fileInfoRespVOList)) {
            List<FileInfoSaveVO> fileInfoSaveVOList = fileInfoRespVOList.stream().map(FileinfoConvert.INSTANCE::respVoToSaveVO).collect(Collectors.toList());
            fileInfoSaveVOList.forEach(fileInfoSaveVO -> {
                fileInfoSaveVO.setSourceId(null);
                fileInfoSaveVO.setId(null);
            });

            saveVO.setFileInfoSaveVOS(fileInfoSaveVOList);
        }

        //调用新增
        createTask(saveVO);
    }

    /**
     * 根据ChronoUnit计算两个日期时间之间相隔日期时间
     *
     * @param start      开始日期时间
     * @param end        结束日期时间
     * @param chronoUnit 日期时间单位
     * @return long 相隔日期时间
     */
    public static long getChronoUnitBetween(LocalDateTime start, LocalDateTime end, ChronoUnit chronoUnit) {
        return Math.abs(start.until(end, chronoUnit));
    }

    /**
     * 根据任务ID更新执行时间
     *
     * @param executTime 执行时间
     * @param id         任务ID
     * @return
     */
    private void updateExecutTimeById(LocalDateTime executTime, Long id) {
        taskInfoRepoProc.updateExecutTimeById(executTime, id);
    }

    /**
     * 根据任务ID更新重复设置信息
     *
     * @param saveVO 入参
     * @return
     */
    private void updateRepeatSetById(TaskInfoSaveVO saveVO) {
        taskInfoRepoProc.updateRepeatSetById(saveVO);
    }

    /**
     * 根据任务ID更新任务状态
     *
     * @param state 任务状态
     * @param id    任务ID
     * @return
     */
    @Override
    @Transactional(rollbackFor = {Exception.class})
    public void updateStateById(String state, Long id) {
        taskInfoRepoProc.updateStateById(state, id);
    }

    /**
     * 根据任务ID更新任务状态、完成时间
     *
     * @param state        任务状态
     * @param completeTime 完成时间
     * @param id           任务ID
     * @return
     */
    @Override
    @Transactional(rollbackFor = {Exception.class})
    public void updateStateAndCompleteTime(String state, LocalDateTime completeTime, Long id) {
        taskInfoRepoProc.updateStateAndCompleteTimeById(state, completeTime, id);
    }

    /**
     * 根据任务ID更新任务进度
     *
     * @param progress 任务进度
     * @param id       任务ID
     * @return
     */
    @Override
    @Transactional(rollbackFor = {Exception.class})
    public void updateProgressById(BigDecimal progress, Long id) {
        taskInfoRepoProc.updateProgressById(progress, id);
    }

    /**
     * 获取当前用户
     *
     * @param
     * @return 当前用户信息
     */
    private SysUserDTO getSysUser() {
        //获取当前用户
        GeneralUserDetails userDetails = SecurityContextUtil.currentUser();
        if (Objects.isNull(userDetails) || Objects.isNull(userDetails.getUser())) {
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "获取当前用户信息失败");
        }
        SysUserDTO sysUser = userDetails.getUser();
        if (ObjectUtils.isEmpty(sysUser)) {
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "用户信息获取失败");
        }

        return sysUser;
    }

    /**
     * 编码集合转换为字符串,以逗号分隔的字符串形式
     *
     * @param codeList 编码集合
     * @return 以逗号分隔的字符串
     */
    private String transitionCodesToStr(List<String> codeList) {
        if (CollectionUtils.isEmpty(codeList)) {
            return null;
        }
        String codeString = codeList.stream().distinct().collect(Collectors.joining(","));

        return codeString;
    }

    /**
     * 字符串转换为编码集合,以逗号分隔的字符串形式
     *
     * @param codeString 以逗号分隔的字符串
     * @return 编码集合
     */
    private List<String> transitionStrToCodes(String codeString) {
        if (StringUtils.isEmpty(codeString)) {
            return Collections.emptyList();
        }
        List<String> codeList = Arrays.asList(codeString.split(","));

        return codeList;
    }


    /**
     * 任务导出分页查询-主表和明细
     *
     * @param pageParam 入参
     * @return 主表和明细信息集合
     */
    @Override
    @SysCodeProc
    public PagingVO<TaskInfoExportRespVO> exportPage(TaskInfoQueryVO pageParam) {

        PagingVO<TaskInfoExportRespVO> pagingVO = taskInfoRepoProc.exportPage(pageParam);
        if (CollectionUtils.isEmpty(pagingVO.getRecords())) {
            return PagingVO.<TaskInfoExportRespVO>builder().total(0L).records(Collections.EMPTY_LIST).build();
        }
        List<TaskInfoExportRespVO> respVOList = pagingVO.getRecords();
        //组装填充任务的相关关联字段信息
        exportTranslateTask(respVOList);
        //组装填充任务明细的相关关联字段信息
        exportTranslateTaskDtl(respVOList);
        //组装处理行政区域信息
        exportTranslateAreaDtl(respVOList);

        return PagingVO.<TaskInfoExportRespVO>builder()
                .total(pagingVO.getTotal())
                .records(respVOList)
                .build();
    }

    @Override
    public PagingVO<TaskInfoDtlRespVO> findPage(Long id) {
        TaskInfoDtlQueryVO param = new TaskInfoDtlQueryVO();
        param.setId(id);
        PagingVO<TaskInfoDtlRespVO> page = taskInfoDtlService.page(param);
        return page;
    }

    /**
     * 导出组装填充任务的相关关联字段信息
     *
     * @param respVOList 主表信息集合
     * @return
     */
    private void exportTranslateTask(List<TaskInfoExportRespVO> respVOList) {
        //公司-编码
        List<String> ouCodeList = respVOList.stream().map(TaskInfoExportRespVO::getOuCode).filter(Objects::nonNull).distinct().collect(Collectors.toList());
        List<OrgOuRpcDTO> ouRpcDTOList = rmiOrgOuService.findOuDtoListByOuCodes(ouCodeList);
        //发布人-ID
        Set<Long> publishUserIdList = respVOList.stream().map(TaskInfoExportRespVO::getPublishUserId).filter(Objects::nonNull).distinct().collect(Collectors.toSet());
        List<SysEmployeeBasicDTO> employeeList = rmiEmployeeRpcService.findEmployeeByIds(publishUserIdList);
        //执行模板-编码关联执行模板名称
        List<String> templateCodeList = respVOList.stream().map(TaskInfoExportRespVO::getExecutTemplateCode).filter(Objects::nonNull).distinct().collect(Collectors.toList());
        List<ExectRecordTempRespVO> recordTempVOList = exectRecordTempService.queryByCodes(templateCodeList);

        respVOList.forEach(respVO -> {
            //拼接任务进度展示
            String progressStr = String.valueOf(respVO.getProgress());
            respVO.setProgressStr(progressStr + "%");
            val ouRpcDTOOptional = ouRpcDTOList.stream().filter(ouRpcDTO -> Objects.equals(ouRpcDTO.getOuCode(), respVO.getOuCode())).findFirst();
            ouRpcDTOOptional.ifPresent(ouRpcDTO -> respVO.setOuName(ouRpcDTO.getOuName()));
            val employeeDTOOptional = employeeList.stream().filter(employeeDTO -> Objects.equals(employeeDTO.getId(), respVO.getPublishUserId())).findFirst();
            employeeDTOOptional.ifPresent(employeeDTO -> respVO.setPublishUser(employeeDTO.getFullName()));
            //执行模板
            if (!CollectionUtils.isEmpty(recordTempVOList)) {
                val recordTempOptional = recordTempVOList.stream().filter(tempRespVO -> Objects.equals(tempRespVO.getTempCode(), respVO.getExecutTemplateCode())).findFirst();
                recordTempOptional.ifPresent(tempRespVO -> respVO.setExecutTemplateName(tempRespVO.getTempName()));
            }

        });

    }

    /**
     * 导出组装填充任务明细的相关关联字段信息
     *
     * @param dtlRespVOList 任务明细数据信息
     * @return
     */
    private void exportTranslateTaskDtl(List<TaskInfoExportRespVO> dtlRespVOList) {
        //所属客户编码
        List<String> dealerCodeList = dtlRespVOList.stream().map(TaskInfoExportRespVO::getDealerCode).filter(Objects::nonNull).distinct().collect(Collectors.toList());
        List<LmSaveCustRespVO> dealerList = crmCustService.getCustInfoByCustCodes(dealerCodeList);
        //执行人编码-业务员-业务员表没有名称，要通过业务员编码查询员工的名称,业务员编码和员工编码是相同的
        Set<String> executeUserCodeList = dtlRespVOList.stream().map(TaskInfoExportRespVO::getExecutUserCode).filter(Objects::nonNull).collect(Collectors.toSet());
        List<SysEmployeeBasicDTO> executeUserList = rmiEmployeeRpcService.findEmployeeByCodes(executeUserCodeList);
        //执行记录-编码,执行记录没有名称字段
        //查询不同业务类型的关联业务编码的信息
        List<OrgStoreDetailRpcDTO> storeList = new ArrayList<>();
        List<LmSaveCustRespVO> custList = new ArrayList<>();
        List<SysEmployeeBasicDTO> employeeList = new ArrayList<>();
        Map<String, List<TaskInfoExportRespVO>> businessTypeListMap = dtlRespVOList.stream().filter(dtlRespVO -> StringUtils.isNotBlank(dtlRespVO.getBusinessType())).collect(Collectors.groupingBy(TaskInfoExportRespVO::getBusinessType));
        for (String key : businessTypeListMap.keySet()) {
            List<String> businessCodes = businessTypeListMap.get(key).stream().map(TaskInfoExportRespVO::getBusinessCode).filter(Objects::nonNull).distinct().collect(Collectors.toList());
            //关联业务-门店
            if (Objects.equals(key, UdcEnum.SALESMAN_TASK_BUSINESS_TYPE_STORE.getValueCode())) {
                storeList = rmiOrgStoreRpcService.findStoreByStoreCodes(businessCodes);
            } else if (Objects.equals(key, UdcEnum.SALESMAN_TASK_BUSINESS_TYPE_DEALER.getValueCode())) {
                custList = crmCustService.getCustInfoByCustCodes(businessCodes);
            } else if (Objects.equals(key, UdcEnum.SALESMAN_TASK_BUSINESS_TYPE_SALESMAN.getValueCode())) {
                Set<String> businessCodeSet = businessCodes.stream().collect(Collectors.toSet());
                employeeList = rmiEmployeeRpcService.findEmployeeByCodes(businessCodeSet);
            }
        }

        for (TaskInfoExportRespVO respVO : dtlRespVOList) {
            //关联业务-门店
            if (Objects.equals(respVO.getBusinessType(), UdcEnum.SALESMAN_TASK_BUSINESS_TYPE_STORE.getValueCode())) {
                val storeOptional = storeList.stream().filter(storeVO -> Objects.equals(storeVO.getStoreCode(), respVO.getBusinessCode())).findFirst();
                storeOptional.ifPresent(storeVO -> {
                    respVO.setBusinessName(storeVO.getStoreName());
                    respVO.setContactName(storeVO.getStoreManager());
                    respVO.setContactPhone(storeVO.getStoreContPhone());
                });
            }
            //关联业务-经销商
            if (Objects.equals(respVO.getBusinessType(), UdcEnum.SALESMAN_TASK_BUSINESS_TYPE_DEALER.getValueCode())) {
                val custOptional = custList.stream().filter(custVO -> Objects.equals(custVO.getCustCode(), respVO.getBusinessCode())).findFirst();
                custOptional.ifPresent(custVO -> {
                    respVO.setBusinessName(custVO.getCustName());
                    respVO.setCustCode2(custVO.getCustCode2());
                    respVO.setContactName(custVO.getContactName());
                    respVO.setContactPhone(custVO.getContactPhone());
                });
            }
            //关联业务-业务员
            if (Objects.equals(respVO.getBusinessType(), UdcEnum.SALESMAN_TASK_BUSINESS_TYPE_SALESMAN.getValueCode())) {
                val employeeOptional = employeeList.stream().filter(employeeVO -> Objects.equals(employeeVO.getCode(), respVO.getBusinessCode())).findFirst();
                employeeOptional.ifPresent(employeeVO -> {
                    respVO.setBusinessName(employeeVO.getFullName());
                    respVO.setContactName(employeeVO.getFullName());
                    respVO.setContactPhone(employeeVO.getPhone());
                });
            }
            //所属客户
            val dealerOptional = dealerList.stream().filter(dealerVO -> Objects.equals(dealerVO.getCustCode(), respVO.getDealerCode())).findFirst();
            dealerOptional.ifPresent(dealerVO -> respVO.setDealerName(dealerVO.getCustName()));
            //执行人
            val executeUserOptional = executeUserList.stream().filter(executeUser -> Objects.equals(executeUser.getCode(), respVO.getExecutUserCode())).findFirst();
            executeUserOptional.ifPresent(executeUser -> respVO.setExecutUser(executeUser.getFullName()));

        }
    }

    /**
     * 组装处理行政区域信息
     *
     * @param dtlRespVOList 任务明细数据信息
     * @return
     */
    private void exportTranslateAreaDtl(List<TaskInfoExportRespVO> dtlRespVOList) {
        //省市
        Set<String> provinces = dtlRespVOList.stream().map(TaskInfoExportRespVO::getProvince).distinct().filter(Objects::nonNull).collect(Collectors.toSet());
        //市
        Set<String> cities = dtlRespVOList.stream().map(TaskInfoExportRespVO::getCity).distinct().filter(Objects::nonNull).collect(Collectors.toSet());
        //区县
        Set<String> counties = dtlRespVOList.stream().map(TaskInfoExportRespVO::getDistrict).distinct().filter(Objects::nonNull).collect(Collectors.toSet());
        Set<String> areaCodes = new HashSet<>();
        areaCodes.addAll(provinces);
        areaCodes.addAll(cities);
        areaCodes.addAll(counties);
        List<SysAreaRespDTO> areaList = rmiSysAreaRpcService.findAreaByCodes(areaCodes);
        dtlRespVOList.forEach(dtlRespVO -> {
            Optional<SysAreaRespDTO> areaOptional1 = areaList.stream().filter(d -> Objects.equals(d.getAreaCode(), dtlRespVO.getProvince())).findFirst();
            areaOptional1.ifPresent(areaRespDTO -> dtlRespVO.setProvinceName(areaRespDTO.getAreaName()));
            Optional<SysAreaRespDTO> areaOptional2 = areaList.stream().filter(d -> Objects.equals(d.getAreaCode(), dtlRespVO.getCity())).findFirst();
            areaOptional2.ifPresent(areaRespDTO -> dtlRespVO.setCityName(areaRespDTO.getAreaName()));
            Optional<SysAreaRespDTO> areaOptional3 = areaList.stream().filter(d -> Objects.equals(d.getAreaCode(), dtlRespVO.getDistrict())).findFirst();
            areaOptional3.ifPresent(areaRespDTO -> dtlRespVO.setDistrictName(areaRespDTO.getAreaName()));

            //拼接省市区展示
            StringJoiner sj = new StringJoiner("");
            if (StringUtils.isNotBlank(dtlRespVO.getProvinceName())){
                sj.add(dtlRespVO.getProvinceName());
            }
            if (StringUtils.isNotBlank(dtlRespVO.getCityName())){
                sj.add(dtlRespVO.getCityName());
            }
            if (StringUtils.isNotBlank(dtlRespVO.getDistrictName())){
                sj.add(dtlRespVO.getDistrictName());
            }
            dtlRespVO.setRegionStr(sj.toString());
        });

    }

}
