package com.elitesland.yst.production.sale.config.cas;

import cn.hutool.core.util.ObjectUtil;
import com.elitescloud.boot.auth.cas.UserSourceProvider;
import com.elitescloud.boot.auth.cas.model.AuthUserDTO;
import com.elitescloud.boot.auth.cas.provider.UserTransferHelper;
import com.elitescloud.boot.auth.config.AuthorizationSdkProperties;
import com.elitescloud.boot.auth.model.Result;
import lombok.extern.log4j.Log4j2;
import org.springframework.core.task.TaskExecutor;
import org.springframework.stereotype.Component;

import javax.validation.constraints.NotBlank;
import javax.validation.constraints.NotNull;
import java.util.Map;
import java.util.function.Supplier;

/**
 * 用户变更后的同步回调.
 * <p>
 * 本地业务需要保存认证服务中的用户ID，该服务中涉及到的id均指的认证服务的用户ID
 *
 * @author Kaiser（wang shao）
 * @date 2022/12/29
 */
@Log4j2
@Component
public class CasUserClientHelper {

    private final AuthorizationSdkProperties sdkProperties;
    private final UserSourceProvider userSourceProvider;
    private final TaskExecutor taskExecutor;
    private final UserTransferHelper userTransferHelper;


    public CasUserClientHelper(AuthorizationSdkProperties sdkProperties, UserSourceProvider userSourceProvider, TaskExecutor taskExecutor) {
        this.sdkProperties = sdkProperties;
        this.userSourceProvider = userSourceProvider;
        this.taskExecutor = taskExecutor;
        this.userTransferHelper = UserTransferHelper.getInstance(sdkProperties.getAuthServer());
    }

    /**
     * 当新增（或修改）用户时的回调
     *
     * @param userDTO 用户信息，userDTO中的id为统一认证服务中的用户ID，为空时新增，不为空时表示修改
     */
    public Long onUpsert(@NotNull AuthUserDTO userDTO) {
        if (!sdkProperties.getCasClient().getEnabled()) {
            // 已禁用
            log.warn("统一身份认证客户端已禁用");
            return null;
        }

        return execute(() -> {
            // 开始调用认证中心接口同步用户

            var syncResult = userTransferHelper.upsertUser(userDTO);

            if (!syncResult.getSuccess() || syncResult.getData() == null) {
                log.error("CAS同步用户账号信息失败：{}, {}", userDTO, syncResult);
                return null;
            }

            // 更新认证中心用户ID
            if (ObjectUtil.equals(userDTO.getId(), syncResult.getData())) {
                // 更新操作
                return syncResult.getData();
            }

            // 新增操作，回写本地的认证服务用户ID
            userSourceProvider.successCallback(Map.of(userDTO.getUsername(), syncResult.getData()));

            return syncResult.getData();
        });
    }
    public void updateMobile(Long userId,String mobile){
        userTransferHelper.updateMobile(userId, mobile);
    }

    public Result<AuthUserDTO> getAuthByMobile(String mobile){
        return userTransferHelper.getUserByMobile(mobile);
    }


    /**
     * 当启用状态修改时的回调
     *
     * @param id      统一认证服务中的用户ID
     * @param enabled 启用状态
     * @return 用户ID
     */
    public Long onEnabled(long id, boolean enabled) {
        if (!sdkProperties.getCasClient().getEnabled()) {
            // 已禁用
            log.warn("统一身份认证客户端已禁用");
            return null;
        }

        return execute(() -> {
            // 更新状态
            userTransferHelper.updateEnabled(id, enabled);
            return id;
        });
    }

    /**
     * 当用户密码修改后的回调
     *
     * @param id       统一认证服务中的用户ID
     * @param password 新密码（明文）
     */
    public Long onUpdatePassword(long id, @NotBlank String password) {
        if (!sdkProperties.getCasClient().getEnabled()) {
            // 已禁用
            log.warn("统一身份认证客户端已禁用");
            return null;
        }

        return execute(() -> {
            // 更新密码
            userTransferHelper.updatePassword(id, password);
            return id;
        });
    }

    /**
     * 删除账号时的回调
     *
     * @param id 统一认证服务中的用户ID
     */
    public Long onDelete(long id) {
        if (!sdkProperties.getCasClient().getEnabled()) {
            // 已禁用
            log.warn("统一身份认证客户端已禁用");
            return null;
        }

        return execute(() -> {
            // 删除
            userTransferHelper.delete(id);
            return id;
        });
    }

    private <T> T execute(Supplier<T> supplier) {
//        if (sdkProperties.getCasClient().getSyncFailThrow()) {
//            // 同步操作
//            return supplier.get();
//        }
        return supplier.get();
        // 异步操作
//        CompletableFuture.supplyAsync(supplier, taskExecutor)
//                .whenComplete((t, e) -> {
//                    if (e != null) {
//                        log.error("用户中心同步异常：", e);
//                    }
//                });
//        return null;
    }
}
