package com.elitesland.yst.production.sale.controller;

import cn.hutool.core.util.StrUtil;
import com.elitesland.yst.production.sale.common.model.CurrentUserDTO;
import com.elitesland.yst.production.sale.core.service.UserService;
import com.elitesland.yst.production.sale.core.util.excel.ExcelExportUtil;
import com.elitesland.yst.production.sale.core.util.excel.ExcelImportUtil;
import com.elitesland.yst.production.sale.core.util.excel.support.ExportColumnParam;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitescloud.boot.common.param.AbstractOrderQueryParam;
import com.elitescloud.boot.exception.BusinessException;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.core.io.FileSystemResource;
import org.springframework.core.io.Resource;
import org.springframework.http.HttpHeaders;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.util.ResourceUtils;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.multipart.MultipartFile;

import javax.servlet.http.HttpServletResponse;
import java.io.File;
import java.io.FileNotFoundException;
import java.net.URLEncoder;
import java.nio.charset.StandardCharsets;
import java.util.List;
import java.util.function.BiFunction;
import java.util.function.Function;

/**
 * controller的基类.
 *
 * @author Kaiser（wang shao）
 * @date 2021-05-08
 */
@Slf4j
@Validated
public abstract class BaseController {

    @Autowired
    private UserService userService;

    /**
     * 获取当前用户信息
     *
     * @return 当前用户信息
     */
    protected CurrentUserDTO currentUser() {
        return UserService.currentUser();
    }

    /**
     * 下载导入模板
     *
     * @param tmpl 模板名称
     */
    protected ResponseEntity<Resource> downloadExcelTmpl(String tmpl) {
        if (StrUtil.isBlank(tmpl)) {
            return ResponseEntity.badRequest().build();
        }

        File file = null;
        try {
            file = ResourceUtils.getFile("classpath:excel/" + tmpl);
        } catch (FileNotFoundException e) {
            log.error("下载模板文件【" + tmpl + "】失败：", e);
            return ResponseEntity.badRequest().build();
        }

        var headers = new HttpHeaders();
        String fileName = URLEncoder.encode(tmpl, StandardCharsets.UTF_8).replace(" ", "%20");
        headers.add(HttpHeaders.CONTENT_DISPOSITION, "attachment;filename*=utf-8''" + fileName);

        return ResponseEntity.ok()
                .headers(headers)
                .contentLength(file.length())
                .contentType(MediaType.parseMediaType("application/vnd.ms-excel"))
                .body(new FileSystemResource(file));
    }

    /**
     * 解析导入数据
     *
     * @param file        导入文件
     * @param cls         数据类型
     * @param headRow     头部行数
     * @param dataTypeRow 数据类型所在行
     * @return 数据列表
     * @throws Exception 解析异常
     */
    protected List<?> importExcel(MultipartFile file, Class<?> cls, Integer headRow, Integer dataTypeRow) throws Exception {
        if (file == null || file.isEmpty()) {
            throw new BusinessException("导入文件为空");
        }

        return ExcelImportUtil.instance(file.getInputStream())
                .headRow(headRow)
                .dataType(cls, dataTypeRow)
                .readAllSync();
    }

    /**
     * 导出数据
     *
     * @param response        response
     * @param fileName        导出文件名，不含后缀
     * @param columnParamList 导出的数据列
     * @param dataProducer    数据生产者
     * @param queryParam      查询参数
     * @param <T>             参数类型
     * @throws Exception 导出异常
     */
    protected <T extends AbstractOrderQueryParam> void export(HttpServletResponse response, String fileName, List<ExportColumnParam> columnParamList,
                                                              Function<T, PagingVO<?>> dataProducer, T queryParam) throws Exception {
        getExportUtilInstance(response, fileName, columnParamList)
                .write("导出数据", (page, pageSize) -> {
                    queryParam.setCurrent(page);
                    queryParam.setSize(pageSize);

                    return dataProducer.apply(queryParam).getRecords();
                });
    }

    /**
     * 导出数据
     *
     * @param response        response
     * @param fileName        导出文件名，不含后缀
     * @param columnParamList 导出的数据列
     * @param dataProducer    数据生产者
     * @param queryParam      查询参数
     * @param otherParam      其它参数
     * @param <T>             参数类型
     * @throws Exception 导出异常
     */
    protected <T extends AbstractOrderQueryParam> void export(HttpServletResponse response, String fileName, List<ExportColumnParam> columnParamList,
                                                              BiFunction<T, Object[], PagingVO<?>> dataProducer, T queryParam, Object[] otherParam) throws Exception {
        getExportUtilInstance(response, fileName, columnParamList)
                .write("导出数据", (page, pageSize) -> {
                    queryParam.setCurrent(page);
                    queryParam.setSize(pageSize);

                    return dataProducer.apply(queryParam, otherParam).getRecords();
                });
    }

    private ExcelExportUtil getExportUtilInstance(HttpServletResponse response, String fileName, List<ExportColumnParam> columnParamList) throws Exception {
        response.setContentType("application/vnd.ms-excel");
        response.setCharacterEncoding(StandardCharsets.UTF_8.name());
        fileName = URLEncoder.encode(fileName, StandardCharsets.UTF_8).replace(" ", "%20");
        response.setHeader("Content-disposition", "attachment;filename*=utf-8''" + fileName + ".xlsx");

        return ExcelExportUtil.instance(response.getOutputStream())
                .batchSize(200)
                .fields(columnParamList);
    }
}
