package com.elitesland.yst.production.sale.service;

import cn.hutool.core.collection.CollUtil;
import cn.hutool.core.collection.CollectionUtil;
import com.elitescloud.boot.auth.util.SecurityContextUtil;
import com.elitescloud.boot.excel.util.ExcelImportUtil;
import com.elitescloud.boot.exception.BusinessException;
import com.elitescloud.cloudt.common.annotation.SysCodeProc;
import com.elitescloud.cloudt.common.base.ApiCode;
import com.elitescloud.cloudt.common.base.ApiResult;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitescloud.boot.common.param.AreaVO;
import com.elitescloud.cloudt.security.entity.GeneralUserDetails;
import com.elitescloud.cloudt.system.dto.SysEmployeeBasicDTO;
import com.elitescloud.cloudt.system.dto.resp.EmployeePageRespDTO;
import com.elitescloud.cloudt.system.dto.resp.EmployeeUnderlingDTO;
import com.elitescloud.cloudt.system.dto.resp.SysAreaRespDTO;
import com.elitescloud.cloudt.system.vo.SysUserDTO;
import com.elitesland.yst.production.sale.api.service.*;
import com.elitesland.yst.production.sale.api.vo.param.salesman.SalesmanInfoSimpleQueryVO;
import com.elitesland.yst.production.sale.api.vo.param.taskinfo.TaskInfoAppQueryVO;
import com.elitesland.yst.production.sale.api.vo.param.taskinfo.TaskInfoDtlQueryVO;
import com.elitesland.yst.production.sale.api.vo.param.taskinfo.TaskInfoQueryVO;
import com.elitesland.yst.production.sale.api.vo.resp.crm.CrmCustBaseRespVO;
import com.elitesland.yst.production.sale.api.vo.resp.crm.LmSaveCustRespVO;
import com.elitesland.yst.production.sale.api.vo.resp.salesman.SalesmanInfoSimpleRespVO;
import com.elitesland.yst.production.sale.api.vo.resp.taskinfo.*;
import com.elitesland.yst.production.sale.api.vo.save.TaskExecutionRecordSaveVO;
import com.elitesland.yst.production.sale.api.vo.save.TaskInfoDtlSaveVO;
import com.elitesland.yst.production.sale.common.constant.ConstantsSale;
import com.elitesland.yst.production.sale.common.constant.UdcEnum;
import com.elitesland.yst.production.sale.convert.TaskInfoDtlConvert;
import com.elitesland.yst.production.sale.entity.SalesmanInfoDO;
import com.elitesland.yst.production.sale.entity.TaskInfoDtlDO;
import com.elitesland.yst.production.sale.repo.TaskInfoDtlRepo;
import com.elitesland.yst.production.sale.repo.TaskInfoDtlRepoProc;
import com.elitesland.yst.production.sale.rmi.ystsupport.RmiOrgAddrService;
import com.elitesland.yst.production.sale.rmi.ystsupport.RmiOrgStoreRpcService;
import com.elitesland.yst.production.sale.rmi.ystsystem.RmiEmployeeRpcService;
import com.elitesland.yst.production.sale.rmi.ystsystem.RmiSysAreaRpcService;
import com.elitesland.yst.production.support.provider.org.dto.OrgAddrAddressRpcDTO;
import com.elitesland.yst.production.support.provider.org.dto.OrgAddressRpcDTO;
import com.elitesland.yst.production.support.provider.org.dto.OrgStoreDetailRpcDTO;
import com.elitesland.yst.production.support.provider.org.param.OrgAddressRpcDtoParam;
import com.google.common.collect.Lists;
import lombok.extern.slf4j.Slf4j;
import lombok.val;
import org.apache.commons.collections4.SetUtils;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Lazy;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;
import org.springframework.web.multipart.MultipartFile;

import java.math.BigDecimal;
import java.math.RoundingMode;
import java.time.LocalDateTime;
import java.util.*;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;

/**
 * <p>
 * 功能说明:业务员任务管理
 * </p>
 *
 * @Author Darren
 * @Date 2023/04/10
 * @Version 1.0
 * @Content:
 */
@Slf4j
@Service
public class TaskInfoDtlServiceImpl implements TaskInfoDtlService {

    @Autowired
    private TaskInfoDtlRepo taskInfoDtlRepo;
    @Autowired
    private TaskInfoDtlRepoProc taskInfoDtlRepoProc;
    @Autowired
    private TaskInfoService taskInfoService;

    @Autowired
    private RmiEmployeeRpcService rmiEmployeeRpcService;
    @Autowired
    private RmiOrgStoreRpcService rmiOrgStoreRpcService;
    @Autowired
    private RmiSysAreaRpcService rmiSysAreaRpcService;
    @Autowired
    private CrmCustService crmCustService;
    @Autowired
    private SalesmanInfoService salesmanInfoService;
    @Autowired
    private RmiOrgAddrService rmiOrgAddrService;
    //@Autowired
    //private CrmCustRpcService crmCustRpcService;

    private ExectRecordService exectRecordService;

    @Autowired
    @Lazy
    public void setExectRecordService(ExectRecordService exectRecordService) {
        this.exectRecordService = exectRecordService;
    }

    private ExectRecordTempService exectRecordTempService;

    @Autowired
    @Lazy
    public void setExectRecordTempService(ExectRecordTempService exectRecordTempService) {
        this.exectRecordTempService = exectRecordTempService;
    }

    /**
     * 任务明细保存
     *
     * @param saveVOList 入参
     * @return 任务明细ID
     */
    @Override
    @Transactional(rollbackFor = {Exception.class})
    public List<Long> saveTaskInfoDtl(List<TaskInfoDtlSaveVO> saveVOList) {
        List<TaskInfoDtlDO> taskInfoDtlDOList = saveVOList.stream().map(TaskInfoDtlConvert.INSTANCE::saveVoToDo).collect(Collectors.toList());

        return taskInfoDtlRepo.saveAll(taskInfoDtlDOList).stream().map(TaskInfoDtlDO::getId).collect(Collectors.toList());
    }

    /**
     * 根据任务主表ID删除任务明细
     *
     * @param masId 任务主表ID
     * @return
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void deleteByMasId(Long masId) {
        taskInfoDtlRepo.deleteByMasId(masId);
    }

    /**
     * 根据任务主表ID查询任务明细
     *
     * @param masId 任务主表ID
     * @return 任务明细
     */
    @Override
    @SysCodeProc
    public List<TaskInfoDtlRespVO> selectByMasId(Long masId) {
        if (Objects.isNull(masId)) {
            return Collections.emptyList();
        }
        TaskInfoDtlQueryVO queryVO = new TaskInfoDtlQueryVO();
        queryVO.setMasId(masId);
        List<TaskInfoDtlRespVO> taskInfoDtlRespVOList = taskInfoDtlRepoProc.selectByQueryVO(queryVO);
        if (CollectionUtils.isEmpty(taskInfoDtlRespVOList)) {
            return Collections.emptyList();
        }

        //组装填充任务明细的相关关联字段信息
        translateTaskDtl(taskInfoDtlRespVOList);
        //组装处理行政区域信息
        translateArea(taskInfoDtlRespVOList);

        return taskInfoDtlRespVOList;
    }

    /**
     * 组装填充任务明细的相关关联字段信息
     *
     * @param dtlRespVOList 任务明细数据信息
     * @return
     */
    private void translateTaskDtl(List<TaskInfoDtlRespVO> dtlRespVOList) {
        //所属客户编码
        List<String> dealerCodeList = dtlRespVOList.stream().map(TaskInfoDtlRespVO::getDealerCode).filter(Objects::nonNull).distinct().collect(Collectors.toList());
        List<LmSaveCustRespVO> dealerList = crmCustService.getCustInfoByCustCodes(dealerCodeList);
        //执行人编码-业务员-业务员表没有名称，要通过业务员编码查询员工的名称,业务员编码和员工编码是相同的
        Set<String> executeUserCodeList = dtlRespVOList.stream().map(TaskInfoDtlRespVO::getExecutUserCode).filter(Objects::nonNull).collect(Collectors.toSet());
        List<SysEmployeeBasicDTO> executeUserList = rmiEmployeeRpcService.findEmployeeByCodes(executeUserCodeList);
        //执行记录-编码,执行记录没有名称字段
        //查询不同业务类型的关联业务编码的信息
        List<OrgStoreDetailRpcDTO> storeList = new ArrayList<>();
        List<LmSaveCustRespVO> custList = new ArrayList<>();
        List<SysEmployeeBasicDTO> employeeList = new ArrayList<>();
        Map<String, List<TaskInfoDtlRespVO>> businessTypeListMap = dtlRespVOList.stream().filter(dtlRespVO -> StringUtils.isNotBlank(dtlRespVO.getBusinessType())).collect(Collectors.groupingBy(TaskInfoDtlRespVO::getBusinessType));
        for (String key : businessTypeListMap.keySet()) {
            List<String> businessCodes = businessTypeListMap.get(key).stream().map(TaskInfoDtlRespVO::getBusinessCode).filter(Objects::nonNull).distinct().collect(Collectors.toList());
            //关联业务-门店
            if (Objects.equals(key, UdcEnum.SALESMAN_TASK_BUSINESS_TYPE_STORE.getValueCode())) {
                storeList = rmiOrgStoreRpcService.findStoreByStoreCodes(businessCodes);
            } else if (Objects.equals(key, UdcEnum.SALESMAN_TASK_BUSINESS_TYPE_DEALER.getValueCode())) {
                custList = crmCustService.getCustInfoByCustCodes(businessCodes);
            } else if (Objects.equals(key, UdcEnum.SALESMAN_TASK_BUSINESS_TYPE_SALESMAN.getValueCode())) {
                Set<String> businessCodeSet = businessCodes.stream().collect(Collectors.toSet());
                employeeList = rmiEmployeeRpcService.findEmployeeByCodes(businessCodeSet);
            }
        }

        for (TaskInfoDtlRespVO respVO : dtlRespVOList) {
            //明细的关联业务编码数据库里以逗号分隔的字符串形式保存
            //目前一行任务明细的业务编码只有一个,前端传businessCode，不使用businessCodes了。
            //List<String> businessCodes = transitionStrToCodes(respVO.getBusinessCode());
            //respVO.setBusinessCodes(businessCodes);

            //关联业务-门店
            if (Objects.equals(respVO.getBusinessType(), UdcEnum.SALESMAN_TASK_BUSINESS_TYPE_STORE.getValueCode())) {
                //业务编码是可多选的,名称进行拼接显示
                //需要支撑域提供批量的根据编码查询门店的接口
                /*List<OrgStoreDetailRpcDTO> storeList = rmiOrgStoreRpcService.findStoreByStoreCodes(respVO.getBusinessCodes());
                String businessName = storeList.stream().filter(storeDTO -> StringUtils.isNotBlank(storeDTO.getStoreName()))
                        .map(OrgStoreDetailRpcDTO::getStoreName).distinct().collect(Collectors.joining("-"));
                respVO.setBusinessName(businessName);
                String contactName = storeList.stream().filter(storeDTO -> StringUtils.isNotBlank(storeDTO.getStoreManager()))
                        .map(OrgStoreDetailRpcDTO::getStoreManager).distinct().collect(Collectors.joining("-"));
                String contactPhone = storeList.stream().filter(storeDTO -> StringUtils.isNotBlank(storeDTO.getStoreContPhone()))
                        .map(OrgStoreDetailRpcDTO::getStoreContPhone).distinct().collect(Collectors.joining("-"));
                respVO.setContactName(contactName);
                respVO.setContactPhone(contactPhone);*/

                val storeOptional = storeList.stream().filter(storeVO -> Objects.equals(storeVO.getStoreCode(), respVO.getBusinessCode())).findFirst();
                storeOptional.ifPresent(storeVO -> {
                    respVO.setBusinessName(storeVO.getStoreName());
                    respVO.setContactName(storeVO.getStoreManager());
                    respVO.setContactPhone(storeVO.getStoreContPhone());
                });
            }
            //关联业务-经销商
            if (Objects.equals(respVO.getBusinessType(), UdcEnum.SALESMAN_TASK_BUSINESS_TYPE_DEALER.getValueCode())) {
                /*List<LmSaveCustRespVO> custList = crmCustService.getCustInfoByCustCodes(respVO.getBusinessCodes());
                String businessName = custList.stream().filter(custRespVO -> StringUtils.isNotBlank(custRespVO.getCustName()))
                        .map(LmSaveCustRespVO::getCustName).distinct().collect(Collectors.joining("-"));
                respVO.setBusinessName(businessName);
                String custCode2 = custList.stream().filter(custRespVO -> StringUtils.isNotBlank(custRespVO.getCustCode2()))
                        .map(LmSaveCustRespVO::getCustCode2).distinct().collect(Collectors.joining("-"));
                respVO.setCustCode2(custCode2);
                String contactName = custList.stream().filter(storeDTO -> StringUtils.isNotBlank(storeDTO.getContactName()))
                        .map(LmSaveCustRespVO::getContactName).distinct().collect(Collectors.joining("-"));
                String contactPhone = custList.stream().filter(storeDTO -> StringUtils.isNotBlank(storeDTO.getContactPhone()))
                        .map(LmSaveCustRespVO::getContactPhone).distinct().collect(Collectors.joining("-"));
                respVO.setContactName(contactName);
                respVO.setContactPhone(contactPhone);*/
                val custOptional = custList.stream().filter(custVO -> Objects.equals(custVO.getCustCode(), respVO.getBusinessCode())).findFirst();
                custOptional.ifPresent(custVO -> {
                    respVO.setBusinessName(custVO.getCustName());
                    respVO.setCustCode2(custVO.getCustCode2());
                    respVO.setContactName(custVO.getContactName());
                    respVO.setContactPhone(custVO.getContactPhone());
                });
            }
            //关联业务-业务员
            if (Objects.equals(respVO.getBusinessType(), UdcEnum.SALESMAN_TASK_BUSINESS_TYPE_SALESMAN.getValueCode())) {
                //业务员表没有名称，要通过业务员编码查询员工的名称,业务员编码和员工编码是相同的
               /* Set<String> businessCodeSet = respVO.getBusinessCodes().stream().collect(Collectors.toSet());
                List<SysEmployeeBasicDTO> employeeList = rmiEmployeeRpcService.findEmployeeByCodes(businessCodeSet);
                String businessName = employeeList.stream().filter(employeeDTO -> StringUtils.isNotBlank(employeeDTO.getFullName()))
                        .map(SysEmployeeBasicDTO::getFullName).distinct().collect(Collectors.joining("-"));
                respVO.setBusinessName(businessName);

                String contactPhone = employeeList.stream().filter(storeDTO -> StringUtils.isNotBlank(storeDTO.getPhone()))
                        .map(SysEmployeeBasicDTO::getPhone).distinct().collect(Collectors.joining("-"));
                respVO.setContactName(businessName);
                respVO.setContactPhone(contactPhone);*/
                val employeeOptional = employeeList.stream().filter(employeeVO -> Objects.equals(employeeVO.getCode(), respVO.getBusinessCode())).findFirst();
                employeeOptional.ifPresent(employeeVO -> {
                    respVO.setBusinessName(employeeVO.getFullName());
                    respVO.setContactName(employeeVO.getFullName());
                    respVO.setContactPhone(employeeVO.getPhone());
                });
            }
            //所属客户
            val dealerOptional = dealerList.stream().filter(dealerVO -> Objects.equals(dealerVO.getCustCode(), respVO.getDealerCode())).findFirst();
            dealerOptional.ifPresent(dealerVO -> respVO.setDealerName(dealerVO.getCustName()));
            //执行人
            val executeUserOptional = executeUserList.stream().filter(executeUser -> Objects.equals(executeUser.getCode(), respVO.getExecutUserCode())).findFirst();
            executeUserOptional.ifPresent(executeUser -> respVO.setExecutUser(executeUser.getFullName()));


        }
    }

    /**
     * 组装处理行政区域信息
     *
     * @param dtlRespVOList 任务明细数据信息
     * @return
     */
    private void translateArea(List<TaskInfoDtlRespVO> dtlRespVOList) {
        //省市
        Set<String> provinces = dtlRespVOList.stream().map(TaskInfoDtlRespVO::getProvince).distinct().filter(Objects::nonNull).collect(Collectors.toSet());
        //市
        Set<String> cities = dtlRespVOList.stream().map(TaskInfoDtlRespVO::getCity).distinct().filter(Objects::nonNull).collect(Collectors.toSet());
        //区县
        Set<String> counties = dtlRespVOList.stream().map(TaskInfoDtlRespVO::getDistrict).distinct().filter(Objects::nonNull).collect(Collectors.toSet());
        Set<String> areaCodes = new HashSet<>();
        areaCodes.addAll(provinces);
        areaCodes.addAll(cities);
        areaCodes.addAll(counties);
        List<SysAreaRespDTO> areaList = rmiSysAreaRpcService.findAreaByCodes(areaCodes);
        dtlRespVOList.forEach(dtlRespVO -> {
            Optional<SysAreaRespDTO> areaOptional1 = areaList.stream().filter(d -> Objects.equals(d.getAreaCode(), dtlRespVO.getProvince())).findFirst();
            areaOptional1.ifPresent(areaRespDTO -> dtlRespVO.setProvinceName(areaRespDTO.getAreaName()));
            Optional<SysAreaRespDTO> areaOptional2 = areaList.stream().filter(d -> Objects.equals(d.getAreaCode(), dtlRespVO.getCity())).findFirst();
            areaOptional2.ifPresent(areaRespDTO -> dtlRespVO.setCityName(areaRespDTO.getAreaName()));
            Optional<SysAreaRespDTO> areaOptional3 = areaList.stream().filter(d -> Objects.equals(d.getAreaCode(), dtlRespVO.getDistrict())).findFirst();
            areaOptional3.ifPresent(areaRespDTO -> dtlRespVO.setDistrictName(areaRespDTO.getAreaName()));

        });

    }

    /**
     * 根据任务明细ID查询任务明细详情数据
     *
     * @param id 任务明细ID
     * @return 任务明细数据
     */
    @Override
    @SysCodeProc
    public TaskInfoDtlRespVO selectById(Long id) {
        if (Objects.isNull(id)) {
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "任务明细ID不能为空!");
        }
        Optional<TaskInfoDtlDO> optional = taskInfoDtlRepo.findById(id);
        if (optional.isEmpty()) {
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "未查询到任务明细数据信息!");
        }
        TaskInfoDtlRespVO taskInfoDtlRespVO = TaskInfoDtlConvert.INSTANCE.doToRespVo(optional.get());
        //组装填充任务明细的相关关联字段信息
        translateTaskDtl(Collections.singletonList(taskInfoDtlRespVO));
        //组装处理行政区域信息
        translateArea(Collections.singletonList(taskInfoDtlRespVO));

        return taskInfoDtlRespVO;
    }

    /**
     * 根据任务明细ID查询任务明细数据
     *
     * @param id 任务明细ID
     * @return 任务明细数据
     */
    @Override
    @SysCodeProc
    public TaskInfoDtlRespVO findById(Long id) {
        if (Objects.isNull(id)) {
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "任务明细ID不能为空!");
        }
        TaskInfoDtlQueryVO dtlQueryVO = new TaskInfoDtlQueryVO();
        dtlQueryVO.setId(id);
        List<TaskInfoDtlRespVO> dtlRespVOList = taskInfoDtlRepoProc.selectByQueryVO(dtlQueryVO);
        if (CollectionUtils.isEmpty(dtlRespVOList)) {
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "未查询到任务明细数据信息!");
        }
        if (dtlRespVOList.size() > 1) {
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "查询到多条任务明细数据信息!");
        }
        TaskInfoDtlRespVO taskInfoDtlRespVO = dtlRespVOList.get(0);

        return taskInfoDtlRespVO;
    }

    /**
     * 根据入参查询任务明细
     *
     * @param queryVO 入参
     * @return 任务明细
     */
    @Override
    @SysCodeProc
    public List<TaskInfoDtlRespVO> selectByParam(TaskInfoDtlQueryVO queryVO) {
        List<TaskInfoDtlRespVO> taskInfoDtlRespVOList = taskInfoDtlRepoProc.selectByQueryVO(queryVO);
        if (CollectionUtils.isEmpty(taskInfoDtlRespVOList)) {
            return Collections.emptyList();
        }
        return taskInfoDtlRespVOList;
    }


    /**
     * 根据主表任务ID更新deleteFlag
     *
     * @param deleteFlag 删除标记
     * @param masId      主表任务ID
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void updateDeleteFlag(Integer deleteFlag, Long masId) {
        taskInfoDtlRepoProc.updateDeleteFlag(deleteFlag, masId);
    }

    /**
     * 根据主表任务ID批量更新deleteFlag
     *
     * @param deleteFlag 删除标记
     * @param masIds     主表任务ID
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void updateDeleteFlagBatch(Integer deleteFlag, List<Long> masIds) {
        taskInfoDtlRepoProc.updateDeleteFlagBatch(deleteFlag, masIds);
    }

    /**
     * 根据主表任务ID更新完成状态
     *
     * @param completeState 完成状态
     * @param masId         主表任务ID
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void updateCompleteStateByMasId(String completeState, Long masId) {
        taskInfoDtlRepoProc.updateCompleteStateByMasId(completeState, masId);
    }

    /**
     * 根据主表任务ID更新完成状态、完成时间
     *
     * @param completeState 完成状态
     * @param completeTime  完成时间
     * @param masId         主表任务ID
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void updateCompleteStateAndTimeByMasId(String completeState, LocalDateTime completeTime, Long masId) {
        taskInfoDtlRepoProc.updateCompleteStateAndCompleteTimeByMasId(completeState, completeTime, masId);
    }

    /**
     * 根据任务明细ID批量更新完成状态、完成时间
     *
     * @param completeState 完成状态
     * @param completeTime  完成时间
     * @param ids           任务明细ID
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void updateCompleteStateAndTimeByIdBatch(String completeState, LocalDateTime completeTime, List<Long> ids) {
        taskInfoDtlRepoProc.updateCompleteStateAndTimeByIdBatch(completeState, completeTime, ids);
    }

    /**
     * 根据任务明细ID批量更新完成状态
     *
     * @param completeState 完成状态
     * @param ids           任务明细ID
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void updateCompleteStateByIdBatch(String completeState, List<Long> ids) {
        taskInfoDtlRepoProc.updateCompleteStateByIdBatch(completeState, ids);
    }

    /**
     * 根据任务明细ID批量更新是否逾期
     *
     * @param delayFlag 是否逾期
     * @param ids       任务明细ID
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void updateDelayFlagByIdBatch(String delayFlag, List<Long> ids) {
        taskInfoDtlRepoProc.updateDelayFlagByIdBatch(delayFlag, ids);
    }

    /**
     * 根据任务明细ID开始执行前置校验
     *
     * @param id            任务明细ID
     * @param checkSignFlag 校验是否任务强制签到标识,中台 true APP端 false
     * @return 任务返给执行的出参
     */
    @Override
    public TaskExecuteRespVO taskDtlExecuteCheck(Long id, Boolean checkSignFlag) {
        //查询任务明细
        Optional<TaskInfoDtlDO> optional = taskInfoDtlRepo.findById(id);
        if (optional.isEmpty()) {
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "未查询到任务明细数据信息!");
        }
        TaskInfoDtlRespVO dtlRespVO = TaskInfoDtlConvert.INSTANCE.doToRespVo(optional.get());
        //针对‘待执行’状态的任务（明细），勾选点击开始执行，校验：
        if (!Objects.equals(dtlRespVO.getCompleteState(), UdcEnum.SALESMAN_TASK_COMPLETION_STATUS_WEE.getValueCode())) {
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "任务明细是待执行状态的才可执行!");
        }
        //查询任务明细主表
        TaskInfoRespVO respVO = taskInfoService.findById(dtlRespVO.getMasId());
        //1、是否到了任务开始时间，如果没有到，则提示用户此任务还未开始；
        if (Objects.isNull(respVO.getStartTime())) {
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "任务开始时间为空!");
        }
        boolean startTimeFlag = LocalDateTime.now().isBefore(respVO.getStartTime());
        if (startTimeFlag) {
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "此任务还未开始!");
        }
        //2、是否执行人是当前登录用户，如果不是，则提示用户：请选择指派给自己的任务明细进行执行！
        //执行人是业务员、业务员和员工的编码一致，可以根据用户获取员工
        SysUserDTO sysUserDTO = getSysUser();
        if (Objects.isNull(dtlRespVO.getExecutUserCode())) {
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "执行人为空!");
        }
        //根据当前用户查询所属员工信息
        SysEmployeeBasicDTO employeeBasicDTO = getEmployeeByUser(sysUserDTO);
        if (!Objects.equals(dtlRespVO.getExecutUserCode(), employeeBasicDTO.getCode())) {
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "请选择指派给自己的任务明细进行执行!");
        }
        //3、是否任务强制签到，如果是，则提示用户：任务需要定位打卡，请在业务员APP进行操作。
        //中台端校验，APP端不校验
        if (checkSignFlag && Objects.equals(respVO.getForceSignFlag(), UdcEnum.SALESMAN_TASK_FORCE_SIGN_FLAG_Y.getValueCode())) {
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "任务需要定位打卡,请在业务员APP进行操作!");
        }

        //目前一行任务明细的业务编码只有一个,前端传businessCode，不使用businessCodes了。
        //List<String> businessCodes = transitionStrToCodes(dtlRespVO.getBusinessCode());
        if (Objects.equals(dtlRespVO.getBusinessType(), UdcEnum.SALESMAN_TASK_BUSINESS_TYPE_STORE.getValueCode())) {
            //关联业务-门店
            List<OrgStoreDetailRpcDTO> storeList = rmiOrgStoreRpcService.findStoreByStoreCodes(Collections.singletonList(dtlRespVO.getBusinessCode()));
            if (!CollectionUtils.isEmpty(storeList)) {
                dtlRespVO.setContactName(storeList.get(0).getStoreManager());
                dtlRespVO.setContactPhone(storeList.get(0).getStoreContPhone());
            }
        } else if (Objects.equals(dtlRespVO.getBusinessType(), UdcEnum.SALESMAN_TASK_BUSINESS_TYPE_DEALER.getValueCode())) {
            //关联业务-经销商
            List<LmSaveCustRespVO> custList = crmCustService.getCustInfoByCustCodes(Collections.singletonList(dtlRespVO.getBusinessCode()));
            if (!CollectionUtils.isEmpty(custList)) {
                dtlRespVO.setContactName(custList.get(0).getContactName());
                dtlRespVO.setContactPhone(custList.get(0).getContactPhone());
            }
        } else if (Objects.equals(dtlRespVO.getBusinessType(), UdcEnum.SALESMAN_TASK_BUSINESS_TYPE_SALESMAN.getValueCode())) {
            //关联业务-业务员
            Set<String> businessCodeSet = Collections.singletonList(dtlRespVO.getBusinessCode()).stream().collect(Collectors.toSet());
            List<SysEmployeeBasicDTO> employeeList = rmiEmployeeRpcService.findEmployeeByCodes(businessCodeSet);
            dtlRespVO.setContactName(dtlRespVO.getBusinessName());
            if (!CollectionUtils.isEmpty(employeeList)) {
                dtlRespVO.setContactPhone(employeeList.get(0).getPhone());
            }
        }

        //校验通过后，则根据执行模版，生成执行记录新增页面。
        return assembleExecute(respVO, dtlRespVO);
    }

    /**
     * 根据当前用户查询所属员工信息
     *
     * @param sysUserDTO 当前用户信息
     * @return 所属员工信息
     */
    private SysEmployeeBasicDTO getEmployeeByUser(SysUserDTO sysUserDTO) {
        if (Objects.isNull(sysUserDTO)) {
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "当前用户信息为空!");
        }
        //根据用户ID获取所属员工
        SysEmployeeBasicDTO employeeBasicDTO = rmiEmployeeRpcService.findEmployeeByUserId(sysUserDTO.getId());
        if (Objects.isNull(employeeBasicDTO)) {
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "当前用户所属员工信息为空!");
        }
        return employeeBasicDTO;
    }

    /**
     * 组装返回执行使用的任务相关信息
     *
     * @param respVO    任务主表信息
     * @param dtlRespVO 任务明细信息
     * @return 任务相关信息
     */
    private TaskExecuteRespVO assembleExecute(TaskInfoRespVO respVO, TaskInfoDtlRespVO dtlRespVO) {
        TaskExecuteRespVO taskExecuteRespVO = new TaskExecuteRespVO();
        taskExecuteRespVO.setTaskId(respVO.getId());
        taskExecuteRespVO.setTaskCode(respVO.getCode());
        taskExecuteRespVO.setTaskName(respVO.getName());
        taskExecuteRespVO.setTaskType(respVO.getType());
        taskExecuteRespVO.setTaskTypeName(respVO.getTypeName());
        taskExecuteRespVO.setStartTime(respVO.getStartTime());
        taskExecuteRespVO.setEndTime(respVO.getEndTime());
        taskExecuteRespVO.setPublishUser(respVO.getPublishUser());
        taskExecuteRespVO.setPublishUserId(respVO.getPublishUserId());
        taskExecuteRespVO.setPublishUserCode(respVO.getPublishUserCode());
        taskExecuteRespVO.setExecutTemplate(respVO.getExecutTemplate());
        taskExecuteRespVO.setExecutTemplateId(respVO.getExecutTemplateId());
        taskExecuteRespVO.setExecutTemplateCode(respVO.getExecutTemplateCode());
        taskExecuteRespVO.setExecutTemplateName(respVO.getExecutTemplateName());
        taskExecuteRespVO.setForceSignFlag(respVO.getForceSignFlag());
        taskExecuteRespVO.setSignInRange(respVO.getSignInRange());
        taskExecuteRespVO.setSignOutRange(respVO.getSignOutRange());

        taskExecuteRespVO.setTaskDtlId(dtlRespVO.getId());
        taskExecuteRespVO.setExecutUser(dtlRespVO.getExecutUser());
        taskExecuteRespVO.setExecutUserId(dtlRespVO.getExecutUserId());
        taskExecuteRespVO.setExecutUserCode(dtlRespVO.getExecutUserCode());
        taskExecuteRespVO.setBusinessType(dtlRespVO.getBusinessType());
        taskExecuteRespVO.setBusinessCode(dtlRespVO.getBusinessCode());
        //目前一行任务明细的业务编码只有一个,前端传businessCode，不使用businessCodes了。
        //taskExecuteRespVO.setBusinessCodes(transitionStrToCodes(dtlRespVO.getBusinessCode()));
        taskExecuteRespVO.setBusinessName(dtlRespVO.getBusinessName());

        taskExecuteRespVO.setBusinessTime(dtlRespVO.getBusinessTime());
        taskExecuteRespVO.setXLon(dtlRespVO.getXLon());
        taskExecuteRespVO.setYLat(dtlRespVO.getYLat());
        taskExecuteRespVO.setCoordType(dtlRespVO.getCoordType());
        taskExecuteRespVO.setEmployeeId(dtlRespVO.getEmployeeId());
        taskExecuteRespVO.setContactName(dtlRespVO.getContactName());
        taskExecuteRespVO.setContactPhone(dtlRespVO.getContactPhone());
        taskExecuteRespVO.setDealerName(dtlRespVO.getDealerName());
        taskExecuteRespVO.setDealerCode(dtlRespVO.getDealerCode());

        return taskExecuteRespVO;
    }


    /**
     * 根据任务明细ID批量关闭前置校验
     *
     * @param ids 任务明细ID
     * @return true 校验通过  false 校验未通过
     */
    @Override
    public Boolean closeCheckTaskBatch(List<Long> ids) {
        //根据任务明细ID批量关闭时查询主表和明细数据并进行校验
        TaskInfoRespVO respVO = closeCheckSelect(ids);
        List<TaskInfoDtlRespVO> dtlVOList = respVO.getDtlRespVOList();
        //关闭时查询是否任务明细有执行记录，如果有的话，弹窗提醒：
        // 此任务明细有进行中的执行记录，标记关闭会将任务明细状态统一标记为已关闭，执行记录状态也统一标记为已关闭，确定要标记关闭嘛？
        //根据执行记录编码查询执行记录信息
        List<String> recordCode = dtlVOList.stream().map(TaskInfoDtlRespVO::getExecutRecordCode).filter(Objects::nonNull).distinct().collect(Collectors.toList());
        boolean recordFlag = taskInfoService.selectExecutionRecordCheck(recordCode);
        if (recordFlag) {
            //有关联的执行记录,则校验不通过
            return false;
        }

        return true;
    }

    /**
     * 根据任务明细ID批量关闭时查询主表和明细数据并进行校验
     *
     * @param ids 任务明细ID
     * @return 任务主表和明细数据
     */
    private TaskInfoRespVO closeCheckSelect(List<Long> ids) {
        TaskInfoDtlQueryVO dtlQueryVO = new TaskInfoDtlQueryVO();
        dtlQueryVO.setIds(ids);
        List<TaskInfoDtlRespVO> dtlVOList = selectByParam(dtlQueryVO);
        if (CollectionUtils.isEmpty(dtlVOList)) {
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "未查询到任务明细数据!");
        }
        //支持批量选择明细状态非‘已关闭’、‘已完成’、‘已取消’的表单行，执行关闭，
        List<String> completionStatusList = Lists.newArrayList(
                UdcEnum.SALESMAN_TASK_COMPLETION_STATUS_CPD.getValueCode(),
                UdcEnum.SALESMAN_TASK_COMPLETION_STATUS_CSD.getValueCode(),
                UdcEnum.SALESMAN_TASK_COMPLETION_STATUS_CCD.getValueCode()
        );
        boolean completionStatusFlag = dtlVOList.stream().anyMatch(dtlVo -> completionStatusList.contains(dtlVo.getCompleteState()));
        if (completionStatusFlag) {
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "明细状态非已关闭、已完成、已取消的才可执行关闭!");
        }
        //主表ID集合
        List<Long> masIds = dtlVOList.stream().map(TaskInfoDtlRespVO::getMasId).filter(Objects::nonNull).distinct().collect(Collectors.toList());
        if (masIds.size() > 1) {
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "请选择相同任务主表的任务明细!");
        }
        //任务主表信息
        TaskInfoRespVO respVO = taskInfoService.findById(masIds.get(0));
        if (Objects.isNull(respVO)) {
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "未查询到任务主表数据!");
        }
        //详情页（任务主表状态非‘已完成’、‘已关闭’、‘已取消’）
        List<String> taskStatusList = Lists.newArrayList(
                UdcEnum.SALESMAN_TASK_STATUS_CPD.getValueCode(),
                UdcEnum.SALESMAN_TASK_STATUS_CCD.getValueCode(),
                UdcEnum.SALESMAN_TASK_STATUS_CSD.getValueCode()
        );
        if (taskStatusList.contains(respVO.getState())) {
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "任务主表状态非已完成、已关闭、已取消的才可执行关闭!");
        }

        respVO.setDtlRespVOList(dtlVOList);

        return respVO;
    }

    /**
     * 根据任务明细ID批量关闭
     *
     * @param ids 任务明细ID
     * @return
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void closeTaskBatch(List<Long> ids) {
        //根据任务明细ID批量关闭时查询主表和明细数据并进行校验
        TaskInfoRespVO respVO = closeCheckSelect(ids);
        List<TaskInfoDtlRespVO> dtlVOList = respVO.getDtlRespVOList();
        //点击确认后，将任务明细状态更新为‘已关闭’状态，相关执行记录将状态更新为‘已关闭’；
        //根据任务明细ID批量更新完成状态-已关闭
        updateCompleteStateByIdBatch(UdcEnum.SALESMAN_TASK_COMPLETION_STATUS_CSD.getValueCode(), ids);
        //相关执行记录将状态更新为‘已关闭’；
        closeUpdateExecuteRecord(dtlVOList);
        //此时任务主表状态判断：如果有明细状态为已完成+已关闭，则主表状态为已完成，如果明细全部为已关闭，
        // 则主表状态更新为已关闭。如果有明细有其他状态的，保持主表任务状态不变。
        //根据任务主表ID查询所有的明细
        List<TaskInfoDtlRespVO> dtlAllVoList = selectByMasId(respVO.getId());
        //先判断如果明细全部为已关闭，
        boolean completionStatusFlag1 = dtlAllVoList.stream().allMatch(dtlVo -> Objects.equals(dtlVo.getCompleteState(), UdcEnum.SALESMAN_TASK_COMPLETION_STATUS_CSD.getValueCode()));
        if (completionStatusFlag1) {
            ///则主表状态更新为已关闭
            taskInfoService.updateStateById(UdcEnum.SALESMAN_TASK_STATUS_CSD.getValueCode(), respVO.getId());
        } else {
            //否则再判断如果有明细状态为已完成+已关闭，
            List<String> completionStatusList = Lists.newArrayList(
                    UdcEnum.SALESMAN_TASK_COMPLETION_STATUS_CPD.getValueCode(),
                    UdcEnum.SALESMAN_TASK_COMPLETION_STATUS_CSD.getValueCode()
            );
            boolean completionStatusFlag = dtlAllVoList.stream().allMatch(dtlVo -> completionStatusList.contains(dtlVo.getCompleteState()));
            if (completionStatusFlag) {
                //则主表状态为已完成
                taskInfoService.updateStateAndCompleteTime(UdcEnum.SALESMAN_TASK_STATUS_CPD.getValueCode(), LocalDateTime.now(), respVO.getId());
            }
        }

    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void del(List<Long> ids){
        taskInfoDtlRepo.deleteAllById(ids);
    }

    /**
     * 关闭-相关执行记录将状态更新为‘已关闭’；
     *
     * @param dtlRespVOList 任务明细信息
     * @return
     */
    private void closeUpdateExecuteRecord(List<TaskInfoDtlRespVO> dtlRespVOList) {
        //更新执行记录
        //获取任务关联的执行记录编码
        List<String> recordCodes = dtlRespVOList.stream().map(TaskInfoDtlRespVO::getExecutRecordCode).filter(Objects::nonNull).distinct().collect(Collectors.toList());
        if (!CollectionUtils.isEmpty(recordCodes)) {
            exectRecordService.close(recordCodes);
        }

    }

    /**
     * 任务明细分页查询
     *
     * @param pageParam 入参
     * @return 任务明细信息集合
     */
    @Override
    @SysCodeProc
    public PagingVO<TaskInfoDtlRespVO> page(TaskInfoDtlQueryVO pageParam) {
        PagingVO<TaskInfoDtlRespVO> pagingVO = taskInfoDtlRepoProc.page(pageParam);
        if (CollectionUtils.isEmpty(pagingVO.getRecords())) {
            return PagingVO.<TaskInfoDtlRespVO>builder().total(0L).records(Collections.EMPTY_LIST).build();
        }
        List<TaskInfoDtlRespVO> dtlRespVOList = pagingVO.getRecords();

        //组装填充任务明细的相关关联字段信息
        translateTaskDtl(dtlRespVOList);
        //组装处理行政区域信息
        translateArea(dtlRespVOList);

        return PagingVO.<TaskInfoDtlRespVO>builder()
                .total(pagingVO.getTotal())
                .records(dtlRespVOList)
                .build();
    }

    /**
     * 明细数据转换处理
     *
     * @param respVOList 入参
     * @return
     */
    private void transitionDtlToCodes(List<TaskInfoDtlRespVO> respVOList) {
        respVOList.forEach(respVO -> {
            //明细的关联业务编码数据库里以逗号分隔的字符串形式保存
            //目前一行任务明细的业务编码只有一个,前端传businessCode，不使用businessCodes了。
            //List<String> businessCodes = transitionStrToCodes(respVO.getBusinessCode());
            //respVO.setBusinessCodes(businessCodes);
        });
    }

    /**
     * 根据任务明细ID批量分配执行人
     *
     * @param saveVO 任务明细入参
     * @return 任务明细ID集合
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public List<Long> updateExecuteUserBatch(TaskInfoDtlSaveVO saveVO) {
        //新增、编辑，详情（任务主表非‘已完成’、‘已关闭’、‘已取消’状态）页面展示此按钮，
        // 支持批量勾选明细状态为‘待发布’、‘待执行’的表单行，允许修改；
        TaskInfoDtlQueryVO dtlQueryVO = new TaskInfoDtlQueryVO();
        dtlQueryVO.setIds(saveVO.getIds());
        List<TaskInfoDtlRespVO> dtlVOList = selectByParam(dtlQueryVO);
        if (CollectionUtils.isEmpty(dtlVOList)) {
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "未查询到任务明细数据!");
        }
        //支持批量勾选明细状态为‘待发布’、‘待执行’的表单行，允许修改，
        List<String> completionStatusList = Lists.newArrayList(
                UdcEnum.SALESMAN_TASK_COMPLETION_STATUS_WRE.getValueCode(),
                UdcEnum.SALESMAN_TASK_COMPLETION_STATUS_WEE.getValueCode()
        );
        boolean completionStatusFlag = dtlVOList.stream().allMatch(dtlVo -> completionStatusList.contains(dtlVo.getCompleteState()));
        if (!completionStatusFlag) {
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "明细状态待发布、待执行的才可分配执行人!");
        }
        //主表ID集合
        List<Long> masIds = dtlVOList.stream().map(TaskInfoDtlRespVO::getMasId).filter(Objects::nonNull).distinct().collect(Collectors.toList());
        if (masIds.size() > 1) {
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "请选择相同任务主表的任务明细!");
        }
        //任务主表信息
        TaskInfoRespVO respVO = taskInfoService.findById(masIds.get(0));
        if (Objects.isNull(respVO)) {
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "未查询到任务主表数据!");
        }
        //新增、编辑，详情（任务主表非‘已完成’、‘已关闭’、‘已取消’状态）
        List<String> taskStatusList = Lists.newArrayList(
                UdcEnum.SALESMAN_TASK_STATUS_CPD.getValueCode(),
                UdcEnum.SALESMAN_TASK_STATUS_CCD.getValueCode(),
                UdcEnum.SALESMAN_TASK_STATUS_CSD.getValueCode()
        );
        if (taskStatusList.contains(respVO.getState())) {
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "任务主表状态非已完成、已关闭、已取消的才可分配执行人!");
        }
        //根据业务员编码查询员工ID
        //Long employeeId = selectEmployeeIdByCode(saveVO.getExecutUserCode());
        //批量更新执行人
        taskInfoDtlRepoProc.updateExecuteUserByIdBatch(saveVO.getExecutUser(), saveVO.getExecutUserId(),
                saveVO.getExecutUserCode(), null, saveVO.getIds());

        return saveVO.getIds();
    }

    /**
     * 根据业务员编码查询员工ID：业务员编码和员工编码一致
     *
     * @param code 业务员编码
     * @return 员工ID
     */
    public Long selectEmployeeIdByCode(String code) {
        if (StringUtils.isBlank(code)) {
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "业务员编码为空!");
        }
        SysEmployeeBasicDTO employeeBasicDTO = rmiEmployeeRpcService.findEmployeeByCode(code);
        if (Objects.isNull(employeeBasicDTO) || Objects.isNull(employeeBasicDTO.getId())) {
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "未查询到业务员的员工信息!");
        }

        return employeeBasicDTO.getId();
    }

    /**
     * 导入任务明细
     * <p>
     * 导入逻辑：
     * 1、校验：业务类型和关联业务编码是否匹配
     * 2、校验：关联业务编码是否系统中存在，且状态为启用
     * 3、导入后，根据业务编码带出关联业务名称、所属客户、营业时间段、省市区、详细地址信息
     * 4、执行人：如果模版维护了业务员编码的话，导入后查询业务员名称展示在系统页面上；如果没有维护则根据关联业务编码带出所属业务员展示
     * 5、完成状态（待发布）、完成时间（为空）、巡店记录（为空）、是否逾期（未逾期），按照任务新增时取默认值/为空值
     *
     * @param file 导入数据
     * @return 组装导入数据返给前端的任务明细信息
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public List<TaskInfoDtlRespVO> importTaskInfoDtl(MultipartFile file) {
        if (file == null || file.isEmpty()) {
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "请选择导入文件!");
        }
        //headRow – 头部数据行数 title总行数，headRow以下便是需要解析的数据
        Integer headRow = 2;
        //dataTypeAttr – 数据类型类的属性所在行，必须在headRow内(与java bean 属性对应的行);
        Integer dataTypeAttr = 2;
        List<TaskInfoDtlImportRespVO> dataList = null;
        try {
            dataList = (List<TaskInfoDtlImportRespVO>) ExcelImportUtil.instance(file.getInputStream())
                    .headRow(headRow)
                    .dataType(TaskInfoDtlImportRespVO.class, dataTypeAttr)
                    .readAllSync();
        } catch (Exception e) {
            log.error("任务明细导入失败:", e);
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "任务明细导入失败:" + e.getMessage());
        }

        if (CollectionUtils.isEmpty(dataList)) {
            //return Collections.EMPTY_LIST;
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "导入数据为空,请检查!");

        }
        //校验必填项
        AtomicInteger sheetCount = new AtomicInteger(1);
        dataList.forEach(dtlRespVO -> {
            sheetCount.addAndGet(1);
            if (StringUtils.isBlank(dtlRespVO.getBusinessTypeName()) || StringUtils.isBlank(dtlRespVO.getBusinessCode())) {
                throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "第" + sheetCount + "行数据存在空值,请检查!");
            }
            if (!ConstantsSale.SALESMAN_TASK_BUSINESS_TYPE_NAME_LIST.contains(dtlRespVO.getBusinessTypeName())){
                throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "第" + sheetCount + "行数据业务类型不存在,请检查!");
            }
        });

       /* //客户信息
        List<LmSaveCustRespVO> custRespVOList = new ArrayList<>();
        //查询业务编码对应的信息
        Map<String, List<TaskInfoImportCommonRespVO>> importListMap = dataList.stream().filter(dtlRespVO -> Objects.nonNull(dtlRespVO.getBusinessTypeName()))
                .collect(Collectors.groupingBy(TaskInfoImportCommonRespVO::getBusinessTypeName));
        for (String key : importListMap.keySet()) {
            if (Objects.equals(UdcEnum.SALESMAN_TASK_BUSINESS_TYPE_STORE.getValueCodeName(), key)) {

            } else if (Objects.equals(UdcEnum.SALESMAN_TASK_BUSINESS_TYPE_DEALER.getValueCodeName(), key)) {
                List<String> dealerCodes = importListMap.get(key).stream().map(TaskInfoImportCommonRespVO::getBusinessCode).filter(Objects::nonNull).distinct().collect(Collectors.toList());
                if (!CollectionUtils.isEmpty(dealerCodes)) {
                     *//*//考虑处理业务编码传多个场景，逗号分隔
                        List<String> custCodes = dealerCodes.stream().map(s -> transitionStrToCodes(s)).flatMap(Collection::stream)
                            .filter(Objects::nonNull).distinct().collect(Collectors.toList());*//*
                    custRespVOList = selectCustVO(dealerCodes);
                }

            } else if (Objects.equals(UdcEnum.SALESMAN_TASK_BUSINESS_TYPE_SALESMAN.getValueCodeName(), key)) {

            }

        }*/
        //业务处理逻辑
        List<TaskInfoDtlRespVO> taskInfoDtlRespVOList = disposeImportTaskDtl(dataList);

        return taskInfoDtlRespVOList;

    }

    /**
     * 任务明细导入的业务处理逻辑
     *
     * @param dataList 导入的数据
     * @return 组装后的任务明细信息
     */
    public List<TaskInfoDtlRespVO> disposeImportTaskDtl(List<TaskInfoDtlImportRespVO> dataList) {
        List<TaskInfoDtlImportRespVO> errorDataList = new ArrayList<>();
        List<TaskInfoDtlImportRespVO> successDataList = new ArrayList<>();
        //根据业务类型进行分组
        Map<String, List<TaskInfoDtlImportRespVO>> importListMap = dataList.stream().filter(dtlRespVO -> Objects.nonNull(dtlRespVO.getBusinessTypeName()))
                .collect(Collectors.groupingBy(TaskInfoDtlImportRespVO::getBusinessTypeName));
        //查询不同业务类型的关联业务编码对应的信息，并校验关联业务编码是否在系统中存在，且状态为启用
        for (String key : importListMap.keySet()) {
            List<TaskInfoDtlImportRespVO> importDataList = importListMap.get(key);
            List<String> businessCodes = importDataList.stream().map(TaskInfoDtlImportRespVO::getBusinessCode).filter(Objects::nonNull).distinct().collect(Collectors.toList());

            if (Objects.equals(UdcEnum.SALESMAN_TASK_BUSINESS_TYPE_STORE.getValueCodeName(), key)) {
                //门店类型的导入
                //任务明细的导入需要在立马项目里再写一个实现类继承TaskInfoDtlServiceImpl
                //来使用返回的结果再重新组装门店的信息	，因为门店客户和业务员信息都在立马里
                List<OrgStoreDetailRpcDTO> storeRespVOList = new ArrayList<>();
                if (!CollectionUtils.isEmpty(businessCodes)) {
                    storeRespVOList = selectStore(businessCodes);
                }
                //校验客户的关联业务编码是否在系统中存在，且状态为启用，并组装部分字段信息
                disposeStoreImport(importDataList, storeRespVOList, errorDataList, successDataList);
            } else if (Objects.equals(UdcEnum.SALESMAN_TASK_BUSINESS_TYPE_DEALER.getValueCodeName(), key)) {
                //客户信息
                List<LmSaveCustRespVO> custRespVOList = new ArrayList<>();
                if (!CollectionUtils.isEmpty(businessCodes)) {
                    //custRespVOList = selectCust(businessCodes);
                    custRespVOList = selectCustByCustCode2(businessCodes);
                }
                //校验客户的关联业务编码是否在系统中存在，且状态为启用，并组装部分字段信息
                disposeCustImport(importDataList, custRespVOList, errorDataList, successDataList);
            } else if (Objects.equals(UdcEnum.SALESMAN_TASK_BUSINESS_TYPE_SALESMAN.getValueCodeName(), key)) {
                //业务员信息
                List<SalesmanInfoSimpleRespVO> salesmanRespVOList = new ArrayList<>();
                if (!CollectionUtils.isEmpty(businessCodes)) {
                    salesmanRespVOList = selectSalesman(businessCodes);
                }
                //校验业务员的关联业务编码是否在系统中存在，且状态为启用，并组装部分字段信息
                disposeSalesmanImport(importDataList, salesmanRespVOList, errorDataList, successDataList);
            }

        }

        //2.校验
        checkImportTaskDtl(errorDataList);
        //3.组装剩余信息
        List<TaskInfoDtlRespVO> taskInfoDtlRespVOList = assembleImportTaskDtl(successDataList);

        return taskInfoDtlRespVOList;
    }

    /**
     * 校验门店的关联业务编码是否在系统中存在，且状态为启用，并组装部分字段信息
     *
     * @param dataList        客户类型的导入数据
     * @param storeRespVOList 门店信息
     * @param errorDataList   校验失败的数据
     * @param successDataList 校验成功的数据
     */
    public void disposeStoreImport(List<TaskInfoDtlImportRespVO> dataList, List<OrgStoreDetailRpcDTO> storeRespVOList,
                                   List<TaskInfoDtlImportRespVO> errorDataList, List<TaskInfoDtlImportRespVO> successDataList) {
        for (TaskInfoDtlImportRespVO dtlRespVO : dataList) {
            Optional<OrgStoreDetailRpcDTO> storeOptional = storeRespVOList.stream().filter(storeRespVO -> Objects.equals(storeRespVO.getStoreCode(), dtlRespVO.getBusinessCode())).findFirst();
            if (storeOptional.isPresent()) {
                OrgStoreDetailRpcDTO storeRespVO = storeOptional.get();
                //存在则赋值
                dtlRespVO.setBusinessType(UdcEnum.SALESMAN_TASK_BUSINESS_TYPE_STORE.getValueCode());
                dtlRespVO.setBusinessTypeName(dtlRespVO.getBusinessTypeName());
                dtlRespVO.setBusinessId(storeRespVO.getId());
                dtlRespVO.setCustCode2(storeRespVO.getCustCode2());
                dtlRespVO.setBusinessName(storeRespVO.getStoreName());
                //dtlRespVO.setDealerId();
                dtlRespVO.setDealerCode(storeRespVO.getCustCode());
                dtlRespVO.setDealerName(storeRespVO.getCustName());
                dtlRespVO.setBusinessTime(storeRespVO.getOpenTimeSpan());
                //dtlRespVO.setAddrNo(storeRespVO.getAddrNo());
                OrgAddrAddressRpcDTO orgAddrAddressRpcDTO = storeRespVO.getAddressRpcDTO();
                if (Objects.nonNull(orgAddrAddressRpcDTO)) {
                    dtlRespVO.setProvince(orgAddrAddressRpcDTO.getProvince());
                    dtlRespVO.setProvinceName(orgAddrAddressRpcDTO.getProvinceName());
                    dtlRespVO.setCity(orgAddrAddressRpcDTO.getCity());
                    dtlRespVO.setCityName(orgAddrAddressRpcDTO.getCityName());
                    dtlRespVO.setDistrict(orgAddrAddressRpcDTO.getCounty());
                    dtlRespVO.setDistrictName(orgAddrAddressRpcDTO.getCountyName());
                    dtlRespVO.setAddress(orgAddrAddressRpcDTO.getDetailAddr());
                }
                //关联执行人ID，用于后续关联查询执行人信息
                //此处不赋值给executUserId或executUserCode，因为后续需要根据这个字段executUserCode是否有值来判断是以导入执行人为准还是关联业务的执行人为准
                dtlRespVO.setAssociatedExecutorCode(storeRespVO.getAgentEmpCode());
                //门店提供经纬度
                if (Objects.nonNull(storeRespVO.getAddressRpcDTO())) {
                    OrgAddrAddressRpcDTO addressRpcDTO = storeRespVO.getAddressRpcDTO();
                    dtlRespVO.setXLon(addressRpcDTO.getXLon());
                    dtlRespVO.setYLat(addressRpcDTO.getYLat());
                    dtlRespVO.setCoordType(null);
                }
                dtlRespVO.setContactName(storeRespVO.getStoreManager());
                dtlRespVO.setContactPhone(storeRespVO.getStoreContPhone());

                successDataList.add(dtlRespVO);
            } else {
                //不存在则记录统一校验提示
                errorDataList.add(dtlRespVO);
            }
        }

    }

    /**
     * 校验客户的关联业务编码是否在系统中存在，且状态为启用，并组装部分字段信息
     *
     * @param dataList        客户类型的导入数据
     * @param custRespVOList  客户信息
     * @param errorDataList   校验失败的数据
     * @param successDataList 校验成功的数据
     */
    public void disposeCustImport(List<TaskInfoDtlImportRespVO> dataList, List<LmSaveCustRespVO> custRespVOList,
                                  List<TaskInfoDtlImportRespVO> errorDataList, List<TaskInfoDtlImportRespVO> successDataList) {
        for (TaskInfoDtlImportRespVO dtlRespVO : dataList) {
            //Optional<LmSaveCustRespVO> custOptional = custRespVOList.stream().filter(custRespVO -> Objects.equals(custRespVO.getCustCode(), dtlRespVO.getBusinessCode())).findFirst();
            Optional<LmSaveCustRespVO> custOptional = custRespVOList.stream().filter(custRespVO -> Objects.equals(custRespVO.getCustCode2(), dtlRespVO.getBusinessCode())).findFirst();

               /* List<String> custCodes = transitionStrToCodes(dtlRespVO.getBusinessCode());
                List<LmSaveCustRespVO> custRespVOList = custCodes.stream().map(s -> {
                    Optional<LmSaveCustRespVO> custOptional = custList.stream().filter(custRespVO -> Objects.equals(custRespVO.getCustCode(),s)).findFirst();
                    if (custOptional.isPresent()){
                        return custOptional.get();
                    }
                    return null;
                }).filter(Objects::nonNull).collect(Collectors.toList());*/

            if (custOptional.isPresent()) {
                LmSaveCustRespVO custRespVO = custOptional.get();
                //存在则赋值
                dtlRespVO.setBusinessType(UdcEnum.SALESMAN_TASK_BUSINESS_TYPE_DEALER.getValueCode());
                dtlRespVO.setBusinessTypeName(dtlRespVO.getBusinessTypeName());
                    /*String businessName = custRespVOList.stream().filter(custResp -> StringUtils.isNotBlank(custResp.getCustName()))
                            .map(LmSaveCustRespVO::getCustName).distinct().collect(Collectors.joining("-"));
                    dtlRespVO.setBusinessName(businessName);*/
                dtlRespVO.setBusinessCode(custRespVO.getCustCode());
                dtlRespVO.setBusinessId(custRespVO.getId());
                dtlRespVO.setCustCode2(custRespVO.getCustCode2());
                dtlRespVO.setBusinessName(custRespVO.getCustName());
                dtlRespVO.setAddrNo(custRespVO.getAddrNo());
                //关联执行人ID，用于后续关联查询执行人信息
                //此处不赋值给executUserId或executUserCode，因为后续需要根据这个字段executUserCode是否有值来判断是以导入执行人为准还是关联业务的执行人为准
                dtlRespVO.setAssociatedExecutorId(custRespVO.getAgentEmpId());

                dtlRespVO.setXLon(null);
                dtlRespVO.setYLat(null);
                dtlRespVO.setCoordType(null);
                dtlRespVO.setContactName(custRespVO.getContactName());
                dtlRespVO.setContactPhone(custRespVO.getContactPhone());

                successDataList.add(dtlRespVO);
            } else {
                //不存在则记录统一校验提示
                errorDataList.add(dtlRespVO);
            }
        }

    }

    /**
     * 校验业务员的关联业务编码是否在系统中存在，且状态为启用，并组装部分字段信息
     *
     * @param dataList           客户类型的导入数据
     * @param salesmanRespVOList 业务员信息
     * @param errorDataList      校验失败的数据
     * @param successDataList    校验成功的数据
     */
    public void disposeSalesmanImport(List<TaskInfoDtlImportRespVO> dataList, List<SalesmanInfoSimpleRespVO> salesmanRespVOList,
                                      List<TaskInfoDtlImportRespVO> errorDataList, List<TaskInfoDtlImportRespVO> successDataList) {
        for (TaskInfoDtlImportRespVO dtlRespVO : dataList) {
            Optional<SalesmanInfoSimpleRespVO> salesmanOptional = salesmanRespVOList.stream().filter(salesmanRespVO -> Objects.equals(salesmanRespVO.getSalesmanNo(), dtlRespVO.getBusinessCode())).findFirst();

            if (salesmanOptional.isPresent()) {
                SalesmanInfoSimpleRespVO salesmanRespVO = salesmanOptional.get();
                //存在则赋值
                dtlRespVO.setBusinessType(UdcEnum.SALESMAN_TASK_BUSINESS_TYPE_SALESMAN.getValueCode());
                dtlRespVO.setBusinessTypeName(dtlRespVO.getBusinessTypeName());
                dtlRespVO.setBusinessId(salesmanRespVO.getId());
                //dtlRespVO.setCustCode2();
                dtlRespVO.setBusinessName(salesmanRespVO.getFullName());
                //dtlRespVO.setAddrNo();
                dtlRespVO.setAddress(salesmanRespVO.getAddress());
                AreaVO areaVO = salesmanRespVO.getAreaVO();
                if (Objects.nonNull(areaVO)) {
                    dtlRespVO.setProvince(areaVO.getProvinceCode());
                    dtlRespVO.setProvinceName(areaVO.getProvinceName());
                    dtlRespVO.setCity(areaVO.getCityCode());
                    dtlRespVO.setCityName(areaVO.getCityName());
                    dtlRespVO.setDistrict(areaVO.getCountyCode());
                    dtlRespVO.setDistrictName(areaVO.getCountyName());
                }
                //关联执行人CODE，用于后续关联查询执行人信息
                dtlRespVO.setAssociatedExecutorCode(salesmanRespVO.getSalesmanNo());
                //此处不赋值给executUserCode，因为后续需要根据这个字段是否有值来判断是以导入执行人为准还是关联业务的执行人为准
                dtlRespVO.setExecutUser(salesmanRespVO.getFullName());
                dtlRespVO.setExecutUserId(salesmanRespVO.getId());

                dtlRespVO.setXLon(null);
                dtlRespVO.setYLat(null);
                dtlRespVO.setCoordType(null);
                //业务员提供联系人和电话
                dtlRespVO.setContactName(salesmanRespVO.getFullName());
                dtlRespVO.setContactPhone(salesmanRespVO.getPhone());

                successDataList.add(dtlRespVO);
            } else {
                //不存在则记录统一校验提示
                errorDataList.add(dtlRespVO);
            }
        }

    }

    /**
     * 组装导入数据剩余的信息
     *
     * @param dataList 导入的数据包含部分组装过的信息
     * @return 组装后的任务明细出参信息
     */
    public List<TaskInfoDtlRespVO> assembleImportTaskDtl(List<TaskInfoDtlImportRespVO> dataList) {
        //List<TaskInfoDtlRespVO> taskInfoDtlRespVOList = new ArrayList<>();
        //客户时的所属业务员存的是ID,关联业务编码关联的执行人
        List<Long> associatedExecutorIds = dataList.stream().map(TaskInfoDtlImportRespVO::getAssociatedExecutorId).filter(Objects::nonNull).distinct().collect(Collectors.toList());
        List<SalesmanInfoSimpleRespVO> custSalesmanVoList = new ArrayList<>();
        if (!CollectionUtils.isEmpty(associatedExecutorIds)) {
            SalesmanInfoSimpleQueryVO salesmanQuery1VO = new SalesmanInfoSimpleQueryVO();
            salesmanQuery1VO.setIds(associatedExecutorIds);
            custSalesmanVoList = salesmanInfoService.simpleQuery(salesmanQuery1VO);
        }

        //查询执行人信息，也就是业务员信息
        //导入时填写的执行人(业务员编码)
        List<String> executUserCodes = dataList.stream().map(TaskInfoDtlImportRespVO::getExecutUserCode).filter(Objects::nonNull).distinct().collect(Collectors.toList());
        //关联业务编码关联的执行人(业务员编码)
        List<String> associatedExecutorCodes = dataList.stream().map(TaskInfoDtlImportRespVO::getAssociatedExecutorCode).filter(Objects::nonNull).distinct().collect(Collectors.toList());
        List executUserCodeList = Stream.of(executUserCodes, associatedExecutorCodes).flatMap(Collection::stream).filter(Objects::nonNull).distinct().collect(Collectors.toList());

        List<SalesmanInfoSimpleRespVO> salesmanVoList = new ArrayList<>();
        if (!CollectionUtils.isEmpty(executUserCodeList)) {
            SalesmanInfoSimpleQueryVO salesmanQuery2VO = new SalesmanInfoSimpleQueryVO();
            salesmanQuery2VO.setCodes(executUserCodeList);
            salesmanVoList = salesmanInfoService.simpleQuery(salesmanQuery2VO);
        }

        //根据地址号查询地址信息
        List<Long> addrNos = dataList.stream().map(TaskInfoDtlImportRespVO::getAddrNo).filter(Objects::nonNull).distinct().collect(Collectors.toList());
        List<OrgAddressRpcDTO> orgAddressRpcDTOList = new ArrayList<>();
        if (!CollectionUtils.isEmpty(addrNos)) {
            OrgAddressRpcDtoParam orgAddressRpcDtoParam = new OrgAddressRpcDtoParam();
            orgAddressRpcDtoParam.setAddrNos(addrNos);
            orgAddressRpcDTOList = rmiOrgAddrService.findAddrAddressListByParam(orgAddressRpcDtoParam);
        }

        List<TaskInfoDtlImportRespVO> errorDataList = new ArrayList<>();
        for (TaskInfoDtlImportRespVO dtlRespVO : dataList) {
            //TaskInfoDtlRespVO taskInfoDtlRespVO = new TaskInfoDtlRespVO();
            //完成状态:待发布
            dtlRespVO.setCompleteState(UdcEnum.SALESMAN_TASK_COMPLETION_STATUS_WRE.getValueCode());
            dtlRespVO.setCompleteStateName(UdcEnum.SALESMAN_TASK_COMPLETION_STATUS_WRE.getValueCodeName());
            //是否逾期:未逾期
            dtlRespVO.setDelayFlag(UdcEnum.SALESMAN_TASK_DELAY_FLAG_N.getValueCode());
            dtlRespVO.setDelayFlagName(UdcEnum.SALESMAN_TASK_DELAY_FLAG_N.getValueCodeName());

            if (Objects.equals(UdcEnum.SALESMAN_TASK_BUSINESS_TYPE_STORE.getValueCodeName(), dtlRespVO.getBusinessTypeName())) {
                fillStoreDtl(dtlRespVO, salesmanVoList,errorDataList);
            } else if (Objects.equals(UdcEnum.SALESMAN_TASK_BUSINESS_TYPE_DEALER.getValueCodeName(), dtlRespVO.getBusinessTypeName())) {
                fillCustDtl(dtlRespVO, orgAddressRpcDTOList, custSalesmanVoList, salesmanVoList,errorDataList);
            } else if (Objects.equals(UdcEnum.SALESMAN_TASK_BUSINESS_TYPE_SALESMAN.getValueCodeName(), dtlRespVO.getBusinessTypeName())) {
                fillSalesmanDtl(dtlRespVO, salesmanVoList,errorDataList);
            }
        }

        //2.校验是否有未匹配到执行人信息
        checkImportTaskDtlExecuteUser(errorDataList);

        List<TaskInfoDtlRespVO> taskInfoDtlDOList = dataList.stream().map(TaskInfoDtlConvert.INSTANCE::importVoToRespVo).collect(Collectors.toList());

        return taskInfoDtlDOList;
    }

    /**
     * 组装填充任务明细数据-门店类型
     *
     * @param dtlRespVO      需要组装的任务明细数据
     * @param salesmanVoList 客户所属业务员和导入时维护的业务员信息
     * @param errorDataList 未匹配到执行人的数据
     */
    public void fillStoreDtl(TaskInfoDtlImportRespVO dtlRespVO, List<SalesmanInfoSimpleRespVO> salesmanVoList,List<TaskInfoDtlImportRespVO> errorDataList) {

        //执行人：如果模版维护了业务员编码的话，导入后查询业务员名称展示在系统页面上；如果没有维护则根据关联业务编码带出所属业务员展示
        if (StringUtils.isBlank(dtlRespVO.getExecutUserCode())) {
            Optional<SalesmanInfoSimpleRespVO> salesmanOptional = salesmanVoList.stream().filter(salesmanVo -> Objects.equals(salesmanVo.getSalesmanNo(), dtlRespVO.getAssociatedExecutorCode())).findFirst();
            if (salesmanOptional.isPresent()) {
                SalesmanInfoSimpleRespVO salesmanVo = salesmanOptional.get();
                dtlRespVO.setExecutUserCode(salesmanVo.getSalesmanNo());
                dtlRespVO.setExecutUser(salesmanVo.getFullName());
                dtlRespVO.setExecutUserId(salesmanVo.getId());
            }
        } else {
            Optional<SalesmanInfoSimpleRespVO> salesmanVoOptional = salesmanVoList.stream().filter(salesmanVo -> Objects.equals(salesmanVo.getSalesmanNo(), dtlRespVO.getExecutUserCode())).findFirst();
            if (salesmanVoOptional.isPresent()) {
                SalesmanInfoSimpleRespVO salesmanVo = salesmanVoOptional.get();
                dtlRespVO.setExecutUserCode(salesmanVo.getSalesmanNo());
                dtlRespVO.setExecutUser(salesmanVo.getFullName());
                dtlRespVO.setExecutUserId(salesmanVo.getId());

            }else {
                //未匹配到执行人的则记录统一校验提示
                errorDataList.add(dtlRespVO);
            }
        }
    }

    /**
     * 组装填充任务明细数据-客户类型
     *
     * @param dtlRespVO            需要组装的任务明细数据
     * @param orgAddressRpcDTOList 地址信息
     * @param custSalesmanVoList   客户所属业务员信息
     * @param salesmanVoList       客户所属业务员和导入时维护的业务员信息
     * @param errorDataList 未匹配到执行人的数据
     */
    public void fillCustDtl(TaskInfoDtlImportRespVO dtlRespVO, List<OrgAddressRpcDTO> orgAddressRpcDTOList,
                            List<SalesmanInfoSimpleRespVO> custSalesmanVoList, List<SalesmanInfoSimpleRespVO> salesmanVoList,
                            List<TaskInfoDtlImportRespVO> errorDataList) {
        //维护地址信息
        Optional<OrgAddressRpcDTO> orgAddressOptional = orgAddressRpcDTOList.stream().filter(orgAddressVo -> Objects.equals(orgAddressVo.getAddrNo(), dtlRespVO.getAddrNo())).findFirst();
        if (orgAddressOptional.isPresent()) {
            OrgAddressRpcDTO orgAddressRpcDTO = orgAddressOptional.get();
            dtlRespVO.setAddress(orgAddressRpcDTO.getDetailAddr());
            dtlRespVO.setCountry(orgAddressRpcDTO.getCountry());
            dtlRespVO.setCountryName(orgAddressRpcDTO.getCountryName());
            dtlRespVO.setProvince(orgAddressRpcDTO.getProvince());
            dtlRespVO.setProvinceName(orgAddressRpcDTO.getProvinceName());
            dtlRespVO.setCity(orgAddressRpcDTO.getCity());
            dtlRespVO.setCityName(orgAddressRpcDTO.getCityName());
            dtlRespVO.setDistrict(orgAddressRpcDTO.getCounty());
            dtlRespVO.setDistrictName(orgAddressRpcDTO.getCountyName());

        }

        //执行人：如果模版维护了业务员编码的话，导入后查询业务员名称展示在系统页面上；如果没有维护则根据关联业务编码带出所属业务员展示
        if (StringUtils.isBlank(dtlRespVO.getExecutUserCode())) {
            Optional<SalesmanInfoSimpleRespVO> custSalesmanOptional = custSalesmanVoList.stream().filter(custSalesmanVo -> Objects.equals(custSalesmanVo.getId(), dtlRespVO.getAssociatedExecutorId())).findFirst();
            if (custSalesmanOptional.isPresent()) {
                SalesmanInfoSimpleRespVO custSalesmanVo = custSalesmanOptional.get();
                dtlRespVO.setExecutUserCode(custSalesmanVo.getSalesmanNo());
                dtlRespVO.setExecutUser(custSalesmanVo.getFullName());
                dtlRespVO.setExecutUserId(custSalesmanVo.getId());
            }
        } else {
            Optional<SalesmanInfoSimpleRespVO> salesmanVoOptional = salesmanVoList.stream().filter(salesmanVo -> Objects.equals(salesmanVo.getSalesmanNo(), dtlRespVO.getExecutUserCode())).findFirst();
            if (salesmanVoOptional.isPresent()) {
                SalesmanInfoSimpleRespVO salesmanVo = salesmanVoOptional.get();
                dtlRespVO.setExecutUserCode(salesmanVo.getSalesmanNo());
                dtlRespVO.setExecutUser(salesmanVo.getFullName());
                dtlRespVO.setExecutUserId(salesmanVo.getId());

            }else {
                //未匹配到执行人的则记录统一校验提示
                errorDataList.add(dtlRespVO);
            }
        }
    }

    /**
     * 组装填充任务明细数据-业务员类型
     *
     * @param dtlRespVO      需要组装的任务明细数据
     * @param salesmanVoList 客户所属业务员和导入时维护的业务员信息
     * @param errorDataList 未匹配到执行人的数据
     */
    public void fillSalesmanDtl(TaskInfoDtlImportRespVO dtlRespVO, List<SalesmanInfoSimpleRespVO> salesmanVoList,List<TaskInfoDtlImportRespVO> errorDataList) {

        //执行人：如果模版维护了业务员编码的话，导入后查询业务员名称展示在系统页面上；如果没有维护则根据关联业务编码带出所属业务员展示
        if (StringUtils.isBlank(dtlRespVO.getExecutUserCode())) {
            Optional<SalesmanInfoSimpleRespVO> salesmanOptional = salesmanVoList.stream().filter(salesmanVo -> Objects.equals(salesmanVo.getSalesmanNo(), dtlRespVO.getAssociatedExecutorCode())).findFirst();
            if (salesmanOptional.isPresent()) {
                SalesmanInfoSimpleRespVO salesmanVo = salesmanOptional.get();
                dtlRespVO.setExecutUserCode(salesmanVo.getSalesmanNo());
                dtlRespVO.setExecutUser(salesmanVo.getFullName());
                dtlRespVO.setExecutUserId(salesmanVo.getId());
            }
        } else {
            Optional<SalesmanInfoSimpleRespVO> salesmanVoOptional = salesmanVoList.stream().filter(salesmanVo -> Objects.equals(salesmanVo.getSalesmanNo(), dtlRespVO.getExecutUserCode())).findFirst();
            if (salesmanVoOptional.isPresent()) {
                SalesmanInfoSimpleRespVO salesmanVo = salesmanVoOptional.get();
                dtlRespVO.setExecutUserCode(salesmanVo.getSalesmanNo());
                dtlRespVO.setExecutUser(salesmanVo.getFullName());
                dtlRespVO.setExecutUserId(salesmanVo.getId());

            }else {
                //未匹配到执行人的则记录统一校验提示
                errorDataList.add(dtlRespVO);
            }
        }
    }

    /**
     * 校验导入数据是否有校验失败的
     *
     * @param errorDataList 校验错误数据的集合
     */
    public void checkImportTaskDtl(List<TaskInfoDtlImportRespVO> errorDataList) {
        if (!CollectionUtils.isEmpty(errorDataList)) {
            String checkResult = errorDataList.stream().map(vo ->
                    "业务类型:(" + vo.getBusinessTypeName() + ")-关联业务编码:(" + vo.getBusinessCode() + ")"
            ).collect(Collectors.joining(";", "关联业务编码不存在:[", "].请检查!"));
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, checkResult);
        }
    }

    /**
     * 校验导入数据是否有未匹配到执行人信息的
     *
     * @param errorDataList 校验错误数据的集合
     */
    public void checkImportTaskDtlExecuteUser(List<TaskInfoDtlImportRespVO> errorDataList) {
        if (!CollectionUtils.isEmpty(errorDataList)) {
            String checkResult = errorDataList.stream().map(vo ->
                    "业务类型:(" + vo.getBusinessTypeName() + ")-关联业务编码:(" + judgmentBusinessCode(vo) + ")"
            ).collect(Collectors.joining(";", "未查询到执行人信息:[", "].请检查!"));
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, checkResult);
        }
    }

    /**
     * 根据业务类型判断关联业务编码取值
     *
     * @param dtlRespVO 出参数据对象
     * @return 关联业务编码
     */
    private String judgmentBusinessCode(TaskInfoDtlImportRespVO dtlRespVO) {
        String businessCode = "";
        if (Objects.equals(UdcEnum.SALESMAN_TASK_BUSINESS_TYPE_STORE.getValueCodeName(), dtlRespVO.getBusinessTypeName())) {
            businessCode = dtlRespVO.getBusinessCode();
        } else if (Objects.equals(UdcEnum.SALESMAN_TASK_BUSINESS_TYPE_DEALER.getValueCodeName(), dtlRespVO.getBusinessTypeName())) {
            //客户时，导入存的关联业务编码是客户号，而关联业务编码BusinessCode转换后存的是客户编码，故展示提示信息时需要展示客户号
            businessCode = StringUtils.isBlank(dtlRespVO.getCustCode2()) ? dtlRespVO.getBusinessCode() : dtlRespVO.getCustCode2();
        } else if (Objects.equals(UdcEnum.SALESMAN_TASK_BUSINESS_TYPE_SALESMAN.getValueCodeName(), dtlRespVO.getBusinessTypeName())) {
            businessCode = dtlRespVO.getBusinessCode();
        }
        return businessCode;
    }

    /**
     * 根据门店编码查询启用的门店信息
     *
     * @param storeCodes 门店编码
     * @return 启用的门店信息
     */
    public List<OrgStoreDetailRpcDTO> selectStore(List<String> storeCodes) {
        if (CollectionUtils.isEmpty(storeCodes)) {
            return Collections.EMPTY_LIST;
        }
        List<OrgStoreDetailRpcDTO> storeRespVOList = rmiOrgStoreRpcService.findStoreByStoreCodes(storeCodes);
        if (CollectionUtils.isEmpty(storeRespVOList)) {
            return Collections.EMPTY_LIST;
        }
        //过滤出启用状态的
        List<OrgStoreDetailRpcDTO> storeVoList = storeRespVOList.stream().filter(storeRespVO -> Objects.equals(storeRespVO.getStoreStatus(), ConstantsSale.STORE_STATUS_ACTIVE))
                .collect(Collectors.toList());

        return storeVoList;
    }

    /**
     * 根据客户编码查询启用的客户信息
     *
     * @param custCodes 客户编码
     * @return 启用的客户信息
     */
    public List<LmSaveCustRespVO> selectCust(List<String> custCodes) {
        if (CollectionUtils.isEmpty(custCodes)) {
            return Collections.EMPTY_LIST;
        }
        List<LmSaveCustRespVO> custRespVOList = crmCustService.getCustInfoByCustCodes(custCodes);
        if (CollectionUtils.isEmpty(custRespVOList)) {
            return Collections.EMPTY_LIST;
        }
        //过滤出启用状态的
        List<LmSaveCustRespVO> custVoList = custRespVOList.stream().filter(custRespVO -> Objects.equals(custRespVO.getCustStatus(), UdcEnum.CRM_CUST_STATUS_ACTIVE.getValueCode()))
                .collect(Collectors.toList());

        return custVoList;
    }

    /**
     * 根据客户号查询启用的客户信息
     *
     * @param custCode2List 客户号
     * @return 启用的客户信息
     */
    public List<LmSaveCustRespVO> selectCustByCustCode2(List<String> custCode2List) {
        if (CollectionUtils.isEmpty(custCode2List)) {
            return Collections.EMPTY_LIST;
        }
        List<LmSaveCustRespVO> custRespVOList = crmCustService.getCustInfoByCustCode2s(custCode2List);
        if (CollectionUtils.isEmpty(custRespVOList)) {
            return Collections.EMPTY_LIST;
        }
        //过滤出启用状态的
        List<LmSaveCustRespVO> custVoList = custRespVOList.stream().filter(custRespVO -> Objects.equals(custRespVO.getCustStatus(), UdcEnum.CRM_CUST_STATUS_ACTIVE.getValueCode()))
                .collect(Collectors.toList());

        if (CollectionUtils.isEmpty(custVoList)) {
            return Collections.EMPTY_LIST;
        }
       /* //客户时的所属业务员存的是ID,关联业务编码关联的执行人
        List<Long> agentEmpIds = custVoList.stream().map(LmSaveCustRespVO::getAgentEmpId).filter(Objects::nonNull).distinct().collect(Collectors.toList());
        List<SalesmanInfoSimpleRespVO> custSalesmanVoList = new ArrayList<>();
        if (!CollectionUtils.isEmpty(agentEmpIds)) {
            SalesmanInfoSimpleQueryVO salesmanQuery1VO = new SalesmanInfoSimpleQueryVO();
            salesmanQuery1VO.setIds(agentEmpIds);
            custSalesmanVoList = salesmanInfoService.simpleQuery(salesmanQuery1VO);
        }*/

        return custVoList;
    }

    /**
     * 根据业务员编码查询启用的业务员信息
     *
     * @param salesmanCodes 业务员编码
     * @return 启用的业务员信息
     */
    public List<SalesmanInfoSimpleRespVO> selectSalesman(List<String> salesmanCodes) {
        if (CollectionUtils.isEmpty(salesmanCodes)) {
            return Collections.EMPTY_LIST;
        }
        SalesmanInfoSimpleQueryVO salesmanQueryVO = new SalesmanInfoSimpleQueryVO();
        salesmanQueryVO.setCodes(salesmanCodes);
        salesmanQueryVO.setEnableStatus(1);
        List<SalesmanInfoSimpleRespVO> salesmanRespVOList = salesmanInfoService.simpleQuery(salesmanQueryVO);
        if (CollectionUtils.isEmpty(salesmanRespVOList)) {
            return Collections.EMPTY_LIST;
        }

        return salesmanRespVOList;
    }

     /*private void disposeImportTaskDtl(List<TaskInfoImportCommonRespVO> dataList, List<LmSaveCustRespVO> custList) {
        List<TaskInfoImportCommonRespVO> errorDataList = new ArrayList<>();
        for (TaskInfoImportCommonRespVO dtlRespVO : dataList) {
            if (Objects.equals(UdcEnum.SALESMAN_TASK_BUSINESS_TYPE_DEALER.getValueCodeName(), dtlRespVO.getBusinessTypeName())) {
                Optional<LmSaveCustRespVO> custOptional = custList.stream().filter(custRespVO -> Objects.equals(custRespVO.getCustCode(), dtlRespVO.getBusinessCode())).findFirst();
               *//* List<String> custCodes = transitionStrToCodes(dtlRespVO.getBusinessCode());
                List<LmSaveCustRespVO> custRespVOList = custCodes.stream().map(s -> {
                    Optional<LmSaveCustRespVO> custOptional = custList.stream().filter(custRespVO -> Objects.equals(custRespVO.getCustCode(),s)).findFirst();
                    if (custOptional.isPresent()){
                        return custOptional.get();
                    }
                    return null;
                }).filter(Objects::nonNull).collect(Collectors.toList());*//*

                if (custOptional.isPresent()) {
                    LmSaveCustRespVO custRespVO = custOptional.get();
                    //存在则赋值
                    dtlRespVO.setBusinessType(UdcEnum.SALESMAN_TASK_BUSINESS_TYPE_DEALER.getValueCode());
                    dtlRespVO.setBusinessTypeName(dtlRespVO.getBusinessTypeName());
                    *//*String businessName = custRespVOList.stream().filter(custResp -> StringUtils.isNotBlank(custResp.getCustName()))
                            .map(LmSaveCustRespVO::getCustName).distinct().collect(Collectors.joining("-"));
                    dtlRespVO.setBusinessName(businessName);*//*
                    dtlRespVO.setBusinessId(custRespVO.getId());
                    dtlRespVO.setCustCode2(custRespVO.getCustCode2());
                    dtlRespVO.setBusinessName(custRespVO.getCustName());
                    dtlRespVO.setAddrNo(custRespVO.getAddrNo());
                    dtlRespVO.setExecutUserId(custRespVO.getAgentEmpId());
                } else {
                    //不存在则记录统一校验提示
                    errorDataList.add(dtlRespVO);
                }
            }
        }

        //2.校验
        checkImportTaskDtl(errorDataList);
        //3.组装剩余信息
        //assembleImportTaskDtl();
    }*/


    /**
     * APP端我的任务-任务明细分页查询
     * <p>
     * 登陆业务员查看自己的任务列表
     *
     * @param pageParam 入参
     * @return 任务明细信息集合
     */
    @Override
    @SysCodeProc
    public PagingVO<TaskInfoAppRespVO> appDtlPage(TaskInfoAppQueryVO pageParam) {
        //获取当前用户信息
        SysUserDTO sysUserDTO = getSysUser();
        //根据当前用户查询所属员工信息
        SysEmployeeBasicDTO employeeBasicDTO = getEmployeeByUser(sysUserDTO);
        //用于业务员查看所属的任务子项列表，从中台后台获取任务明细执行人为业务员
        pageParam.setExecutUserCode(employeeBasicDTO.getCode());


        return appDtlCommonPage(pageParam);
    }

    /**
     * APP端团队任务列表-任务明细分页查询
     * <p>
     * 找下级业务员 登陆业务员下级团队的任务列表
     *
     * @param queryVO 入参
     * @return 任务明细信息集合
     */
    @Override
    @SysCodeProc
    public PagingVO<TaskInfoAppRespVO> appDtlTeamPage(TaskInfoAppQueryVO queryVO) {
        //获取当前用户信息
        SysUserDTO sysUserDTO = getSysUser();
        //根据当前用户查询所属员工信息
        SysEmployeeBasicDTO employeeBasicDTO = getEmployeeByUser(sysUserDTO);

        //获取下属员工
        /*List<EmployeeUnderlingDTO> subEmployeeList = rmiEmployeeRpcService.getUnderlingByCode(employeeBasicDTO.getCode(), true, true);
        if (CollectionUtils.isEmpty(subEmployeeList)) {
            return PagingVO.<TaskInfoAppRespVO>builder().total(0L).records(Collections.EMPTY_LIST).build();
        }
        List<String> subEmployeeCodeList = subEmployeeList.stream().map(EmployeeUnderlingDTO::getCode).filter(Objects::nonNull).distinct().collect(Collectors.toList());
        if (CollectionUtils.isEmpty(subEmployeeCodeList)) {
            return PagingVO.<TaskInfoAppRespVO>builder().total(0L).records(Collections.EMPTY_LIST).build();
        }
        queryVO.setExecutUserCodes(subEmployeeCodeList);*/

        if (Objects.isNull(employeeBasicDTO.getCode())){
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "业务员编码为空");
        }
        //获取下属员工
        List<String> subEmployeeCodeList = queryAllSubEmployeeByCode(employeeBasicDTO.getCode());
        if (CollectionUtils.isEmpty(subEmployeeCodeList)) {
            return PagingVO.<TaskInfoAppRespVO>builder().total(0L).records(Collections.EMPTY_LIST).build();
        }
        queryVO.setExecutUserCodes(subEmployeeCodeList);

        return appDtlCommonPage(queryVO);
    }

    private List<String> queryAllSubEmployeeByCode(String salesmanNo){
        List<EmployeeUnderlingDTO> resultData = rmiEmployeeRpcService.getUnderlingByCode(salesmanNo,Boolean.TRUE,Boolean.TRUE);
        if (CollectionUtils.isEmpty(resultData)) {
            return Collections.EMPTY_LIST;
        }
        Set<String> codeList = new HashSet<>();
        //不包含当前登陆人
        //codeList.add(salesmanNo);
        if (!CollectionUtils.isEmpty(resultData)) {
            List<EmployeeUnderlingDTO> listEmp = new ArrayList<>();
            List<EmployeeUnderlingDTO> res = this.queryAllSubTreeToList(resultData,listEmp);
            codeList.addAll(res.stream().map(EmployeeUnderlingDTO::getCode).collect(Collectors.toList()));
        }

       return codeList.stream().filter(Objects::nonNull).collect(Collectors.toList());
    }

    private List<EmployeeUnderlingDTO> queryAllSubTreeToList(List<EmployeeUnderlingDTO> source ,List<EmployeeUnderlingDTO> obj){
        source.stream().forEach(v ->{
            EmployeeUnderlingDTO t = new EmployeeUnderlingDTO();
            BeanUtils.copyProperties(v,t);
            t.setUnderlingList(new ArrayList<>());
            obj.add(t);
            if(CollectionUtil.isNotEmpty(v.getUnderlingList())){
                queryAllSubTreeToList(v.getUnderlingList(),obj);
            }
        });
        return obj;
    }

    /**
     * APP端-任务明细分页查询共用部分
     * <p>
     * 根据任务明细的完成状态区分TAB页（包括：待执行、进行中、待审核、已完成、审核拒绝、已逾期），每个TAB页的展示逻辑：
     * a、待执行：按照紧急程度优先级展示，同一个紧急程度内的任务单据根据截止时间正序展示。
     * b、待审核：按照紧急程度优先级展示，同一个紧急程度内的任务单据根据截止时间正序展示。
     * c、审核拒绝：按照紧急程度优先级展示，同一个紧急程度内的任务单据根据截止时间正序展示。
     * d、已完成：按照任务创建时间倒序展示已完成状态的任务列表。
     * e、已逾期：按照紧急程度优先级展示，同一个紧急程度内的任务单据根据截止时间正序展示。
     * f、进行中：按照紧急程度优先级展示，同一个紧急程度内的任务单据根据截止时间正序展示。
     * g、全部：
     * 先按照状态：已逾期>进行中>待执行>审核拒绝>待审核>已完成；
     * 同状态中的任务，再按照紧急程度优先级排序；
     * 同状态同紧急程度的任务列表，最后按照任务截止时间正序排序。
     * 紧急程度优先级：按照紧急程度UDC的编码1、2、3、.....，这样的优先级在前端展示。
     *
     * @param queryVO 入参
     * @return 任务明细信息集合
     */
    @SysCodeProc
    private PagingVO<TaskInfoAppRespVO> appDtlCommonPage(TaskInfoAppQueryVO queryVO) {
        //任务明细完成状态非‘待发布’、‘已关闭’、‘已取消’的数据；
        List<String> completionStatusList = Lists.newArrayList(
                UdcEnum.SALESMAN_TASK_COMPLETION_STATUS_WEE.getValueCode(), UdcEnum.SALESMAN_TASK_COMPLETION_STATUS_IPS.getValueCode(),
                UdcEnum.SALESMAN_TASK_COMPLETION_STATUS_WAT.getValueCode(), UdcEnum.SALESMAN_TASK_COMPLETION_STATUS_ARN.getValueCode(),
                UdcEnum.SALESMAN_TASK_COMPLETION_STATUS_CPD.getValueCode()
        );
        queryVO.setCompleteStates(completionStatusList);

        //输入任务名称和执行人进行模糊查询
        if (StringUtils.isNotBlank(queryVO.getQueryKeyword())) {
            //根据执行人名称查询员工的全名(FullName)进行模糊匹配，目前只有分页的接口，默认1000条
            List<EmployeePageRespDTO> employeePageRespDTOList = rmiEmployeeRpcService.findEmployeeByFullName(queryVO.getQueryKeyword());
            //List<SysEmployeeBasicDTO> employeeBasicDTOList = new ArrayList<>();
            //employeeBasicDTOList.add(employeeDTO1);
            if (!CollectionUtils.isEmpty(employeePageRespDTOList)) {
                List<String> executUserCodes = employeePageRespDTOList.stream().map(EmployeePageRespDTO::getCode).filter(Objects::nonNull).distinct().collect(Collectors.toList());
                queryVO.setExecutUserCodeList(executUserCodes);
            }

        }


        PagingVO pagingVO = null;
        if (queryVO.getAllStateSign()) {
            //查询全部明细状态
            pagingVO = taskInfoDtlRepoProc.selectAppTask(queryVO);
        } else if (Objects.equals(queryVO.getCompleteState(), UdcEnum.SALESMAN_TASK_COMPLETION_STATUS_CPD.getValueCode())) {
            //查询已完成状态
            pagingVO = taskInfoDtlRepoProc.selectAppTaskComplete(queryVO);
        } else if (Objects.equals(queryVO.getCompleteState(), ConstantsSale.SALESMAN_TASK_DELAY_FLAG_OVERDUE)) {
            //查询已逾期,前端不好处理，按照状态的值传过来
            //任务明细完成状态非‘待发布’、‘已关闭’、‘已取消’、‘已完成’的数据；
            List<String> delayFlagCompletionStatusList = Lists.newArrayList(
                    UdcEnum.SALESMAN_TASK_COMPLETION_STATUS_WEE.getValueCode(), UdcEnum.SALESMAN_TASK_COMPLETION_STATUS_IPS.getValueCode(),
                    UdcEnum.SALESMAN_TASK_COMPLETION_STATUS_WAT.getValueCode(), UdcEnum.SALESMAN_TASK_COMPLETION_STATUS_ARN.getValueCode());
            queryVO.setCompleteStates(delayFlagCompletionStatusList);
            queryVO.setDelayFlag(UdcEnum.SALESMAN_TASK_DELAY_FLAG_Y.getValueCode());
            //查询已逾期时，前端传的是OVERDUE，明细状态没有这个的，故要置为空，避免作为查询条件
            queryVO.setCompleteState(null);
            pagingVO = taskInfoDtlRepoProc.selectAppTaskOverdue(queryVO);
        }else if (completionStatusList.contains(queryVO.getCompleteState())) {
            //查询其它状态
            pagingVO = taskInfoDtlRepoProc.selectAppTaskOther(queryVO);
        }

        if (Objects.isNull(pagingVO) || CollectionUtils.isEmpty(pagingVO.getRecords())) {
            return PagingVO.<TaskInfoAppRespVO>builder().total(0L).records(Collections.EMPTY_LIST).build();
        }
        List<TaskInfoAppRespVO> dtlRespVOList = pagingVO.getRecords();

        //组装填充APP端任务的相关关联字段信息
        translateAppTask(dtlRespVOList);
        //组装处理APP端行政区域信息
        translateAppArea(dtlRespVOList);

        return PagingVO.<TaskInfoAppRespVO>builder()
                .total(pagingVO.getTotal())
                .records(dtlRespVOList)
                .build();
    }

    /**
     * 组装填充APP端任务的相关关联字段信息
     *
     * @param appRespVOS APP端任务数据信息
     * @return
     */
    private void translateAppTask(List<TaskInfoAppRespVO> appRespVOS) {
        //执行记录-编码,执行记录没有名称
        //List<String> executeRecordCodeList = appRespVOS.stream().filter(Objects::nonNull).map(TaskInfoAppRespVO::getExecutRecordCode).distinct().collect(Collectors.toList());
        //执行人编码-业务员-业务员表没有名称，要通过业务员编码查询员工的名称,业务员编码和员工编码是相同的
        Set<String> executeUserCodeList = appRespVOS.stream().map(TaskInfoAppRespVO::getExecutUserCode).filter(Objects::nonNull).collect(Collectors.toSet());
        List<SysEmployeeBasicDTO> executeUserList = rmiEmployeeRpcService.findEmployeeByCodes(executeUserCodeList);
        //执行模板-编码
        List<String> templateCodeList = appRespVOS.stream().map(TaskInfoAppRespVO::getExecutTemplateCode).filter(Objects::nonNull).distinct().collect(Collectors.toList());
        //关联执行模板名称
        List<ExectRecordTempRespVO> recordTempVOList = exectRecordTempService.queryByCodes(templateCodeList);

        //查询不同业务类型的关联业务编码的信息
        List<OrgStoreDetailRpcDTO> storeList = new ArrayList<>();
        List<LmSaveCustRespVO> custList = new ArrayList<>();
        List<SysEmployeeBasicDTO> employeeList = new ArrayList<>();
        Map<String, List<TaskInfoAppRespVO>> businessTypeListMap = appRespVOS.stream().filter(dtlRespVO -> StringUtils.isNotBlank(dtlRespVO.getBusinessType())).collect(Collectors.groupingBy(TaskInfoAppRespVO::getBusinessType));
        for (String key : businessTypeListMap.keySet()) {
            List<String> businessCodes = businessTypeListMap.get(key).stream().map(TaskInfoAppRespVO::getBusinessCode).filter(Objects::nonNull).distinct().collect(Collectors.toList());
            //关联业务-门店
            if (Objects.equals(key, UdcEnum.SALESMAN_TASK_BUSINESS_TYPE_STORE.getValueCode())) {
                storeList = rmiOrgStoreRpcService.findStoreByStoreCodes(businessCodes);
            } else if (Objects.equals(key, UdcEnum.SALESMAN_TASK_BUSINESS_TYPE_DEALER.getValueCode())) {
                custList = crmCustService.getCustInfoByCustCodes(businessCodes);
            } else if (Objects.equals(key, UdcEnum.SALESMAN_TASK_BUSINESS_TYPE_SALESMAN.getValueCode())) {
                Set<String> businessCodeSet = businessCodes.stream().collect(Collectors.toSet());
                employeeList = rmiEmployeeRpcService.findEmployeeByCodes(businessCodeSet);
            }
        }

        for (TaskInfoAppRespVO respVO : appRespVOS) {
            //明细的关联业务编码数据库里以逗号分隔的字符串形式保存
            //目前一行任务明细的业务编码只有一个,前端传businessCode，不使用businessCodes了。
           /* List<String> businessCodes = transitionStrToCodes(respVO.getBusinessCode());
            respVO.setBusinessCodes(businessCodes);*/

            //关联业务-门店
            if (Objects.equals(respVO.getBusinessType(), UdcEnum.SALESMAN_TASK_BUSINESS_TYPE_STORE.getValueCode())) {
                //业务编码是可多选的,名称进行拼接显示
                //需要支撑域提供批量的根据编码查询门店的接口
               /* List<OrgStoreDetailRpcDTO> storeList = rmiOrgStoreRpcService.findStoreByStoreCodes(respVO.getBusinessCodes());
                String businessName = storeList.stream().filter(storeDTO -> StringUtils.isNotBlank(storeDTO.getStoreName()))
                        .map(OrgStoreDetailRpcDTO::getStoreName).distinct().collect(Collectors.joining("-"));
                respVO.setBusinessName(businessName);*/
                val storeOptional = storeList.stream().filter(storeVO -> Objects.equals(storeVO.getStoreCode(), respVO.getBusinessCode())).findFirst();
                storeOptional.ifPresent(storeVO -> {
                    respVO.setBusinessName(storeVO.getStoreName());
                    respVO.setContactName(storeVO.getStoreManager());
                    respVO.setContactPhone(storeVO.getStoreContPhone());
                });
            }
            //关联业务-经销商
            if (Objects.equals(respVO.getBusinessType(), UdcEnum.SALESMAN_TASK_BUSINESS_TYPE_DEALER.getValueCode())) {
                /*List<LmSaveCustRespVO> custList = crmCustService.getCustInfoByCustCodes(respVO.getBusinessCodes());
                String businessName = custList.stream().filter(custRespVO -> StringUtils.isNotBlank(custRespVO.getCustName()))
                        .map(LmSaveCustRespVO::getCustName).distinct().collect(Collectors.joining("-"));
                respVO.setBusinessName(businessName);
                String custCode2 = custList.stream().filter(custRespVO -> StringUtils.isNotBlank(custRespVO.getCustCode2()))
                        .map(LmSaveCustRespVO::getCustCode2).distinct().collect(Collectors.joining("-"));
                respVO.setCustCode2(custCode2);*/
                val custOptional = custList.stream().filter(custVO -> Objects.equals(custVO.getCustCode(), respVO.getBusinessCode())).findFirst();
                custOptional.ifPresent(custVO -> {
                    respVO.setBusinessName(custVO.getCustName());
                    respVO.setCustCode2(custVO.getCustCode2());
                    respVO.setContactName(custVO.getContactName());
                    respVO.setContactPhone(custVO.getContactPhone());
                });
            }
            //关联业务-业务员
            if (Objects.equals(respVO.getBusinessType(), UdcEnum.SALESMAN_TASK_BUSINESS_TYPE_SALESMAN.getValueCode())) {
                //业务员表没有名称，要通过业务员编码查询员工的名称,业务员编码和员工编码是相同的
               /* Set<String> businessCodeSet = respVO.getBusinessCodes().stream().collect(Collectors.toSet());
                List<SysEmployeeBasicDTO> employeeList = rmiEmployeeRpcService.findEmployeeByCodes(businessCodeSet);
                String businessName = employeeList.stream().filter(employeeDTO -> StringUtils.isNotBlank(employeeDTO.getFullName()))
                        .map(SysEmployeeBasicDTO::getFullName).distinct().collect(Collectors.joining("-"));
                respVO.setBusinessName(businessName);*/
                val employeeOptional = employeeList.stream().filter(employeeVO -> Objects.equals(employeeVO.getCode(), respVO.getBusinessCode())).findFirst();
                employeeOptional.ifPresent(employeeVO -> {
                    respVO.setBusinessName(employeeVO.getFullName());
                    respVO.setContactName(employeeVO.getFullName());
                    respVO.setContactPhone(employeeVO.getPhone());
                });
            }
            //执行模板
            if (!CollectionUtils.isEmpty(recordTempVOList)) {
                val recordTempOptional = recordTempVOList.stream().filter(tempRespVO -> Objects.equals(tempRespVO.getTempCode(), respVO.getExecutTemplateCode())).findFirst();
                recordTempOptional.ifPresent(tempRespVO -> respVO.setExecutTemplateName(tempRespVO.getTempName()));
            }
            //执行人
            val executeUserOptional = executeUserList.stream().filter(executeUser -> Objects.equals(executeUser.getCode(), respVO.getExecutUserCode())).findFirst();
            executeUserOptional.ifPresent(executeUser -> respVO.setExecutUser(executeUser.getFullName()));


        }
    }

    /**
     * 组装处理APP端行政区域信息
     *
     * @param appRespVOS APP端任务数据信息
     * @return
     */
    private void translateAppArea(List<TaskInfoAppRespVO> appRespVOS) {
        //省市
        Set<String> provinces = appRespVOS.stream().map(TaskInfoAppRespVO::getProvince).distinct().filter(Objects::nonNull).collect(Collectors.toSet());
        //市
        Set<String> cities = appRespVOS.stream().map(TaskInfoAppRespVO::getCity).distinct().filter(Objects::nonNull).collect(Collectors.toSet());
        //区县
        Set<String> counties = appRespVOS.stream().map(TaskInfoAppRespVO::getDistrict).distinct().filter(Objects::nonNull).collect(Collectors.toSet());
        Set<String> areaCodes = new HashSet<>();
        areaCodes.addAll(provinces);
        areaCodes.addAll(cities);
        areaCodes.addAll(counties);
        List<SysAreaRespDTO> areaList = rmiSysAreaRpcService.findAreaByCodes(areaCodes);
        appRespVOS.forEach(dtlRespVO -> {
            Optional<SysAreaRespDTO> areaOptional1 = areaList.stream().filter(d -> Objects.equals(d.getAreaCode(), dtlRespVO.getProvince())).findFirst();
            areaOptional1.ifPresent(areaRespDTO -> dtlRespVO.setProvinceName(areaRespDTO.getAreaName()));
            Optional<SysAreaRespDTO> areaOptional2 = areaList.stream().filter(d -> Objects.equals(d.getAreaCode(), dtlRespVO.getCity())).findFirst();
            areaOptional2.ifPresent(areaRespDTO -> dtlRespVO.setCityName(areaRespDTO.getAreaName()));
            Optional<SysAreaRespDTO> areaOptional3 = areaList.stream().filter(d -> Objects.equals(d.getAreaCode(), dtlRespVO.getDistrict())).findFirst();
            areaOptional3.ifPresent(areaRespDTO -> dtlRespVO.setDistrictName(areaRespDTO.getAreaName()));

        });

    }

    /**
     * APP端根据执行人编码和任务类型查询角标的数量
     * <p>
     * 根据任务类型UDC增加图标，汇总此任务类型下，此业务员，待执行的任务的汇总，
     * 点击后进入任务列表（任务类型+待执行状态筛选出来的任务列表）
     *
     * @param queryVO 入参
     * @return 不同类型的角标的汇总数量
     */
    @Override
    public List<TaskInfoCommonAppRespVO> queryAngleMark(TaskInfoAppQueryVO queryVO) {
        //前端把业务员编码传过来
        String executeUserCode = queryVO.getExecutUserCode();
        List<String> typeList = queryVO.getTypeList();
        if (StringUtils.isBlank(executeUserCode)) {
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "业务员编码为空!");
        }
        if (CollectionUtils.isEmpty(typeList)) {
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "任务类型为空!");
        }
        //无论哪种情况都给前端返回数据，不是空List
        List<TaskInfoCommonAppRespVO> commonAppRespVOList = new ArrayList<>();
        typeList.forEach(s -> {
            TaskInfoCommonAppRespVO commonAppRespVO = new TaskInfoCommonAppRespVO();
            commonAppRespVO.setAngleMarkType(s);
            commonAppRespVO.setAngleMarkCount(0);
            commonAppRespVOList.add(commonAppRespVO);
        });

        //1.先根据业务员+待执行状态查询任务的明细，
        TaskInfoDtlQueryVO taskInfoDtlQueryVO = new TaskInfoDtlQueryVO();
        taskInfoDtlQueryVO.setExecutUserCode(executeUserCode);
        taskInfoDtlQueryVO.setCompleteState(UdcEnum.SALESMAN_TASK_COMPLETION_STATUS_WEE.getValueCode());
        List<TaskInfoDtlRespVO> taskInfoDtlRespVOList = taskInfoDtlRepoProc.selectByQueryVO(taskInfoDtlQueryVO);
        if (CollectionUtils.isEmpty(taskInfoDtlRespVOList)) {
            return commonAppRespVOList;
        }
        //2.然后根据明细里的主表ID+任务类型查询主表，若为空，则返回0
        List<Long> masIdList = taskInfoDtlRespVOList.stream().map(TaskInfoDtlRespVO::getMasId).filter(Objects::nonNull).distinct().collect(Collectors.toList());
        TaskInfoQueryVO taskInfoQueryVO = new TaskInfoQueryVO();
        taskInfoQueryVO.setIds(masIdList);
        taskInfoQueryVO.setTypeList(typeList);
        List<TaskInfoRespVO> taskInfoRespVOList = taskInfoService.selectByParam(taskInfoQueryVO);
        if (CollectionUtils.isEmpty(taskInfoRespVOList)) {
            return commonAppRespVOList;
        }
        //3.主表不为空的话，则按照主表的ID过滤明细返回的数据，进行求和
       /* List<TaskInfoDtlRespVO> dtlRespVOList = taskInfoDtlRespVOList.stream().parallel().filter(dtlRespVO -> taskInfoRespVOList.stream()
                .anyMatch(respVO-> Objects.equals(dtlRespVO.getMasId(), respVO.getId()))).collect(Collectors.toList());
        if (CollectionUtils.isEmpty(dtlRespVOList)){
            return commonAppRespVOList;
        }*/

        HashMap<String, Integer> map = new HashMap<>();
        taskInfoDtlRespVOList.forEach(dtlRespVO -> {
            Optional<TaskInfoRespVO> respVOOptional = taskInfoRespVOList.stream().filter(respVO -> Objects.equals(dtlRespVO.getMasId(), respVO.getId())).findFirst();
            if (respVOOptional.isPresent()) {
                TaskInfoRespVO taskInfoRespVO = respVOOptional.get();
                if (!map.isEmpty() && map.containsKey(taskInfoRespVO.getType())) {
                    Integer angleMarkCount = map.get(taskInfoRespVO.getType());
                    map.put(taskInfoRespVO.getType(), angleMarkCount + 1);
                } else {
                    map.put(taskInfoRespVO.getType(), 1);
                }
            }
        });

        //4.匹配任务类型进行赋值
        commonAppRespVOList.forEach(commonAppRespVO -> {
            if (!map.isEmpty() && map.containsKey(commonAppRespVO.getAngleMarkType())) {
                Integer angleMarkCount = map.get(commonAppRespVO.getAngleMarkType());
                commonAppRespVO.setAngleMarkCount(angleMarkCount);
            }
        });

        return commonAppRespVOList;

    }

    /**
     * 执行记录保存、提交、审核拒绝、审核通过、关闭时：
     * 根据任务明细ID
     * 更新明细(状态、完成时间、执行记录信息)
     * 更新主表(状态、完成时间、任务进度)
     * <p>
     * 后面执行记录第一次保存/提交后存记录编码
     * 当执行记录提交时，相应的将执行的任务明细完成状态更新为‘待审核’，同时将主表任务状态由‘待执行’更新为‘进行中’
     * <p>
     * 当执行记录审核通过时，更新任务明细完成状态为‘已完成’，同时判断任务明细完成状态是否全部为‘已完成’，
     * 如果是，将任务主表状态更新为‘已完成’，如果不是保持主表任务状态不变；
     * 当任务明细执行状态均为已完成和已关闭时，任务主表状态为已完成；
     *
     * <p>
     * 当有一个甚至全部的任务明细执行状态为待审核、审核拒绝状态时，任务主表状态更新为进行中；
     * <p>
     * 当任务执行关闭操作时，相应的状态也要更新为关闭
     * <p>
     * 当任务明细执行记录审批通过后更新任务完成状态和完成时间；
     * 当任务最后一个明细执行记录审批通过时，不仅更新任务完成状态和完成时间，同时更新任务主表的完成时间
     * 当任务明细全部完成式，主表记录最后完成的一条明细的完成时间，主表记录所有任务明细最终的完成时间
     * 明细记录明细完成状态时的时间
     * <p>
     * 明细记录此条明细的完成时间；
     * 巡店任务类型：当执行记录提交后更新时间；
     * 门店提报类型：标记完成时更新时间
     * <p>
     * 计算更新任务进度，默认为0%，统计任务明细：已完成/总数量算百分比；小数点后4位
     *
     * @param saveVO 任务执行记录更新任务对象入参
     * @return
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void executionRecordCallback(TaskExecutionRecordSaveVO saveVO) {
        if (StringUtils.isBlank(saveVO.getExecuteType())) {
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "业务操作执行类型为空!");
        }
        if (Objects.isNull(saveVO.getTaskDtlId())) {
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "任务明细ID为空!");
        }
        if (!ConstantsSale.TASK_EXECUTION_RECORD_LIST.contains(saveVO.getExecuteType())) {
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "未匹配到业务操作执行类型!");
        }
        //任务明细数据
        TaskInfoDtlRespVO dtlRespVO = findById(saveVO.getTaskDtlId());
        if (Objects.isNull(dtlRespVO)) {
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "未查询到任务明细数据!");
        }
        TaskInfoRespVO respVO = taskInfoService.findById(dtlRespVO.getMasId());
        if (Objects.isNull(respVO)) {
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "未查询到任务数据!");
        }

        switch (saveVO.getExecuteType()) {
            case ConstantsSale.TASK_EXECUTION_RECORD_SAVE:
                executionRecordSaveCallback(saveVO, dtlRespVO);
                break;
            case ConstantsSale.TASK_EXECUTION_RECORD_SUBMIT:
                executionRecordSubmitCallback(saveVO, dtlRespVO);
                break;
            case ConstantsSale.TASK_EXECUTION_RECORD_REFUSE:
                executionRecordRefuseCallback(saveVO);
                break;
            case ConstantsSale.TASK_EXECUTION_RECORD_PASS:
                executionRecordPassCallback(saveVO, dtlRespVO);
                break;
            case ConstantsSale.TASK_EXECUTION_RECORD_CLOSE:
                executionRecordCloseCallback(saveVO, dtlRespVO);
                break;
        }

    }

    @Override
    public void releaseDtl(Long id, String valueCode) {
        taskInfoDtlRepo.releaseDtl(id, valueCode);
    }

    /**
     * 执行记录第一次保存后存执行记录信息
     * 根据任务明细ID更新明细执行记录信息
     * 相应的将执行的任务明细完成状态更新为‘进行中’
     *
     * @param saveVO    任务执行记录更新任务对象入参
     * @param dtlRespVO 任务明细数据
     * @return
     */
    private void executionRecordSaveCallback(TaskExecutionRecordSaveVO saveVO, TaskInfoDtlRespVO dtlRespVO) {
        if (StringUtils.isBlank(saveVO.getExecutRecordCode())) {
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "执行记录编码为空!");
        }
       /* if (StringUtils.isBlank(saveVO.getExecutRecordName())) {
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "执行记录名称为空!");
        }*/
        //执行记录第一次保存后存记录编码信息
        taskInfoDtlRepoProc.updateExecuteRecordById(saveVO.getExecutRecordCode(),
                saveVO.getExecutRecordId(), saveVO.getExecutRecordName(), saveVO.getTaskDtlId());
        //相应的将执行的任务明细完成状态更新为‘进行中’
        taskInfoDtlRepoProc.updateCompleteStateById(UdcEnum.SALESMAN_TASK_COMPLETION_STATUS_IPS.getValueCode(), saveVO.getTaskDtlId());

        //同时将主表任务状态由‘待执行’更新为‘进行中’
        taskInfoService.updateStateById(UdcEnum.SALESMAN_TASK_STATUS_IPS.getValueCode(), dtlRespVO.getMasId());
    }

    /**
     * 执行记录第一次提交后存执行记录信息
     * 当执行记录提交时，相应的将执行的任务明细完成状态更新为‘待审核’，同时将主表任务状态由‘待执行’更新为‘进行中’
     * 执行记录提交时更新任务明细和主表信息
     * 根据任务明细ID
     * 更新明细(状态、执行记录信息)
     * 更新主表(状态)
     *
     * @param saveVO    任务执行记录更新任务对象入参
     * @param dtlRespVO 任务明细数据
     * @return
     */
    private void executionRecordSubmitCallback(TaskExecutionRecordSaveVO saveVO, TaskInfoDtlRespVO dtlRespVO) {
        //调用保存方法逻辑
        executionRecordSaveCallback(saveVO, dtlRespVO);
        //记录此条明细的完成时间；巡店任务类型：当执行记录提交后更新时间；门店提报类型：标记完成时更新时间
        if (Objects.equals(dtlRespVO.getBusinessType(), UdcEnum.SALESMAN_TASK_TYPE_DST.getValueCode())) {
            //相应的将执行的任务明细完成状态更新为‘待审核’
            //taskInfoDtlRepoProc.updateCompleteStateAndTimeById(UdcEnum.SALESMAN_TASK_COMPLETION_STATUS_WAT.getValueCode(), LocalDateTime.now(), saveVO.getTaskDtlId());
            taskInfoDtlRepoProc.updateCompleteStateById(UdcEnum.SALESMAN_TASK_COMPLETION_STATUS_WAT.getValueCode(), saveVO.getTaskDtlId());

        } else {
            //相应的将执行的任务明细完成状态更新为‘待审核’
            taskInfoDtlRepoProc.updateCompleteStateById(UdcEnum.SALESMAN_TASK_COMPLETION_STATUS_WAT.getValueCode(), saveVO.getTaskDtlId());
        }

        //同时将主表任务状态由‘待执行’更新为‘进行中’
        taskInfoService.updateStateById(UdcEnum.SALESMAN_TASK_STATUS_IPS.getValueCode(), dtlRespVO.getMasId());
    }

    /**
     * <p>
     * 执行记录审核拒绝时更新任务明细
     * 根据任务明细ID
     * 更新明细(状态)
     *
     * @param saveVO 任务执行记录更新任务对象入参
     * @return
     */
    private void executionRecordRefuseCallback(TaskExecutionRecordSaveVO saveVO) {
        //更新任务明细完成状态为‘审核拒绝’
        taskInfoDtlRepoProc.updateCompleteStateById(UdcEnum.SALESMAN_TASK_COMPLETION_STATUS_ARN.getValueCode(), saveVO.getTaskDtlId());
    }

    /**
     * 当执行记录审核通过时，更新任务明细完成状态为‘已完成’，同时判断任务明细完成状态是否全部为‘已完成’，
     * 如果是，将任务主表状态更新为‘已完成’，如果不是保持主表任务状态不变；
     * 当任务明细执行状态均为已完成和已关闭时，任务主表状态为已完成；
     * <p>
     * 执行记录审核通过时更新任务明细和主表信息
     * 根据任务明细ID
     * 更新明细(状态、完成时间)
     * 更新主表(状态、完成时间、任务进度)
     *
     * @param saveVO    任务执行记录更新任务对象入参
     * @param dtlRespVO 任务明细数据
     * @return
     */
    private void executionRecordPassCallback(TaskExecutionRecordSaveVO saveVO, TaskInfoDtlRespVO dtlRespVO) {
        //记录此条明细的完成时间；巡店任务类型：当执行记录提交后更新时间；门店提报类型：标记完成时更新时间
        if (Objects.equals(dtlRespVO.getBusinessType(), (UdcEnum.SALESMAN_TASK_TYPE_DST.getValueCode()))) {
            //更新任务明细完成状态为‘已完成’
            //taskInfoDtlRepoProc.updateCompleteStateById(UdcEnum.SALESMAN_TASK_COMPLETION_STATUS_CPD.getValueCode(), saveVO.getTaskDtlId());
            taskInfoDtlRepoProc.updateCompleteStateAndTimeById(UdcEnum.SALESMAN_TASK_COMPLETION_STATUS_CPD.getValueCode(), LocalDateTime.now(), saveVO.getTaskDtlId());

        } else {
            //更新任务明细完成状态为‘已完成’、完成时间
            taskInfoDtlRepoProc.updateCompleteStateAndTimeById(UdcEnum.SALESMAN_TASK_COMPLETION_STATUS_CPD.getValueCode(), LocalDateTime.now(), saveVO.getTaskDtlId());
        }
        //查询所有任务明细
        List<TaskInfoDtlRespVO> dtlRespVOList = selectByMasId(dtlRespVO.getMasId());
        if (CollectionUtils.isEmpty(dtlRespVOList)) {
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "未查询到任务明细数据!");
        }

        //同时判断任务明细完成状态是否全部为‘已完成’或已关闭，
        boolean completeStateFlag = dtlRespVOList.stream().allMatch(vo -> checkAllCompleteState(vo.getCompleteState()));
        //如果是，将任务主表状态更新为‘已完成’，如果不是保持主表任务状态不变；
        if (completeStateFlag) {
            taskInfoService.updateStateAndCompleteTime(UdcEnum.SALESMAN_TASK_STATUS_CPD.getValueCode(), LocalDateTime.now(), dtlRespVO.getMasId());
        }

        //计算更新任务进度
        //完成状态的明细
        List<TaskInfoDtlRespVO> dtlCompleteVOList = dtlRespVOList.stream().filter(vo -> Objects.equals(vo.getCompleteState(), UdcEnum.SALESMAN_TASK_COMPLETION_STATUS_CPD.getValueCode())).collect(Collectors.toList());
        BigDecimal completeSize = Objects.isNull(dtlCompleteVOList.size()) ? BigDecimal.ZERO : BigDecimal.valueOf(dtlCompleteVOList.size());
        BigDecimal allSize = Objects.isNull(dtlRespVOList.size()) ? BigDecimal.ZERO : BigDecimal.valueOf(dtlRespVOList.size());
        BigDecimal progress = completeSize.divide(allSize, 4, RoundingMode.HALF_UP).setScale(4, RoundingMode.HALF_UP);
        BigDecimal finallyProgress = progress.multiply(BigDecimal.valueOf(100));
        //更新任务进度
        taskInfoService.updateProgressById(finallyProgress, dtlRespVO.getMasId());

    }

    private boolean checkAllCompleteState(String completeState) {
        if (StringUtils.isBlank(completeState)) {
            return false;
        }
        if (Objects.equals(completeState, UdcEnum.SALESMAN_TASK_COMPLETION_STATUS_CPD.getValueCode())
                || Objects.equals(completeState, UdcEnum.SALESMAN_TASK_COMPLETION_STATUS_CSD.getValueCode())) {
            return true;
        }
        return false;
    }

    /**
     * 根据任务主表ID计算更新任务进度
     *
     * @param id 任务主表ID
     * @return
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void updateProgressById(Long id) {
        //查询所有任务明细
        List<TaskInfoDtlRespVO> dtlRespVOList = selectByMasId(id);
        if (CollectionUtils.isEmpty(dtlRespVOList)) {
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "未查询到任务明细数据!");
        }
        //计算更新任务进度
        //完成状态的明细
        List<TaskInfoDtlRespVO> dtlCompleteVOList = dtlRespVOList.stream().filter(vo -> Objects.equals(vo.getCompleteState(), UdcEnum.SALESMAN_TASK_COMPLETION_STATUS_CPD.getValueCode())).collect(Collectors.toList());
        BigDecimal completeSize = Objects.isNull(dtlCompleteVOList.size()) ? BigDecimal.ZERO : BigDecimal.valueOf(dtlCompleteVOList.size());
        BigDecimal allSize = Objects.isNull(dtlRespVOList.size()) ? BigDecimal.ZERO : BigDecimal.valueOf(dtlRespVOList.size());
        BigDecimal progress = completeSize.divide(allSize, 4, RoundingMode.HALF_UP).setScale(4, RoundingMode.HALF_UP);
        BigDecimal finallyProgress = progress.multiply(BigDecimal.valueOf(100));
        //更新任务进度
        taskInfoService.updateProgressById(finallyProgress, id);
    }

    /**
     * 执行记录关闭时更新任务明细
     * 根据任务明细ID
     * 更新明细(状态)
     *
     * @param saveVO    任务执行记录更新任务对象入参
     * @param dtlRespVO 任务明细数据
     * @return
     */
    private void executionRecordCloseCallback(TaskExecutionRecordSaveVO saveVO, TaskInfoDtlRespVO dtlRespVO) {
        //更新任务明细完成状态为‘关闭’
        taskInfoDtlRepoProc.updateCompleteStateById(UdcEnum.SALESMAN_TASK_COMPLETION_STATUS_CSD.getValueCode(), saveVO.getTaskDtlId());
        //查询所有任务明细
        List<TaskInfoDtlRespVO> dtlRespVOList = selectByMasId(dtlRespVO.getMasId());
        if (CollectionUtils.isEmpty(dtlRespVOList)) {
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "未查询到任务明细数据!");
        }

        //同时判断任务明细完成状态是否全部为‘已关闭’，
        boolean completeStateFlag = dtlRespVOList.stream().allMatch(vo -> Objects.equals(vo.getCompleteState(), UdcEnum.SALESMAN_TASK_COMPLETION_STATUS_CSD.getValueCode()));
        //如果是，将任务主表状态更新为‘已关闭’，如果不是保持主表任务状态不变；
        if (completeStateFlag) {
            taskInfoService.updateStateAndCompleteTime(UdcEnum.SALESMAN_TASK_STATUS_CSD.getValueCode(), LocalDateTime.now(), dtlRespVO.getMasId());
        }
    }

    /**
     * 获取当前用户
     *
     * @param
     * @return 当前用户信息
     */
    private SysUserDTO getSysUser() {
        //获取当前用户
        GeneralUserDetails userDetails = SecurityContextUtil.currentUser();
        if (Objects.isNull(userDetails) || Objects.isNull(userDetails.getUser())) {
            throw new BusinessException(ApiCode.UNAUTHORIZED, "获取当前用户信息失败");
        }
        SysUserDTO sysUser = userDetails.getUser();
        if (ObjectUtils.isEmpty(sysUser)) {
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "用户信息获取失败");
        }

        return sysUser;
    }

    /**
     * 编码集合转换为字符串,以逗号分隔的字符串形式
     *
     * @param codeList 编码集合
     * @return 以逗号分隔的字符串
     */
    private String transitionCodesToStr(List<String> codeList) {
        if (CollectionUtils.isEmpty(codeList)) {
            return null;
        }
        String codeString = codeList.stream().distinct().collect(Collectors.joining(","));

        return codeString;
    }

    /**
     * 字符串转换为编码集合,以逗号分隔的字符串形式
     *
     * @param codeString 以逗号分隔的字符串
     * @return 编码集合
     */
    private List<String> transitionStrToCodes(String codeString) {
        if (StringUtils.isEmpty(codeString)) {
            return Collections.emptyList();
        }
        List<String> codeList = Arrays.asList(codeString.split(","));

        return codeList;
    }


    @Override
    @Transactional(rollbackFor = Exception.class)
    public ApiResult<Object> importTaskDtl(MultipartFile file,Long masId) {
        if (file == null || file.isEmpty()) {
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "请选择导入文件!");
        }
        //headRow – 头部数据行数 title总行数，headRow以下便是需要解析的数据
        Integer headRow = 2;
        //dataTypeAttr – 数据类型类的属性所在行，必须在headRow内(与java bean 属性对应的行);
        Integer dataTypeAttr = 2;
        List<TaskInfoDtlImportRespVO> dataList = null;
        try {
            dataList = (List<TaskInfoDtlImportRespVO>) ExcelImportUtil.instance(file.getInputStream())
                    .headRow(headRow)
                    .dataType(TaskInfoDtlImportRespVO.class, dataTypeAttr)
                    .readAllSync();
        } catch (Exception e) {
            log.error("任务明细导入失败:", e);
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "任务明细导入失败:" + e.getMessage());
        }

        if (CollectionUtils.isEmpty(dataList)) {
            //return Collections.EMPTY_LIST;
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "导入数据为空,请检查!");

        }
        Set<String> repeatCode = this.findDuplicateStrings(dataList.stream().map(TaskInfoDtlImportRespVO :: getBusinessCode).collect(Collectors.toList()));
        if(!repeatCode.isEmpty()){
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "导入数据重复" + repeatCode + ",请检查!");
        }
        TaskInfoDtlQueryVO queryVO = new TaskInfoDtlQueryVO();
        queryVO.setMasId(masId);
        List<TaskInfoDtlDO> detailData =  taskInfoDtlRepoProc.getByMasId(masId);
        if(!detailData.isEmpty()){
            List<String> impotCodes = dataList.stream().map(TaskInfoDtlImportRespVO :: getBusinessCode).collect(Collectors.toList());
            List<String> detailCodeList = detailData.stream().map(TaskInfoDtlDO :: getBusinessCode).collect(Collectors.toList());
            List<String> repeatCodes = new ArrayList<>();
            for(String code2 : detailCodeList){
                if(impotCodes.contains(code2)){
                    repeatCodes.add(code2);
                }
            }
            if (org.apache.commons.collections4.CollectionUtils.isNotEmpty(repeatCodes)){
                throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "导入数据" + repeatCodes + "已存在,请检查!");
            }
            //            List<String> containList = detailCodeList.stream().filter(detailData :: contains).collect(Collectors.toList());
//            if (org.apache.commons.collections4.CollectionUtils.isNotEmpty(containList)) {
//                throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "导入数据" + containList + "已存在,请检查!");
//            }

        }
        //校验必填项
        AtomicInteger sheetCount = new AtomicInteger(1);
        dataList.forEach(dtlRespVO -> {
            sheetCount.addAndGet(1);
            if (StringUtils.isBlank(dtlRespVO.getBusinessTypeName()) || StringUtils.isBlank(dtlRespVO.getBusinessCode())) {
                throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "第" + sheetCount + "行数据存在空值,请检查!");
            }
            if (!ConstantsSale.SALESMAN_TASK_BUSINESS_TYPE_NAME_LIST.contains(dtlRespVO.getBusinessTypeName())){
                throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "第" + sheetCount + "行数据业务类型不存在,请检查!");
            }
        });

       /* //客户信息
        List<LmSaveCustRespVO> custRespVOList = new ArrayList<>();
        //查询业务编码对应的信息
        Map<String, List<TaskInfoImportCommonRespVO>> importListMap = dataList.stream().filter(dtlRespVO -> Objects.nonNull(dtlRespVO.getBusinessTypeName()))
                .collect(Collectors.groupingBy(TaskInfoImportCommonRespVO::getBusinessTypeName));
        for (String key : importListMap.keySet()) {
            if (Objects.equals(UdcEnum.SALESMAN_TASK_BUSINESS_TYPE_STORE.getValueCodeName(), key)) {

            } else if (Objects.equals(UdcEnum.SALESMAN_TASK_BUSINESS_TYPE_DEALER.getValueCodeName(), key)) {
                List<String> dealerCodes = importListMap.get(key).stream().map(TaskInfoImportCommonRespVO::getBusinessCode).filter(Objects::nonNull).distinct().collect(Collectors.toList());
                if (!CollectionUtils.isEmpty(dealerCodes)) {
                     *//*//考虑处理业务编码传多个场景，逗号分隔
                        List<String> custCodes = dealerCodes.stream().map(s -> transitionStrToCodes(s)).flatMap(Collection::stream)
                            .filter(Objects::nonNull).distinct().collect(Collectors.toList());*//*
                    custRespVOList = selectCustVO(dealerCodes);
                }

            } else if (Objects.equals(UdcEnum.SALESMAN_TASK_BUSINESS_TYPE_SALESMAN.getValueCodeName(), key)) {

            }

        }*/
        //业务处理逻辑
        return disposeImportDtl(dataList, masId);

    }

    public static Set<String> findDuplicateStrings(List<String> list) {
        return list.stream()
                .collect(Collectors.groupingBy(Function.identity(), Collectors.counting()))
                .entrySet().stream()
                .filter(entry -> entry.getValue() > 1)
                .map(Map.Entry::getKey)
                .collect(Collectors.toSet());
    }
    /**
     * 任务明细导入的业务处理逻辑
     *
     * @param dataList 导入的数据
     * @return 组装后的任务明细信息
     */
    public ApiResult<Object> disposeImportDtl(List<TaskInfoDtlImportRespVO> dataList,Long masId) {
        List<TaskInfoDtlImportRespVO> errorDataList = new ArrayList<>();
        List<TaskInfoDtlImportRespVO> successDataList = new ArrayList<>();
        //根据业务类型进行分组
        Map<String, List<TaskInfoDtlImportRespVO>> importListMap = dataList.stream().filter(dtlRespVO -> Objects.nonNull(dtlRespVO.getBusinessTypeName()))
                .collect(Collectors.groupingBy(TaskInfoDtlImportRespVO::getBusinessTypeName));
        //查询不同业务类型的关联业务编码对应的信息，并校验关联业务编码是否在系统中存在，且状态为启用
        for (String key : importListMap.keySet()) {
            List<TaskInfoDtlImportRespVO> importDataList = importListMap.get(key);
            List<String> businessCodes = importDataList.stream().map(TaskInfoDtlImportRespVO::getBusinessCode).filter(Objects::nonNull).distinct().collect(Collectors.toList());

            if (Objects.equals(UdcEnum.SALESMAN_TASK_BUSINESS_TYPE_STORE.getValueCodeName(), key)) {
                //门店类型的导入
                //任务明细的导入需要在立马项目里再写一个实现类继承TaskInfoDtlServiceImpl
                //来使用返回的结果再重新组装门店的信息	，因为门店客户和业务员信息都在立马里
                List<OrgStoreDetailRpcDTO> storeRespVOList = new ArrayList<>();
                if (!CollectionUtils.isEmpty(businessCodes)) {
                    storeRespVOList = selectStore(businessCodes);
                }
                //校验客户的关联业务编码是否在系统中存在，且状态为启用，并组装部分字段信息
                disposeStoreImport(importDataList, storeRespVOList, errorDataList, successDataList);
            } else if (Objects.equals(UdcEnum.SALESMAN_TASK_BUSINESS_TYPE_DEALER.getValueCodeName(), key)) {
                //客户信息
                List<LmSaveCustRespVO> custRespVOList = new ArrayList<>();
                if (!CollectionUtils.isEmpty(businessCodes)) {
                    //custRespVOList = selectCust(businessCodes);
                    custRespVOList = selectCustByCustCode2(businessCodes);
                }
                //校验客户的关联业务编码是否在系统中存在，且状态为启用，并组装部分字段信息
                disposeCustImport(importDataList, custRespVOList, errorDataList, successDataList);
            } else if (Objects.equals(UdcEnum.SALESMAN_TASK_BUSINESS_TYPE_SALESMAN.getValueCodeName(), key)) {
                //业务员信息
                List<SalesmanInfoSimpleRespVO> salesmanRespVOList = new ArrayList<>();
                if (!CollectionUtils.isEmpty(businessCodes)) {
                    salesmanRespVOList = selectSalesman(businessCodes);
                }
                //校验业务员的关联业务编码是否在系统中存在，且状态为启用，并组装部分字段信息
                disposeSalesmanImport(importDataList, salesmanRespVOList, errorDataList, successDataList);
            }

        }

        //2.校验
        checkImportTaskDtl(errorDataList);
        //3.组装剩余信息
        List<TaskInfoDtlRespVO> taskInfoDtlRespVOList = assembleImportTaskDtl(successDataList);
        List<TaskInfoDtlDO> list = new ArrayList<>();
        taskInfoDtlRespVOList.forEach(taskDtl ->{
            TaskInfoDtlDO taskInfoDtl = new TaskInfoDtlDO();
            taskInfoDtl.setMasId(masId);
            taskInfoDtl.setBusinessCode(taskDtl.getBusinessCode());
            taskInfoDtl.setBusinessType(taskDtl.getBusinessType());
            taskInfoDtl.setBusinessName(taskDtl.getBusinessName());
            taskInfoDtl.setDealerCode(taskDtl.getDealerCode());
            taskInfoDtl.setAddress(taskDtl.getAddress());
            taskInfoDtl.setExecutUser(taskDtl.getExecutUser());
            taskInfoDtl.setCompleteState(UdcEnum.SALESMAN_TASK_STATUS_WRE.getValueCode());
            taskInfoDtl.setExecutUserCode(taskDtl.getExecutUserCode());
            taskInfoDtl.setCustCode2(taskDtl.getCustCode2());
            list.add(taskInfoDtl);
        });
        taskInfoDtlRepo.saveAll(list);
        return ApiResult.ok();
    }

}
