package com.elitesland.yst.production.sale.service.shop;

import cn.hutool.core.lang.Assert;
import com.elitescloud.cloudt.common.base.ApiCode;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitescloud.boot.exception.BusinessException;
import com.elitescloud.cloudt.core.security.util.DataAuthJpaUtil;
import com.elitesland.yst.production.sale.api.service.shop.BipRotationChartService;
import com.elitesland.yst.production.sale.api.vo.param.shop.BipRotationChartQueryParamVO;
import com.elitesland.yst.production.sale.api.vo.resp.shop.BipRetationChartVO;
import com.elitesland.yst.production.sale.api.vo.save.shop.BipRetationChartSaveVO;
import com.elitesland.yst.production.sale.common.constant.UdcEnum;
import com.elitesland.yst.production.sale.common.model.CurrentUserDTO;
import com.elitesland.yst.production.sale.convert.shop.BipRetationChartConvert;
import com.elitesland.yst.production.sale.core.service.UserService;
import com.elitesland.yst.production.sale.entity.BipRetationChartDO;
import com.elitesland.yst.production.sale.entity.QBipRetationChartDO;
import com.elitesland.yst.production.sale.repo.shop.BipRetationChartRepo;
import com.elitesland.yst.production.sale.repo.shop.BipRotationChartRepoProc;
import com.elitesland.yst.production.sale.repo.shop.MktDiscountOffsetRepoProc;
import com.elitesland.yst.production.support.provider.org.dto.OrgUserEmpInfoRpcDTO;
import com.elitesland.yst.production.support.provider.org.service.OrgEmpRpcService;
import com.querydsl.core.types.ExpressionUtils;
import com.querydsl.core.types.Predicate;
import lombok.RequiredArgsConstructor;
import lombok.val;
import org.apache.commons.lang3.StringUtils;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.time.LocalDateTime;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.stream.Collectors;

/**
 * <p>
 * 轮播图实现类
 * </p>
 *
 * @author Shadow
 * @since 2021-08-11 15:20:05
 */
@Service
@RequiredArgsConstructor
public class BipRotationChartServiceImpl implements BipRotationChartService {

    private final OrgEmpRpcService orgEmpRpcService;

    private final BipRetationChartRepo bipRetationChartRepo;
    private final BipRotationChartRepoProc bipRotationChartRepoProc;
    private final MktDiscountOffsetRepoProc mktDiscountOffsetRepoProc;

    private final QBipRetationChartDO qBipRetationChartDO = QBipRetationChartDO.bipRetationChartDO;

    private final Map<String, String> statusMap = new HashMap<>();{
        statusMap.put(UdcEnum.COM_STATUS_ACTIVEORNO_ACTIVE.getValueCode(), UdcEnum.COM_STATUS_ACTIVEORNO_ACTIVE.getValueCodeName());
        statusMap.put(UdcEnum.COM_STATUS_ACTIVEORNO_INACTIVE.getValueCode(), UdcEnum.COM_STATUS_ACTIVEORNO_INACTIVE.getValueCodeName());
        statusMap.put(UdcEnum.COM_STATUS_RETAION_DRAFT.getValueCode(), UdcEnum.COM_STATUS_RETAION_DRAFT.getValueCodeName());
    }

    @Override
    public PagingVO<BipRetationChartVO> search(BipRotationChartQueryParamVO param) {
        Predicate predicate = bipRotationChartRepoProc.where(param);
        predicate = ExpressionUtils.and(predicate, DataAuthJpaUtil.dataAuthJpaPredicate(qBipRetationChartDO.getMetadata()));

        val ret = bipRetationChartRepo.findAll(predicate, param.getPageRequest());
        val vos = ret.getContent().stream().map(BipRetationChartConvert.INSTANCE::doToVO).collect(Collectors.toList());

        // 状态名称
        for (BipRetationChartVO vo : vos) {
            if (StringUtils.isNotBlank(vo.getStatus())) {
                vo.setStatusName(statusMap.get(vo.getStatus()));
            }
        }

        return PagingVO.<BipRetationChartVO>builder()
                .total(ret.getTotalElements())
                .records(vos)
                .build();
    }

    @Override
    public List<BipRetationChartVO> homeRotation(Long ouId) {
        return bipRotationChartRepoProc.homeSlideshow(ouId);
    }

    @Override
    public Optional<BipRetationChartVO> findIdOne(Long id) {
        return bipRetationChartRepo.findById(id).map(BipRetationChartConvert.INSTANCE::doToVO);
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public Long createOne(BipRetationChartSaveVO param) {
        Assert.notNull(param.getOffsetCode(), "请输入满减折扣编码!");

        OrgUserEmpInfoRpcDTO empInfo = getCurrentEmpInfo();
        BipRetationChartDO bipRetationChartDO = BipRetationChartConvert.INSTANCE.saveToDo(param);
        // 防止修改
        bipRetationChartDO.setId(null);
        // 设置数据权限
        bipRetationChartDO.setSecBuId(empInfo.getBuId());
        bipRetationChartDO.setSecOuId(empInfo.getOuId());
        bipRetationChartDO.setSecUserId(empInfo.getUserId());
        bipRetationChartDO.setStatus(UdcEnum.COM_STATUS_RETAION_DRAFT.getValueCode());

        // 校验满减是否生效
        Assert.isTrue(mktDiscountOffsetRepoProc.checkValid(param.getOffsetCode()), "该满减折扣未生效!");

        return bipRetationChartRepo.save(bipRetationChartDO).getId();
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void update(BipRetationChartSaveVO param) {
        BipRetationChartDO objDo = BipRetationChartConvert.INSTANCE.saveToDo(param);
        Optional<BipRetationChartDO> optional = bipRetationChartRepo.findById(objDo.getId());
        if (optional.isPresent()) {
            objDo.setId(optional.get().getId());
            bipRetationChartRepo.save(objDo);
        } else {
            throw new BusinessException(ApiCode.FAIL, "修改失败，数据不存在" + objDo.getId());
        }
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void updateDeleteFlag(Long id) {
        Optional<BipRetationChartDO> optionalItem = bipRetationChartRepo.findById(id);
        if (optionalItem.isPresent()) {
            BipRetationChartDO retationChartDO = optionalItem.get();
            if (UdcEnum.COM_STATUS_ACTIVEORNO_ACTIVE.getValueCode().equals(retationChartDO.getStatus())) {
                throw new BusinessException("启用状态的轮播图不能删除");
            }
            retationChartDO.setDeleteFlag(1);
            bipRetationChartRepo.save(retationChartDO);
        } else {
            throw new BusinessException(ApiCode.FAIL, "修改失败，数据不存在" + id);
        }
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void deleteBatch(List<Long> list) {
        list.forEach(this::updateDeleteFlag);
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void changeStatus(List<Long> ids, String status) {
        // 激活校验，同一时间段同一优先权只能存在一个
        if (UdcEnum.COM_STATUS_ACTIVEORNO_ACTIVE.getValueCode().equals(status)) {
            List<BipRetationChartDO> slideshows = bipRetationChartRepo.findAllById(ids);
            for (BipRetationChartDO slideshow : slideshows) {
                LocalDateTime now = LocalDateTime.now();
                // 只能激活生效时间内的轮播图
                if (now.isAfter(slideshow.getEtime()) || now.isBefore(slideshow.getStime())) {
                    throw new BusinessException("激活错误，当前时间不在该轮播图的生效时间之内！");
                }
                // 重复数据检验
                if (!bipRotationChartRepoProc.judgeActive(slideshow)) {
                    throw new BusinessException("激活错误，同一时间同优先级的轮播图已存在！");
                }
            }
        }

        // 更新状态
        if (UdcEnum.COM_STATUS_ACTIVEORNO_ACTIVE.getValueCode().equals(status) ||
                UdcEnum.COM_STATUS_ACTIVEORNO_INACTIVE.getValueCode().equals(status)) {
            bipRotationChartRepoProc.updateStatusBatch(ids, status);
        } else {
            throw new BusinessException(ApiCode.FAIL, "修改失败，状态值错误:" + status);
        }

    }

    /**
     * 获取当前登陆人的信息
     *
     * @return 用户信息
     */
    private OrgUserEmpInfoRpcDTO getCurrentEmpInfo() {
        CurrentUserDTO currentUser = UserService.currentUser();
        if (currentUser == null) {
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "当前用户已失效,请重登录重试");
        }
        OrgUserEmpInfoRpcDTO userEmpInfo = orgEmpRpcService.findUserEmpInfo(currentUser.getUserId());
        if (userEmpInfo == null) {
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "无法获取当前用户信息");
        }
        return userEmpInfo;
    }

}


