package com.elitesland.yst.production.sale.service.shop;

import cn.hutool.core.lang.Assert;
import cn.hutool.core.util.ObjectUtil;
import cn.hutool.core.util.StrUtil;
import com.el.coordinator.boot.fsm.model.vo.FileObjRespVO;
import com.elitesland.yst.production.sale.api.service.shop.BipCustUserBindService;
import com.elitesland.yst.production.sale.api.service.shop.BipItemCategoryService;
import com.elitesland.yst.production.sale.api.vo.resp.shop.BipItemCategoryRespVO;
import com.elitesland.yst.production.sale.api.vo.resp.shop.ItemCatTreeAllRespVO;
import com.elitesland.yst.production.sale.api.vo.resp.shop.ItemCatTreeRespVO;
import com.elitesland.yst.production.sale.api.vo.save.shop.BipItemCategorySaveVO;
import com.elitesland.yst.production.sale.common.constant.ConstantsCache;
import com.elitesland.yst.production.sale.convert.shop.BipItemCategoryConvert;
import com.elitesland.yst.production.sale.core.service.BaseServiceImpl;
import com.elitesland.yst.production.sale.core.util.TreeDataUtil;
import com.elitesland.yst.production.sale.entity.BipItemCategoryDO;
import com.elitesland.yst.production.sale.repo.shop.BipItemCategoryRepo;
import com.elitesland.yst.production.sale.repo.shop.BipItemCategoryRepoProc;
import com.elitesland.yst.production.sale.repo.shop.BipItemRepoProc;
import com.elitescloud.cloudt.common.base.ApiResult;
import com.elitescloud.boot.exception.BusinessException;
//import com.elitescloud.cloudt.core.security.util.SecurityUtil;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.cache.annotation.CacheEvict;
import org.springframework.cache.annotation.Cacheable;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import javax.validation.Valid;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;
import java.util.stream.Collectors;

/**
 * @auther JinXK
 * @date 2021/8/6 12:07
 */
@Service
@RequiredArgsConstructor
@Slf4j
@Valid
public class BipItemCategoryServiceImpl extends BaseServiceImpl implements BipItemCategoryService {

    private final BipItemCategoryRepo itemCategoryRepo;
    private final BipItemCategoryRepoProc itemCategoryRepoProc;
    private final BipItemRepoProc itemRepoProc;

    @Autowired
    private BipCustUserBindService bipCustUserBindService;

    private static final BipItemCategoryConvert CONVERT = BipItemCategoryConvert.INSTANCE;
    /**
     * 分类的默认根节点ID
     */
    private static final long ROOT = -1L;
    private static final String CACHE_NAME = ConstantsCache.ITEM_CAT_NAME;
    private static final String CACHE_KEY_TREE = "'tree'";

    @CacheEvict(value = CACHE_NAME, allEntries = true, condition = "#result.success")
    @Transactional(rollbackFor = Exception.class)
    @Override
    public ApiResult<Long> save(BipItemCategorySaveVO saveVO) {
        try {
            validateForSave(saveVO, null);
        } catch (IllegalArgumentException e) {
            log.info("保存商品分类校验不通过", e);
            return ApiResult.fail("保存失败，" + e.getMessage());
        }

        // vo转do
        var doo = CONVERT.vo2DO(saveVO);

        // 设置等级
        fillLevel(doo);

        itemCategoryRepo.save(doo);

        return ApiResult.ok(doo.getId());
    }

    @CacheEvict(value = CACHE_NAME, allEntries = true, condition = "#result.success")
    @Transactional(rollbackFor = Exception.class)
    @Override
    public ApiResult<Long> update(BipItemCategorySaveVO saveVO, Long id) {
        try {
            validateForSave(saveVO, id);
        } catch (IllegalArgumentException e) {
            log.info("修改商品分类校验不通过", e);
            return ApiResult.fail("修改失败，" + e.getMessage());
        }

        var category = itemCategoryRepo.findById(id).orElseThrow(new BusinessException("分类不存在"));

        // vo转do
        CONVERT.copyVO2DO(saveVO, category);
        if (category.getLevel() == null) {
            fillLevel(category);
        }

        itemCategoryRepo.save(category);

        // 如果名称修改，则修改对应商品的
        if (!StrUtil.equals(category.getName(), saveVO.getName())) {
            itemRepoProc.updateCategoryName(id, category.getName(), category.getLevel());
        }

        return ApiResult.ok(id);
    }

    @CacheEvict(value = CACHE_NAME, allEntries = true, condition = "#result.success")
    @Transactional(rollbackFor = Exception.class)
    @Override
    public ApiResult<Long> updateEnable(Long id) {
        var enable = itemCategoryRepoProc.getEnable(id);
        itemCategoryRepoProc.updateEnable(id, !Boolean.TRUE.equals(enable));

        return ApiResult.ok(id);
    }

    @CacheEvict(value = CACHE_NAME, allEntries = true, condition = "#result.success")
    @Transactional(rollbackFor = Exception.class)
    @Override
    public ApiResult<Long> delete(Long id) {
        var exists = itemCategoryRepoProc.existsById(id);
        if (!exists) {
            // 不存在则直接返回
            return ApiResult.fail("数据不存在");
        }
        exists = itemCategoryRepoProc.existsByPid(id);
        if (exists) {
            return ApiResult.fail("删除失败，请先删除下级分类");
        }

        var level = itemCategoryRepoProc.getLevel(id);
        exists = itemRepoProc.existsByCategoryId(id, level);
        if (exists) {
            return ApiResult.fail("删除失败，请先删除分类下的商品");
        }

        itemCategoryRepoProc.delete(id);
        return ApiResult.ok(id);
    }

    @Override
    public ApiResult<BipItemCategoryRespVO> get(Long id) {
        var doo = itemCategoryRepo.findById(id).orElse(null);
        if (doo == null) {
            return ApiResult.fail("商品分类不存在");
        }

        var respVO = CONVERT.do2VO(doo);
        // 设置父节点
        if (respVO.getPid() == ROOT) {
            respVO.setPName("");
        } else {
            respVO.setPName(itemCategoryRepoProc.getName(respVO.getPid()));
        }

        // 分类图片
        if (StrUtil.isNotBlank(respVO.getPicFileCode())) {
            com.el.coordinator.core.common.api.ApiResult<FileObjRespVO<String>> picResult = fileService.get(respVO.getPicFileCode());
            if (!picResult.isSuccess()) {
                log.error("获取分类图片失败：{}", picResult.getMsg());
            }
            respVO.setPicFileInfo(picResult.getData());
        }

        return ApiResult.ok(respVO);
    }

    @Override
    public ApiResult<List<ItemCatTreeAllRespVO>> treeAll() {
        // 先查出所有未删除且启用的
        var doList = itemCategoryRepo.findAll();
        if (doList.isEmpty()) {
            return ApiResult.ok(Collections.emptyList());
        }

        // 转树形列表
        var respVOList = do2TreeAll(doList);

        return ApiResult.ok(respVOList);
    }

    @Override
    public ApiResult<List<ItemCatTreeAllRespVO>> treeAll(Long ouId) {
        if (ouId == null) {
            return ApiResult.ok(Collections.emptyList());
        }
        // 先查出所有未删除且启用的
        var doList = itemCategoryRepoProc.queryByOuId(ouId);
        if (doList.isEmpty()) {
            return ApiResult.ok(Collections.emptyList());
        }

        // 转树形列表
        var respVOList = do2TreeAll(doList);

        return ApiResult.ok(respVOList);
    }

    @Cacheable(value = CACHE_NAME, keyGenerator = ConstantsCache.KEY_GENERATOR_CURRENT_USER_CUST_OU)
    @Override
    public ApiResult<List<ItemCatTreeRespVO>> treeOfOu(Long ouId) {
//        if (ouId == null) {
//            // 获取绑定的公司
//            var user = SecurityContextUtil.currentUser();
//            if (user == null) {
//                return ApiResult.ok(Collections.emptyList());
//            }
//            var custResult = bipCustUserBindService.getCustOfUserBind(user.getUser().getId());
//            if (custResult.getData() == null || custResult.getData().getOuId() == null) {
//                return ApiResult.ok(Collections.emptyList());
//            }
//            ouId = custResult.getData().getOuId();
//        }
//        // 先查出所有未删除且启用的
//        var doList = itemCategoryRepoProc.queryByOuId(ouId).stream().filter(BipItemCategoryDO::getEnable).collect(Collectors.toList());
//        if (doList.isEmpty()) {
//            return ApiResult.ok(Collections.emptyList());
//        }
//
//        // 转树形列表
//        var respVOList = do2Tree(doList);
//
//        return ApiResult.ok(respVOList);
        return null;
    }

    @Cacheable(value = CACHE_NAME, key = CACHE_KEY_TREE)
    @Override
    public ApiResult<List<ItemCatTreeRespVO>> tree() {
        // 先查出所有未删除且启用的
        var doList = itemCategoryRepo.findAll().stream().filter(BipItemCategoryDO::getEnable).collect(Collectors.toList());
        if (doList.isEmpty()) {
            return ApiResult.ok(Collections.emptyList());
        }

        // 转树形列表
        var respVOList = do2Tree(doList);

        return ApiResult.ok(respVOList);
    }

    private List<ItemCatTreeRespVO> do2Tree(List<BipItemCategoryDO> doList) {
        var respVOList = CONVERT.do2TreeVO(doList);
        TreeDataUtil<ItemCatTreeRespVO> treeDataUtil = new TreeDataUtil<>(respVOList, ItemCatTreeRespVO::getId, ItemCatTreeRespVO::getPid,
                ItemCatTreeRespVO::setChildren, Comparator.comparingInt(ItemCatTreeRespVO::getSortNo));
        return ((List<ItemCatTreeRespVO>) treeDataUtil.getRoots()).stream().filter(t -> t.getPid() == ROOT).collect(Collectors.toList());
    }

    private List<ItemCatTreeAllRespVO> do2TreeAll(List<BipItemCategoryDO> doList) {
        var respVOList = CONVERT.do2TreeAllVO(doList);
        TreeDataUtil<ItemCatTreeAllRespVO> treeDataUtil = new TreeDataUtil<>(respVOList, ItemCatTreeAllRespVO::getId, ItemCatTreeAllRespVO::getPid,
                ItemCatTreeAllRespVO::setChildren, Comparator.comparingInt(ItemCatTreeAllRespVO::getSortNo));
        return  ((List<ItemCatTreeAllRespVO>) treeDataUtil.getRoots()).stream().filter(t -> t.getPid() == ROOT).collect(Collectors.toList());
    }

    private void validateForSave(BipItemCategorySaveVO saveVO, Long id) {
        var isAdd = id == null;

        // 判断修改的数据是否存在
        var exists = false;
        if (!isAdd) {
            exists = itemCategoryRepoProc.existsById(id);
            Assert.isTrue(exists, "修改的数据不存在");
        }

        // 判断父节点是否存在
        saveVO.setPid(ObjectUtil.defaultIfNull(saveVO.getPid(), ROOT));
        if (saveVO.getPid() != null && saveVO.getPid() != ROOT) {
            exists = itemCategoryRepoProc.existsById(saveVO.getPid());
            Assert.isTrue(exists, "上级分类不存在");
        }
        if (!isAdd) {
            // 修改时不可修改上级分类
            var pidOld = itemCategoryRepoProc.getPid(id);
            Assert.isTrue(pidOld.longValue() == saveVO.getPid(), "上级分类不可修改");
        }

        // 判断编号是否存在
        exists = isAdd ? itemCategoryRepoProc.existsByCode(saveVO.getCode()) : itemCategoryRepoProc.existsByCode(saveVO.getCode(), id);
        Assert.isFalse(exists, "编号已存在");

        // 判断序号是否存在
        if (saveVO.getPid() == ROOT) {
            exists = isAdd ? itemCategoryRepoProc.existsBySortNo(saveVO.getOuId(), saveVO.getPid(), saveVO.getSortNo()) : itemCategoryRepoProc.existsBySortNo(saveVO.getOuId(), saveVO.getPid(), saveVO.getSortNo(), id);
        } else {
            exists = isAdd ? itemCategoryRepoProc.existsBySortNo(saveVO.getPid(), saveVO.getSortNo()) : itemCategoryRepoProc.existsBySortNo(saveVO.getPid(), saveVO.getSortNo(), id);
        }
        Assert.isFalse(exists, "序号已存在");

        saveVO.setEnable(ObjectUtil.defaultIfNull(saveVO.getEnable(), false));

        // 判断图片是否存在
        if (StrUtil.isNotBlank(saveVO.getPicFileCode())) {
            exists = isExistsFile(saveVO.getPicFileCode());
            Assert.isTrue(exists, "未找到有效图片，请从新上传");
        }
    }

    private void fillLevel(BipItemCategoryDO doo) {
        int level = doo.getPid() == ROOT ? 1 : (itemCategoryRepoProc.getLevel(doo.getPid())) + 1;
        doo.setLevel(level);
    }
}
