package com.elitesland.yst.production.sale.service.shop;

import cn.hutool.core.collection.CollUtil;
import com.elitescloud.cloudt.common.base.ApiResult;
import com.elitescloud.boot.exception.BusinessException;
import com.elitesland.yst.production.sale.api.dto.BipItemDTO;
import com.elitesland.yst.production.sale.api.dto.BipItemSkuDTO;
import com.elitesland.yst.production.sale.api.service.shop.BipItemService;
import com.elitesland.yst.production.sale.convert.shop.BipItemConvert;
import com.elitesland.yst.production.sale.core.service.BaseServiceImpl;
import com.elitesland.yst.production.sale.entity.BipItemDO;
import com.elitesland.yst.production.sale.entity.BipItemSkuDO;
import com.elitesland.yst.production.sale.entity.QBipItemDO;
import com.elitesland.yst.production.sale.event.ItemStockUpdateEvent;
import com.elitesland.yst.production.sale.repo.shop.*;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.redisson.api.RLock;
import org.redisson.api.RedissonClient;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.util.Collections;
import java.util.Comparator;
import java.util.List;
import java.util.Map;
import java.util.concurrent.TimeUnit;
import java.util.function.BiConsumer;
import java.util.stream.Collectors;

/**
 * <p>
 * 小程序商品业务实现
 * </p>
 *
 * @author Shadow
 * @since 2021-08-10 16:10:59
 */
@Service
@RequiredArgsConstructor
@Slf4j
public class BipItemServiceImpl extends BaseServiceImpl implements BipItemService {

    private final BipItemRepo itemRepo;
    private final BipItemRepoProc itemRepoProc;
    private final BipItemExtRepo itemExtRepo;
    private final BipItemExtRepoProc itemExtRepoProc;
    private final BipItemPicRepo itemPicRepo;
    private final BipItemPicRepoProc itemPicRepoProc;
    private final BipItemSkuRepo itemSkuRepo;
    private final BipItemSkuRepoProc itemSkuRepoProc;
    private final BipItemCategoryRepoProc itemCategoryRepoProc;

    private static final BipItemConvert CONVERT = BipItemConvert.INSTANCE;
    private static final QBipItemDO ITEM_DO = QBipItemDO.bipItemDO;

    // 分布式锁
    private final RedissonClient redissonClient;

    @Transactional(rollbackFor = Exception.class)
    @Override
    public ApiResult<Long> updateStock(Long skuId, Integer stock) {
        try {
            stockLockAndUpdate(skuId, stock, itemSkuRepoProc::updateStock);
        } catch (Exception e) {
            throw new BusinessException("更新商品库存失败，请稍后再试", e);
        }

        // 发送库存变更事件
        publishEventSync(new ItemStockUpdateEvent(this, skuId, stock > 0));
        return ApiResult.ok(skuId);
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public ApiResult<Long> incrementStock(Long skuId, Integer increment) {
        try {
            stockLockAndUpdate(skuId, increment, itemSkuRepoProc::incrementStock);
        } catch (Exception e) {
            throw new BusinessException("更新商品库存失败，请稍后再试", e);
        }

        // 发送库存变更事件
        publishEventSync(new ItemStockUpdateEvent(this, skuId, increment > 0));
        return ApiResult.ok(skuId);
    }

    @Override
    public ApiResult<List<BipItemSkuDTO>> querySku(List<Long> skuIds) {
        var skuList = listSku(skuIds);
        return ApiResult.ok(skuList);
    }

    @Override
    public ApiResult<List<BipItemDTO>> querySkuWithItem(List<Long> skuIds) {
        var skus = itemSkuRepo.findAllById(skuIds);
        if (skus.isEmpty()) {
            return ApiResult.ok(Collections.emptyList());
        }

        var itemIds = skus.parallelStream().map(BipItemSkuDO::getBipItemId).collect(Collectors.toList());

        var items = itemRepo.findAllById(itemIds);

        var itemDTOs = convertItem(items, skus);
        return ApiResult.ok(itemDTOs);
    }

    @Override
    public ApiResult<List<BipItemDTO>> queryItem(List<Long> itemIds) {
        var items = listItem(itemIds, true);
        return ApiResult.ok(items);
    }

    @Override
    public ApiResult<List<BipItemDTO>> queryItemWithoutSku(List<Long> itemIds) {
        var items = listItem(itemIds, false);
        return ApiResult.ok(items);
    }

    private List<BipItemDTO> listItem(List<Long> itemIds, boolean withSku) {
        var items = itemRepo.findAllById(itemIds);
        if (items.isEmpty()) {
            return Collections.emptyList();
        }

        List<BipItemSkuDO> skus = withSku ? itemSkuRepoProc.queryByBipItemId(itemIds) : null;
        return convertItem(items, skus);
    }

    private List<BipItemDTO> convertItem(List<BipItemDO> items, List<BipItemSkuDO> skus) {
        Map<Long, List<BipItemSkuDTO>> skuMap = null;
        if (CollUtil.isNotEmpty(skus)) {
            skuMap = convertSku(skus).stream().collect(Collectors.groupingBy(BipItemSkuDTO::getBipItemId));
        }

        Map<Long, List<BipItemSkuDTO>> finalSkuMap = skuMap;
        return items.stream().map(t -> {
            var dto = CONVERT.do2DTO(t);
            if (finalSkuMap != null) {
                dto.setSkuList(finalSkuMap.get(t.getId()));
            }
            return dto;
        }).collect(Collectors.toList());
    }

    private List<BipItemSkuDTO> listSku(List<Long> skuIds) {
        var skus = itemSkuRepo.findAllById(skuIds);
        return convertSku(skus);
    }

    private List<BipItemSkuDTO> convertSku(List<BipItemSkuDO> skus) {
        if (skus.isEmpty()) {
            return Collections.emptyList();
        }

        return skus.stream().sorted(Comparator.comparingInt(BipItemSkuDO::getSortNo))
                .map(t -> {
                    var dto = CONVERT.skuDO2DTO(t);
                    dto.setAttrList(convert2List(t.getAttr(), BipItemSkuDTO.Attr.class));
                    return dto;
                }).collect(Collectors.toList());
    }

    private void stockLockAndUpdate(Long skuId, Integer value, BiConsumer<Long, Integer> consumer) throws Exception {
        RLock lock = null;
        try {
            lock = redissonClient.getLock("yst_sale_item_stock_" + skuId.toString());
            if (lock.tryLock(1, 1, TimeUnit.MINUTES)) {
                consumer.accept(skuId, value);
                lock.unlock();
            }
        } finally {
            if (lock != null && lock.isLocked()) {
                lock.unlock();
            }
        }
    }
}


