package com.elitesland.yst.production.sale.service;

import com.elitescloud.boot.exception.BusinessException;
import com.elitescloud.cloudt.common.base.ApiResult;
import com.elitesland.yst.production.sale.api.service.DeCustAccountService;
import com.elitesland.yst.production.sale.api.vo.param.crm.CrmCustAccountParam;
import com.elitesland.yst.production.sale.api.vo.resp.crm.CustAccountVO;
import com.elitesland.yst.production.sale.dto.ModifyStoreMobile;
import com.elitesland.yst.production.sale.dto.StoreAccountSaveRpcDTO;
import com.elitesland.yst.production.sale.dto.param.StoreAccountSaveRpcParam;
import com.elitesland.yst.production.sale.entity.CustAccountDO;
import com.elitesland.yst.production.sale.entity.QCustAccountDO;
import com.elitesland.yst.production.sale.repo.CustAccountRepo;
import com.elitesland.yst.production.sale.repo.CustAccountRepoProc;
import com.querydsl.core.types.dsl.BooleanExpression;
import jodd.util.StringUtil;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.CollectionUtils;
import org.springframework.stereotype.Service;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RestController;

import java.util.*;
import java.util.stream.Collectors;

@Service
@Slf4j
@RequiredArgsConstructor
@RestController
@RequestMapping(CrmCustAccountRpcService.URI)
public class CrmCustAccountRpcServiceImpl implements CrmCustAccountRpcService {

    private final DeCustAccountService deCustAccountService;

    private final CustAccountRepoProc custAccountRepoProc;

    private final CustAccountRepo custAccountRepo;

    @Override
    public CustAccountVO getCustAccountByUserId(Long userId) {
        return deCustAccountService.getCustAccountByUserId(userId);
    }

    /**
     * 代码逻辑调整：原逻辑 根据custCode 取1条（limit 1）
     * 现逻辑 根据custCode取数据集合，取id最小的一条
     * 当前确认的业务应用逻辑点：1.销售域创建客户账号  2.消息通知
     *
     * @param custCodes
     * @return
     */
    @Override
    public List<CustAccountVO> getCustAccountBycustCode(List<String> custCodes) {
        if (CollectionUtils.isEmpty(custCodes)) {
            throw new BusinessException("客户编号为空");
        }
        List<CustAccountVO> accountByCustCodeList = custAccountRepoProc.getAccountByCustCode(custCodes);
        List<CustAccountVO> list = new ArrayList<>();
        if (CollectionUtils.isNotEmpty(accountByCustCodeList)) {
            Map<String, List<CustAccountVO>> listMap = accountByCustCodeList.stream().collect(Collectors.groupingBy(CustAccountVO::getCustCode));
            listMap.entrySet().forEach(i ->
                i.getValue().stream().max(Comparator.comparing(CustAccountVO::getId))
                        .ifPresent(v -> list.add(v))
            );
        }
        return list;
    }

    @Override
    public ApiResult<List<StoreAccountSaveRpcDTO>> saveStoreAccount(List<StoreAccountSaveRpcParam> param) {
        if (CollectionUtils.isEmpty(param)) {
            return ApiResult.ok();
        }
        var data = param.stream().map(this::saveStoreAccountQuietLy).collect(Collectors.toList());
        return ApiResult.ok(data);
    }

    @Override
    public ApiResult<String> modifyStoreMobile(ModifyStoreMobile modifyStoreMobile) {
        if (Objects.isNull(modifyStoreMobile) || StringUtil.isBlank(modifyStoreMobile.getUserName()) || StringUtil.isBlank(modifyStoreMobile.getMobile())) {
            throw new BusinessException("参数不能为空！");
        }
        QCustAccountDO qCustAccountDO = QCustAccountDO.custAccountDO;
        BooleanExpression e = qCustAccountDO.userName.eq(modifyStoreMobile.getUserName()).and(qCustAccountDO.deleteFlag.eq(0));
        Optional<CustAccountDO> optional = custAccountRepo.findOne(e);
        if (optional.isPresent()) {
            CustAccountDO custAccountDO = optional.get();
            custAccountDO.setMobile(modifyStoreMobile.getMobile());
            custAccountRepo.save(custAccountDO);
            return ApiResult.ok(custAccountDO.getCustCode());
        }
        return null;
    }

    private StoreAccountSaveRpcDTO saveStoreAccountQuietLy(StoreAccountSaveRpcParam param) {
        StoreAccountSaveRpcDTO resp = new StoreAccountSaveRpcDTO();
        resp.setStoreCode(param.getStoreCode());
        resp.setUserName(param.getStoreCode());
        resp.setFailureFlag(Boolean.FALSE);
        try {
            this.saveStoreAccount(param);
        } catch (Exception e) {
            log.warn("========>保存门店账号出错:{}", e.getMessage(), e);
            resp.setFailureFlag(Boolean.TRUE);
            resp.setFailureMsg(e.getMessage());
        }
        return resp;
    }

    private void saveStoreAccount(StoreAccountSaveRpcParam param) {
        CrmCustAccountParam createUserParam = new CrmCustAccountParam();
        String storeCode = param.getStoreCode();
        createUserParam.setCustCode(storeCode);
        createUserParam.setCustName(param.getStoreName());
        createUserParam.setMobile(param.getStoreContPhone());
        createUserParam.setUserName(storeCode);

        //根据这两个常量确定门店账户角色
        createUserParam.setDealerType("STORE"); //参考销售域表.de_account_role
        createUserParam.setBusinessType("ADMIN");

        createUserParam.setEnable(Objects.isNull(param.getEnable()) || param.getEnable() ? Boolean.TRUE : Boolean.FALSE);
        createUserParam.setSubAcFlag(Boolean.FALSE);

        createUserParam.setCreateFlag(notExists(createUserParam.getUserName()));

        deCustAccountService.createUser(createUserParam);
    }

    private Boolean notExists(String userName) {
        return !custAccountRepoProc.exists(userName, null, "");
    }
}
