package com.elitesland.yst.production.sale.repo.shop;

import com.elitesland.yst.production.sale.api.dto.BipItemCatsDTO;
import com.elitesland.yst.production.sale.common.constant.ConstantsSale;
import com.elitesland.yst.production.sale.entity.BipItemDO;
import com.elitesland.yst.production.sale.entity.QBipItemDO;
import com.elitescloud.boot.exception.BusinessException;
import com.querydsl.core.types.Projections;
import com.querydsl.core.types.dsl.NumberPath;
import com.querydsl.core.types.dsl.StringPath;
import com.querydsl.jpa.impl.JPAQueryFactory;
import org.springframework.stereotype.Repository;

import java.math.BigDecimal;
import java.time.LocalDateTime;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

/**
 * <p>
 * 功能说明
 * </p>
 *
 * @author Shadow
 * @since 2021-08-10 16:16:25
 */
@Repository
public class BipItemRepoProc {

    private final JPAQueryFactory jpaQueryFactory;

    private static final QBipItemDO DO = QBipItemDO.bipItemDO;

    public BipItemRepoProc(JPAQueryFactory jpaQueryFactory) {
        this.jpaQueryFactory = jpaQueryFactory;
    }

    /**
     * 更新分类名称
     *
     * @param categoryId   分类ID
     * @param categoryName 分类名称
     * @param level        分类级别
     */
    public void updateCategoryName(Long categoryId, String categoryName, Integer level) {
        jpaQueryFactory.update(DO)
                .set(getCategoryName(level), categoryName)
                .where(getCategory(level).eq(categoryId))
                .execute();
    }

    /**
     * 更新状态
     *
     * @param id    ID
     * @param state 状态
     */
    public void updateState(Long id, String state) {
        jpaQueryFactory.update(DO)
                .set(DO.state, state)
                .where(DO.id.eq(id))
                .execute();
    }

    /**
     * 批量更新状态
     *
     * @param ids   ID
     * @param state 状态
     */
    public void updateState(List<Long> ids, String state) {
        jpaQueryFactory.update(DO)
                .set(DO.state, state)
                .where(DO.id.in(ids))
                .execute();
    }

    /**
     * 更新销量
     *
     * @param id        商品ID
     * @param numSale   销量
     * @param numSale90 90天销量
     */
    public void updateSale(Long id, Long numSale, Long numSale90) {
        jpaQueryFactory.update(DO)
                .set(DO.numSale, numSale)
                .set(DO.numSale90, numSale90)
                .where(DO.id.eq(id))
                .execute();
    }

    /**
     * 更新好评率
     *
     * @param id           商品ID
     * @param numEval      总评价数
     * @param numEvalGood  好评数量
     * @param rateEvalGood 好评率
     */
    public void updateRateEvalGood(Long id, Long numEval, Long numEvalGood, BigDecimal rateEvalGood) {
        jpaQueryFactory.update(DO)
                .set(DO.numEval, numEval)
                .set(DO.numEvalGood, numEvalGood)
                .set(DO.rateEvalGood, rateEvalGood)
                .where(DO.id.eq(id))
                .execute();
    }

    /**
     * 更新评分
     *
     * @param id        商品ID
     * @param scoreEval 评分
     */
    public void updateScoreEval(Long id, BigDecimal scoreEval) {
        jpaQueryFactory.update(DO)
                .set(DO.scoreEval, scoreEval)
                .where(DO.id.eq(id))
                .execute();
    }

    /**
     * 更新是否缺库存
     *
     * @param id        商品ID
     * @param lackStock 是否缺库存
     */
    public void updateLackStock(Long id, boolean lackStock) {
        jpaQueryFactory.update(DO)
                .set(DO.lackStock, lackStock)
                .where(DO.id.eq(id))
                .execute();
    }

    /**
     * 伪删除
     *
     * @param id ID
     */
    public void deleteFake(Long id) {
        jpaQueryFactory.update(DO)
                .set(DO.deleteFlag, ConstantsSale.COMMON_DELETE_YSE)
                .set(DO.onShelf, false)
                .set(DO.timeOffShelf, LocalDateTime.now())
                .set(DO.remark, "删除而下架")
                .where(DO.id.eq(id))
                .execute();
    }

    /**
     * 伪删除
     *
     * @param ids ID
     */
    public void deleteFake(List<Long> ids) {
        jpaQueryFactory.update(DO)
                .set(DO.deleteFlag, ConstantsSale.COMMON_DELETE_YSE)
                .set(DO.onShelf, false)
                .set(DO.timeOffShelf, LocalDateTime.now())
                .set(DO.remark, "删除而下架")
                .where(DO.id.in(ids))
                .execute();
    }

    /**
     * 删除商品
     *
     * @param id 商品ID
     */
    public void delete(Long id) {
        jpaQueryFactory.delete(DO)
                .where(DO.id.eq(id))
                .execute();
    }

    /**
     * 删除商品
     *
     * @param ids 商品ID
     */
    public void delete(List<Long> ids) {
        jpaQueryFactory.delete(DO)
                .where(DO.id.in(ids))
                .execute();
    }

    /**
     * 根据ID判断是否存在
     *
     * @param id ID
     * @return 是否存在
     */
    public boolean existsById(Long id) {
        return jpaQueryFactory.select(DO.id)
                .from(DO)
                .where(DO.id.eq(id))
                .limit(1)
                .fetchOne() != null;
    }

    /**
     * 根据上架编号判断是否存在
     *
     * @param shelfCode 上架编号
     * @return 是否存在
     */
    public boolean existsByShelfCode(String shelfCode) {
        return jpaQueryFactory.select(DO.id)
                .from(DO)
                .where(DO.shelfCode.eq(shelfCode).and(DO.deleteFlag.eq(ConstantsSale.COMMON_DELETE_NO)))
                .limit(1)
                .fetchOne() != null;
    }

    public BipItemDO withoutMainImage(List<Long> ids) {
        return jpaQueryFactory.select(DO)
                .from(DO)
                .where(DO.id.in(ids).and(DO.mainPicFileCode.isNull().or(DO.mainPicFileCode.eq(""))))
                .limit(1)
                .fetchOne();
    }

    /**
     * 根据编号判断是否存在
     *
     * @param shelfCode 上架编号
     * @param id        ID
     * @return 是否存在
     */
    public boolean existsByShelfCode(String shelfCode, Long id) {
        return jpaQueryFactory.select(DO.id)
                .from(DO)
                .where(DO.shelfCode.eq(shelfCode).and(DO.id.ne(id)).and(DO.deleteFlag.eq(ConstantsSale.COMMON_DELETE_NO)))
                .limit(1)
                .fetchOne() != null;
    }

    /**
     * 根据商品判断是否存在
     *
     * @param itemId 商品ID
     * @return 是否存在
     */
    public boolean existsByItemId(Long itemId) {
        return jpaQueryFactory.select(DO.id)
                .from(DO)
                .where(DO.itemId.eq(itemId).and(DO.deleteFlag.eq(ConstantsSale.COMMON_DELETE_NO)))
                .limit(1)
                .fetchOne() != null;
    }

    /**
     * 根据商品判断是否存在
     *
     * @param itemId 商品ID
     * @param id     ID
     * @return 是否存在
     */
    public boolean existsByItemId(Long itemId, Long id) {
        return jpaQueryFactory.select(DO.id)
                .from(DO)
                .where(DO.itemId.eq(itemId).and(DO.id.ne(id)).and(DO.deleteFlag.eq(ConstantsSale.COMMON_DELETE_NO)))
                .limit(1)
                .fetchOne() != null;
    }

    /**
     * 根据商品分类判断是否存在
     *
     * @param categoryId 商品分类ID
     * @param level      商品分类等级
     * @return 是否存在
     */
    public boolean existsByCategoryId(Long categoryId, Integer level) {
        NumberPath<Long> condition = getCategory(level);
        return jpaQueryFactory.select(DO.id)
                .from(DO)
                .where(condition.eq(categoryId).and(DO.deleteFlag.eq(ConstantsSale.COMMON_DELETE_NO)))
                .limit(1)
                .fetchOne() != null;
    }

    /**
     * 过滤出存在的数据
     *
     * @param ids 商品ID
     * @return 存在的数据
     */
    public List<Long> filterExists(List<Long> ids) {
        return jpaQueryFactory.select(DO.id)
                .from(DO)
                .where(DO.id.in(ids))
                .fetch();
    }

    /**
     * 获取商品中心的ID
     *
     * @param id ID
     * @return 商品ID
     */
    public Long getItemId(Long id) {
        return jpaQueryFactory.select(DO.itemId)
                .from(DO)
                .where(DO.id.eq(id))
                .fetchOne();
    }

    /**
     * 获取商品ID
     *
     * @param itemId 商品中心的ID
     * @return 商品ID
     */
    public Long getIdByItemId(Long itemId) {
        return jpaQueryFactory.select(DO.id)
                .from(DO)
                .where(DO.itemId.eq(itemId))
                .limit(1)
                .fetchOne();
    }

    /**
     * 获取状态
     *
     * @param id ID
     * @return 状态
     */
    public String getState(Long id) {
        return jpaQueryFactory.select(DO.state)
                .from(DO)
                .where(DO.id.eq(id))
                .fetchOne();
    }

    /**
     * 获取状态
     *
     * @param ids ID
     * @return ID与状态的映射
     */
    public Map<Long, String> getState(List<Long> ids) {
        return jpaQueryFactory.select(DO.id, DO.state)
                .from(DO)
                .where(DO.id.in(ids))
                .fetch()
                .stream()
                .collect(Collectors.toMap(t -> t.get(DO.id), t -> t.get(DO.state), (t1, t2) -> t1))
                ;
    }

    /**
     * 获取是否上架过
     *
     * @param id ID
     * @return 是否上架过
     */
    public Boolean getShelf(Long id) {
        return jpaQueryFactory.select(DO.shelf)
                .from(DO)
                .where(DO.id.eq(id))
                .fetchOne();
    }

    /**
     * 获取是否上架过
     *
     * @param ids ID
     * @return ID与是否上架过的映射
     */
    public Map<Long, Boolean> getShelf(List<Long> ids) {
        return jpaQueryFactory.select(DO.id, DO.shelf)
                .from(DO)
                .where(DO.id.in(ids))
                .fetch()
                .stream()
                .collect(Collectors.toMap(t -> t.get(DO.id), t -> t.get(DO.shelf), (t1, t2) -> t1))
                ;
    }

    /**
     * 获取商品的分类
     *
     * @param id 商品ID
     * @return 分类
     */
    public BipItemCatsDTO getItemCategory(Long id) {
        return jpaQueryFactory.select(Projections.bean(BipItemCatsDTO.class,
                        DO.categoryId1, DO.categoryId2, DO.categoryId3, DO.categoryName1, DO.categoryName2, DO.categoryName3))
                .from(DO)
                .where(DO.id.eq(id))
                .fetchOne();
    }

    public NumberPath<Long> getCategory(Integer level) {
        switch (level) {
            case 1:
                return DO.categoryId1;
            case 2:
                return DO.categoryId2;
            case 3:
                return DO.categoryId3;
            default:
                throw new BusinessException("未知商品分类");
        }
    }

    private StringPath getCategoryName(Integer level) {
        switch (level) {
            case 1:
                return DO.categoryName1;
            case 2:
                return DO.categoryName2;
            case 3:
                return DO.categoryName3;
            default:
                throw new BusinessException("未知商品分类");
        }
    }

    public BipItemDO getItemForImage(String code, String ouCode) {
        return jpaQueryFactory.select(DO).from(DO).where(DO.itemCode.eq(code).and(DO.ouCode.eq(ouCode))).fetchOne();
    }

    public List<BipItemDO> getItemByCodes(List<String> codes) {
        return jpaQueryFactory.select(DO).from(DO).where(DO.itemCode.in(codes)).fetch();
    }

    public List<Long> findByIdAndOuCode(List<Long> ids,String buCode){
        return jpaQueryFactory.select(DO.id).from(DO).where(DO.id.in(ids).and(DO.ouCode.eq(buCode))).fetch();
    }

    public List<Long> findIdByItemIds(List<Long> ids){
        return jpaQueryFactory.select(DO.id).from(DO).where(DO.id.in(ids).and(DO.deleteFlag.eq(0))).fetch();
    }

    public List<Long> findOuIdByItemIds(List<Long> ids){
        return jpaQueryFactory.select(DO.ouId).from(DO).where(DO.id.in(ids).and(DO.deleteFlag.eq(0))).fetch();
    }

}

