package com.elitesland.support.provider.pri.cache;

import cn.hutool.core.collection.CollectionUtil;
import cn.hutool.core.date.LocalDateTimeUtil;
import cn.hutool.core.lang.Assert;
import cn.hutool.core.map.MapUtil;
import cn.hutool.extra.spring.SpringUtil;
import com.alibaba.fastjson.JSON;
import com.elitescloud.boot.core.base.UdcProvider;
import com.elitescloud.boot.exception.BusinessException;
import com.elitescloud.boot.tenant.client.common.TenantClient;
import com.elitescloud.boot.util.StrUtil;
import com.elitescloud.cloudt.common.base.ApiResult;
import com.elitesland.support.provider.item.dto.ItmItemIcDTO;
import com.elitesland.support.provider.item.service.ItmItemRpcService;
import com.elitesland.support.provider.pri.cache.client.*;
import com.elitesland.support.provider.pri.service.PriPriceRpcService;
import com.elitesland.support.provider.pri.service.SuppPriPriceRpcService;
import com.elitesland.support.provider.pri.service.dto.PriPriceRpcDTO;
import com.elitesland.support.provider.pri.service.param.ItmPriPriceRpcDtoParam;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;

import java.math.BigDecimal;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.util.*;
import java.util.stream.Collectors;

@Slf4j
public class PriPriceCacheService {

    private static ItmItemRpcService itmItemRpcService;
    private static UdcProvider udcProvider;
    private static SuppPriPriceRpcService priPriceRpcService;

    private static String priFetchMethod;

    public static List<String> priceTypes = List.of("MAIN_PRICE", "SALE_PRICE", "CHANNEL_PRICE", "INTERNAL_SETTLEMENT_PRICE", "NEGO_PRICE", "PROM_PRICE", "RETAIL_PRICE");

    /**
     * 基础域内部调用
     *
     * @param params
     * @param rpcServiceImpl 传PriPriceRpcServiceImpl实例，避免基础域内部走rpc
     * @return
     */
    public static ApiResult<List<PriPriceRpcDTO>> findPriPrice(List<ItmPriPriceRpcDtoParam> params, SuppPriPriceRpcService rpcServiceImpl, ItmItemRpcService itemServiceImpl) {
        priPriceRpcService = rpcServiceImpl;
        itmItemRpcService = itemServiceImpl;
        return findPriPrice(params);
    }

    /**
     * 其他外部业务调用获取价格
     *
     * @param params
     * @return
     */
    public static ApiResult<List<PriPriceRpcDTO>> findPriPrice(List<ItmPriPriceRpcDtoParam> params) {
        log.info("缓存取价接口，参数：" + JSON.toJSONString(params));

        if (priFetchMethod == null) {
            priFetchMethod = SpringUtil.getProperty("yst.support.price.ftype", "redis");
        }

        // 配置非Redis取价模式，或Redis价格缓存未完成 走数据库取价
        if (!"redis".equals(priFetchMethod) || !PriPriceCacheClient.cacheFinished()) {
            if (priPriceRpcService == null) {
                priPriceRpcService = SpringUtil.getBean(SuppPriPriceRpcService.class);
            }
            return priPriceRpcService.findPriPrice(params);
        }

        long start = System.currentTimeMillis();
        if (CollectionUtil.isEmpty(params)) {
            return ApiResult.ok(new ArrayList<>());
        }

        // 检查参数合法性
        checkFindDtoByParamParameter(params);

        List<PriPriceRpcDTO> result = new ArrayList<>();

        // 非查询当天生效且启用状态价格的请求走原查询逻辑
        List<ItmPriPriceRpcDtoParam> notTodayParams = params.stream().filter(i -> i.getValidTime() != null && !LocalDateTimeUtil.isSameDay(LocalDateTime.now(), i.getValidTime()))
                .collect(Collectors.toList());
        List<ItmPriPriceRpcDtoParam> notEnableParams = params.stream().filter(i -> i.getPriceStatus() != null && Objects.equals(i.getPriceStatus(), "DISABLE"))
                .collect(Collectors.toList());
        if (!notTodayParams.isEmpty() || !notEnableParams.isEmpty()) {
            notTodayParams.addAll(notEnableParams);
            if (priPriceRpcService == null) {
                priPriceRpcService = SpringUtil.getBean(SuppPriPriceRpcService.class);
            }
            List<PriPriceRpcDTO> dbFindPriceList = priPriceRpcService.findPriPrice(notTodayParams).computeData();
            result.addAll(dbFindPriceList);
            List<String> dbParamUuids = notTodayParams.stream().map(i -> i.getUuid()).collect(Collectors.toList());
            params = params.stream().filter(i -> !dbParamUuids.contains(i.getUuid())).collect(Collectors.toList());
        }
        if (params.isEmpty()) {
            log.info("缓存获取价格结果，无缓存查询：" + JSON.toJSONString(result));
            return ApiResult.ok(result);
        }

        Long tenantId = TenantClient.getSessionTenant().getId();
        String dateStr = LocalDateTimeUtil.format(LocalDate.now(), "yyyyMMdd");

        // 如果商品编码为空，填充商品编码
        fillUpItem(params, null);
        Map<String, String> flowNoCache = getFlowNoCache();
        List<PriPriceConfCache> confCache = PriPriceCacheClient.findConfCache();
        Map<String, List<PriPriceConfCache>> priConfigMap = confCache.stream().collect(Collectors.groupingBy(i -> i.getPriceType()));

        Map<String, ArrayList<String>> redisKeyMap = new HashMap<>();
        Map<String, String> priceTypePathMap = new HashMap<>();
        List<String> redisKeys = new ArrayList<>();
        for (ItmPriPriceRpcDtoParam param : params) {
            List<String> currentPriceTypes = priceTypes;
            // 指令查询某种类型价格，则只查询该价格类型
            if (StringUtils.isNotBlank(param.getPriceType())) {
                currentPriceTypes = Arrays.asList(param.getPriceType());
            }
            for (String priceType : priConfigMap.keySet()) {
                if (!currentPriceTypes.contains(priceType)) {
                    continue;
                }
                for (PriPriceConfCache priPriceConfCache : priConfigMap.get(priceType)) {
                    List<String> elementPaths = new ArrayList<>();
                    List<String> elementPathValues = new ArrayList<>();

                    if (priPriceConfCache.getPriceElement().contains("OU_CODE")) {
                        if (!MapUtil.isEmpty(flowNoCache) && !Objects.isNull(flowNoCache.get("OU_CODE"))){
                            String value = param.getOuCode();
                            if (!StrUtil.isBlank(value)) {
                                elementPaths.add("OU_CODE");
                                elementPathValues.add(value);
                            }
                        }
                    }

                    if (priPriceConfCache.getPriceElement().contains("ITEM_CODE")) {
                        if (!MapUtil.isEmpty(flowNoCache) && !Objects.isNull(flowNoCache.get("ITEM_CODE"))){
                            String value = param.getItemCode();
                            if (!StrUtil.isBlank(value)) {
                                elementPaths.add("ITEM_CODE");
                                elementPathValues.add(value);
                            }
                        }
                    }

                    if (priPriceConfCache.getPriceElement().contains("CUSTOM_CODE")) {
                        if (!MapUtil.isEmpty(flowNoCache) && !Objects.isNull(flowNoCache.get("CUSTOM_CODE"))){
                            String value = param.getCustCode();
                            if (!StrUtil.isBlank(value)) {
                                elementPaths.add("CUSTOM_CODE");
                                elementPathValues.add(value);
                            }
                        }
                    }

                    if (priPriceConfCache.getPriceElement().contains("SALE_ORG")) {
                        if (!MapUtil.isEmpty(flowNoCache) && !Objects.isNull(flowNoCache.get("SALE_ORG"))){
                            String value = param.getSaleOrg();
                            if (!StrUtil.isBlank(value)) {
                                elementPaths.add("SALE_ORG");
                                elementPathValues.add(value);
                            }
                        }
                    }

                    if (priPriceConfCache.getPriceElement().contains("DISTRIBUTION_CHANNEL")) {
                        if (!MapUtil.isEmpty(flowNoCache) && !Objects.isNull(flowNoCache.get("DISTRIBUTION_CHANNEL"))){
                            String value = param.getSaleChannel();
                            if (!StrUtil.isBlank(value)) {
                                elementPaths.add("DISTRIBUTION_CHANNEL");
                                elementPathValues.add(value);
                            }
                        }
                    }


                    if (priPriceConfCache.getPriceElement().contains("REGION")) {
                        if (!MapUtil.isEmpty(flowNoCache) && !Objects.isNull(flowNoCache.get("REGION"))){
                            String value = param.getRegion();
                            if (!StrUtil.isBlank(value)) {
                                elementPaths.add("REGION");
                                elementPathValues.add(value);
                            }
                        }
                    }

                    if (priPriceConfCache.getPriceElement().contains("UOM")) {
                        if (!MapUtil.isEmpty(flowNoCache) && !Objects.isNull(flowNoCache.get("UOM"))){
                            String value = param.getUom();
                            if (!StrUtil.isBlank(value)) {
                                elementPaths.add("UOM");
                                elementPathValues.add(value);
                            }
                        }
                    }

                    if (priPriceConfCache.getPriceElement().contains("CURR_CODE")) {
                        if (!MapUtil.isEmpty(flowNoCache) && !Objects.isNull(flowNoCache.get("CURR_CODE"))){
                            String value = param.getCurrCode();
                            if (!StrUtil.isBlank(value)) {
                                elementPaths.add("CURR_CODE");
                                elementPathValues.add(value);
                            }
                        }
                    }
                    if (elementPaths.isEmpty()) {
                        continue;
                    }
                    if (!allBeIncluded(priPriceConfCache.getPriceElement(), elementPaths)) {
                        continue;
                    }

                    String elementPathStr = String.join(":", elementPaths);
                    String redisKey = PriPriceCache.getCacheKey(tenantId, dateStr, priceType, elementPathStr, String.join(":", elementPathValues));
                    if (!redisKeyMap.containsKey(param.getUuid())) {
                        redisKeyMap.put(param.getUuid(), new ArrayList<>());
                    }
                    redisKeyMap.get(param.getUuid()).add(redisKey);
                    redisKeys.add(redisKey);

                    priceTypePathMap.put(priceType + priPriceConfCache.getPriority().intValue(), elementPathStr);
                }
            }
        }

        if (redisKeyMap.isEmpty()) {
            log.info("缓存获取价格结果：" + JSON.toJSONString(result));
            return ApiResult.ok(result);
        }

        List<String> priceClsList = params.stream().filter(i -> StringUtils.isBlank(i.getPriceType())).map(i -> i.getPriceCls()).distinct().collect(Collectors.toList());
        Map<String, PriPricePriorityConfCache> priorityConfDOMap = null;
        if (!priceClsList.isEmpty()) {
            List<PriPricePriorityConfCache> priorityConfigList = PriPriceCacheClient.findPriorityConfCache();
            log.info("价格优先级取价查询结果；" + JSON.toJSONString(priorityConfigList));
            priorityConfDOMap = priorityConfigList.stream().collect(Collectors.toMap(i -> i.getPriceCls(), i -> i, (o, n) -> n));
        }

        List<PriPriceCache> allPrice = PriPriceCacheClient.findPrice(redisKeys);
        Map<String, PriPriceCache> priceCacheMap = allPrice.stream().collect(Collectors.toMap(i -> i.getCacheKey(tenantId, dateStr), i -> i, (o, n) -> n));

        Map<String, ItmPriPriceRpcDtoParam> paramMap = params.stream().collect(Collectors.toMap(i -> i.getUuid(), i -> i));
        for (String uuid : redisKeyMap.keySet()) {
            List<PriPriceCache> currentPrice = new ArrayList<>(); //PriPriceCacheClient.findPrice(redisKeyMap.get(uuid));
            for (String key : redisKeyMap.get(uuid)) {
                if (priceCacheMap.containsKey(key) && priceCacheMap.get(key) != null) {
                    currentPrice.add(priceCacheMap.get(key));
                }
            }
            PriPricePriorityConfCache priPricePriorityConfCache = null;
            if (priorityConfDOMap != null) {
                priPricePriorityConfCache = priorityConfDOMap.get(paramMap.get(uuid).getPriceCls());
            }
            result.addAll(buildResult(paramMap.get(uuid), currentPrice, priConfigMap, priPricePriorityConfCache, priceTypePathMap));
        }
        log.info("缓存获取价格结果：" + JSON.toJSONString(result) + "，耗时：" + (System.currentTimeMillis() - start));
        return ApiResult.ok(result);
    }

    public static List<PriPriceRpcDTO> buildResult(ItmPriPriceRpcDtoParam param, List<PriPriceCache> priceResult, Map<String, List<PriPriceConfCache>> priConfigMap,
                                             PriPricePriorityConfCache priPricePriorityConfCache, Map<String, String> priceTypePathMap) {
        List<PriPriceCache> caches = filterPriceRpcDtoByPriceCfg(priceResult, priceTypePathMap, priConfigMap);
        List<PriPriceRpcDTO> currentPriceResult = PriceClientConvert.INSTANCE.cacheToRpcDTO(caches);
        if (StringUtils.isNotBlank(param.getPriceType())) {
            for (PriPriceRpcDTO priPriceRpcDTO : currentPriceResult) {
                priPriceRpcDTO.setPriceCls(param.getPriceCls());
                priPriceRpcDTO.setUuid(param.getUuid());
            }
            return currentPriceResult;
        } else {
            String priceCls = param.getPriceCls();
            if (priPricePriorityConfCache == null) {
                log.warn("价格类别" + priceCls + "无取价配置，忽略请求：" + JSON.toJSONString(param));
            } else {
                PriPriceRpcDTO priceConfigResult = findPriceConfigResult(currentPriceResult, priPricePriorityConfCache);
                if (priceConfigResult != null) {
                    priceConfigResult.setUuid(param.getUuid());
                    return List.of(priceConfigResult);
                }
            }
        }
        return new ArrayList<>();
    }

    private static PriPriceRpcDTO findPriceConfigResult(List<PriPriceRpcDTO> currentPriceResult, PriPricePriorityConfCache confDO) {
        PriPriceRpcDTO firstPriPriceRpcDTO =  CollectionUtils.isEmpty(currentPriceResult) ? null : currentPriceResult.get(0);
        Map<String, PriPriceRpcDTO> priceTypeMap = CollectionUtils.isEmpty(currentPriceResult) ? new HashMap<>() :
                currentPriceResult.stream().collect(Collectors.toMap(PriPriceRpcDTO::getPriceType, t -> t, (t1, t2) -> t1));


        String[] split = confDO.getPricePriority().split("&&");
        // 按优先级
        PriPriceRpcDTO result = null;
        if (confDO.getConfigType().equals("10")) {
            for (String priceType : split) {
                //PriPriceRpcDTO priPriceRpcDTO = currentPriceResult.stream().filter(i -> i.getPriceType().equals(priceType)).findFirst().orElseGet(() -> null);
                PriPriceRpcDTO priPriceRpcDTO = priceTypeMap.get(priceType);
                if (priPriceRpcDTO != null) {
                    result = priPriceRpcDTO;
                    break;
                }
            }
        } else {
            BigDecimal totalPrice = BigDecimal.ZERO;
            BigDecimal totalNetPrice = BigDecimal.ZERO;
            boolean hasImportantPrice = true;
            for (int i = 0; i < split.length; i = i + 2) {
                int k = i;
                //PriPriceRpcDTO priPriceRpcDTO = currentPriceResult.stream().filter(j -> j.getPriceType().equals(split[k])).findFirst().orElseGet(() -> null);
                PriPriceRpcDTO priPriceRpcDTO = priceTypeMap.get(split[k]);
                if (priPriceRpcDTO == null) {
                    if (priPriceRpcDTO.getPriceType().equals(confDO.getImportantPriceType())) {
                        log.warn("无重点关注价：" + confDO.getImportantPriceType());
                        hasImportantPrice = false;
                        break;
                    } else {
                        continue;
                    }
                }
                if (i == 0) {
                    totalPrice = totalPrice.add(priPriceRpcDTO.getPrice());
                    totalNetPrice = totalNetPrice.add(priPriceRpcDTO.getNetPrice());
                } else {
                    // 加
                    if (Objects.equals(split[i - 1], "ADD")) {
                        totalPrice = totalPrice.add(priPriceRpcDTO.getPrice());
                        totalNetPrice = totalNetPrice.add(priPriceRpcDTO.getNetPrice());
                    } else {
                        totalPrice = totalPrice.subtract(priPriceRpcDTO.getPrice());
                        totalNetPrice = totalNetPrice.subtract(priPriceRpcDTO.getNetPrice());
                    }
                }
            }
            if (hasImportantPrice && Objects.nonNull(firstPriPriceRpcDTO)) {
                //result = PriPriceConvert.INSTANCE.rpcDtoToRpcPriceDto(currentPriceResult.get(0));
                result = PriceClientConvert.INSTANCE.rpcDtoToRpcPriceDto(firstPriPriceRpcDTO);
                result.setPrice(totalPrice);
                result.setNetPrice(totalNetPrice);
                result.setChildPrice(currentPriceResult);
                result.setPriceType(null);
                result.setPriceResultNote(confDO.getImportantPriceType());
            }
        }
        if (result == null && StringUtils.isNotBlank(confDO.getBasePriceType())) {
            //result = currentPriceResult.stream().filter(i -> i.getPriceType().equals(confDO.getBasePriceType())).findFirst().orElseGet(() -> null);
            result = priceTypeMap.get(confDO.getBasePriceType());
            if (result != null) {
                result.setPriceResultNote("取兜底价 " + confDO.getBasePriceType());
            } else {
                //result.setPriceResultNote("取兜底价 " + confDO.getBasePriceType() + " 未配置该价格");
                log.info("取兜底价 " + confDO.getBasePriceType() + " 未配置该价格");
            }
        }
        if (result != null) {
            result.setPriceCls(confDO.getPriceCls());
        }
        return result;
    }

    public static List<PriPriceCache> filterPriceRpcDtoByPriceCfg(List<PriPriceCache> priceRpcDTOList, Map<String, String> priceTypePathMap,
                                                                   Map<String, List<PriPriceConfCache>> priConfigMap) {
        List<PriPriceCache> result = new ArrayList<>();
        if (CollectionUtils.isEmpty(priceRpcDTOList)) {
            return result;
        }
        for (String priceType : priConfigMap.keySet()) {
            List<PriPriceConfCache> priPriceConfCaches = priConfigMap.get(priceType);
            priPriceConfCaches.sort(new Comparator<PriPriceConfCache>() {
                @Override
                public int compare(PriPriceConfCache o1, PriPriceConfCache o2) {
                    return o1.getPriority() - o2.getPriority();
                }
            });
            for (PriPriceConfCache priPriceConfCach : priPriceConfCaches) {
                String elementCfgPath = priceTypePathMap.get(priceType + priPriceConfCach.getPriority().intValue());
                if (StrUtil.isBlank(elementCfgPath)) {
                    continue;
                }

                PriPriceCache priPriceRpcDTO = priceRpcDTOList.stream()
                        .filter(item -> priceType.equals(item.getPriceType()) && elementCfgPath.equals(item.getElementPath()))
                        .findFirst().orElseGet(() -> null);
                if (priPriceRpcDTO != null) {
                    result.add(priPriceRpcDTO);
                    break;
                }
            }
        }
        return result;
    }

    private static boolean allBeIncluded(String priceElementCfg, List<String> elementPaths) {
        if (StringUtils.isBlank(priceElementCfg)) {
            return true;
        }
        String[] split = priceElementCfg.split("\\+");
        for (int i = 0; i < split.length; i++) {
            if (!elementPaths.contains(split[i])) {
                return false;
            }
        }
        return true;
    }

    public static void checkFindDtoByParamParameter(List<ItmPriPriceRpcDtoParam> param) {
        param.stream().forEach(item -> {
            if (StringUtils.isBlank(item.getPriceType())) {
                Assert.isTrue(StringUtils.isNotBlank(item.getPriceCls()), "价格类别不能为空");
            }
            Assert.isTrue(StringUtils.isNotBlank(item.getItemCode()) || item.getItemId() != null, "商品id和商品编码不能同时为空");
            Assert.isTrue(StringUtils.isNotBlank(item.getUom()), "单位不能为空");
            Assert.isTrue(StringUtils.isNotBlank(item.getCurrCode()), "货币编码不能为空");
            if (StrUtil.isBlank(item.getUuid())) {
                item.setUuid(UUID.randomUUID().toString());
            }
        });
        List<ItmPriPriceRpcDtoParam> collect = param.stream().filter(i -> StringUtils.isNotBlank(i.getUuid())).collect(Collectors.toList());
        Assert.isTrue(collect.size() == collect.stream().distinct().count(), "取价传参中id标识不允许重复");
    }

    public static void fillUpItem(List<ItmPriPriceRpcDtoParam> params, ItmItemRpcService rpcService) {
        List<Long> itemIds = params.stream().filter(item -> item.getItemId() != null && StringUtils.isBlank(item.getItemCode()))
                .map(item -> item.getItemId()).distinct().collect(Collectors.toList());
        if (!CollectionUtils.isEmpty(itemIds)) {
            if (itmItemRpcService == null) {
                if (rpcService != null) {
                    itmItemRpcService = rpcService;
                } else {
                    itmItemRpcService = SpringUtil.getBean(ItmItemRpcService.class);
                }
            }
            List<ItmItemIcDTO> icItem = itmItemRpcService.findIcItem(itemIds);
            List<Long> existIds = icItem.stream().map(item -> item.getId()).collect(Collectors.toList());
            List<Long> notExistsIds = itemIds.stream().filter(i -> !existIds.contains(i)).collect(Collectors.toList());
            if (!notExistsIds.isEmpty()) {
                throw new BusinessException("商品id" + JSON.toJSONString(notExistsIds) + "不存在，请检查");
            }
            Map<Long, String> icMap = icItem.stream().collect(Collectors.toMap(i -> i.getId(), i -> i.getItemCode()));
            for (ItmPriPriceRpcDtoParam param : params) {
                if (StrUtil.isNotBlank(param.getItemCode())) {
                    continue;
                }
                param.setItemCode(icMap.get(param.getItemId()));
            }
        }
    }

    public static Map<String, String> getFlowNoCache() {
        if (udcProvider == null) {
            udcProvider = SpringUtil.getBean(UdcProvider.class);
        }
        Map<String, String> map = udcProvider.getValueMapByUdcCode("yst-supp", "FIXED_PRICE_ELEMENT");
        return map;
    }
}
