package com.elitesland.zhiyuan.openapi.outer.nc.interceptor;

import com.dtflys.forest.exceptions.ForestRuntimeException;
import com.dtflys.forest.http.ForestRequest;
import com.dtflys.forest.http.ForestResponse;
import com.dtflys.forest.interceptor.Interceptor;
import com.elitesland.zhiyuan.openapi.outer.nc.config.NcProperties;
import com.elitesland.zhiyuan.openapi.outer.nc.service.NcService;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Component;

/**
 * NC Forest请求拦截器
 * 自动添加NC接口所需的URL参数
 *
 * @author duwh
 */
@Slf4j
@Component
@RequiredArgsConstructor
public class NcForestInterceptor implements Interceptor<Object> {

    private final NcProperties ncProperties;
    private final NcService ncService;

    @Override
    public boolean beforeExecute(ForestRequest request) {
        try {
            // 检查是否启用NC接口
            if (!ncProperties.getEnabled()) {
                log.warn("NC接口未启用，跳过请求");
                return false;
            }

            // 生成时间戳和token
            String timestamp = String.valueOf(System.currentTimeMillis());
            String token = ncService.getToken(ncProperties.getSecrect(), ncProperties.getUserCode(), timestamp);

            // NC接口通过URL参数传递认证信息，不是请求头
            // 添加URL参数
            request.addQuery("appid", ncProperties.getAppId());
            request.addQuery("from", ncProperties.getFrom());
            request.addQuery("usercode", ncProperties.getUserCode());
            request.addQuery("timestamp", timestamp);
            request.addQuery("token", token);

            // 注释掉请求头方式，实际NC接口通过URL参数传递认证信息，不使用请求头
            // request.addHeader("appid", ncProperties.getAppId());
            // request.addHeader("timestamp", timestamp);
            // request.addHeader("token", token);
            // request.addHeader("from", ncProperties.getFrom());
            // 实际请求格式: http://10.8.100.69:8090/service/muap/api?appid=SCMZT0001&from=SCMZT&usercode=zy_test01&timestamp=1758605757461&token=E163D7C38C0C56F2E1D4D931F9AE07756BD407AE
            
            // 只保留Content-Type请求头
            request.addHeader("Content-Type", "application/json");

            log.info("Forest请求NC接口: {}", request.getUrl());
            log.info("Forest URL参数: appid={}, from={}, usercode={}, timestamp={}, token={}", 
                    ncProperties.getAppId(), ncProperties.getFrom(), ncProperties.getUserCode(), timestamp, token);

            return true;
        } catch (Exception e) {
            log.error("Forest拦截器处理URL参数失败", e);
            throw new ForestRuntimeException("设置NC URL参数失败", e);
        }
    }

    @Override
    public void onSuccess(Object data, ForestRequest request, ForestResponse response) {
        log.info("Forest调用NC接口成功，响应状态: {}", response.getStatusCode());
        log.info("Forest响应内容: {}", data);
    }

    @Override
    public void onError(ForestRuntimeException ex, ForestRequest request, ForestResponse response) {
        log.error("Forest调用NC接口失败，状态码: {}, 错误: {}", 
                response != null ? response.getStatusCode() : "unknown", ex.getMessage());
        log.error("Forest请求详情: {}", request.toString());
    }
}