package com.elitescloud.cloudt.authorization.api.client.client.config;

import com.elitescloud.cloudt.authorization.api.client.client.OAuthUserProvider;
import com.elitescloud.cloudt.authorization.api.client.client.config.filter.OAuthClientTokenFilter;
import com.elitescloud.cloudt.authorization.api.client.client.service.AuthServerRequestHolder;
import com.elitescloud.cloudt.authorization.api.client.client.service.OAuthClientUserService;
import com.fasterxml.jackson.databind.ObjectMapper;
import org.springframework.beans.factory.ObjectProvider;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.boot.autoconfigure.condition.ConditionalOnClass;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.context.annotation.Bean;
import org.springframework.util.Assert;
import org.springframework.web.reactive.function.client.WebClient;
import reactor.netty.http.client.HttpClient;

/**
 * 客户端自动化配置.
 *
 * @author Kaiser（wang shao）
 * @date 2021/10/22
 */
@ConditionalOnProperty(prefix = OAuthClientProperties.CONFIG_PREFIX, name = "enabled", havingValue = "true")
@ConditionalOnClass({WebClient.class, HttpClient.class})
@EnableConfigurationProperties(OAuthClientProperties.class)
public class OAuthClientAutoConfiguration {

    @Value("${spring.security.oauth2.resourceserver.jwt.issuer-uri:#{null}}")
    private String serverUri;

    private final OAuthClientProperties clientProperties;

    public OAuthClientAutoConfiguration(OAuthClientProperties clientProperties) {
        this.clientProperties = clientProperties;
        Assert.hasText(serverUri, "issuerUri不能为空");
    }

    @Bean
    public OAuthClientUserService oauthClientUserService(ObjectProvider<OAuthUserProvider> userServiceObjectProvider,
                                                         ObjectMapper objectMapper) {
        var userProvider = userServiceObjectProvider.getIfUnique();
        Assert.notNull(userProvider, "请配置用户服务：OAuthUserProvider");
        var authServerRequestHolder = new AuthServerRequestHolder(clientProperties, serverUri, objectMapper);

        return new OAuthClientUserService(clientProperties, userProvider, objectMapper, authServerRequestHolder);
    }

    @Bean
    public OAuthClientTokenFilter oauthClientTokenFilter(OAuthClientUserService userService) {
        Assert.hasText(clientProperties.getPath(), "token路径不能为空");
        return new OAuthClientTokenFilter(clientProperties, userService);
    }
}
