package com.elitescloud.cloudt.authorization.api.client.config.security.configurer.filter;

import com.elitescloud.cloudt.authorization.api.client.common.SecurityConstants;
import com.elitescloud.cloudt.authorization.api.client.config.security.handler.DefaultAuthenticationEntryPointHandler;
import lombok.extern.log4j.Log4j2;
import org.springframework.lang.NonNull;
import org.springframework.security.authentication.AbstractAuthenticationToken;
import org.springframework.security.authentication.AuthenticationDetailsSource;
import org.springframework.security.authentication.AuthenticationManager;
import org.springframework.security.authentication.AuthenticationServiceException;
import org.springframework.security.core.Authentication;
import org.springframework.security.core.AuthenticationException;
import org.springframework.security.core.context.SecurityContext;
import org.springframework.security.core.context.SecurityContextHolder;
import org.springframework.security.web.AuthenticationEntryPoint;
import org.springframework.security.web.authentication.AuthenticationFailureHandler;
import org.springframework.security.web.authentication.WebAuthenticationDetailsSource;
import org.springframework.security.web.context.NullSecurityContextRepository;
import org.springframework.security.web.context.SecurityContextRepository;
import org.springframework.util.Assert;
import org.springframework.web.filter.OncePerRequestFilter;

import javax.servlet.FilterChain;
import javax.servlet.ServletException;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.io.IOException;

/**
 * 自定义身份识别过滤器父类.
 * <p>
 * 一般用于登录后从请求中解析当前用户身份
 *
 * @author Kaiser（wang shao）
 * @date 2022/7/2
 */
@Log4j2
public abstract class AbstractAuthorizationFilter<T extends AbstractAuthenticationToken> extends OncePerRequestFilter {

    private AuthenticationManager authenticationManager;
    private AuthenticationEntryPoint authenticationEntryPoint = new DefaultAuthenticationEntryPointHandler(null);
    private AuthenticationDetailsSource<HttpServletRequest, ?> authenticationDetailsSource = new WebAuthenticationDetailsSource();
    private SecurityContextRepository securityContextRepository = new NullSecurityContextRepository();
    private AuthenticationFailureHandler authenticationFailureHandler = (request, response, exception) -> {
        if (exception instanceof AuthenticationServiceException) {
            throw exception;
        }
        this.authenticationEntryPoint.commence(request, response, exception);
    };

    public AbstractAuthorizationFilter(AuthenticationManager authenticationManager) {
        this.authenticationManager = authenticationManager;
    }

    /**
     * 解析身份令牌
     * <p>
     * 为空时继续下一个filter
     *
     * @param request
     * @return
     * @throws AuthenticationException 认证异常
     */
    protected abstract T obtain(HttpServletRequest request) throws AuthenticationException;

    @Override
    protected void doFilterInternal(HttpServletRequest request, HttpServletResponse response, FilterChain filterChain) throws ServletException, IOException {
        // 解析AuthenticationToken
        T authenticationToken = null;
        try {
            authenticationToken = obtain(request);
        } catch (AuthenticationException e) {
            log.error("解析AuthenticationToken异常：", e);
            this.authenticationEntryPoint.commence(request, response, e);
            return;
        }

        if (authenticationToken == null) {
            log.trace("{}未解析出有效AuthenticationToken", request::getRequestURI);

            SecurityContext context = SecurityContextHolder.createEmptyContext();
            context.setAuthentication(SecurityConstants.AUTHENTICATION_ANONYMOUS);
            SecurityContextHolder.setContext(context);
            this.securityContextRepository.saveContext(context, request, response);
            filterChain.doFilter(request, response);
            return;
        }

        // 解析成功
        authenticationToken.setDetails(authenticationDetailsSource.buildDetails(request));

        // 开始认证
        try {
            Authentication authenticationResult = authenticationManager.authenticate(authenticationToken);
            SecurityContext context = SecurityContextHolder.createEmptyContext();
            context.setAuthentication(authenticationResult);
            SecurityContextHolder.setContext(context);
            this.securityContextRepository.saveContext(context, request, response);
            log.debug("设置当前认证用户：{}", authenticationResult::getName);
            filterChain.doFilter(request, response);
        } catch (AuthenticationException failed) {
            if (log.isDebugEnabled()) {
                log.debug("用户认证失败：{}", authenticationToken);
            }
            this.authenticationFailureHandler.onAuthenticationFailure(request, response, failed);
        } finally {
            SecurityContextHolder.clearContext();
        }
    }

    @Override
    public void afterPropertiesSet() throws ServletException {
        Assert.notNull(authenticationManager, "authenticationManager不能为空");
        super.afterPropertiesSet();
    }

    public void setAuthenticationManager(@NonNull AuthenticationManager authenticationManager) {
        this.authenticationManager = authenticationManager;
    }

    public void setAuthenticationEntryPoint(@NonNull AuthenticationEntryPoint authenticationEntryPoint) {
        this.authenticationEntryPoint = authenticationEntryPoint;
    }

    public void setAuthenticationDetailsSource(@NonNull AuthenticationDetailsSource<HttpServletRequest, ?> authenticationDetailsSource) {
        this.authenticationDetailsSource = authenticationDetailsSource;
    }

    public void setSecurityContextRepository(SecurityContextRepository securityContextRepository) {
        this.securityContextRepository = securityContextRepository;
    }

    public void setAuthenticationFailureHandler(AuthenticationFailureHandler authenticationFailureHandler) {
        this.authenticationFailureHandler = authenticationFailureHandler;
    }
}
