package com.elitescloud.cloudt.authorization.api.client.config.security.configurer.provider;

import cn.hutool.core.text.CharSequenceUtil;
import cn.hutool.core.util.ObjectUtil;
import com.elitescloud.cloudt.authorization.api.client.AuthenticationClaim;
import com.elitescloud.cloudt.authorization.api.client.common.AuthorizationException;
import com.elitescloud.cloudt.authorization.api.client.common.InterceptUri;
import com.elitescloud.cloudt.authorization.api.client.common.SecurityConstants;
import com.elitescloud.cloudt.authorization.api.client.config.AuthorizationProperties;
import com.elitescloud.cloudt.authorization.api.client.config.support.AuthenticationCache;
import com.elitescloud.cloudt.authorization.api.client.principal.AuthorizedClient;
import com.elitescloud.cloudt.authorization.api.client.token.BearerTokenAuthenticationToken;
import com.elitescloud.cloudt.context.util.HttpServletUtil;
import com.elitescloud.cloudt.security.entity.GeneralUserDetails;
import lombok.extern.log4j.Log4j2;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Lazy;
import org.springframework.security.authentication.AnonymousAuthenticationToken;
import org.springframework.security.authentication.AuthenticationProvider;
import org.springframework.security.core.Authentication;
import org.springframework.security.core.AuthenticationException;
import org.springframework.security.core.authority.AuthorityUtils;
import org.springframework.security.oauth2.jwt.BadJwtException;
import org.springframework.security.oauth2.jwt.Jwt;
import org.springframework.security.oauth2.jwt.JwtDecoder;
import org.springframework.security.oauth2.server.resource.InvalidBearerTokenException;
import org.springframework.security.web.servlet.util.matcher.MvcRequestMatcher;
import org.springframework.security.web.util.matcher.AntPathRequestMatcher;
import org.springframework.security.web.util.matcher.RequestMatcher;
import org.springframework.web.servlet.handler.HandlerMappingIntrospector;

import java.time.Clock;
import java.time.Duration;
import java.time.Instant;
import java.time.temporal.ChronoUnit;
import java.util.*;

/**
 * bearerToken身份认证.
 *
 * @author Kaiser（wang shao）
 * @date 2022/7/2
 */
@Log4j2
public class BearerTokenAuthenticationProvider implements AuthenticationProvider {
    private static final Duration DEFAULT_MAX_CLOCK_SKEW = Duration.of(60, ChronoUnit.SECONDS);
    private final Clock clock = Clock.systemUTC();
    private final List<RequestMatcher> allowRequestList = new ArrayList<>();

    private AuthenticationCache authenticationCache;
    private JwtDecoder jwtDecoder;
    private AuthorizationProperties authorizationProperties;
    private HandlerMappingIntrospector handlerMappingIntrospector;

    @Override
    public Authentication authenticate(Authentication authentication) throws AuthenticationException {
        BearerTokenAuthenticationToken authenticationToken = (BearerTokenAuthenticationToken) authentication;
        Jwt jwt = decodeToken(authenticationToken);
        String principalType = jwt.getClaimAsString(AuthenticationClaim.KEY_PRINCIPAL_TYPE);

        // 认证成功
        BearerTokenAuthenticationToken authorizationResult = new BearerTokenAuthenticationToken(authenticationToken.getToken(), Collections.emptyList());

        authorizationResult.setAuthorizedClient(AuthorizedClient.buildByJwt(jwt));

        if (CharSequenceUtil.equals(principalType, AuthenticationClaim.VALUE_PRINCIPAL_USER)) {
            // 如果是用户认证，则获取用户信息
            GeneralUserDetails user = authenticationCache.getUserDetail(authenticationToken.getToken());
            if (user == null) {
                if (allow()) {
                    // 允许访问
                    return SecurityConstants.AUTHENTICATION_ANONYMOUS;
                }
                log.warn("无效token：{}", authenticationToken.getToken());
                throw new AuthorizationException("当前用户还未认证或身份认证已过期");
            }

            authorizationResult.setPrincipal(user);
        } else {
            // 校验token是否有效
            validateExpires(jwt);
        }

        return authorizationResult;
    }

    @Override
    public boolean supports(Class<?> authentication) {
        return BearerTokenAuthenticationToken.class.isAssignableFrom(authentication);
    }

    private boolean allow() {
        if (authorizationProperties.getAnonymousEnabled()) {
            // 允许匿名访问
            return true;
        }

        // 是否在白名单内
        if (allowRequestList.isEmpty()) {
            // 初始化白名单
            initAllowRequestList();
        }
        if (allowRequestList.isEmpty()) {
            return false;
        }

        var request= HttpServletUtil.currentRequest();
        for (RequestMatcher requestMatcher : allowRequestList) {
            if (requestMatcher.matches(request)) {
                return true;
            }
        }

        return false;
    }

    private void initAllowRequestList() {
        Set<String> allowList = new HashSet<>();
        // 配置的白名单
        if (authorizationProperties.getAllowList() != null) {
            allowList.addAll(authorizationProperties.getAllowList());
        }
        // 内置白名单
        allowList.addAll(InterceptUri.getAllowUri());
        if (!allowList.isEmpty()) {
            HandlerMappingIntrospector introspector = ObjectUtil.defaultIfNull(handlerMappingIntrospector, new HandlerMappingIntrospector());
            for (String s : allowList) {
                if (adapterMvcRequestMatch(s)) {
                    allowRequestList.add(new MvcRequestMatcher(introspector, s));
                } else {
                    allowRequestList.add(new AntPathRequestMatcher(s));
                }
            }
        }
    }

    /**
     * 是否适配mvc匹配模式
     *
     * @param pattern
     * @return
     */
    private boolean adapterMvcRequestMatch(String pattern) {
        if (pattern == null) {
            return false;
        }
        // mvc匹配模式下，**只能位于结尾
        var indexDoubleWildcard = pattern.indexOf("**");
        return indexDoubleWildcard == pattern.length() - 2;
    }

    private Jwt decodeToken(BearerTokenAuthenticationToken bearer) {
        try {
            return jwtDecoder.decode(bearer.getToken());
        } catch (BadJwtException e) {
            log.error("解析token异常：", e);
            throw new InvalidBearerTokenException("不支持的token", e);
        }
    }

    private void validateExpires(Jwt jwt) {
        Instant expiry = jwt.getExpiresAt();
        if (expiry != null) {
            if (Instant.now(this.clock).minus(DEFAULT_MAX_CLOCK_SKEW).isAfter(expiry)) {
                throw new AuthorizationException("身份认证已过期");
            }
        }
        Instant notBefore = jwt.getNotBefore();
        if (notBefore != null) {
            if (Instant.now(this.clock).plus(DEFAULT_MAX_CLOCK_SKEW).isBefore(notBefore)) {
                throw new AuthorizationException("身份认证还未生效");
            }
        }
    }

    private boolean renewal() {
        return authorizationProperties.getTokenRenewal() != null && authorizationProperties.getTokenRenewal().getSeconds() > 0;
    }

    @Autowired
    public void setAuthenticationCache(AuthenticationCache authenticationCache) {
        this.authenticationCache = authenticationCache;
    }

    @Autowired
    public void setJwtDecoder(JwtDecoder jwtDecoder) {
        this.jwtDecoder = jwtDecoder;
    }

    @Autowired
    public void setAuthorizationProperties(AuthorizationProperties authorizationProperties) {
        this.authorizationProperties = authorizationProperties;
    }

    @Autowired(required = false)
    @Lazy
    public void setHandlerMappingIntrospector(HandlerMappingIntrospector handlerMappingIntrospector) {
        this.handlerMappingIntrospector = handlerMappingIntrospector;
    }
}
