package com.elitescloud.cloudt.authorization.api.client.util;

import com.elitescloud.cloudt.context.SpringContextHolder;
import com.fasterxml.jackson.databind.ObjectMapper;
import lombok.extern.log4j.Log4j2;
import org.springframework.http.MediaType;
import org.springframework.util.LinkedMultiValueMap;
import org.springframework.util.MultiValueMap;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.nio.charset.StandardCharsets;
import java.util.HashMap;
import java.util.Map;

/**
 * Servlet工具类.
 *
 * @author Kaiser（wang shao）
 * @date 2021/12/31
 */
@Log4j2
public class HttpServletUtil {

    private static final Object LOCK = new Object();
    private static final Map<Class<?>, Object> SHARED_OBJECTS = new HashMap<>();

    private HttpServletUtil() {
    }

    /**
     * response返回json格式内容
     *
     * @param response response
     * @param object   待返回的对象
     * @throws Exception 异常
     */
    public static void writeJson(HttpServletResponse response, Object object) throws Exception {
        response.setCharacterEncoding(StandardCharsets.UTF_8.name());
        response.setContentType(MediaType.APPLICATION_JSON_VALUE);

        var writer = response.getWriter();

        if (object instanceof String) {
            writer.write((String) object);
        } else {
            writer.write(getObjectMapper().writerWithDefaultPrettyPrinter().writeValueAsString(object));
        }

        writer.flush();
        writer.close();
    }

    /**
     * response返回json格式内容
     * <p>
     * 忽略异常
     *
     * @param response response
     * @param object   待返回的对象
     */
    public static void writeJsonIgnoreException(HttpServletResponse response, Object object) {
        try {
            writeJson(response, object);
        } catch (Exception e) {
            log.info("返回前端信息失败：", e);
        }
    }

    /**
     * 解析请求参数
     *
     * @param request
     * @return
     */
    public static MultiValueMap<String, String> getParameters(HttpServletRequest request) {
        Map<String, String[]> parameterMap = request.getParameterMap();
        MultiValueMap<String, String> parameters = new LinkedMultiValueMap<>(parameterMap.size());
        parameterMap.forEach((key, values) -> {
            if (values.length > 0) {
                for (String value : values) {
                    parameters.add(key, value);
                }
            }
        });
        return parameters;
    }

    private static ObjectMapper getObjectMapper() {
        var clazz = ObjectMapper.class;
        ObjectMapper objectMapper = getSharedObject(clazz);
        if (objectMapper != null) {
            return objectMapper;
        }
        synchronized (LOCK) {
            objectMapper = SpringContextHolder.getBean(clazz);
            setSharedObject(clazz, objectMapper);
        }
        return objectMapper;
    }

    private static <C> C getSharedObject(Class<C> clazz) {
        return (C) SHARED_OBJECTS.get(clazz);
    }

    private static <C> void setSharedObject(Class<C> clazz, C object) {
        SHARED_OBJECTS.put(clazz, object);
    }
}
