package com.elitescloud.cloudt.authorization.api.client.config;

import com.elitescloud.cloudt.authorization.api.client.config.security.AbstractServletSecurityConfig;
import com.elitescloud.cloudt.authorization.api.client.config.security.OAuth2ResourceServletSecurityConfig;
import com.elitescloud.cloudt.authorization.api.client.config.security.SingleClientServletSecurityConfig;
import com.elitescloud.cloudt.authorization.api.client.config.support.AuthenticationCache;
import com.elitescloud.cloudt.authorization.api.client.config.support.RedisAuthenticationCache;
import com.elitescloud.cloudt.authorization.api.client.config.support.dubbo.SecurityDubboConfig;
import com.elitescloud.cloudt.authorization.api.client.config.support.springcloud.SecuritySpringCloudConfig;
import com.elitescloud.cloudt.authorization.api.client.tool.RedisHelper;
import com.elitescloud.cloudt.authorization.api.client.util.JwtUtil;
import com.elitescloud.cloudt.common.config.cache.RedisCacheAutoConfiguration;
import com.elitescloud.cloudt.common.util.RedisUtils;
import com.elitescloud.cloudt.context.redis.RedisWrapper;
import com.elitescloud.cloudt.context.threadpool.support.ContextTransfer;
import com.nimbusds.jose.jwk.RSAKey;
import lombok.extern.log4j.Log4j2;
import org.springframework.beans.factory.ObjectProvider;
import org.springframework.boot.autoconfigure.AutoConfigureAfter;
import org.springframework.boot.autoconfigure.condition.ConditionalOnBean;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Import;
import org.springframework.core.Ordered;
import org.springframework.core.annotation.Order;
import org.springframework.security.config.annotation.web.builders.HttpSecurity;
import org.springframework.security.core.context.SecurityContext;
import org.springframework.security.core.context.SecurityContextHolder;
import org.springframework.security.crypto.bcrypt.BCryptPasswordEncoder;
import org.springframework.security.crypto.factory.PasswordEncoderFactories;
import org.springframework.security.crypto.password.DelegatingPasswordEncoder;
import org.springframework.security.crypto.password.PasswordEncoder;
import org.springframework.security.oauth2.core.DelegatingOAuth2TokenValidator;
import org.springframework.security.oauth2.jwt.JwtDecoder;
import org.springframework.security.oauth2.jwt.NimbusJwtDecoder;
import org.springframework.security.web.SecurityFilterChain;

import java.util.Collections;

/**
 * 认证中心客户端自动化配置.
 *
 * @author Kaiser（wang shao）
 * @date 2022/6/20
 */
@EnableConfigurationProperties(AuthorizationProperties.class)
@Import({AuthorizationClientAutoConfiguration.OnEnableSecurity.class, AuthorizationClientAutoConfiguration.OnDisableSecurity.class})
@AutoConfigureAfter({RedisCacheAutoConfiguration.class})
@Log4j2
public class AuthorizationClientAutoConfiguration {

    @Bean
    public PasswordEncoder passwordEncoder() {
        DelegatingPasswordEncoder encoder = (DelegatingPasswordEncoder) PasswordEncoderFactories.createDelegatingPasswordEncoder();
        encoder.setDefaultPasswordEncoderForMatches(new BCryptPasswordEncoder());
        return encoder;
    }

    /**
     * 用户认证信息上下文传递器
     *
     * @return 上下文传递器
     */
    @Bean
    public ContextTransfer<SecurityContext> contextTransferSecurityContext() {
        return new ContextTransfer<>() {
            @Override
            public SecurityContext getContext() {
                return SecurityContextHolder.getContext();
            }

            @Override
            public void setContext(SecurityContext context) {
                SecurityContextHolder.setContext(context);
            }

            @Override
            public void clearContext() {
                SecurityContextHolder.clearContext();
            }
        };
    }

    /**
     * 启用安全配置
     */
    @ConditionalOnProperty(prefix = AuthorizationProperties.CONFIG_PREFIX, name = "enabled", havingValue = "true", matchIfMissing = true)
    @Import({SingleClientServletSecurityConfig.class, OAuth2ResourceServletSecurityConfig.class,
            SecuritySpringCloudConfig.class, SecurityDubboConfig.class})
    static class OnEnableSecurity {
        private final AuthorizationProperties authorizationProperties;

        public OnEnableSecurity(AuthorizationProperties authorizationProperties) {
            this.authorizationProperties = authorizationProperties;
            log.info("启用安全配置");
        }

        /**
         * redis操作工具
         *
         * @return
         */
        @Bean
        @ConditionalOnBean({RedisUtils.class})
        @ConditionalOnMissingBean
        public RedisHelper redisHelper(RedisUtils redisUtils,
                                       ObjectProvider<RedisWrapper> wrapperObjectProvider) {
            return new RedisHelper(redisUtils, wrapperObjectProvider.getIfAvailable());
        }

        /**
         * 身份缓存服务
         *
         * @param redisHelper
         * @return
         */
        @Bean
        @ConditionalOnMissingBean
        @ConditionalOnBean(RedisHelper.class)
        public AuthenticationCache defaultAuthenticationCache(RedisHelper redisHelper) {
            return new RedisAuthenticationCache(redisHelper);
        }

        /**
         * jwt解密工具
         *
         * @param rsaKey
         * @return
         */
        @Bean
        @ConditionalOnMissingBean
        public JwtDecoder jwtDecoder(RSAKey rsaKey) {
            JwtDecoder jwtDecoder = JwtUtil.buildJwtDecoder(rsaKey);
            if (jwtDecoder instanceof NimbusJwtDecoder) {
                NimbusJwtDecoder nimbusJwtDecoder = (NimbusJwtDecoder) jwtDecoder;
                if (authorizationProperties.getTokenRenewal() != null && authorizationProperties.getTokenRenewal().toSeconds() > 0) {
                    // 自动刷新，则关闭对过期时间的校验
                    nimbusJwtDecoder.setJwtValidator(new DelegatingOAuth2TokenValidator<>(Collections.emptyList()));
                }
            }
            return jwtDecoder;
        }
    }


    /**
     * 禁用安全配置
     */
    @ConditionalOnProperty(prefix = AuthorizationProperties.CONFIG_PREFIX, name = "enabled", havingValue = "false")
    static class OnDisableSecurity {
        public OnDisableSecurity() {
            log.info("禁用安全配置");
        }

        @Bean
        @ConditionalOnMissingBean(name = AbstractServletSecurityConfig.SECURITY_CHAIN_DEFAULT)
        @Order(Ordered.HIGHEST_PRECEDENCE)
        public SecurityFilterChain defaultFilterChain(HttpSecurity http) throws Exception {
            http.authorizeRequests().anyRequest().permitAll();

            return http.build();
        }
    }
}
