package com.elitescloud.cloudt.authorization.api.client.config.security.configurer.provider;

import cn.hutool.core.text.CharSequenceUtil;
import com.elitescloud.cloudt.authorization.api.client.AuthenticationClaim;
import com.elitescloud.cloudt.authorization.api.client.common.AuthorizationException;
import com.elitescloud.cloudt.authorization.api.client.config.AuthorizationProperties;
import com.elitescloud.cloudt.authorization.api.client.config.support.AuthenticationCache;
import com.elitescloud.cloudt.authorization.api.client.principal.AuthorizedClient;
import com.elitescloud.cloudt.authorization.api.client.token.BearerTokenAuthenticationToken;
import com.elitescloud.cloudt.security.entity.GeneralUserDetails;
import lombok.extern.log4j.Log4j2;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.security.authentication.AuthenticationProvider;
import org.springframework.security.core.Authentication;
import org.springframework.security.core.AuthenticationException;
import org.springframework.security.oauth2.jwt.BadJwtException;
import org.springframework.security.oauth2.jwt.Jwt;
import org.springframework.security.oauth2.jwt.JwtDecoder;
import org.springframework.security.oauth2.server.resource.InvalidBearerTokenException;

import java.time.Clock;
import java.time.Duration;
import java.time.Instant;
import java.time.temporal.ChronoUnit;
import java.util.Collections;

/**
 * bearerToken身份认证.
 *
 * @author Kaiser（wang shao）
 * @date 2022/7/2
 */
@Log4j2
public class BearerTokenAuthenticationProvider implements AuthenticationProvider {
    private static final Duration DEFAULT_MAX_CLOCK_SKEW = Duration.of(60, ChronoUnit.SECONDS);
    private final Clock clock = Clock.systemUTC();

    private AuthenticationCache authenticationCache;
    private JwtDecoder jwtDecoder;
    private AuthorizationProperties authorizationProperties;

    @Override
    public Authentication authenticate(Authentication authentication) throws AuthenticationException {
        BearerTokenAuthenticationToken authenticationToken = (BearerTokenAuthenticationToken) authentication;
        Jwt jwt = decodeToken(authenticationToken);
        String principalType = jwt.getClaimAsString(AuthenticationClaim.KEY_PRINCIPAL_TYPE);

        // 认证成功
        BearerTokenAuthenticationToken authorizationResult = new BearerTokenAuthenticationToken(authenticationToken.getToken(), Collections.emptyList());

        authorizationResult.setAuthorizedClient(AuthorizedClient.buildByJwt(jwt));

        if (CharSequenceUtil.equals(principalType, AuthenticationClaim.VALUE_PRINCIPAL_USER)) {
            // 如果是用户认证，则获取用户信息
            GeneralUserDetails user = authenticationCache.getUserDetail(authenticationToken.getToken());
            if (user == null) {
                throw new AuthorizationException("当前用户还未认证或身份认证已过期");
            }

            authorizationResult.setPrincipal(user);
        } else {
            // 校验token是否有效
            validateExpires(jwt);
        }

        return authorizationResult;
    }

    @Override
    public boolean supports(Class<?> authentication) {
        return BearerTokenAuthenticationToken.class.isAssignableFrom(authentication);
    }

    private Jwt decodeToken(BearerTokenAuthenticationToken bearer) {
        try {
            return jwtDecoder.decode(bearer.getToken());
        } catch (BadJwtException e) {
            log.error("解析token异常：", e);
            throw new InvalidBearerTokenException("不支持的token", e);
        }
    }

    private void validateExpires(Jwt jwt) {
        Instant expiry = jwt.getExpiresAt();
        if (expiry != null) {
            if (Instant.now(this.clock).minus(DEFAULT_MAX_CLOCK_SKEW).isAfter(expiry)) {
                throw new AuthorizationException("身份认证已过期");
            }
        }
        Instant notBefore = jwt.getNotBefore();
        if (notBefore != null) {
            if (Instant.now(this.clock).plus(DEFAULT_MAX_CLOCK_SKEW).isBefore(notBefore)) {
                throw new AuthorizationException("身份认证还未生效");
            }
        }
    }

    private boolean renewal() {
        return authorizationProperties.getTokenRenewal() != null && authorizationProperties.getTokenRenewal().getSeconds() > 0;
    }

    @Autowired
    public void setAuthenticationCache(AuthenticationCache authenticationCache) {
        this.authenticationCache = authenticationCache;
    }

    @Autowired
    public void setJwtDecoder(JwtDecoder jwtDecoder) {
        this.jwtDecoder = jwtDecoder;
    }

    @Autowired
    public void setAuthorizationProperties(AuthorizationProperties authorizationProperties) {
        this.authorizationProperties = authorizationProperties;
    }
}
