package com.elitescloud.cloudt.authorization.api.client.config.support.springcloud;

import com.elitescloud.cloudt.authorization.api.client.util.JwtUtil;
import com.nimbusds.jose.jwk.RSAKey;
import org.springframework.boot.autoconfigure.condition.ConditionalOnClass;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingClass;
import org.springframework.boot.context.properties.ConfigurationProperties;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.cloud.bootstrap.BootstrapConfiguration;
import org.springframework.cloud.bootstrap.encrypt.KeyProperties;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Import;
import org.springframework.core.io.Resource;

import java.security.KeyStore;

/**
 * springCloud相关安全配置.
 *
 * @author Kaiser（wang shao）
 * @date 2022/7/3
 */
@Import({SecuritySpringCloudConfig.AuthorizationOnSpringCloudConfig.class, SecuritySpringCloudConfig.AuthorizationOnNoSpringCloudConfig.class})
public class SecuritySpringCloudConfig {

    /**
     * 如果是SpringCloud项目
     */
    @ConditionalOnClass(BootstrapConfiguration.class)
    static class AuthorizationOnSpringCloudConfig {
        /**
         * 构建RSAKey
         * <p>
         * 用于创建Jwt的加密和解密
         *
         * @param keyProperties keystore配置
         * @return rsaKey
         */
        @Bean
        @ConditionalOnMissingBean
        public RSAKey rsaKeySource(KeyProperties keyProperties) {
            var keyStoreSpring = keyProperties.getKeyStore();
            KeyStore keyStore = JwtUtil.loadKeystore(keyStoreSpring.getLocation(), keyStoreSpring.getType(),
                    keyStoreSpring.getPassword(), keyStoreSpring.getAlias(), keyStoreSpring.getSecret());

            return JwtUtil.loadRSAKey(keyStore, keyStoreSpring.getAlias(), keyStoreSpring.getSecret());
        }
    }

    /**
     * 如果不是SpringCloud项目
     */
    @ConditionalOnMissingClass("org.springframework.cloud.bootstrap.BootstrapConfiguration")
    @EnableConfigurationProperties(AuthorizationOnNoSpringCloudConfig.CustomKeyProperties.class)
    static class AuthorizationOnNoSpringCloudConfig {
        /**
         * 构建RSAKey
         * <p>
         * 用于创建Jwt的加密和解密
         *
         * @param keyProperties keystore配置
         * @return rsaKey
         */
        @Bean
        @ConditionalOnMissingBean
        public RSAKey rsaKeySource(CustomKeyProperties keyProperties) {
            var keyStoreSpring = keyProperties.getKeyStore();
            KeyStore keyStore = JwtUtil.loadKeystore(keyStoreSpring.getLocation(), keyStoreSpring.getType(),
                    keyStoreSpring.getPassword(), keyStoreSpring.getAlias(), keyStoreSpring.getSecret());

            return JwtUtil.loadRSAKey(keyStore, keyStoreSpring.getAlias(), keyStoreSpring.getSecret());
        }

        /**
         * springCloud中的KeyProperties
         * {@link org.springframework.cloud.bootstrap.encrypt.KeyProperties}
         */
        @ConfigurationProperties(CustomKeyProperties.PREFIX)
        static class CustomKeyProperties {
            private static final String PREFIX = "encrypt";

            /**
             * A symmetric key. As a stronger alternative, consider using a keystore.
             */
            private String key;

            /**
             * A salt for the symmetric key, in the form of a hex-encoded byte array. As a
             * stronger alternative, consider using a keystore.
             */
            private String salt = "deadbeef";

            /**
             * Flag to say that a process should fail if there is an encryption or decryption
             * error.
             */
            private boolean failOnError = true;

            /**
             * The key store properties for locating a key in a Java Key Store (a file in a format
             * defined and understood by the JVM).
             */
            private KeyStore keyStore = new KeyStore();

            public boolean isFailOnError() {
                return this.failOnError;
            }

            public void setFailOnError(boolean failOnError) {
                this.failOnError = failOnError;
            }

            public String getKey() {
                return this.key;
            }

            public void setKey(String key) {
                this.key = key;
            }

            public String getSalt() {
                return this.salt;
            }

            public void setSalt(String salt) {
                this.salt = salt;
            }

            public KeyStore getKeyStore() {
                return this.keyStore;
            }

            public void setKeyStore(KeyStore keyStore) {
                this.keyStore = keyStore;
            }

            /**
             * Key store properties.
             */
            public static class KeyStore {

                /**
                 * Location of the key store file, e.g. classpath:/keystore.jks.
                 */
                private Resource location;

                /**
                 * Password that locks the keystore.
                 */
                private String password;

                /**
                 * Alias for a key in the store.
                 */
                private String alias;

                /**
                 * Secret protecting the key (defaults to the same as the password).
                 */
                private String secret;

                /**
                 * The KeyStore type. Defaults to jks.
                 */
                private String type = "jks";

                public String getAlias() {
                    return this.alias;
                }

                public void setAlias(String alias) {
                    this.alias = alias;
                }

                public Resource getLocation() {
                    return this.location;
                }

                public void setLocation(Resource location) {
                    this.location = location;
                }

                public String getPassword() {
                    return this.password;
                }

                public String getType() {
                    return type;
                }

                public void setPassword(String password) {
                    this.password = password;
                }

                public String getSecret() {
                    return this.secret == null ? this.password : this.secret;
                }

                public void setSecret(String secret) {
                    this.secret = secret;
                }

                public void setType(String type) {
                    this.type = type;
                }

            }
        }
    }
}
