package com.elitescloud.cloudt.authorization.core;

import com.elitescloud.cloudt.authorization.api.client.common.CodeChallengeMethod;
import com.elitescloud.cloudt.authorization.api.client.util.JwtUtil;
import com.elitescloud.cloudt.authorization.api.client.util.ProofKeyUtil;
import com.nimbusds.jose.jwk.RSAKey;
import lombok.extern.log4j.Log4j2;
import org.springframework.lang.NonNull;
import org.springframework.security.oauth2.jwt.JwtDecoder;
import org.springframework.security.oauth2.jwt.JwtEncoder;

import java.io.IOException;
import java.net.URLEncoder;
import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;
import java.util.Base64;

/**
 * 认证工具类.
 *
 * @author Kaiser（wang shao）
 * @date 2021/12/31
 */
@Log4j2
public class AuthorizationUtil {

    private static final Charset CHARSET = StandardCharsets.UTF_8;

    private AuthorizationUtil() {
    }

    /**
     * 加密基本认证信息
     * <p>
     * 加密结果可直接放到http请求头{@link org.springframework.http.HttpHeaders.AUTHORIZATION}中
     *
     * @param clientId     clientID
     * @param clientSecret clientSecret
     * @return 加密字符串
     */
    public static String encodeBasicAuth(String clientId, String clientSecret) throws IOException {
        clientId = URLEncoder.encode(clientId, CHARSET);
        clientSecret = URLEncoder.encode(clientSecret, CHARSET);
        String credentialsString = clientId + ":" + clientSecret;
        byte[] encodedBytes = Base64.getEncoder().encode(credentialsString.getBytes(CHARSET));
        String baseAuth = new String(encodedBytes, CHARSET);
        return "Basic " + baseAuth;
    }

    /**
     * pkce算法
     *
     * @param codeVerifier
     * @return
     * @throws Exception
     */
    public static String codeChallenge(String codeVerifier) throws Exception {
        return ProofKeyUtil.codeChallenge(CodeChallengeMethod.S256, codeVerifier);
    }

    /**
     * 构建jwt解析器
     *
     * @param rsaKey rsaKey
     * @return decoder
     */
    public static JwtDecoder buildJwtDecoder(@NonNull RSAKey rsaKey) {
        return JwtUtil.buildJwtDecoder(rsaKey);
    }

    /**
     * 构建jwt解析器
     *
     * @param rsaKey rsaKey
     * @return decoder
     */
    public static JwtEncoder buildJwtEncoder(@NonNull RSAKey rsaKey) {
        return JwtUtil.buildJwtEncoder(rsaKey);
    }
}
