package com.elitescloud.cloudt.authorization.sdk.cas.provider;

import com.elitescloud.cloudt.authorization.sdk.cas.model.AuthUserDTO;
import com.elitescloud.cloudt.authorization.sdk.cas.task.ClientTokenHolder;
import com.elitescloud.cloudt.authorization.sdk.model.Result;
import com.elitescloud.cloudt.authorization.sdk.util.RestTemplateFactory;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.core.ParameterizedTypeReference;
import org.springframework.http.*;
import org.springframework.http.client.ClientHttpRequestExecution;
import org.springframework.http.client.ClientHttpRequestInterceptor;
import org.springframework.http.client.ClientHttpResponse;
import org.springframework.lang.NonNull;
import org.springframework.util.Assert;
import org.springframework.web.client.RestClientException;
import org.springframework.web.client.RestTemplate;

import javax.validation.constraints.NotBlank;
import javax.validation.constraints.NotEmpty;
import javax.validation.constraints.NotNull;
import java.io.IOException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;

/**
 * 用户传输服务.
 *
 * @author Kaiser（wang shao）
 * @date 2022/12/16
 */
public final class UserTransferHelper {
    private static final Logger LOG = LoggerFactory.getLogger(UserTransferHelper.class);

    private static UserTransferHelper transferHelper = null;
    private final RestTemplate restTemplate;

    private UserTransferHelper(String authServer) {
        this.restTemplate = RestTemplateFactory.instance(builder -> builder.rootUri(authServer)
                .additionalInterceptors(new AuthorizationInterceptor())
        );
    }

    /**
     * 创建工具实例
     *
     * @param authServer 认证服务器地址
     * @return
     */
    public static UserTransferHelper getInstance(@NotBlank String authServer) {
        if (transferHelper == null) {
            synchronized (UserTransferHelper.class) {
                if (transferHelper == null) {
                    transferHelper = new UserTransferHelper(authServer);
                }
            }
        }
        return transferHelper;
    }

    /**
     * 同步用户
     *
     * @param userInfo 用户信息
     * @return 用户ID
     */
    public Result<Long> upsertUser(@NotNull AuthUserDTO userInfo) {
        Assert.notNull(userInfo, "用户信息为空");

        return remoteExchange(CasUrlConstant.URI_USER_UPSERT, HttpMethod.POST, new HttpEntity<>(userInfo),
                new ParameterizedTypeReference<>() {
                });
    }

    /**
     * 批量同步用户
     *
     * @param rollBackAllOnException 出现异常，是否回滚所有
     * @param userInfoList           用户信息列表
     * @return username map userId
     */
    public Result<HashMap<String, Long>> upsertUserBatch(boolean rollBackAllOnException, @NotEmpty List<AuthUserDTO> userInfoList) {
        Assert.notEmpty(userInfoList, "用户信息为空");

        return remoteExchange(CasUrlConstant.URI_USER_UPSERT_BATCH, HttpMethod.POST, new HttpEntity<>(userInfoList),
                new ParameterizedTypeReference<>() {
                }, rollBackAllOnException);
    }

    /**
     * 更新用户状态
     *
     * @param userId  用户ID
     * @param enabled 启用状态
     * @return 用户ID
     */
    public Result<Long> updateEnabled(long userId, boolean enabled) {
        return remoteExchange(CasUrlConstant.URI_USER_UPDATE_ENABLED, HttpMethod.PATCH, null,
                new ParameterizedTypeReference<>() {
                }, userId, enabled);
    }

    /**
     * 修改用户的密码
     *
     * @param userId 用户账号ID
     * @param pd     密码，明文
     * @return 用户ID
     */
    public Result<Long> updatePassword(long userId, @NotBlank String pd) {
        Assert.hasText(pd, "密码为空");

        return remoteExchange(CasUrlConstant.URI_USER_UPDATE_PWD, HttpMethod.PATCH, null,
                new ParameterizedTypeReference<>() {
                }, userId, pd);
    }

    /**
     * 修改用户的账号到期时间
     *
     * @param userId      用户账号ID
     * @param expiredTime 账号过期时间，格式yyyy-MM-dd HH:mm:ss
     * @return 用户ID
     */
    public Result<Long> updateExpiredTime(long userId, @NotBlank String expiredTime) {
        Assert.hasText(expiredTime, "过期时间为空");

        return remoteExchange(CasUrlConstant.URI_USER_UPDATE_EXPIRED_TIME, HttpMethod.PATCH, null,
                new ParameterizedTypeReference<>() {
                }, userId, expiredTime);
    }


    /**
     * 修改用户的密码到期时间
     *
     * @param userId      用户账号ID
     * @param expiredTime 密码过期时间，格式yyyy-MM-dd HH:mm:ss
     * @return 用户ID
     */
    public Result<Long> updatePwdExpiredTime(long userId, String expiredTime) {
        Assert.hasText(expiredTime, "过期时间为空");
        return remoteExchange(CasUrlConstant.URI_USER_UPDATE_PWD_EXPIRED_TIME, HttpMethod.PATCH, null,
                new ParameterizedTypeReference<>() {
                }, userId, expiredTime);
    }

    /**
     * 删除用户账号
     *
     * @param userId 用户账号ID
     * @return 用户账号ID
     */
    public Result<Long> delete(long userId) {
        return remoteExchange(CasUrlConstant.URI_USER_DELETE, HttpMethod.DELETE, null,
                new ParameterizedTypeReference<>() {
                }, userId);
    }

    /**
     * 根据ID获取用户信息
     *
     * @param userId 用户账号ID
     * @return 用户信息
     */
    public Result<AuthUserDTO> getUser(long userId) {
        return remoteExchange(CasUrlConstant.URI_USER_GET, HttpMethod.GET, null,
                new ParameterizedTypeReference<>() {
                }, userId);
    }

    /**
     * 获取用户列表
     *
     * @param userIds 用户账号ID
     * @return 用户信息
     */
    public Result<ArrayList<AuthUserDTO>> getUserList(@NotEmpty List<Long> userIds) {
        Assert.notEmpty(userIds, "用户信息为空");

        return remoteExchange(CasUrlConstant.URI_USER_LIST, HttpMethod.POST, new HttpEntity<>(userIds),
                new ParameterizedTypeReference<>() {
                });
    }

    private <T> T nullToDefault(T obj, T defaultObj) {
        return obj == null ? defaultObj : obj;
    }

    private <T> T remoteExchange(String url, HttpMethod httpMethod, HttpEntity<?> httpEntity, ParameterizedTypeReference<T> responseType,
                                 Object... param) {
        ResponseEntity<T> response = null;
        try {
            response = restTemplate.exchange(url, httpMethod, httpEntity, responseType, param);
        } catch (RestClientException e) {
            LOG.error("认证授权服务器调用失败：", e);
            throw new RuntimeException("认证授权服务器异常", e);
        }

        if (response.getStatusCode() != HttpStatus.OK) {
            LOG.error("调用文件服务器接口失败：{}, {}", url, response);
            throw new RuntimeException("调用文件服务器接口失败");
        }

        return response.getBody();
    }

    static class AuthorizationInterceptor implements ClientHttpRequestInterceptor {

        @NonNull
        @Override
        public ClientHttpResponse intercept(HttpRequest request, @NonNull byte[] body, @NonNull ClientHttpRequestExecution execution) throws IOException {
            HttpHeaders headers = request.getHeaders();
            var token = ClientTokenHolder.getToken();
            if (token == null) {
                throw new IllegalStateException("Token获取失败");
            }

            // 设置token
            headers.add(HttpHeaders.AUTHORIZATION, token.getTokenType() + " " + token.getAccessToken());
            return execution.execute(request, body);
        }
    }
}
