package com.elitescloud.boot.util;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.core.ParameterizedTypeReference;
import org.springframework.http.HttpEntity;
import org.springframework.http.HttpMethod;
import org.springframework.http.HttpStatus;
import org.springframework.http.ResponseEntity;
import org.springframework.util.Assert;
import org.springframework.web.client.RestClientException;
import org.springframework.web.client.RestTemplate;

import javax.validation.constraints.NotBlank;
import javax.validation.constraints.NotNull;

/**
 * RestTemplate工具类.
 *
 * @author Kaiser（wang shao）
 * @date 2023/4/29
 */
public class RestTemplateHelper {
    private static final Logger LOG = LoggerFactory.getLogger(RestTemplateHelper.class);

    private final RestTemplate restTemplate;

    private RestTemplateHelper(RestTemplate restTemplate) {
        Assert.notNull(restTemplate, "restTemplate为空");
        this.restTemplate = restTemplate;
    }

    /**
     * 构建实例
     *
     * @param restTemplate restTemplate实例
     * @return restTemplate
     */
    public static RestTemplateHelper instance(@NotNull RestTemplate restTemplate) {
        return new RestTemplateHelper(restTemplate);
    }

    /**
     * exchange
     *
     * @param url           url
     * @param httpMethod    请求方法
     * @param httpEntity    请求体
     * @param responseType  响应类型
     * @param uriParameters 路径参数
     * @param <T>           内容类型
     * @return 返回结果
     */
    public <T> T exchange(@NotBlank String url, @NotNull HttpMethod httpMethod, HttpEntity<?> httpEntity, ParameterizedTypeReference<T> responseType,
                          Object... uriParameters) {
        ResponseEntity<T> response = null;
        try {
            response = restTemplate.exchange(url, httpMethod, httpEntity, responseType, uriParameters);
        } catch (RestClientException e) {
            LOG.error(url + "调用失败：", e);
            throw new RuntimeException("远程服务器异常", e);
        }

        if (response.getStatusCode() != HttpStatus.OK) {
            LOG.error("调用接口失败：{}, {}", url, response);
            throw new RuntimeException("调用远程接口失败");
        }
        LOG.info("接口{}调用成功", url);

        return response.getBody();
    }
}
