package com.elitescloud.boot.util;

import org.springframework.util.StringUtils;

import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.time.*;
import java.time.format.DateTimeFormatter;
import java.time.temporal.WeekFields;
import java.util.Date;
import java.util.TimeZone;

/**
 * 时间日期工具.
 *
 * @author Kaiser（wang shao）
 * @date 2021/12/24
 */
public class DatetimeUtil {

    public static final String FORMATTER_DATETIME_STR = "yyyy-MM-dd HH:mm:ss";
    /**
     * 常用日期时间格式
     */
    public static final DateTimeFormatter FORMATTER_DATETIME = DateTimeFormatter.ofPattern(FORMATTER_DATETIME_STR);
    public static final String FORMATTER_DATE_STR = "yyyy-MM-dd";
    public static final DateTimeFormatter FORMATTER_DATE = DateTimeFormatter.ofPattern(FORMATTER_DATE_STR);
    public static final String FORMATTER_TIME_STR = "HH:mm:ss";
    public static final DateTimeFormatter FORMATTER_TIME = DateTimeFormatter.ofPattern(FORMATTER_TIME_STR);
    public static final String FORMATTER_DATETIME_LONG_STR = "yyyyMMddHHmmssSSS";
    public static final DateTimeFormatter FORMATTER_DATETIME_LONG = DateTimeFormatter.ofPattern(FORMATTER_DATETIME_LONG_STR);
    public static final SimpleDateFormat FORMAT_DATE = new SimpleDateFormat(FORMATTER_DATE_STR);
    public static final SimpleDateFormat FORMAT_DATETIME = new SimpleDateFormat(FORMATTER_DATETIME_STR);

    /**
     * 服务器时区
     */
    public static final ZoneOffset SYS_ZONE_OFFSET = ZoneOffset.of("+8");
    public static final TimeZone SYS_TIME_ZONE = TimeZone.getTimeZone(SYS_ZONE_OFFSET);

    protected DatetimeUtil() {
    }

    /**
     * 当前时间字符串
     *
     * @return 当前时间字符串
     */
    public static String currentTimeStr() {
        return FORMATTER_DATETIME.format(LocalDateTime.now());
    }

    /**
     * 当前时间字符串
     *
     * @return 当前时间字符串
     */
    public static String currentTimeLong() {
        return FORMATTER_DATETIME_LONG.format(LocalDateTime.now());
    }

    /**
     * 转换字符串
     *
     * @param dateTime 时间
     * @return 字符串
     */
    public static String toStr(LocalDateTime dateTime) {
        if (dateTime == null) {
            return null;
        }
        return FORMATTER_DATETIME.format(dateTime);
    }

    /**
     * 转换字符串
     *
     * @param date 时间
     * @return 字符串
     */
    public static String toStr(LocalDate date) {
        if (date == null) {
            return null;
        }
        return FORMATTER_DATE.format(date);
    }

    /**
     * 转换字符串
     *
     * @param time 时间
     * @return 字符串
     */
    public static String toStr(LocalTime time) {
        if (time == null) {
            return null;
        }
        return FORMATTER_TIME.format(time);
    }

    /**
     * 转换字符串
     *
     * @param time 时间
     * @return 字符串
     */
    public static String toStr(Date time) {
        if (time == null) {
            return null;
        }
        return FORMAT_DATETIME.format(time);
    }

    /**
     * date转LocalDateTime
     *
     * @param date date
     * @return LocalDateTime
     */
    public static LocalDateTime date2LocalDateTime(Date date) {
        if (date == null) {
            return null;
        }
        return date.toInstant().atOffset(SYS_ZONE_OFFSET).toLocalDateTime();
    }

    /**
     * localDateTime转date
     *
     * @param dateTime localDateTime
     * @return date
     */
    public static Date localDateTime2Date(LocalDateTime dateTime) {
        if (dateTime == null) {
            return null;
        }
        return Date.from(dateTime.atZone(SYS_ZONE_OFFSET).toInstant());
    }

    /**
     * instant转LocalDateTime
     *
     * @param instant instant
     * @return LocalDateTime
     */
    public static LocalDateTime toLocalDateTime(Instant instant) {
        if (instant == null) {
            return null;
        }
        return instant.atZone(SYS_ZONE_OFFSET).toLocalDateTime();
    }

    /**
     * LocalDateTime转Instant
     *
     * @param localDateTime LocalDateTime
     * @return instant
     */
    public static Instant toInstant(LocalDateTime localDateTime) {
        if (localDateTime == null) {
            return null;
        }
        return localDateTime.toInstant(SYS_ZONE_OFFSET);
    }

    /**
     * 日期字符串转日期
     *
     * @param date 日期字符串
     * @return 日期
     */
    public static LocalDate parseLocalDate(String date) {
        if (isBlank(date)) {
            return null;
        }

        return LocalDate.parse(date, FORMATTER_DATE);
    }

    /**
     * 日期时间字符串转日期
     *
     * @param datetime 日期时间字符串
     * @return 日期时间
     */
    public static LocalDateTime parseLocalDateTime(String datetime) {
        if (isBlank(datetime)) {
            return null;
        }

        return LocalDateTime.parse(datetime, FORMATTER_DATETIME);
    }

    /**
     * 时间字符串转日期
     *
     * @param localTime 时间字符串
     * @return 时间
     */
    public static LocalTime parseLocalTime(String localTime) {
        if (isBlank(localTime)) {
            return null;
        }

        return LocalTime.parse(localTime, FORMATTER_TIME);
    }

    /**
     * 日期字符串转date
     *
     * @param date 日期格式
     * @return date
     */
    public static Date parseForDate(String date) {
        try {
            return FORMAT_DATE.parse(date);
        } catch (ParseException e) {
            throw new IllegalArgumentException("日期格式不正确");
        }
    }

    /**
     * 日期字符串转date
     *
     * @param date 日期字符串
     * @return date
     */
    public static Date parseForDatetime(String date) {
        if (isBlank(date)) {
            return null;
        }
        try {
            return FORMAT_DATETIME.parse(date);
        } catch (ParseException e) {
            throw new IllegalArgumentException("日期格式不正确");
        }
    }

    /**
     * 获取指定日期是一年中的第几周
     * <p>
     * 除了weekOfWeekBasedYear，还有一个weekOfYear，它们两个的具体区别为:
     * <p>
     * 如果一周涉及到了跨年，如2021.12.31是周五，2022.01.01是周六，这涉及到了年份的变化。
     * weekOfWeekBasedYear会把2021.12.31、2022.01.01（新年第一天）看做是2021年的最后一周，即看做同一周；
     * <p>
     * weekOfYear则会把2021.12.31看做是2021年的最后一周，把2022.01.01看最是2022年的第一周，即当做两周。
     *
     * @param date 指定日期
     * @return 第几周
     */
    public static int getWeekOfWeekBasedYear(LocalDate date) {
        return date.get(WeekFields.ISO.weekOfWeekBasedYear());
    }

    /**
     * 获取最小的时间
     *
     * @param time1 时间1
     * @param time2 时间2
     * @return 最小的时间
     */
    public static LocalDateTime min(LocalDateTime time1, LocalDateTime time2) {
        if (time1 == null) {
            return time2;
        }
        if (time2 == null) {
            return time1;
        }

        return time1.isBefore(time2) ? time1 : time2;
    }

    /**
     * 获取最小的时间
     *
     * @param time1 时间1
     * @param time2 时间2
     * @return 最小的时间
     */
    public static LocalDate min(LocalDate time1, LocalDate time2) {
        if (time1 == null) {
            return time2;
        }
        if (time2 == null) {
            return time1;
        }

        return time1.isBefore(time2) ? time1 : time2;
    }

    private static boolean isBlank(String str) {
        return !StringUtils.hasText(str);
    }
}
