package com.elitescloud.boot.util;

import com.elitescloud.boot.support.DiscoveryClientHttpRequestInterceptor;
import org.apache.http.conn.ssl.NoopHostnameVerifier;
import org.apache.http.conn.ssl.SSLConnectionSocketFactory;
import org.apache.http.conn.ssl.TrustSelfSignedStrategy;
import org.apache.http.impl.client.HttpClientBuilder;
import org.apache.http.impl.client.HttpClients;
import org.apache.http.ssl.SSLContextBuilder;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.boot.web.client.RestTemplateBuilder;
import org.springframework.boot.web.client.RestTemplateCustomizer;
import org.springframework.http.HttpMethod;
import org.springframework.http.client.ClientHttpRequestFactory;
import org.springframework.http.client.ClientHttpResponse;
import org.springframework.http.client.HttpComponentsClientHttpRequestFactory;
import org.springframework.http.converter.json.MappingJackson2HttpMessageConverter;
import org.springframework.http.converter.xml.MappingJackson2XmlHttpMessageConverter;
import org.springframework.lang.NonNull;
import org.springframework.util.ClassUtils;
import org.springframework.web.client.DefaultResponseErrorHandler;
import org.springframework.web.client.RestTemplate;
import org.springframework.web.util.DefaultUriBuilderFactory;
import org.springframework.web.util.UriTemplateHandler;

import javax.validation.constraints.NotBlank;
import java.io.IOException;
import java.net.URI;
import java.time.Duration;
import java.util.Arrays;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * RestTemplate.
 *
 * @author Kaiser（wang shao）
 * @date 2022/7/14
 */
public class RestTemplateFactory {
    private static final Logger LOG = LoggerFactory.getLogger(RestTemplateFactory.class);
    private static final boolean EXISTS_SPRING_CLOUD = isPresentClass("org.springframework.cloud.client.discovery.DiscoveryClient", null);


    private RestTemplateFactory() {
    }

    /**
     * 创建RestTemplate实例
     *
     * @return restTemplate
     */
    public static RestTemplate instance() {
        return instance(null);
    }

    /**
     * 创建RestTemplate实例
     *
     * @return restTemplate
     */
    public static RestTemplate instance(Function<RestTemplateBuilder, RestTemplateBuilder> restTemplateBuilderFunction) {
        var builder = initializeBuilder();
        if (restTemplateBuilderFunction != null) {
            builder = restTemplateBuilderFunction.apply(builder);
        }
        return builder.build();
    }

    /**
     * 动态获取服务端地址的RestTemplate实例
     *
     * @param restTemplateBuilderFunction 自定义的RestTemplateBuilder
     * @param serviceNames                服务名称
     * @return RestTemplate
     */
    public static RestTemplate dynamicInstance(Function<RestTemplateBuilder, RestTemplateBuilder> restTemplateBuilderFunction, String... serviceNames) {
        var builder = initializeBuilder();
        if (restTemplateBuilderFunction != null) {
            builder = restTemplateBuilderFunction.apply(builder);
        }
        if (EXISTS_SPRING_CLOUD) {
            builder = builder.additionalInterceptors(new DiscoveryClientHttpRequestInterceptor(Arrays.stream(serviceNames).collect(Collectors.toSet())));
        } else {
            LOG.warn("未启用SpringCloud，无法从服务注册中心获取服务");
        }
        return builder.build();
    }

    private static RestTemplateBuilder initializeBuilder() {
        return new RestTemplateBuilder()
                .requestFactory(RestTemplateFactory::getClientHttpRequestFactory)
                .uriTemplateHandler(uriTemplateHandler())
                .errorHandler(new DefaultResponseErrorHandler() {
                    @Override
                    public void handleError(@NonNull URI url, @NonNull HttpMethod method, @NonNull ClientHttpResponse response) throws IOException {
                        LOG.error("RestTemplate请求异常：{}，{}，{}", method, url, response.getStatusCode());
                        super.handleError(url, method, response);
                    }
                })
                .customizers(restTemplateCustomizer())
                ;
    }

    private static UriTemplateHandler uriTemplateHandler() {
        DefaultUriBuilderFactory handler = new DefaultUriBuilderFactory();
        handler.setEncodingMode(DefaultUriBuilderFactory.EncodingMode.TEMPLATE_AND_VALUES);
        return handler;
    }

    private static ClientHttpRequestFactory getClientHttpRequestFactory() {
        SSLConnectionSocketFactory sslConnectionSocketFactory = null;

        try {
            SSLContextBuilder sslContextBuilder = new SSLContextBuilder();
            sslContextBuilder.loadTrustMaterial(null, new TrustSelfSignedStrategy());
            sslConnectionSocketFactory = new SSLConnectionSocketFactory(sslContextBuilder.build(), NoopHostnameVerifier.INSTANCE);
        } catch (Exception e) {
            e.printStackTrace();
        }

        HttpClientBuilder httpClientBuilder = HttpClients.custom();
        httpClientBuilder.setSSLSocketFactory(sslConnectionSocketFactory);
        httpClientBuilder.disableAutomaticRetries();

        var requestFactory = new HttpComponentsClientHttpRequestFactory();
        requestFactory.setHttpClient(httpClientBuilder.build());
        requestFactory.setConnectTimeout((int) Duration.ofSeconds(10).toMillis());
        requestFactory.setReadTimeout((int) Duration.ofSeconds(60).toMillis());
        return requestFactory;
    }

    private static RestTemplateCustomizer restTemplateCustomizer() {
        var objectMapper = ObjectMapperFactory.instance();

        return template -> {
            MappingJackson2HttpMessageConverter jsonConvert = null;

            var xmlConvertIndex = -1;
            int index = 0;
            for (var convert : template.getMessageConverters()) {
                if (convert instanceof MappingJackson2XmlHttpMessageConverter) {
                    xmlConvertIndex = index;
                }
                if (convert instanceof MappingJackson2HttpMessageConverter) {
                    jsonConvert = (MappingJackson2HttpMessageConverter) convert;
                }
                index++;
            }
            if (jsonConvert == null) {
                jsonConvert = new MappingJackson2HttpMessageConverter(objectMapper);
                template.getMessageConverters().add(jsonConvert);
            } else {
                jsonConvert.setObjectMapper(objectMapper);
            }

            if (xmlConvertIndex >= 0) {
                template.getMessageConverters().remove(jsonConvert);
                template.getMessageConverters().add(xmlConvertIndex - 1, jsonConvert);
            }
        };
    }

    /**
     * 是否存在指定的类
     *
     * @param className   类的全限定名
     * @param classLoader classLoader
     * @return 是否存在
     */
    private static boolean isPresentClass(@NotBlank String className, ClassLoader classLoader) {
        if (classLoader == null) {
            classLoader = ClassUtils.getDefaultClassLoader();
        }
        try {
            Class.forName(className, false, classLoader);
            return true;
        } catch (Throwable ex) {
            return false;
        }
    }
}
