package com.elitescloud.boot.util;

import com.elitescloud.boot.common.OpenApiException;
import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.ObjectMapper;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.core.ParameterizedTypeReference;
import org.springframework.http.HttpEntity;
import org.springframework.http.HttpMethod;
import org.springframework.http.HttpStatus;
import org.springframework.http.ResponseEntity;
import org.springframework.util.Assert;
import org.springframework.util.ObjectUtils;
import org.springframework.util.StringUtils;
import org.springframework.web.client.RestClientException;
import org.springframework.web.client.RestTemplate;

import javax.validation.constraints.NotBlank;
import javax.validation.constraints.NotNull;

/**
 * RestTemplate工具类.
 *
 * @author Kaiser（wang shao）
 * @date 2023/4/29
 */
public class RestTemplateHelper {
    private static final Logger LOG = LoggerFactory.getLogger(RestTemplateHelper.class);

    private final RestTemplate restTemplate;
    private final ObjectMapper objectMapper;

    private RestTemplateHelper(RestTemplate restTemplate, ObjectMapper objectMapper) {
        Assert.notNull(restTemplate, "restTemplate为空");
        this.restTemplate = restTemplate;
        this.objectMapper = objectMapper == null ? ObjectMapperFactory.instance() : objectMapper;
    }

    /**
     * 构建实例
     *
     * @param restTemplate restTemplate实例
     * @return restTemplate
     */
    public static RestTemplateHelper instance(@NotNull RestTemplate restTemplate) {
        return new RestTemplateHelper(restTemplate, null);
    }

    /**
     * 构建实例
     *
     * @param restTemplate restTemplate实例
     * @return restTemplate
     */
    public static RestTemplateHelper instance(@NotNull RestTemplate restTemplate, ObjectMapper objectMapper) {
        return new RestTemplateHelper(restTemplate, objectMapper);
    }

    /**
     * exchange
     *
     * @param url           url
     * @param httpMethod    请求方法
     * @param httpEntity    请求体
     * @param responseType  响应类型
     * @param uriParameters 路径参数
     * @param <T>           内容类型
     * @return 返回结果
     */
    public <T> T exchange(@NotBlank String url, @NotNull HttpMethod httpMethod, HttpEntity<?> httpEntity, ParameterizedTypeReference<T> responseType,
                          Object... uriParameters) {
        ResponseEntity<T> response = null;
        try {
            response = restTemplate.exchange(url, httpMethod, httpEntity, responseType, this.normalizeUriParameters(uriParameters));
        } catch (RestClientException e) {
            LOG.error(url + "调用失败：{}", e.getMessage(), e);
            throw new RuntimeException("远程服务器异常", e);
        }

        if (response.getStatusCode() != HttpStatus.OK) {
            LOG.error("调用接口失败：{}, {}", url, response);
            throw new OpenApiException("调用远程接口失败", response.getStatusCodeValue());
        }
        LOG.info("接口{}调用成功", url);

        return response.getBody();
    }

    /**
     * exchange
     *
     * @param url           url
     * @param httpMethod    请求方法
     * @param httpEntity    请求体
     * @param responseType  响应类型
     * @param uriParameters 路径参数
     * @param <T>           内容类型
     * @return 返回结果
     */
    public <T> T exchangeSafely(@NotBlank String url, @NotNull HttpMethod httpMethod, HttpEntity<?> httpEntity, TypeReference<T> responseType,
                                Object... uriParameters) {
        ResponseEntity<String> response = null;
        try {
            response = restTemplate.exchange(url, httpMethod, httpEntity, String.class, this.normalizeUriParameters(uriParameters));
        } catch (RestClientException e) {
            LOG.error(url + "调用失败：{}", e.getMessage(), e);
            throw new RuntimeException("远程服务器异常", e);
        }

        if (response.getStatusCode() != HttpStatus.OK) {
            LOG.error("调用接口失败：{}, {}", url, response);
            throw new OpenApiException("调用远程接口失败：" + response.getBody(), response.getStatusCodeValue());
        }

        LOG.info("接口{}调用成功", url);

        if (!StringUtils.hasText(response.getBody())) {
            return null;
        }
        try {
            return objectMapper.readValue(response.getBody(), responseType);
        } catch (JsonProcessingException e) {
            LOG.error("响应结果转换异常：{}", response.getBody(), e);
            throw new IllegalArgumentException("响应结果转换异常", e);
        }
    }

    private Object[] normalizeUriParameters(Object[] parameters) {
        if (parameters == null) {
            return new Object[0];
        }
        if (parameters.length == 0) {
            return parameters;
        }

        Object[] result = new Object[parameters.length];
        Object param = null;
        for (int i = 0, len = parameters.length; i < len; i++) {
            param = parameters[i];
            result[i] = param == null ? "" : param;
        }
        return result;
    }
}
