package com.elitescloud.cloudt.common.base;

import io.swagger.annotations.ApiModelProperty;

import javax.validation.constraints.NotNull;
import java.io.Serializable;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;

/**
 * 分页信息.
 *
 * @author Kaiser（wang shao）
 * @date 2022/8/10
 */
public class PagingVO<T extends Serializable> implements Serializable {
    private static final long serialVersionUID = -4792752946712260031L;

    /**
     * 总记录数
     */
    @ApiModelProperty(value = "总记录数")
    private long total;
    /**
     * 汇总数据 用于分页查询 并且需要全数据量汇总数据存储
     * 汇总数据需要单独查询单纯存储，如果有数据权限需要考虑和分页查询数据权限一致
     * key 前后端约定的汇总取值名称如：价格，金额，数量。 采用英文，建议使用汇总的英文字段为规范这种
     * value 汇总值
     **/
    @ApiModelProperty(value = "汇总数据")
    private Map<String, Object> aggregatedData;
    /**
     * 数据列表
     */
    @ApiModelProperty(value = "数据列表")
    private List<T> records;


    public PagingVO() {
        this(0L, null);
    }

    public PagingVO(long total, List<T> records) {
        this.total = total;
        this.records = records == null ? Collections.emptyList() : records;
    }

    public PagingVO(long total, List<T> records, Map<String, Object> aggregatedData) {
        this.total = total;
        this.records = records == null ? Collections.emptyList() : records;
        this.aggregatedData = records == null ? Collections.emptyMap() : aggregatedData;
    }

    /**
     * 构建器
     *
     * @param <T> 数据类型
     * @return 结果
     */
    public static <T extends Serializable> PagingVO<T> builder() {
        return new PagingVO<>();
    }

    public Map<String, Object> getAggregatedData() {
        return this.aggregatedData;
    }

    /**
     * 构建
     *
     * @return 结果
     */
    public PagingVO<T> build() {
        return this;
    }

    public long getTotal() {
        return total;
    }

    public PagingVO<T> setTotal(long total) {
        this.total = total;
        return this;
    }

    public PagingVO<T> total(long total) {
        return setTotal(total);
    }

    public List<T> getRecords() {
        return records;
    }

    public PagingVO<T> setAggregatedData(Map<String, Object> aggregatedData) {
        this.aggregatedData = aggregatedData;
        return this;
    }

    public PagingVO<T> setRecords(List<T> records) {
        this.records = records;
        return this;
    }


    public PagingVO<T> records(List<T> records) {
        return setRecords(records);
    }

    /**
     * 空记录
     *
     * @param <T> 数据类型
     * @return 结果
     */
    public static <T extends Serializable> PagingVO<T> empty() {
        return new PagingVO<>();
    }

    /**
     * 转换list对象
     *
     * @param mapper 转换器
     * @param <R>    结果类型
     * @return 结果
     */
    public <R extends Serializable> PagingVO<R> map(@NotNull Function<T, R> mapper) {
        if (isEmpty()) {
            return new PagingVO<>();
        }

        List<R> result = records.stream().map(mapper).collect(Collectors.toList());
        return new PagingVO<>(total, result);
    }

    /**
     * 遍历list对象
     *
     * @param consumer 消费者
     */
    public void each(Consumer<T> consumer) {
        if (isEmpty()) {
            return;
        }

        for (T record : records) {
            consumer.accept(record);
        }
    }

    /**
     * 转换为stream
     *
     * @return stream
     */
    public Stream<T> stream() {
        if (this.isEmpty()) {
            return Stream.empty();
        }

        return this.records.stream();
    }

    /**
     * 判断数据是否为空
     *
     * @return 是否为空
     */
    public boolean isEmpty() {
        return isEmpty(records);
    }

    /**
     * 判断是否不为空
     *
     * @return 是否不为空
     */
    public boolean isNotEmpty() {
        return !isEmpty(records);
    }

    /**
     * 大小
     *
     * @return 数据量
     */
    public int size() {
        return isEmpty() ? 0 : records.size();
    }

    @Override
    public String toString() {
        return "Paging{" +
                "total=" + total +
                ", records=" + records +
                '}';
    }

    private boolean isEmpty(Collection<?> collection) {
        return collection == null || collection.isEmpty();
    }
}
