package com.elitescloud.boot.auth.client;

import com.elitescloud.boot.auth.client.common.AuthClientException;
import com.elitescloud.boot.auth.client.config.support.ClientAuthorizationInterceptor;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.datatype.jsr310.deser.LocalDateDeserializer;
import com.fasterxml.jackson.datatype.jsr310.deser.LocalDateTimeDeserializer;
import com.fasterxml.jackson.datatype.jsr310.ser.LocalDateSerializer;
import com.fasterxml.jackson.datatype.jsr310.ser.LocalDateTimeSerializer;
import org.apache.http.conn.ssl.NoopHostnameVerifier;
import org.apache.http.conn.ssl.SSLConnectionSocketFactory;
import org.apache.http.conn.ssl.TrustSelfSignedStrategy;
import org.apache.http.impl.client.HttpClientBuilder;
import org.apache.http.impl.client.HttpClients;
import org.apache.http.ssl.SSLContextBuilder;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.InitializingBean;
import org.springframework.boot.web.client.RestTemplateBuilder;
import org.springframework.boot.web.client.RestTemplateCustomizer;
import org.springframework.core.ParameterizedTypeReference;
import org.springframework.http.HttpEntity;
import org.springframework.http.HttpMethod;
import org.springframework.http.HttpStatus;
import org.springframework.http.ResponseEntity;
import org.springframework.http.client.ClientHttpRequestFactory;
import org.springframework.http.client.HttpComponentsClientHttpRequestFactory;
import org.springframework.http.converter.json.Jackson2ObjectMapperBuilder;
import org.springframework.http.converter.json.MappingJackson2HttpMessageConverter;
import org.springframework.http.converter.xml.MappingJackson2XmlHttpMessageConverter;
import org.springframework.util.Assert;
import org.springframework.web.client.RestClientException;
import org.springframework.web.client.RestTemplate;
import org.springframework.web.util.DefaultUriBuilderFactory;
import org.springframework.web.util.UriTemplateHandler;

import javax.validation.constraints.NotBlank;
import javax.validation.constraints.NotNull;
import java.time.Duration;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;

/**
 * 云梯认证客户端.
 *
 * @author Kaiser（wang shao）
 * @date 2023/8/22
 */
public class CloudtAuthClient implements InitializingBean {
    private static final Logger LOG = LoggerFactory.getLogger(CloudtAuthClient.class);

    private final CloudtAuthProperties properties;
    private final RestTemplate restTemplate;

    public CloudtAuthClient(CloudtAuthProperties properties) {
        this.properties = properties;
        this.restTemplate = this.restTemplateInstance();
    }

    /**
     * 请求接口
     *
     * @param url           接口路径
     * @param method        请求方式
     * @param requestBody   请求体
     * @param responseType  响应类型
     * @param uriParameters url参数
     * @param <T>           响应类型
     * @return 响应结果
     */
    public <T> T exchange(@NotBlank String url, @NotNull HttpMethod method, Object requestBody, @NotNull ParameterizedTypeReference<T> responseType, Object... uriParameters) {
        ResponseEntity<T> response = null;
        try {
            response = restTemplate.exchange(url, method, new HttpEntity<>(requestBody, null),
                    responseType, uriParameters);
        } catch (RestClientException e) {
            LOG.error(url + "调用失败：", e);
            throw new RuntimeException("远程服务器异常", e);
        }

        if (response.getStatusCode() != HttpStatus.OK) {
            LOG.error("调用接口失败：{}, {}", url, response);
            throw new AuthClientException("调用远程接口失败");
        }
        LOG.info("接口{}调用成功", url);

        return response.getBody();
    }

    @Override
    public void afterPropertiesSet() throws Exception {
        Assert.notNull(properties, "认证配置为空");
        Assert.notNull(properties.getServerAddr(), "服务端地址为空，请确认已配置" + CloudtAuthProperties.CONFIG_PREFIX + ".server-addr");
        Assert.notNull(properties.getClientId(), "客户端ID为空，请确认已配置" + CloudtAuthProperties.CONFIG_PREFIX + ".client-id");
        Assert.notNull(properties.getClientSecret(), "客户端密码为空，请确认已配置" + CloudtAuthProperties.CONFIG_PREFIX + ".client-secret");
    }

    private RestTemplate restTemplateInstance() {
        return new RestTemplateBuilder()
                .rootUri(properties.getServerAddr())
                .requestFactory(this::getClientHttpRequestFactory)
                .uriTemplateHandler(this.uriTemplateHandler())
                .customizers(restTemplateCustomizer())
                .additionalInterceptors(new ClientAuthorizationInterceptor())
                .build();
    }

    private ClientHttpRequestFactory getClientHttpRequestFactory() {
        SSLConnectionSocketFactory sslConnectionSocketFactory = null;

        try {
            SSLContextBuilder sslContextBuilder = new SSLContextBuilder();
            sslContextBuilder.loadTrustMaterial(null, new TrustSelfSignedStrategy());
            sslConnectionSocketFactory = new SSLConnectionSocketFactory(sslContextBuilder.build(), NoopHostnameVerifier.INSTANCE);
        } catch (Exception e) {
            throw new IllegalStateException("客户端初始化失败", e);
        }

        HttpClientBuilder httpClientBuilder = HttpClients.custom();
        httpClientBuilder.setSSLSocketFactory(sslConnectionSocketFactory);

        var requestFactory = new HttpComponentsClientHttpRequestFactory();
        requestFactory.setHttpClient(httpClientBuilder.build());
        requestFactory.setConnectTimeout((int) Duration.ofSeconds(10).toMillis());
        requestFactory.setReadTimeout((int) Duration.ofSeconds(60).toMillis());
        return requestFactory;
    }

    private UriTemplateHandler uriTemplateHandler() {
        DefaultUriBuilderFactory handler = new DefaultUriBuilderFactory();
        handler.setEncodingMode(DefaultUriBuilderFactory.EncodingMode.TEMPLATE_AND_VALUES);
        return handler;
    }

    private RestTemplateCustomizer restTemplateCustomizer() {
        ObjectMapper objectMapper = this.objectMapperInstance();
        return template -> {
            MappingJackson2HttpMessageConverter jsonConvert = null;

            var xmlConvertIndex = -1;
            int index = 0;
            for (var convert : template.getMessageConverters()) {
                if (convert instanceof MappingJackson2XmlHttpMessageConverter) {
                    xmlConvertIndex = index;
                }
                if (convert instanceof MappingJackson2HttpMessageConverter) {
                    jsonConvert = (MappingJackson2HttpMessageConverter) convert;
                }
                index++;
            }
            if (jsonConvert == null) {
                jsonConvert = new MappingJackson2HttpMessageConverter(objectMapper);
                template.getMessageConverters().add(jsonConvert);
            } else {
                jsonConvert.setObjectMapper(objectMapper);
            }

            if (xmlConvertIndex >= 0) {
                template.getMessageConverters().remove(jsonConvert);
                template.getMessageConverters().add(xmlConvertIndex - 1, jsonConvert);
            }
        };
    }

    private ObjectMapper objectMapperInstance() {
        return Jackson2ObjectMapperBuilder.json()
                .serializerByType(LocalDateTime.class, new LocalDateTimeSerializer(DateTimeFormatter.ofPattern("yyyy-MM-dd HH:mm:ss")))
                .serializerByType(LocalDate.class, new LocalDateSerializer(DateTimeFormatter.ofPattern("yyyy-MM-dd")))
                .deserializerByType(LocalDateTime.class, new LocalDateTimeDeserializer(DateTimeFormatter.ofPattern("yyyy-MM-dd HH:mm:ss")))
                .deserializerByType(LocalDate.class, new LocalDateDeserializer(DateTimeFormatter.ofPattern("yyyy-MM-dd")))
                .simpleDateFormat("yyyy-MM-dd HH:mm:ss")
                .failOnUnknownProperties(false)
                .failOnEmptyBeans(false)
                .build();
    }
}
