package com.elitescloud.boot.auth.factory.common;

import cn.hutool.core.lang.Assert;
import cn.hutool.core.text.CharSequenceUtil;
import com.elitescloud.boot.SpringContextHolder;
import com.elitescloud.boot.common.param.CodeNameParam;
import com.elitescloud.boot.redis.util.RedisUtils;
import com.elitescloud.boot.util.JSONUtil;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.validation.constraints.NotBlank;
import javax.validation.constraints.NotNull;
import java.util.ArrayList;
import java.util.List;
import java.util.ServiceLoader;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

/**
 * 认证客户端.
 *
 * @author Kaiser（wang shao）
 * @date 2025/5/26 周一
 */
public class AuthClientHolder {
    private static final Logger logger = LoggerFactory.getLogger(AuthClientHolder.class);

    private static List<AuthClientProvider<?>> authClientProviders;

    /**
     * 获取所有的认证客户端类型.
     *
     * @return
     */
    public static List<CodeNameParam> getAuthClientTypes() {
        return getAuthClientProviders().stream()
                .map(t -> new CodeNameParam(t.code(), t.name()))
                .collect(Collectors.toList());
    }

    /**
     * 认证
     *
     * @param authClientType
     * @param authPropertyJson
     * @return
     */
    public static AuthResult authenticate(@NotBlank String authClientType, @NotBlank String authPropertyJson) {
        logger.info("认证类型：{}", authClientType);
        AuthClientProvider authClientProvider = getAuthClientProvider(authClientType);
        Assert.notNull(authClientProvider, "认证类型不存在");

        BaseAuthProperty property = (BaseAuthProperty) JSONUtil.json2Obj(authPropertyJson, authClientProvider.propertyClass(), true, () -> "认证参数格式有误");
        return execAuth(authClientProvider, property);
    }

    /**
     * 认证
     *
     * @param authClientType
     * @param authProperty
     * @return
     */
    public static AuthResult authenticate(@NotBlank String authClientType, @NotNull BaseAuthProperty authProperty) {
        logger.info("认证类型：{}", authClientType);
        AuthClientProvider authClientProvider = getAuthClientProvider(authClientType);
        Assert.notNull(authClientProvider, "认证类型不存在");
        Assert.isAssignable(authClientProvider.propertyClass(), authProperty.getClass(), "认证参数格式有误");

        return execAuth(authClientProvider, authProperty);
    }

    private static AuthResult execAuth(AuthClientProvider authClientProvider,  BaseAuthProperty authProperty) {
        // 优先从缓存获取
        RedisUtils redisUtils = null;
        if (CharSequenceUtil.isNotBlank(authProperty.getPropKey())) {
            redisUtils = SpringContextHolder.getBean(RedisUtils.class);
            AuthResult authResult = (AuthResult) redisUtils.get(authProperty.getPropKey());
            if (authResult != null) {
                return authResult;
            }
        }

        AuthResult authResult =  authClientProvider.authenticate(authProperty);
        // 添加缓存
        if (authResult != null && authResult.getTtl() != null && CharSequenceUtil.isNotBlank(authProperty.getPropKey())) {
            redisUtils.set(authProperty.getPropKey(), authResult, authResult.getTtl().toSeconds(), TimeUnit.SECONDS);
        }
        return authResult;
    }

    private static AuthClientProvider<?> getAuthClientProvider(String authClientType) {
        for (AuthClientProvider<?> authClientProvider : getAuthClientProviders()) {
            if (authClientProvider.code().equals(authClientType)) {
                return authClientProvider;
            }
        }
        return null;
    }

    private static List<AuthClientProvider<?>> getAuthClientProviders() {
        if (authClientProviders == null) {
            authClientProviders = new ArrayList<>();
            ServiceLoader.load(AuthClientProvider.class).forEach(authClientProviders::add);
        }
        return authClientProviders;
    }
}
