package com.elitescloud.boot.auth.factory.provider;

import cn.hutool.core.lang.Assert;
import com.elitescloud.boot.auth.factory.common.AuthClientProvider;
import com.elitescloud.boot.auth.factory.common.AuthResult;
import com.elitescloud.boot.auth.factory.common.constant.TokenPositionEnum;
import com.elitescloud.boot.auth.factory.provider.properties.OAuth2AuthProperty;
import com.elitescloud.boot.util.RestTemplateFactory;
import org.springframework.core.ParameterizedTypeReference;
import org.springframework.http.HttpEntity;
import org.springframework.http.HttpMethod;
import org.springframework.http.ResponseEntity;
import org.springframework.util.LinkedMultiValueMap;
import org.springframework.util.MultiValueMap;
import org.springframework.web.client.RestTemplate;

import java.time.Duration;
import java.util.Collections;
import java.util.Map;

/**
 * OAuth2的认证.
 *
 * @author Kaiser（wang shao）
 * @date 2025/5/26 周一
 */
public class OAuth2AuthClientProvider implements AuthClientProvider<OAuth2AuthProperty> {

    private final RestTemplate restTemplate = RestTemplateFactory.instance();

    @Override
    public String code() {
        return "OAuth2";
    }

    @Override
    public String name() {
        return "OAuth2";
    }

    @Override
    public Class<OAuth2AuthProperty> propertyClass() {
        return OAuth2AuthProperty.class;
    }

    @Override
    public AuthResult authenticate(OAuth2AuthProperty property) {
        Assert.notBlank(property.getTokenUrl(), "tokenUrl为空");

        // 组织请求参数
        MultiValueMap<String, Object> postParam = new LinkedMultiValueMap<>(8);
        postParam.add("grant_type", "client_credentials");
        postParam.add("client_id", property.getClientId());
        postParam.add("client_secret", property.getClientSecret());

        ResponseEntity<Map<String, Object>> authResult = null;
        try {
            authResult = restTemplate.exchange(property.getTokenUrl(), HttpMethod.POST, new HttpEntity<>(postParam, null),
                    new ParameterizedTypeReference<>() {});
        } catch (Exception e) {
            throw new RuntimeException("获取认证Token异常", e);
        }
        if (authResult.getBody() == null) {
            throw new RuntimeException("获取认证Token失败：" + authResult.getStatusCode());
        }
        Map<String, Object> tokenResult = authResult.getBody();

        AuthResult result = new AuthResult();
        result.setTokenPosition(TokenPositionEnum.HEADER);
        result.setTokenName("Authorization");
        result.setTokenValue(tokenResult.getOrDefault("token_type", "Bearer").toString() + " " + tokenResult.getOrDefault("access_token", "").toString());
        int ttl = Integer.parseInt(tokenResult.getOrDefault("expires_in", 0).toString());
        result.setTtl(Duration.ofSeconds(ttl));
        result.setExtra(Collections.emptyMap());

        return result;
    }
}
