package com.elitescloud.boot.auth.provider.config.servlet.oauth2.handler;

import cn.hutool.core.text.CharSequenceUtil;
import com.elitescloud.boot.auth.client.common.OAuth2ClientConstant;
import com.elitescloud.boot.auth.client.config.AuthorizationProperties;
import com.elitescloud.boot.auth.provider.config.servlet.oauth2.OAuth2AuthorizationCodeRequestCache;
import com.elitescloud.boot.auth.provider.security.handler.LogoutRedirectHandler;
import com.elitescloud.boot.auth.resolver.UniqueRequestResolver;
import org.springframework.security.oauth2.core.endpoint.OAuth2ParameterNames;
import org.springframework.security.oauth2.server.authorization.client.RegisteredClientRepository;
import org.springframework.security.web.savedrequest.RequestCache;
import org.springframework.security.web.savedrequest.SavedRequest;
import org.springframework.util.StringUtils;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

/**
 * OAuth2登出后重定向至登录页.
 *
 * @author Kaiser（wang shao）
 * @date 3/2/2023
 */
public class OAuth2ServerLogoutRedirectHandler implements LogoutRedirectHandler {

    private final RegisteredClientRepository registeredClientRepository;
    private final AuthorizationProperties authorizationProperties;
    private final OAuth2AuthorizationCodeRequestCache authorizationCodeRequestCache;
    private UniqueRequestResolver uniqueRequestResolver = null;
    private RequestCache requestCache;

    public OAuth2ServerLogoutRedirectHandler(RegisteredClientRepository registeredClientRepository, AuthorizationProperties authorizationProperties, OAuth2AuthorizationCodeRequestCache authorizationCodeRequestCache) {
        this.registeredClientRepository = registeredClientRepository;
        this.authorizationProperties = authorizationProperties;
        this.authorizationCodeRequestCache = authorizationCodeRequestCache;
    }

    @Override
    public String determineUrlToUseForThisRequest(HttpServletRequest request, HttpServletResponse response) {
        // 获取认证请求中的重定向url
        SavedRequest savedRequest = null;
        if (requestCache != null) {
            savedRequest = requestCache.getRequest(request, response);
        }
        String clientId = null;
        if (savedRequest != null) {
            String redirectUri = this.obtainRequestParam(savedRequest, OAuth2ParameterNames.REDIRECT_URI);
            if (StringUtils.hasText(redirectUri)) {
                return redirectUri;
            }
            clientId = this.obtainRequestParam(savedRequest, OAuth2ParameterNames.CLIENT_ID);
        }

        // 获取所属客户端
        if (CharSequenceUtil.isBlank(clientId)) {
            clientId = this.obtainClientId(request);
        }
        if (CharSequenceUtil.isBlank(clientId)) {
            return null;
        }

        // 获取登录页
        var client = registeredClientRepository.findByClientId(clientId);
        if (client == null) {
            return null;
        }
        String loginUrl = client.getClientSettings().getSetting(OAuth2ClientConstant.SETTING_LOGIN_URL);
        if (StringUtils.hasText(loginUrl)) {
            return loginUrl;
        }

        // 返回默认登录页
        return authorizationProperties.getLoginPage();
    }

    private String obtainRequestParam(SavedRequest request, String param) {
        String[] values = request.getParameterValues(param);
        if (values != null && values.length > 0) {
            return values[0];
        }
        return null;
    }

    private String obtainClientId(HttpServletRequest request) {
        if (uniqueRequestResolver != null) {
            String reqId = uniqueRequestResolver.analyze(request);
            if (StringUtils.hasText(reqId)) {
                var authenticationToken = authorizationCodeRequestCache.getAuthenticationToken(reqId);
                if (authenticationToken != null) {
                    return authenticationToken.getClientId();
                }
            }
        }

        return null;
    }

    public void setUniqueRequestResolver(UniqueRequestResolver uniqueRequestResolver) {
        this.uniqueRequestResolver = uniqueRequestResolver;
    }

    public void setRequestCache(RequestCache requestCache) {
        this.requestCache = requestCache;
    }
}
